/*
 * Copyright 1987 by MIT Student Information Processing Board
 *
 * For copyright info, see mit-sipb-copyright.h.
 */
#include "ss_internal.h"
#include "mit-sipb-copyright.h"

extern void perror();
static char no_malloc[] = "Can't malloc";

/*
 * parse(line_ptr, argc_ptr)
 *
 * Function:
 *	Parses line, dividing at whitespace, into tokens, returns
 *	the "argc" and "argv" values.
 * Arguments:
 *	line_ptr (char *)
 *		Pointer to text string to be parsed.
 *	argc_ptr (int *)
 *		Where to put the "argc" (number of tokens) value.
 * Returns:
 *	argv (char **)
 *		Series of pointers to parsed tokens.
 */

static char **parse (line_ptr, argc_ptr)
	char *line_ptr;
	int *argc_ptr;
{
	register char **argv;
	register int argc, parse_mode;

	argv = (char **) malloc (sizeof(char *));
	if (argv == (char **)NULL) {
		perror(no_malloc);
		return(argv);
	}
	*argv = (char *)NULL;

	argc = 0;

	parse_mode = 0;		/* flushing whitespace */

	while (*line_ptr != '\0') {
		if (*line_ptr == ' ' || *line_ptr == '\t') {
			/* white space */
			if (parse_mode != 0) {
				*line_ptr = '\0'; /* mark end of token */
				parse_mode = 0;	/* reset mode to whitespace */
			}
		}
		else {
			/* not whitespace */
			if (parse_mode != 1) {
				argv = (char **) realloc((char *)argv,
							 (unsigned)
							 ((argc + 2)*
							  sizeof(char *)));
				if (argv == (char **)NULL) {
					perror(no_malloc);
					exit(1);
				}
				argv[argc] = line_ptr;
				argv[argc+1] = (char *)NULL;
				argc++;
				parse_mode = 1;
			}
		}
		line_ptr++;
	}
	*argc_ptr = argc;
	return(argv);
}


/*
 * check_command(cmd, string)
 *
 * Function:
 *	Checks to see if the command name "string" is among those in
 *	the specified subsystem request table entry.
 * Arguments:
 *	cmd (ss_request_entry *)
 *		pointer to the subsystem request entry
 *	string (char *)
 *		command string to be searched for
 * Returns:
 *	(int)
 *		zero if found, -1 otherwise
 * Notes:
 */

static int check_command(cmd, string)
	ss_request_entry *cmd;
	char *string;
{
	char **name;

	for (name = cmd->command_names; *name; name++)
		if (!strcmp(*name, string))
			return(0);
	return(-1);
}


/*
 * get_request(tbl, idx)
 *
 * Function:
 *	Gets the idx'th request from the request table pointed to
 *	by tbl.
 * Arguments:
 *	tbl (ss_request_entry *)
 *		pointer to request table
 *	idx (int)
 *		index into table
 * Returns:
 *	(ss_request_entry *)
 *		pointer to request table entry
 * Notes:
 *	Has been replaced by a macro.
 */

#define	get_request(tbl,idx)	((tbl) -> requests + (idx))

/*
 * check_request_table(rqtbl, argc, argv, sci_idx)
 *
 * Function:
 *	If the command string in argv[0] is in the request table, execute
 *	the commands and return error code 0.  Otherwise, return error
 *	code ss_et_command_not_found.
 * Arguments:
 *	rqtbl (ss_request_table *)
 *		pointer to request table
 *	argc (int)
 *		number of elements in argv[]
 *	argv (char *[])
 *		argument string array
 *	sci_idx (int)
 *		ss-internal index for subsystem control info structure
 * Returns:
 *	(int)
 *		zero if command found, ss_et_command_not_found otherwise
 * Notes:
 */

static int check_request_table (rqtbl, argc, argv, sci_idx)
	register ss_request_table *rqtbl;
	int argc;
	char *argv[];
	int sci_idx;
{
	register ss_request_entry *request;
	register ss_data *info;
	int i;

	info = ss_info(sci_idx);
	info->argc = argc;
	info->argv = argv;
	for (i = 0; (request = get_request(rqtbl, i))->command_names; i++) {
		if (check_command(request, argv[0]) == 0) {
			info->current_request = request->command_names[0];
			(request->function)(argc, argv, sci_idx);
			info->current_request = (char *)NULL;
			return(0);
		}
	}
	return(SS_ET_COMMAND_NOT_FOUND);
}


/*
 * really_execute_command(sci_idx, argc, argv)
 *
 * Function:
 *	Fills in the argc, argv values in the subsystem entry and 
 *	call the appropriate routine.
 * Arguments:
 *	sci_idx (int)
 *		ss-internal index for subsystem control info structure
 *	argc (int)
 *		number of arguments in argument list
 *	argv (char *[])
 *		parsed argument list
 *
 * Returns:
 *	(int)
 *		Zero if successful, ss_et_command_not_found otherwise.
 * Notes:
 */

static int
really_execute_command(sci_idx, argc, argv)
	int sci_idx;
	int argc;
	char *argv[];
{
	register ss_request_table **rqtbl;

	for (rqtbl = ss_info(sci_idx)->rqt_tables; *rqtbl; rqtbl++) {
		if (check_request_table (*rqtbl, argc, argv, sci_idx) == 0)
			return(0);
	}
	return(SS_ET_COMMAND_NOT_FOUND);
}	

/*
 * ss_execute_command(sci_idx, argv)
 *
 * Function:
 *	Executes a parsed command list within the subsystem.
 * Arguments:
 *	sci_idx (int)
 *		ss-internal index for subsystem control info structure
 *	argv (char *[])
 *		parsed argument list
 * Returns:
 *	(int)
 *		Zero if successful, ss_et_command_not_found otherwise.
 * Notes:
 */

ss_execute_command(sci_idx, argv)
	int sci_idx;
	register char *argv[];
{
	register char **argp;

	for (argp = argv; *argp; argp++)
		;
	return(really_execute_command(sci_idx, argp - argv, argv));
}

/*
 * ss_execute_line(sci_idx, line_ptr, code_ptr)
 *
 * Function:
 *	Parses and executes a command line within a subsystem.
 * Arguments:
 *	sci_idx (int)
 *		ss-internal index for subsystem control info structure
 *	line_ptr (char *)
 *		Pointer to command line to be parsed.
 * Returns:
 *	(void)
 * Notes:
 */

void
ss_execute_line (sci_idx, line_ptr, code_ptr)
	int sci_idx;
	char *line_ptr;
	register int *code_ptr;
{
	register char **argv;
	int argc, i;

	/* flush leading whitespace */
	while (line_ptr[0] == ' ' || line_ptr[0] == '\t')
	     line_ptr++;

	/* check if should be sent to shell */
	i = 0;
	if (line_ptr[0] == '.' && line_ptr[1] == '.')
	     i = 2;
	else if (line_ptr[0] == '!')
	     i = 1;
	if (i) {
	     line_ptr += i;
	     if (ss_info(sci_idx)->escape_disabled)
		  ss_perror(sci_idx, SS_ET_ESCAPE_DISABLED, (char *)NULL);
	     else
		  system(line_ptr);
	     return;
	}

	/* parse it */
	argv = parse(line_ptr, &argc);
	if (argc == 0) {
		*code_ptr = 0;
		return;
	}

	/* look it up in the request tables, execute if found */
	*code_ptr = really_execute_command(sci_idx, argc, argv);
	return;
}
