/*	$NetBSD: rlogin.c,v 1.7 1995/05/17 13:35:50 mycroft Exp $	*/

/*
 * Copyright (c) 1983, 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * rlogin - remote login
 */
#include "bsd_locl.h"

RCSID("$Id: rlogin.c,v 1.39 1996/08/17 14:26:47 assar Exp $");

#include <termios.h>

#if defined(NeXT)               /* NEXTSTEP 3.3 */
#include <sys/ioctl.h>
#endif

#if defined(__osf__)		/* OSF/1 */
#include <sys/ioctl.h>
#endif

#if defined(__hpux)		/* HP-UX */
#include <sys/proc.h>
#include <sys/tty.h>
#endif

#if defined(_AIX)		/* AIX */
#include <sys/ioctl.h>
#endif

#if SunOS == 4
#ifndef _POSIX_VDISABLE
#define _POSIX_VDISABLE 0
#endif /* _POSIX_VDISABLE */
#include <sys/ttold.h>
#endif

#if SunOS == 5
#include <sys/tty.h>
#include <sys/ptyvar.h>
#endif

CREDENTIALS cred;
Key_schedule schedule;
int use_kerberos = 1, doencrypt;
char dst_realm_buf[REALM_SZ], *dest_realm = NULL;

#ifndef TIOCPKT_WINDOW
#define	TIOCPKT_WINDOW	0x80
#endif

/* concession to Sun */
#ifndef SIGUSR1
#define	SIGUSR1	30
#endif

#ifndef CCEQ
#define c2uc(x) ((unsigned char) x)
#define CCEQ__(val, c)	(c == val ? val != c2uc(_POSIX_VDISABLE) : 0)
#define CCEQ(val, c) CCEQ__(c2uc(val), c2uc(c))
#endif

int eight, rem;
struct termios deftty;

int noescape;
char escapechar = '~';

#ifdef OLDSUN
struct winsize {
	unsigned short ws_row, ws_col;
	unsigned short ws_xpixel, ws_ypixel;
};

/*
 * The following routine provides compatibility (such as it is) between older
 * Suns and others.  Suns have only a `ttysize', so we convert it to a winsize.
 */
int
get_window_size(fd, wp)
	int fd;
	struct winsize *wp;
{
	struct ttysize ts;
	int error;

	if ((error = ioctl(0, TIOCGSIZE, &ts)) != 0)
		return (error);
	wp->ws_row = ts.ts_lines;
	wp->ws_col = ts.ts_cols;
	wp->ws_xpixel = 0;
	wp->ws_ypixel = 0;
	return (0);
}
#else /* !OLDSUN */
#define	get_window_size(fd, wp)	ioctl(fd, TIOCGWINSZ, wp)
#endif

struct	winsize winsize;

int parent, rcvcnt;
char rcvbuf[8 * 1024];

static void
warning(const char *fmt, const char *arg)
{
    char *rstar_no_warn = getenv("RSTAR_NO_WARN");
    if (rstar_no_warn == 0)
	rstar_no_warn = "";
    if (strncmp(rstar_no_warn, "yes", 3) != 0) {
	(void)fprintf(stderr, "rlogin: warning, using standard rlogin: ");
	fprintf(stderr, fmt, arg);
	(void)fprintf(stderr, ".\n");
    }
}

#ifdef OLDSUN
int		get_window_size __P((int, struct winsize *));
#endif

int child;

static void
echo(register char c)
{
	register char *p;
	char buf[8];

	p = buf;
	c &= 0177;
	*p++ = escapechar;
	if (c < ' ') {
		*p++ = '^';
		*p++ = c + '@';
	} else if (c == 0177) {
		*p++ = '^';
		*p++ = '?';
	} else
		*p++ = c;
	*p++ = '\r';
	*p++ = '\n';
	(void)write(STDOUT_FILENO, buf, p - buf);
}

static void
mode(int f)
{
	struct termios tty;

	switch (f) {
	case 0:
		(void)tcsetattr(0, TCSANOW, &deftty);
		break;
	case 1:
		(void)tcgetattr(0, &deftty);
		tty = deftty;
		/* This is loosely derived from sys/compat/tty_compat.c. */
		tty.c_lflag &= ~(ECHO|ICANON|ISIG|IEXTEN);
		tty.c_iflag &= ~ICRNL;
		tty.c_oflag &= ~OPOST;
		tty.c_cc[VMIN] = 1;
		tty.c_cc[VTIME] = 0;
		if (eight) {
			tty.c_iflag &= IXOFF;
			tty.c_cflag &= ~(CSIZE|PARENB);
			tty.c_cflag |= CS8;
		}
		(void)tcsetattr(0, TCSANOW, &tty);
		break;
	default:
		return;
	}
}

static void
done(int status)
{
	int w, wstatus;

	mode(0);
	if (child > 0) {
		/* make sure catch_child does not snap it up */
		(void)signal(SIGCHLD, SIG_DFL);
		if (kill(child, SIGKILL) >= 0)
			while ((w = wait(&wstatus)) > 0 && w != child);
	}
	exit(status);
}

static
RETSIGTYPE
catch_child(int foo)
{
	int status;
	int pid;

	for (;;) {
		pid = waitpid(-1, &status, WNOHANG|WUNTRACED);
		if (pid == 0)
			return;
		/* if the child (reader) dies, just quit */
		if (pid < 0 || (pid == child && !WIFSTOPPED(status)))
			done(WTERMSIG(status) | WEXITSTATUS(status));
	}
	/* NOTREACHED */
}

static void
msg(char *str)
{
	(void)fprintf(stderr, "rlogin: %s\r\n", str);
}

/*
 * There is a race in the SunOS5 rlogind. If the slave end has not yet
 * been opened by the child when setting tty size the size is reset to
 * zero when the child opens it. Therefore we send the window update
 * twice.
 */

static int tty_kludge = 1;

/* Return the number of OOB bytes processed. */
static
int
oob_real()
{
	struct termios tty;
	int atmark, n, out, rcvd;
	char waste[BUFSIZ], mark;

	out = O_RDWR;
	rcvd = 0;
	if (recv(rem, &mark, 1, MSG_OOB) < 0) {
		return -1;
	}
	if (mark & TIOCPKT_WINDOW) {
		/* Let server know about window size changes */
		(void)kill(parent, SIGUSR1);
	} else if (tty_kludge) {
		/* Let server know about window size changes */
		(void)kill(parent, SIGUSR1);
		tty_kludge = 0;
	}
	if (!eight && (mark & TIOCPKT_NOSTOP)) {
		(void)tcgetattr(0, &tty);
		tty.c_iflag &= ~IXON;
		(void)tcsetattr(0, TCSANOW, &tty);
	}
	if (!eight && (mark & TIOCPKT_DOSTOP)) {
		(void)tcgetattr(0, &tty);
		tty.c_iflag |= (deftty.c_iflag & IXON);
		(void)tcsetattr(0, TCSANOW, &tty);
	}
	if (mark & TIOCPKT_FLUSHWRITE) {
#ifdef TCOFLUSH
		(void) tcflush(1, TCOFLUSH);
#else
		(void)ioctl(1, TIOCFLUSH, (char *)&out);
#endif
		for (;;) {
			if (ioctl(rem, SIOCATMARK, &atmark) < 0) {
				(void)fprintf(stderr, "rlogin: ioctl: %s.\n",
				    strerror(errno));
				break;
			}
			if (atmark)
				break;
			n = read(rem, waste, sizeof (waste));
			if (n <= 0)
				break;
		}
		/*
		 * Don't want any pending data to be output, so clear the recv
		 * buffer.  If we were hanging on a write when interrupted,
		 * don't want it to restart.  If we were reading, restart
		 * anyway.
		 */
		rcvcnt = 0;
	}

	/* oob does not do FLUSHREAD (alas!) */
	return 1;
}

/* reader: read from remote: line -> 1 */
static int
reader(void)
{
	int n, remaining;
	char *bufp;

	bufp = rcvbuf;
	for (;;) {
	        fd_set readfds, exceptfds;
		while ((remaining = rcvcnt - (bufp - rcvbuf)) > 0) {
			n = write(STDOUT_FILENO, bufp, remaining);
			if (n < 0) {
				if (errno != EINTR)
					return (-1);
				continue;
			}
			bufp += n;
		}
		bufp = rcvbuf;
		rcvcnt = 0;

		FD_ZERO (&readfds);
		FD_SET (rem, &readfds);
		FD_ZERO (&exceptfds);
		FD_SET (rem, &exceptfds);
		if (select(rem+1, &readfds, 0, &exceptfds, 0) == -1)
		    {
			if (errno == EINTR)
			    continue; /* Got signal */
			else
			    {
				msg("select failed mysteriously\n");
				exit(1);
			    }
		    }

		if (!FD_ISSET(rem, &exceptfds) && !FD_ISSET(rem, &readfds))
		  {
		    fprintf(stderr, "select: nothing to read?\n");
		    continue;
		  }

		if (FD_ISSET(rem, &exceptfds))
		  if (oob_real() >= 1)
		    continue;	/* First check if there is more OOB data. */

		if (!FD_ISSET(rem, &readfds))
		    continue;	/* Nothing to read. */

		if (doencrypt)
#ifdef NOENCRYPTION
			rcvcnt = read(rem, rcvbuf, sizeof (rcvbuf));
#else
			rcvcnt = des_enc_read(rem, rcvbuf, sizeof(rcvbuf), schedule, &cred.session);
#endif
		else
			rcvcnt = read(rem, rcvbuf, sizeof (rcvbuf));
		if (rcvcnt == 0)
			return (0);
		if (rcvcnt < 0) {
			if (errno == EINTR)
				continue;
			(void)fprintf(stderr, "rlogin: read: %s.\n",
			    strerror(errno));
			return (-1);
		}
	}
}

/*
 * Send the window size to the server via the magic escape
 */
static void
sendwindow(void)
{
	struct winsize *wp;
	char obuf[4 + sizeof (struct winsize)];

	wp = (struct winsize *)(obuf+4);
	obuf[0] = 0377;
	obuf[1] = 0377;
	obuf[2] = 's';
	obuf[3] = 's';
	wp->ws_row = htons(winsize.ws_row);
	wp->ws_col = htons(winsize.ws_col);
	wp->ws_xpixel = htons(winsize.ws_xpixel);
	wp->ws_ypixel = htons(winsize.ws_ypixel);

	if(doencrypt)
#ifdef NOENCRYPTION
		(void)write(rem, obuf, sizeof(obuf));
#else
		(void)des_enc_write(rem, obuf, sizeof(obuf), schedule, &cred.session);
#endif
	else
		(void)write(rem, obuf, sizeof(obuf));
}

static
RETSIGTYPE
sigwinch(int foo)
{
	struct winsize ws;

	if (get_window_size(0, &ws) == 0 &&
	    memcmp(&ws, &winsize, sizeof(ws))) {
		winsize = ws;
		sendwindow();
	}
}

static void
stop(int all)
{
	mode(0);
	(void)signal(SIGCHLD, SIG_IGN);
	(void)kill(all ? 0 : getpid(), SIGTSTP);
	(void)signal(SIGCHLD, catch_child);
	mode(1);
	kill(SIGWINCH, getpid()); /* check for size changes, if caught */
}

/*
 * writer: write to remote: 0 -> line.
 * ~.				terminate
 * ~^Z				suspend rlogin process.
 * ~<delayed-suspend char>	suspend rlogin process, but leave reader alone.
 */
static void
writer(void)
{
	register int bol, local, n;
	char c;

	bol = 1;			/* beginning of line */
	local = 0;
	for (;;) {
		n = read(STDIN_FILENO, &c, 1);
		if (n <= 0) {
			if (n < 0 && errno == EINTR)
				continue;
			break;
		}
		/*
		 * If we're at the beginning of the line and recognize a
		 * command character, then we echo locally.  Otherwise,
		 * characters are echo'd remotely.  If the command character
		 * is doubled, this acts as a force and local echo is
		 * suppressed.
		 */
		if (bol) {
			bol = 0;
			if (!noescape && c == escapechar) {
				local = 1;
				continue;
			}
		} else if (local) {
			local = 0;
			if (c == '.' || CCEQ(deftty.c_cc[VEOF], c)) {
				echo(c);
				break;
			}
			if (CCEQ(deftty.c_cc[VSUSP], c)) {
				bol = 1;
				echo(c);
				stop(1);
				continue;
			}
#ifdef VDSUSP
			/* Is VDSUSP called something else on Linux?
			 * Perhaps VDELAY is a better thing? */		
			if (CCEQ(deftty.c_cc[VDSUSP], c)) {
				bol = 1;
				echo(c);
				stop(0);
				continue;
			}
#endif /* VDSUSP */
			if (c != escapechar)
				if (doencrypt)
#ifdef NOENCRYPTION
					(void)write(rem, &escapechar, 1);
#else
					(void)des_enc_write(rem, &escapechar, 1, schedule, &cred.session);
#endif
				else
					(void)write(rem, &escapechar, 1);
		}

		if (doencrypt) {
#ifdef NOENCRYPTION
			if (write(rem, &c, 1) == 0) {
#else
			if (des_enc_write(rem, &c, 1, schedule, &cred.session) == 0) {
#endif
				msg("line gone");
				break;
			}
		} else
			if (write(rem, &c, 1) == 0) {
				msg("line gone");
				break;
			}
		bol = CCEQ(deftty.c_cc[VKILL], c) ||
		    CCEQ(deftty.c_cc[VEOF], c) ||
		    CCEQ(deftty.c_cc[VINTR], c) ||
		    CCEQ(deftty.c_cc[VSUSP], c) ||
		    c == '\r' || c == '\n';
	}
}

static
RETSIGTYPE
lostpeer(int foo)
{
	(void)signal(SIGPIPE, SIG_IGN);
	msg("\aconnection closed.");
	done(1);
}

/*
 * This is called in the parent when the reader process gets the
 * out-of-band (urgent) request to turn on the window-changing
 * protocol. It is signalled from the child(reader).
 */
static
RETSIGTYPE
sigusr1(int foo)
{
        /*
	 * Now we now daemon supports winsize hack,
	 */
	sendwindow();
	(void)signal(SIGWINCH, sigwinch); /* so we start to support it */
}

static void
doit(void)
{
	(void)signal(SIGINT, SIG_IGN);
	(void)signal(SIGHUP, SIG_IGN);
	(void)signal(SIGQUIT, SIG_IGN);

	(void)signal(SIGCHLD, catch_child);

	/*
	 * Child sends parent this signal for window size hack.
	 */
	(void)signal(SIGUSR1, sigusr1);

	(void)signal(SIGPIPE, lostpeer);

	mode(1);
	parent = getpid();
	child = fork();
	if (child == -1) {
		(void)fprintf(stderr, "rlogin: fork: %s.\n", strerror(errno));
		done(1);
	}
	if (child == 0) {
	        (void)signal(SIGCHLD, SIG_IGN);
	        (void)signal(SIGTTOU, SIG_IGN);
		if (reader() == 0) {
			msg("connection closed.");
			exit(0);
		}
		sleep(1);
		msg("\aconnection closed.");
		exit(1);
	}

	writer();
	msg("closed connection.");
	done(0);
}

static void
usage(void)
{
	(void)fprintf(stderr,
	    "usage: rlogin [ -%s]%s[-e char] [ -l username ] host\n",
	    "8DEKLdx", " [-k realm] ");
	exit(1);
}

static u_int
getescape(register char *p)
{
	long val;
	int len;

	if ((len = strlen(p)) == 1)	/* use any single char, including '\' */
		return ((u_int)*p);
					/* otherwise, \nnn */
	if (*p == '\\' && len >= 2 && len <= 4) {
		val = strtol(++p, NULL, 8);
		for (;;) {
			if (!*++p)
				return ((u_int)val);
			if (*p < '0' || *p > '8')
				break;
		}
	}
	msg("illegal option value -- e");
	usage();
	return 0;
}

static int get_login_port(int kerberos, int encryption)
{
  struct servent *sp;
  char *service="login";
  int port=htons(513);

  if(kerberos && encryption){
    service="eklogin";
    port=htons(2105);
  }
  
  if(kerberos && !encryption){
    service="klogin";
    port=htons(543);
  }
  
  sp=getservbyname(service, "tcp");
  if(sp == NULL){
    fprintf(stderr, "rlogin: can't get entry for %s service, using default port %d.\n",
	    service, ntohs(port));
  }else{
    port=sp->s_port;
  }
  return port;
}

int
main(int argc, char **argv)
{
	extern char *optarg;
	extern int optind;
	struct passwd *pw;
	int sv_port;
	struct termios tty;
	int argoff, ch, dflag, Dflag, one, uid;
	char *host, *p, *user, term[1024];

	argoff = dflag = Dflag = 0;
	one = 1;
	host = user = NULL;

	if ((p = strrchr(argv[0], '/')))
		++p;
	else
		p = argv[0];

	if (strcmp(p, "rlogin"))
		host = p;

	/* handle "rlogin host flags" */
	if (!host && argc > 2 && argv[1][0] != '-') {
		host = argv[1];
		argoff = 1;
	}

#define	OPTIONS	"8DEKLde:k:l:x"
	while ((ch = getopt(argc - argoff, argv + argoff, OPTIONS)) != EOF)
		switch(ch) {
		case '8':
			eight = 1;
			break;
		case 'D':
		        Dflag = 1;
			break;
		case 'E':
			noescape = 1;
			break;
		case 'K':
			use_kerberos = 0;
			break;
		case 'd':
			dflag = 1;
			break;
		case 'e':
			noescape = 0;
			escapechar = getescape(optarg);
			break;
		case 'k':
			dest_realm = dst_realm_buf;
			(void)strncpy(dest_realm, optarg, REALM_SZ);
			break;
		case 'l':
			user = optarg;
			break;
		case 'x':
			doencrypt = 1;
			break;
		case '?':
		default:
			usage();
		}
	optind += argoff;
	argc -= optind;
	argv += optind;

	/* if haven't gotten a host yet, do so */
	if (!host && !(host = *argv++))
		usage();

	if (*argv)
		usage();

	if (!(pw = getpwuid(uid = getuid()))) {
		(void)fprintf(stderr, "rlogin: unknown user id.\n");
		exit(1);
	}
	if (!user)
		user = pw->pw_name;


	sv_port=get_login_port(use_kerberos, doencrypt);

	p = getenv("TERM");
	strncpy(term, p ? p : "network", sizeof(term));
	term[sizeof(term) - 1] = '\0';
	if (tcgetattr(0, &tty) == 0) {
		char tmp[17];

		sprintf (tmp, "/%ld", (long) cfgetospeed(&tty));
		if (strlen (term) + strlen (tmp) < sizeof(term) - 1)
			strcat (term, tmp);
	}

	(void)get_window_size(0, &winsize);

      try_connect:
	if (use_kerberos) {
		struct hostent *hp;

		/* Fully qualify hostname (needed for krb_realmofhost). */
		hp = gethostbyname(host);
		if (hp != NULL && !(host = strdup(hp->h_name))) {
			(void)fprintf(stderr, "rlogin: %s\n",
			    strerror(ENOMEM));
			exit(1);
		}

		rem = KSUCCESS;
		errno = 0;
		if (dest_realm == NULL)
			dest_realm = krb_realmofhost(host);

		if (doencrypt)
			rem = krcmd_mutual(&host, sv_port, user, term, 0,
			    dest_realm, &cred, schedule);
		else
			rem = krcmd(&host, sv_port, user, term, 0,
			    dest_realm);
		if (rem < 0) {
			use_kerberos = 0;
			sv_port=get_login_port(use_kerberos, doencrypt);
			if (errno == ECONNREFUSED)
			  warning("remote host doesn't support Kerberos", "");
			if (errno == ENOENT)
			  warning("can't provide Kerberos auth data", "");
			goto try_connect;
		}
	} else {
		if (doencrypt) {
			(void)fprintf(stderr,
			    "rlogin: the -x flag requires Kerberos authentication.\n");
			exit(1);
		}
		if (geteuid() != 0) {
		        fprintf(stderr, "rlogin: not installed setuid root, only root may use non kerberized rlogin\n");
			exit(1);
		}
		rem = rcmd(&host, sv_port, pw->pw_name, user, term, 0);
	}

	if (rem < 0)
		exit(1);

	if (dflag &&
	    setsockopt(rem, SOL_SOCKET, SO_DEBUG, (void *)&one,
		       sizeof(one)) < 0)
		(void)fprintf(stderr, "rlogin: setsockopt: %s.\n",
		    strerror(errno));
	if (Dflag &&
	    setsockopt(rem, IPPROTO_TCP, TCP_NODELAY, (void *)&one,
		       sizeof(one)) < 0)
	        perror("rlogin: setsockopt NODELAY (ignored)");
#ifdef IP_TOS
	one = IPTOS_LOWDELAY;
	if (setsockopt(rem, IPPROTO_IP, IP_TOS, (void *)&one, sizeof(int)) < 0)
		perror("rlogin: setsockopt TOS (ignored)");
#endif /* IP_TOS */

	(void)setuid(uid);
	doit();
	return 0;
}
