/*
 * int krb_equiv(u_int32_t ipaddr_a, u_int32_t ipaddr_b);
 *
 * Given two IP adresses return true if they match
 * or are considered to belong to the same host.
 *
 * For example if /etc/krb.equiv looks like
 *
 *    130.237.223.3   192.16.126.3    # alv alv1
 *    130.237.223.4   192.16.126.4    # byse byse1
 *    130.237.228.152 192.16.126.9    # topsy topsy1
 *
 * krb_equiv(alv, alv1) would return true but
 * krb_equiv(alv, byse1) would not.
 *
 * A comment starts with an '#' and ends with '\n'.
 *
 */
#include "krb_locl.h"

RCSID("$Id: krb_equiv.c,v 1.9 1996/03/25 13:09:37 bg Exp $");

int krb_ignore_ip_address = 0;

int
krb_equiv(u_int32_t a, u_int32_t b)
{
  FILE *fil;
  char line[256];
  int hit_a, hit_b;
  int iscomment;
  
  if (a == b)			/* trivial match, also the common case */
    return 1;
  
  if (krb_ignore_ip_address)
    return 1;			/* if we have decided not to compare */

  a = ntohl(a);
  b = ntohl(b);

  fil = fopen(KRB_EQUIV, "r");
  if (fil == NULL)		/* open failed */
    return 0;
  
  hit_a = hit_b = 0;
  iscomment = 0;
  while (fgets(line, sizeof(line)-1, fil) != NULL) /* for each line */
    {
      char *t = line;
      int len = strlen(t);
      
      /* for each item on this line */
      while (*t != 0)		/* more addresses on this line? */
	if (*t == '\n') {
	  iscomment = hit_a = hit_b = 0;
	  break;
	} else if (iscomment)
	  t = line + len - 1;
	else if (*t == '#') {		/* rest is comment */
	  iscomment = 1;
	  ++t;
	} else if (*t == '\\' ) /* continuation */
	  break;
	else if (isspace(*t))	/* skip space */
	  t++;
	else if (isdigit(*t))	/* an address? */
	  {
	    u_int32_t tmp;
	    u_int32_t tmpa, tmpb, tmpc, tmpd;
	    
	    sscanf(t, "%d.%d.%d.%d", &tmpa, &tmpb, &tmpc, &tmpd);
	    tmp = (tmpa << 24) | (tmpb << 16) | (tmpc << 8) | tmpd;

	    while (*t == '.' || isdigit(*t)) /* done with this address */
	      t++;

	    if (tmp != -1) {	/* an address (and not broadcast) */
	      u_int32_t mask = ~0;

	      if (*t == '/') {
		++t;
		mask <<= 32 - atoi(t);

		while(isdigit(*t))
		  ++t;
	      }

	      if ((tmp & mask) == (a & mask))
		hit_a = 1;
	      if ((tmp & mask) == (b & mask))
		hit_b = 1;
	      if (hit_a && hit_b) {
		fclose(fil);
		return 1;
	      }
	    }
	  }
	else
	  ++t;		/* garbage on this line, skip it */

    }

  fclose(fil);
  return 0;
}
