/*
 * RFC2367 PF_KEYv2 Key management API message parser
 * Copyright (C) 1999  Richard Guy Briggs.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * RCSID $Id: pfkey_v2_parser.c,v 1.24 1999/12/09 23:23:00 rgb Exp $
 */

/*
 *		Template from klips/net/ipsec/ipsec/ipsec_netlink.c.
 */

char pfkey_v2_parser_c_version[] = "$Id: pfkey_v2_parser.c,v 1.24 1999/12/09 23:23:00 rgb Exp $";

#include <linux/config.h>
#include <linux/version.h>

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/interrupt.h> /* mark_bh */

#include <linux/netdevice.h>   /* struct device, and other headers */
#include <linux/etherdevice.h> /* eth_type_trans */
#include <linux/ip.h>          /* struct iphdr */
#include <linux/skbuff.h>
#include <freeswan.h>
#ifdef SPINLOCK
#ifdef SPINLOCK_23
#include <linux/spinlock.h> /* *lock* */
#else /* SPINLOCK_23 */
#include <asm/spinlock.h> /* *lock* */
#endif /* SPINLOCK_23 */
#endif /* SPINLOCK */
#ifdef NET_21
#include <asm/uaccess.h>
#include <linux/in6.h>
#define ip_chk_addr inet_addr_type
#define IS_MYADDR RTN_LOCAL
#endif
#include <asm/checksum.h>
#include <net/ip.h>
#ifdef NETLINK_SOCK
#include <linux/netlink.h>
#else
#include <net/netlink.h>
#endif

#include <linux/random.h>	/* get_random_bytes() */

#include "radij.h"
#include "ipsec_encap.h"
#include "ipsec_radij.h"
#include "ipsec_netlink.h"
#include "ipsec_xform.h"
#include "ipsec_ah.h"
#include "ipsec_esp.h"

#include <pfkeyv2.h>
#include <pfkey.h>


#define SENDERR(_x) do { error = -(_x); goto errlab; } while (0)

#ifndef min
#define min(a,b)	(((a)<(b))?(a):(b))
#endif

extern int des_set_key(caddr_t, caddr_t);

struct sklist_t {
	struct socket *sk;
	struct sklist_t* next;
} pfkey_sklist_head, *pfkey_sklist, *pfkey_sklist_prev;

__u32 pfkey_msg_seq = 0;

static int pfkey_sa_process(struct sadb_ext *pfkey_ext, struct tdb *tdb1, struct tdb *tdb2)
{
	struct sadb_sa *pfkey_sa = (struct sadb_sa *)pfkey_ext;
	int error = 0;
	struct tdb* tdbp;
	
	switch(pfkey_ext->sadb_ext_type) {
	case SADB_EXT_SA:
		tdbp = tdb1;
		break;
	case SADB_X_EXT_SA2:
		tdbp = tdb2;
		break;
	default:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug: pfkey_sa_process: invalid exttype=%d.\n",
			    pfkey_ext->sadb_ext_type);
		SENDERR(EINVAL);
	}

	tdbp->tdb_said.spi = pfkey_sa->sadb_sa_spi;
	tdbp->tdb_replaywin = pfkey_sa->sadb_sa_replay;
	tdbp->tdb_state = pfkey_sa->sadb_sa_state;
	tdbp->tdb_flags = pfkey_sa->sadb_sa_flags;
	tdbp->tdb_replaywin_lastseq = tdbp->tdb_replaywin_bitmap = 0;

	switch(tdbp->tdb_said.proto) {
		case IPPROTO_AH:
			tdbp->tdb_authalg = pfkey_sa->sadb_sa_auth;
			tdbp->tdb_encalg = SADB_EALG_NONE;

			break;
		case IPPROTO_ESP:
			tdbp->tdb_authalg = pfkey_sa->sadb_sa_auth;
			tdbp->tdb_encalg = pfkey_sa->sadb_sa_encrypt;

			break;
		case IPPROTO_IPIP:
			tdbp->tdb_authalg = AH_NONE;
			tdbp->tdb_encalg = ESP_NONE;
			break;
#ifdef CONFIG_IPSEC_COMP
		case IPPROTO_COMP:
			break;
#endif /* CONFIG_IPSEC_COMP */
		default:
			SENDERR(EINVAL);
	}

errlab:
	return error;
}

static int
pfkey_lifetime_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
	int error = 0;
	struct sadb_lifetime *pfkey_lifetime = (struct sadb_lifetime *)pfkey_ext;

	switch(pfkey_lifetime->sadb_lifetime_exttype) {
	case SADB_EXT_LIFETIME_CURRENT:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_lifetime_process: "
			    "lifetime_current not supported yet.\n");
  		SENDERR(EINVAL);
  		break;
	case SADB_EXT_LIFETIME_HARD:
		if(pfkey_lifetime->sadb_lifetime_allocations &&
		   (!tdbp->tdb_lifetime_allocations_h ||
		    (pfkey_lifetime->sadb_lifetime_allocations < tdbp->tdb_lifetime_allocations_h))) {
			tdbp->tdb_lifetime_allocations_h =
				pfkey_lifetime->sadb_lifetime_allocations;
			if(!tdbp->tdb_lifetime_allocations_s &&
			   (tdbp->tdb_lifetime_allocations_h < tdbp->tdb_lifetime_allocations_s)) {
				tdbp->tdb_lifetime_allocations_s = tdbp->tdb_lifetime_allocations_h;
			}
		}
		if(pfkey_lifetime->sadb_lifetime_bytes &&
		   (!tdbp->tdb_lifetime_bytes_h ||
		    (pfkey_lifetime->sadb_lifetime_bytes < tdbp->tdb_lifetime_bytes_h))) {
			tdbp->tdb_lifetime_bytes_h =
				pfkey_lifetime->sadb_lifetime_bytes;
			if(!tdbp->tdb_lifetime_bytes_s &&
			   (tdbp->tdb_lifetime_bytes_h < tdbp->tdb_lifetime_bytes_s)) {
				tdbp->tdb_lifetime_bytes_s = tdbp->tdb_lifetime_bytes_h;
			}
		}
		if(pfkey_lifetime->sadb_lifetime_addtime &&
		   (!tdbp->tdb_lifetime_addtime_h ||
		    (pfkey_lifetime->sadb_lifetime_addtime < tdbp->tdb_lifetime_addtime_h))) {
			tdbp->tdb_lifetime_addtime_h =
				pfkey_lifetime->sadb_lifetime_addtime;
			if(tdbp->tdb_lifetime_addtime_s &&
			   (tdbp->tdb_lifetime_addtime_h < tdbp->tdb_lifetime_addtime_s)) {
				tdbp->tdb_lifetime_addtime_s = tdbp->tdb_lifetime_addtime_h;
			}
		}
		if(pfkey_lifetime->sadb_lifetime_usetime &&
		   (!tdbp->tdb_lifetime_usetime_h ||
		    (pfkey_lifetime->sadb_lifetime_usetime < tdbp->tdb_lifetime_usetime_h))) {
			tdbp->tdb_lifetime_usetime_h =
				pfkey_lifetime->sadb_lifetime_usetime;
			if(tdbp->tdb_lifetime_usetime_s &&
			   (tdbp->tdb_lifetime_usetime_h < tdbp->tdb_lifetime_usetime_s)) {
				tdbp->tdb_lifetime_usetime_s = tdbp->tdb_lifetime_usetime_h;
			}
		}
		break;
	case SADB_EXT_LIFETIME_SOFT:
		if(pfkey_lifetime->sadb_lifetime_allocations &&
		   (!tdbp->tdb_lifetime_allocations_s ||
		    (pfkey_lifetime->sadb_lifetime_allocations < tdbp->tdb_lifetime_allocations_s))) {
			tdbp->tdb_lifetime_allocations_s =
				pfkey_lifetime->sadb_lifetime_allocations;
			if(tdbp->tdb_lifetime_allocations_h &&
			   (tdbp->tdb_lifetime_allocations_h < tdbp->tdb_lifetime_allocations_s)) {
				tdbp->tdb_lifetime_allocations_s = tdbp->tdb_lifetime_allocations_h;
			}
		}
		if(pfkey_lifetime->sadb_lifetime_bytes &&
		   (!tdbp->tdb_lifetime_bytes_s ||
		    (pfkey_lifetime->sadb_lifetime_bytes < tdbp->tdb_lifetime_bytes_s))) {
			tdbp->tdb_lifetime_bytes_s =
				pfkey_lifetime->sadb_lifetime_bytes;
			if(tdbp->tdb_lifetime_bytes_h &&
			   (tdbp->tdb_lifetime_bytes_h < tdbp->tdb_lifetime_bytes_s)) {
				tdbp->tdb_lifetime_bytes_s = tdbp->tdb_lifetime_bytes_h;
			}
		}
		if(pfkey_lifetime->sadb_lifetime_addtime &&
		   (!tdbp->tdb_lifetime_addtime_s ||
		    (pfkey_lifetime->sadb_lifetime_addtime < tdbp->tdb_lifetime_addtime_s))) {
			tdbp->tdb_lifetime_addtime_s =
				pfkey_lifetime->sadb_lifetime_addtime;
			if(tdbp->tdb_lifetime_addtime_h &&
			   (tdbp->tdb_lifetime_addtime_h < tdbp->tdb_lifetime_addtime_s)) {
				tdbp->tdb_lifetime_addtime_s = tdbp->tdb_lifetime_addtime_h;
			}
		}
		if(pfkey_lifetime->sadb_lifetime_usetime &&
		   (!tdbp->tdb_lifetime_usetime_s ||
		    (pfkey_lifetime->sadb_lifetime_usetime < tdbp->tdb_lifetime_usetime_s))) {
			tdbp->tdb_lifetime_usetime_s =
				pfkey_lifetime->sadb_lifetime_usetime;
			if(tdbp->tdb_lifetime_usetime_h &&
			   (tdbp->tdb_lifetime_usetime_h < tdbp->tdb_lifetime_usetime_s)) {
				tdbp->tdb_lifetime_usetime_s = tdbp->tdb_lifetime_usetime_h;
			}
		}
		break;
	default:
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_lifetime_build: invalid exttype=%d.\n",
			pfkey_ext->sadb_ext_type);
		SENDERR(EINVAL);
	}

errlab:
	return error;
}

static int
pfkey_address_process(struct sadb_ext *pfkey_ext, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;
	int saddr_len = 0;
	char ipaddr_txt[ADDRTOA_BUF];
	unsigned char **sap;
	struct sadb_address *pfkey_address = (struct sadb_address *)pfkey_ext;
	struct sockaddr* s = (struct sockaddr*)((void*)pfkey_address + sizeof(*pfkey_address));
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_address_process:\n");
	
	switch(s->sa_family) {
	case AF_INET:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: found address family=%d, AF_INET.\n",
			    s->sa_family);
		saddr_len = sizeof(struct sockaddr_in);
		addrtoa(((struct sockaddr_in*)s)->sin_addr, 0, ipaddr_txt, sizeof(ipaddr_txt));
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: found address=%s.\n",
			    ipaddr_txt);
		break;
#if defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE)
	case AF_INET6:
		saddr_len = sizeof(struct sockaddr_in6);
		break;
#endif /* defined(CONFIG_IPV6) || defined(CONFIG_IPV6_MODULE) */
	default:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: s->sa_family=%d not supported.\n",
			    s->sa_family);
		SENDERR(EPFNOSUPPORT);
	}
	
	switch(pfkey_address->sadb_address_exttype) {
	case SADB_EXT_ADDRESS_SRC:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: found src address.\n");
		sap = (unsigned char **)&(tdb1->tdb_addr_s);
		tdb1->tdb_addr_s_size = saddr_len;
		break;
	case SADB_EXT_ADDRESS_DST:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: found dst address.\n");
		sap = (unsigned char **)&(tdb1->tdb_addr_d);
		tdb1->tdb_addr_d_size = saddr_len;
		break;
	case SADB_EXT_ADDRESS_PROXY:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: found proxy address.\n");
		sap = (unsigned char **)&(tdb1->tdb_addr_p);
		tdb1->tdb_addr_p_size = saddr_len;
		break;
	case SADB_X_EXT_ADDRESS_DST2:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: found 2nd dst address.\n");
		sap = (unsigned char **)&(tdb2->tdb_addr_d);
		tdb2->tdb_addr_d_size = saddr_len;
		break;
	default:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_address_process: unrecognised ext_type=%d.\n",
			    pfkey_address->sadb_address_exttype);
		SENDERR(EINVAL);
	}
	
	if(!(*sap = kmalloc(saddr_len, GFP_KERNEL))) {
		SENDERR(ENOMEM);
	}
	memcpy(*sap, s, saddr_len);
	
	if(pfkey_address->sadb_address_exttype == SADB_EXT_ADDRESS_DST) {
		if(s->sa_family == AF_INET) {
			tdb1->tdb_said.dst.s_addr = ((struct sockaddr_in*)(tdb1->tdb_addr_d))->sin_addr.s_addr;
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_address_process: tdb1->tdb_said.dst.s_addr=%08x,\n"
				    "klips_debug:                     ((struct sockaddr_in*)(tdb1->tdb_addr_d))->sin_addr.s_addr=%08x,\n",
				    tdb1->tdb_said.dst.s_addr,
				    ((struct sockaddr_in*)(tdb1->tdb_addr_d))->sin_addr.s_addr
				);
			addrtoa(((struct sockaddr_in*)(tdb1->tdb_addr_d))->sin_addr/*tdb1->tdb_said.dst*/, 0, ipaddr_txt, sizeof(ipaddr_txt));
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_address_process: tdb_said.dst set to %s.\n",
				    ipaddr_txt);
		} else {
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_address_process: uh, tdb_said.dst doesn't do address family=%d yet, said will be invalid.\n",
				    s->sa_family);
		}
	}
	
	/* XXX check if port!=0 */
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_address_process: successful.\n");
 errlab:
	return error;
}

static int
pfkey_key_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
        int error = 0;
        struct sadb_key *pfkey_key = (struct sadb_key *)pfkey_ext;
	
        switch(pfkey_key->sadb_key_exttype) {
        case SADB_EXT_KEY_AUTH:
                tdbp->tdb_key_bits_a = pfkey_key->sadb_key_bits;
		if(!(tdbp->tdb_key_a = kmalloc(DIVUP(pfkey_key->sadb_key_bits, 8), GFP_KERNEL))) {
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_key_process: memory allocation error.\n");
			SENDERR(ENOMEM);
		}
		memcpy(tdbp->tdb_key_a,
		       (char*)pfkey_key + sizeof(struct sadb_key),
		       DIVUP(pfkey_key->sadb_key_bits, 8));
		break;
	case SADB_EXT_KEY_ENCRYPT: /* Key(s) */
		tdbp->tdb_key_bits_e = pfkey_key->sadb_key_bits;
		if(!(tdbp->tdb_key_e = kmalloc(DIVUP(pfkey_key->sadb_key_bits, 8), GFP_KERNEL))) {
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_key_process: memory allocation error.\n");
			SENDERR(ENOMEM);
		}
		memcpy(tdbp->tdb_key_e,
		       (char*)pfkey_key + sizeof(struct sadb_key),
		       DIVUP(pfkey_key->sadb_key_bits, 8));
		break;
	default:
		SENDERR(EINVAL);
 	}

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_key_process: success.\n");
errlab:
	return error;
}

static int
pfkey_ident_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
        int error = 0;
        struct sadb_ident *pfkey_ident = (struct sadb_ident *)pfkey_ext;
	
	switch(pfkey_ident->sadb_ident_exttype) {
	case SADB_EXT_IDENTITY_SRC:
		tdbp->tdb_ident_type_s = pfkey_ident->sadb_ident_type;
		tdbp->tdb_ident_id_s = pfkey_ident->sadb_ident_id;
		tdbp->tdb_ident_len_s = pfkey_ident->sadb_ident_len -
			sizeof(struct sadb_ident) / IPSEC_PFKEYv2_ALIGN;
		if(!(tdbp->tdb_ident_data_s = kmalloc(pfkey_ident->sadb_ident_len * IPSEC_PFKEYv2_ALIGN, GFP_KERNEL))) {
			SENDERR(ENOMEM);
		}
		memcpy(tdbp->tdb_ident_data_s,
		       (char*)pfkey_ident + sizeof(struct sadb_ident),
		       pfkey_ident->sadb_ident_len * IPSEC_PFKEYv2_ALIGN);
		break;
	case SADB_EXT_IDENTITY_DST: /* Identity(ies) */
		tdbp->tdb_ident_type_d = pfkey_ident->sadb_ident_type;
		tdbp->tdb_ident_id_d = pfkey_ident->sadb_ident_id;
		tdbp->tdb_ident_len_d = pfkey_ident->sadb_ident_len -
			sizeof(struct sadb_ident) / IPSEC_PFKEYv2_ALIGN;
		if(!(tdbp->tdb_ident_data_d = kmalloc(pfkey_ident->sadb_ident_len * IPSEC_PFKEYv2_ALIGN, GFP_KERNEL))) {
			SENDERR(ENOMEM);
		}
		memcpy(tdbp->tdb_ident_data_d,
		       (char*)pfkey_ident + sizeof(struct sadb_ident),
		       pfkey_ident->sadb_ident_len * IPSEC_PFKEYv2_ALIGN);
		break;
	default:
		SENDERR(EINVAL);
 	}
errlab:
	return error;
}

static int
pfkey_sens_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
        int error = 0;
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_sens_process: Sorry, I can't process exttype=%d yet.\n",
		    pfkey_ext->sadb_ext_type);
        SENDERR(EINVAL); /* don't process these yet */
 errlab:
        return error;
}

static int
pfkey_prop_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
        int error = 0;
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_prop_process: Sorry, I can't process exttype=%d yet.\n",
		    pfkey_ext->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */
	
 errlab:
	return error;
}

static int
pfkey_supported_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
        int error = 0;

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_supported_process: Sorry, I can't process exttype=%d yet.\n",
		pfkey_ext->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */

errlab:
	return error;
}

static int
pfkey_spirange_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
        int error = 0;

/* errlab: */
	return error;
}

static int
pfkey_x_kmprivate_process(struct sadb_ext *pfkey_ext, struct tdb *tdbp, struct tdb *tdb2)
{
	int error = 0;

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_x_kmprivate_process: Sorry, I can't process exttype=%d yet.\n",
		pfkey_ext->sadb_ext_type);
	SENDERR(EINVAL); /* don't process these yet */

errlab:
	return error;
}

static int
pfkey_x_satype_process(struct sadb_ext *pfkey_ext, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;
	struct sadb_x_satype *pfkey_x_satype = (struct sadb_x_satype *)pfkey_ext;

	if(!(tdb2->tdb_proto = sadb_satype2proto[pfkey_x_satype->sadb_x_satype_satype])) {
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_x_satype_process: proto lookup from satype=%d failed.\n",
			pfkey_x_satype->sadb_x_satype_satype);
		SENDERR(EINVAL);
	}

errlab:
	return error;
}


int
pfkey_tdb_init(struct tdb *tdbp, struct sadb_ext **extensions)
{
        int i;
        int error = 0;
        char sa[SATOA_BUF];
	char ipaddr_txt[ADDRTOA_BUF];
	char ipaddr2_txt[ADDRTOA_BUF];
	unsigned char kb[AHMD596_BLKLEN];

	satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);

        KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_tdb_init: (pfkey defined) called for SA:%s\n", sa);

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_tdb_init: calling init routine of %s%s%s\n",
		    TDB_XFORM_NAME(tdbp));
	
	switch(tdbp->tdb_proto) {
		
#ifdef CONFIG_IPSEC_IPIP
	case IPPROTO_IPIP: {
		addrtoa(((struct sockaddr_in*)(tdbp->tdb_addr_s))->sin_addr,
			0,
			ipaddr_txt, sizeof(ipaddr_txt));
		addrtoa(((struct sockaddr_in*)(tdbp->tdb_addr_d))->sin_addr,
			0,
			ipaddr2_txt, sizeof(ipaddr_txt));
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_tdb_init: "
			    "(pfkey defined) IPIP tdb set for %s->%s.\n",
			    ipaddr_txt,
			    ipaddr2_txt);
	}
	break;
#endif /* !CONFIG_IPSEC_IPIP */
#ifdef CONFIG_IPSEC_AH
	case IPPROTO_AH:
		switch(tdbp->tdb_authalg) {
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
		case AH_MD5: {
			unsigned char *akp;
			
			if(tdbp->tdb_key_bits_a != (AHMD596_KLEN * 8)) {
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_tdb_init: incorrect key size: %d bits"
					    "-- must be %d bits\n"/*octets (bytes)\n"*/,
					    tdbp->tdb_key_bits_a, AHMD596_KLEN * 8);
				SENDERR(EINVAL);
			}
			
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: hmac md5-96 key is 0x%08lx %08lx %08lx %08lx\n",
				    ntohl(*(((__u32 *)ed->ame_key)+0)),
				    ntohl(*(((__u32 *)ed->ame_key)+1)),
				    ntohl(*(((__u32 *)ed->ame_key)+2)),
				    ntohl(*(((__u32 *)ed->ame_key)+3)));
#endif
			
			tdbp->tdb_auth_bits = AHMD596_ALEN * 8;
			
			/* save the pointer to the key material */
			akp = tdbp->tdb_key_a;
			
			if((tdbp->tdb_key_a = (caddr_t)
			    kmalloc((tdbp->tdb_key_a_size = sizeof(struct md5_ctx)),
				    GFP_ATOMIC)) == NULL) {
				SENDERR(ENOMEM);
			}
			MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx);
			
			for (i = 0; i < DIVUP(tdbp->tdb_key_bits_a, 8); i++) {
				kb[i] = akp[i] ^ 0x36;
			}
			for (; i < AHMD596_BLKLEN; i++) {
				kb[i] = 0x36;
			}
			MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx,
				  kb,
				  AHMD596_BLKLEN);
			MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx);
			for (i = 0; i < AHMD596_BLKLEN; i++) {
				kb[i] ^= (0x36 ^ 0x5c);
			}
			MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx,
				  kb,
				  AHMD596_BLKLEN);
			
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: MD5 ictx=0x%08x %08x %08x %08x"
				    " octx=0x%08x %08x %08x %08x\n",
				    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
				    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 1),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 2),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 3));
#endif
			
				/* zero key buffer -- paranoid */
			memset(akp, 0, DIVUP(tdbp->tdb_key_bits_a, BITS_PER_OCTET));
		}
		break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
		case AH_SHA: {
			
			unsigned char *akp;
			
			if(tdbp->tdb_key_bits_a != (AHSHA196_KLEN * 8)) {
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_tdb_init: incorrect key size: %d bits"
					    "-- must be %d bits\n"/*octets (bytes)\n"*/,
					    tdbp->tdb_key_bits_a, AHSHA196_KLEN * 8);
				SENDERR(EINVAL);
			}
			
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: hmac sha1-96 key is 0x%08lx %08lx %08lx %08lx\n",
				    ntohl(*(((__u32 *)ed->ame_key)+0)),
				    ntohl(*(((__u32 *)ed->ame_key)+1)),
				    ntohl(*(((__u32 *)ed->ame_key)+2)),
				    ntohl(*(((__u32 *)ed->ame_key)+3)));
#endif
			
			tdbp->tdb_auth_bits = AHSHA196_ALEN * 8;
			
			/* save the pointer to the key material */
			akp = tdbp->tdb_key_a;
			
			if((tdbp->tdb_key_a = (caddr_t)
			    kmalloc((tdbp->tdb_key_a_size = sizeof(struct sha1_ctx)),
				    GFP_ATOMIC)) == NULL) {
				SENDERR(ENOMEM);
			}

			SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx);
			
			for (i = 0; i < DIVUP(tdbp->tdb_key_bits_a, 8); i++) {
				kb[i] = akp[i] ^ 0x36;
			}
			for (; i < AHMD596_BLKLEN; i++) {
				kb[i] = 0x36;
			}
			SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx,
				  kb,
				  AHSHA196_BLKLEN);
			
			SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx);
			for (i = 0; i < AHSHA196_BLKLEN; i++) {
				kb[i] ^= (0x36 ^ 0x5c);
			}
			SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx,
				  kb,
				  AHSHA196_BLKLEN);
			
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: SHA1 ictx=0x%08x %08x %08x %08x"
				    " octx=0x%08x %08x %08x %08x\n", 
				    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
				    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx),
						    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 1),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 2),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 3));
			
				/* zero key buffer -- paranoid */
			memset(akp, 0, DIVUP(tdbp->tdb_key_bits_a, BITS_PER_OCTET));
		}
		break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
		default:
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: "
				    "authalg=%d support not available in the kernel",
				    tdbp->tdb_authalg);
			SENDERR(EINVAL);
		}
#endif /* CONFIG_IPSEC_AH */
#ifdef CONFIG_IPSEC_ESP
	case IPPROTO_ESP: {
		unsigned char *akp, *ekp;
		
		switch(tdbp->tdb_encalg) {
#ifdef CONFIG_IPSEC_ENC_DES
		case ESP_DES:
#endif /* CONFIG_IPSEC_ENC_DES */
#ifdef CONFIG_IPSEC_ENC_3DES
		case ESP_3DES:
#endif /* CONFIG_IPSEC_ENC_3DES */
#if defined(CONFIG_IPSEC_ENC_DES) || defined(CONFIG_IPSEC_ENC_3DES)
			if((tdbp->tdb_iv = (caddr_t)
			    kmalloc((tdbp->tdb_iv_size = EMT_ESPDES_IV_SZ), GFP_ATOMIC)) == NULL) {
				SENDERR(ENOMEM);
			}
			get_random_bytes((void *)tdbp->tdb_iv, EMT_ESPDES_IV_SZ);
			break;
#endif /* defined(CONFIG_IPSEC_ENC_DES) || defined(CONFIG_IPSEC_ENC_3DES) */
		case ESP_NONE:
#ifdef CONFIG_IPSEC_ENC_NULL
		case ESP_NULL:
#endif /* CONFIG_IPSEC_ENC_NULL */
			break;
		default:
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: "
				    "encalg=%d support not available in the kernel",
				    tdbp->tdb_encalg);
			SENDERR(EINVAL);
		}
		
		switch(tdbp->tdb_encalg) {
#ifdef CONFIG_IPSEC_ENC_DES
		case ESP_DES:
			if(tdbp->tdb_key_bits_e != (EMT_ESPDES_KEY_SZ * 8)) {
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_tdb_init: incorrect encryption"
					    "key size: %d bits -- must be %d bits\n"/*octets (bytes)\n"*/,
					    tdbp->tdb_key_bits_e, EMT_ESPDES_KEY_SZ * 8);
				SENDERR(EINVAL);
			}
			
			/* save encryption key pointer */
			ekp = tdbp->tdb_key_e;
			
			if((tdbp->tdb_key_e = (caddr_t)
			    kmalloc((tdbp->tdb_key_e_size = sizeof(struct des_eks)),
				    GFP_ATOMIC)) == NULL) {
				SENDERR(ENOMEM);
			}
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: des key 1 is 0x%08lx%08lx\n",
				    ntohl(*((__u32 *)ed->eme_key)),
				    ntohl(*((__u32 *)ed->eme_key + 1)));
#endif
			error = des_set_key((caddr_t)ekp, (caddr_t)(tdbp->tdb_key_e));
			if (error == -1)
				printk("klips_debug:pfkey_tdb_init: parity error in des key\n");
			else if (error == -2)
				printk("klips_debug:pfkey_tdb_init: illegal weak des key\n");
			if (error) {
				memset(tdbp->tdb_key_e, 0, sizeof(struct des_eks));
				kfree_s(tdbp->tdb_key_e, sizeof(struct des_eks));
				memset(ekp, 0, DIVUP(tdbp->tdb_key_bits_e, BITS_PER_OCTET));
				SENDERR(EINVAL);
			}
			memset(ekp, 0, DIVUP(tdbp->tdb_key_bits_e, BITS_PER_OCTET));
			break;
#endif /* CONFIG_IPSEC_ENC_DES */
#ifdef CONFIG_IPSEC_ENC_3DES
		case ESP_3DES:
			if(tdbp->tdb_key_bits_e != (EMT_ESP3DES_KEY_SZ * 8)) {
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_tdb_init: incorrect encryption"
					    "key size: %d bits -- must be %d bits\n"/*octets (bytes)\n"*/,
					    tdbp->tdb_key_bits_e, EMT_ESP3DES_KEY_SZ * 8);
				SENDERR(EINVAL);
			}
			
			/* save encryption key pointer */
			ekp = tdbp->tdb_key_e;
			
			if((tdbp->tdb_key_e = (caddr_t)
			    kmalloc((tdbp->tdb_key_e_size = 3 * sizeof(struct des_eks)),
				    GFP_ATOMIC)) == NULL) {
				SENDERR(ENOMEM);
			}
			
			for(i = 0; i < 3; i++) {
#if 0 /* we don't really want to print these unless there are really big problems */
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_tdb_init: 3des key %d/3 is 0x%08lx%08lx\n",
					    i + 1,
					    ntohl(*((__u32 *)ed->eme_key + i * 2)),
					    ntohl(*((__u32 *)ed->eme_key + i * 2 + 1)));
#endif
				error = des_set_key((caddr_t)ekp + EMT_ESPDES_KEY_SZ * i,
						    (caddr_t)&((struct des_eks*)(tdbp->tdb_key_e))[i]);
				if (error == -1)
					printk("klips_debug:pfkey_tdb_init: parity error in des key %d/3\n", i + 1);
				else if (error == -2)
					printk("klips_debug:pfkey_tdb_init: illegal weak des key %d/3\n", i + 1);
				if (error) {
					memset(tdbp->tdb_key_e, 0, 3 * sizeof(struct des_eks));
					kfree_s(tdbp->tdb_key_e, 3 * sizeof(struct des_eks));
					memset(ekp, 0, DIVUP(tdbp->tdb_key_bits_e, BITS_PER_OCTET));
					SENDERR(EINVAL);
				}
			}

			/* paranoid */
			memset(ekp, 0, DIVUP(tdbp->tdb_key_bits_e, BITS_PER_OCTET));
			
			break;
#endif /* CONFIG_IPSEC_ENC_3DES */
#ifdef CONFIG_IPSEC_ENC_NULL
		case ESP_NULL:
#endif /* CONFIG_IPSEC_ENC_NULL */
		case ESP_NONE:
			break;
		default:
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: "
				    "encalg=%d support not available in the kernel",
				    tdbp->tdb_encalg);
			SENDERR(EINVAL);
		}
		
		switch(tdbp->tdb_authalg) {
#ifdef CONFIG_IPSEC_AUTH_HMAC_MD5
		case AH_MD5:
			if(tdbp->tdb_key_bits_a != (AHMD596_KLEN * 8)) {
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_tdb_init: incorrect authorisation"
					    " key size: %d bits -- must be %d bits\n"/*octets (bytes)\n"*/,
					    tdbp->tdb_key_bits_a, AHMD596_KLEN * 8);
				SENDERR(EINVAL);
			}
			
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: hmac md5-96 key is 0x%08lx %08lx %08lx %08lx\n",
				    ntohl(*(((__u32 *)(tdbp->tdb_key_a))+0)),
				    ntohl(*(((__u32 *)(tdbp->tdb_key_a))+1)),
				    ntohl(*(((__u32 *)(tdbp->tdb_key_a))+2)),
				    ntohl(*(((__u32 *)(tdbp->tdb_key_a))+3)));
#endif
			tdbp->tdb_auth_bits = AHMD596_ALEN * 8;
			
			/* save the pointer to the key material */
			akp = tdbp->tdb_key_a;
			
			if((tdbp->tdb_key_a = (caddr_t)
			    kmalloc((tdbp->tdb_key_a_size = sizeof(struct md5_ctx)),
				    GFP_ATOMIC)) == NULL) {
				SENDERR(ENOMEM);
			}
			MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx);
			
			for (i = 0; i < DIVUP(tdbp->tdb_key_bits_a, 8); i++) {
				kb[i] = akp[i] ^ 0x36;
			}
			for (; i < AHMD596_BLKLEN; i++) {
				kb[i] = 0x36;
			}
			MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx,
				  kb,
				  AHMD596_BLKLEN);
			MD5Init(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx);
			for (i = 0; i < AHMD596_BLKLEN; i++) {
				kb[i] ^= (0x36 ^ 0x5c);
			}
			MD5Update(&((struct md5_ctx*)(tdbp->tdb_key_a))->octx,
				  kb,
				  AHMD596_BLKLEN);
			
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: MD5 ictx=0x%08x %08x %08x %08x"
				    " octx=0x%08x %08x %08x %08x\n",
				    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
				    *((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 1),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 2),
				    *(((__u32*)&((struct md5_ctx*)(tdbp->tdb_key_a))->octx) + 3));
#endif
			/* paranoid */
			memset(akp, 0, DIVUP(tdbp->tdb_key_bits_a, BITS_PER_OCTET));
			break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_MD5 */
#ifdef CONFIG_IPSEC_AUTH_HMAC_SHA1
		case AH_SHA:
			if(tdbp->tdb_key_bits_a != (AHSHA196_KLEN * 8)) {
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_tdb_init: incorrect authorisation"
					    " key size: %d bits -- must be %d bits\n"/*octets (bytes)\n"*/,
					    tdbp->tdb_key_bits_a, AHSHA196_KLEN * 8);
				SENDERR(EINVAL);
			}
			
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: hmac sha1-96 key is 0x%08lx %08lx %08lx %08lx\n",
				    ntohl(*(((__u32 *)ed->ame_key)+0)),
				    ntohl(*(((__u32 *)ed->ame_key)+1)),
				    ntohl(*(((__u32 *)ed->ame_key)+2)),
				    ntohl(*(((__u32 *)ed->ame_key)+3)));
#endif
			tdbp->tdb_auth_bits = AHSHA196_ALEN * 8;
			
			/* save the pointer to the key material */
			akp = tdbp->tdb_key_a;

			if((tdbp->tdb_key_a = (caddr_t)
			    kmalloc((tdbp->tdb_key_a_size = sizeof(struct sha1_ctx)),
				    GFP_ATOMIC)) == NULL) {
				SENDERR(ENOMEM);
			}

			SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx);
			
			for (i = 0; i < DIVUP(tdbp->tdb_key_bits_a, 8); i++) {
				kb[i] = akp[i] ^ 0x36;
			}
			for (; i < AHMD596_BLKLEN; i++) {
				kb[i] = 0x36;
			}
			SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx,
				  kb,
				  AHSHA196_BLKLEN);
			
			SHA1Init(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx);
			for (i = 0; i < AHSHA196_BLKLEN; i++) {
				kb[i] ^= (0x36 ^ 0x5c);
			}
			SHA1Update(&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx,
				  kb,
				  AHSHA196_BLKLEN);
			
#if 0 /* we don't really want to print these unless there are really big problems */
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: SHA1 ictx=0x%08x %08x %08x %08x"
				    " octx=0x%08x %08x %08x %08x\n",
				    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 1),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 2),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->ictx) + 3),
				    *((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 1),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 2),
				    *(((__u32*)&((struct sha1_ctx*)(tdbp->tdb_key_a))->octx) + 3));
#endif
			memset(akp, 0, DIVUP(tdbp->tdb_key_bits_a, BITS_PER_OCTET));
			break;
#endif /* CONFIG_IPSEC_AUTH_HMAC_SHA1 */
		case AH_NONE:
			break;
		default:
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_tdb_init: "
				    "authalg=%d support not available in the kernel",
				    tdbp->tdb_authalg);
			SENDERR(EINVAL);
		}
	}
			break;
#endif /* !CONFIG_IPSEC_ESP */
#ifdef CONFIG_IPSEC_COMP
	case IPPROTO_COMP:
#endif /* CONFIG_IPSEC_COMP */
	default:
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_tdb_init: "
			    "proto=%d unknown",
			    tdbp->tdb_proto);
		SENDERR(EINVAL);
	}
	
 errlab:
	return(error);
}

int
pfkey_getspi_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdbp, struct tdb *tdb2)
{
	int error = 0;
	ipsec_spi_t minspi = 256, maxspi = -1L;
	int found_avail = 0;
	struct tdb *tdbq;
	char sa[SATOA_BUF];

	if(!tdbp) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_getspi_parse: error, tdbp pointer NULL\n");
		SENDERR(EINVAL);
	}

	if(extensions[SADB_EXT_SPIRANGE]) {
		minspi = ((struct sadb_spirange *)extensions[SADB_EXT_SPIRANGE])->sadb_spirange_min;
		maxspi = ((struct sadb_spirange *)extensions[SADB_EXT_SPIRANGE])->sadb_spirange_max;
	}

	if(maxspi == minspi) {
		tdbp->tdb_spi = maxspi;
		if((tdbq = gettdb(&(tdbp->tdb_said)))) {
			satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug: pfkey_getspi_parse: EMT_GETSPI found an old Tunnel Descriptor Block\n"
				    "klips_debug:                for SA: %s, delete it first.\n", sa);
			SENDERR(EEXIST);
		} else {
			found_avail = 1;
		}
	} else {
		int i = 0;
		while( ( i < (maxspi - minspi)) && !found_avail ) {
			get_random_bytes((void*) &(tdbp->tdb_spi),
					 /* sizeof(tdbp->tdb_spi) */
					 ( maxspi - minspi + 256 ) / 256 );
			tdbp->tdb_spi = htonl(minspi +
					      (tdbp->tdb_spi %
					      (maxspi - minspi + 1)));
			i++;
			tdbq = gettdb(&(tdbp->tdb_said));
			if(!tdbq) {
				found_avail = 1;
			}
		}
	}

	satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);

	if (!found_avail) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug: pfkey_getspi_parse: found an old Tunnel Descriptor Block\n"
			    "klips_debug:                for SA: %s, delete it first.\n", sa);
		SENDERR(EEXIST);
	}

	if(ip_chk_addr((unsigned long)tdbp->tdb_dst.s_addr) == IS_MYADDR) {
		tdbp->tdb_flags |= EMT_INBOUND;
	}

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_getspi_parse: existing Tunnel Descriptor Block not found (this\n"
		    "klips_debug:                is good) for SA: %s, %s-bound, allocating.\n",
		    sa, tdbp->tdb_flags & EMT_INBOUND ? "in" : "out");
	
	/* XXX tdbp->tdb_rcvif = &(enc_softc[em->em_if].enc_if);*/
	tdbp->tdb_rcvif = NULL;
	tdbp->tdb_lifetime_addtime_c = jiffies/HZ;

	tdbp->tdb_state = SADB_SASTATE_LARVAL;

	if(!tdbp->tdb_lifetime_allocations_c) {
		tdbp->tdb_lifetime_allocations_c += 1;
	}

	puttdb(tdbp);
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_getspi_parse: successful for SA: %s\n", sa);
	
 errlab:
	tdbp->tdb_state = SADB_SASTATE_DEAD;
	return error;
}

int
pfkey_update_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdbp, struct tdb *tdb2)
{
	int error = 0;
	struct tdb* tdbq;
	char sa[SATOA_BUF];
	unsigned long flags; /* save irq state for spinlock */

	if(((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_state != SADB_SASTATE_MATURE) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_update_parse: "
			    "error, sa_state=%d must be MATURE=%d\n",
			    ((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_state,
			    SADB_SASTATE_MATURE);
		SENDERR(EINVAL);
	}

	if(!tdbp) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_update_parse: error, tdbp pointer NULL\n");
		SENDERR(EINVAL);
	}

	satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);

	spin_lock_irqsave(&tdb_lock, flags);

	tdbq = gettdb(&(tdbp->tdb_said));
	if (!tdbq) {
		spin_unlock_irqrestore(&tdb_lock, flags);
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug: pfkey_update_parse: reserved Tunnel Descriptor Block\n"
			    "klips_debug:            for SA: %s not found.  Call SADB_GETSPI first or call SADB_ADD instead.\n", sa);
		SENDERR(EEXIST);
	}

	if(ip_chk_addr((unsigned long)tdbp->tdb_dst.s_addr) == IS_MYADDR) {
		tdbp->tdb_flags |= EMT_INBOUND;
	}

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_update_parse: existing Tunnel Descriptor Block found (this\n"
		    "klips_debug:                is good) for SA: %s, %s-bound, updating.\n",
		    sa, tdbp->tdb_flags & EMT_INBOUND ? "in" : "out");
	
	/* XXX tdbp->tdb_rcvif = &(enc_softc[em->em_if].enc_if);*/
	tdbp->tdb_rcvif = NULL;
	if ((error = pfkey_tdb_init(tdbp, extensions))) {
		spin_unlock_irqrestore(&tdb_lock, flags);
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_update_parse: "
			    "not successful for SA: %s, deleting.\n", sa);
		ipsec_tdbwipe(tdbp);
		SENDERR(-error);
	}

	if((error = deltdbchain(tdbq))) {
		spin_unlock_irqrestore(&tdb_lock, flags);
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_update_parse: "
			    "error, trouble deleting intermediate tdb for SA=%s.\n", sa);
		SENDERR(-error);
	}

	spin_unlock_irqrestore(&tdb_lock, flags);

	puttdb(tdbp);
	ipsec_tdbwipe(tdbq);
	
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_update_parse: successful for SA: %s\n", sa);
	
#if 0
	if ((error = ipsec_makeroute(&(em->em_eaddr), &(em->em_emask), em->em_ersaid)))
		SENDERR(-error);
#endif

 errlab:
	return error;
}

int
pfkey_add_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdbp, struct tdb *tdb2)
{
	int error = 0;
	struct tdb* tdbq;
	char sa[SATOA_BUF];

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_add_parse: parsing add message\n");

	if(((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_state != SADB_SASTATE_MATURE) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_add_parse: "
			    "error, sa_state=%d must be MATURE=%d\n",
			    ((struct sadb_sa*)extensions[SADB_EXT_SA])->sadb_sa_state,
			    SADB_SASTATE_MATURE);
		SENDERR(EINVAL);
	}

	if(!tdbp) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_add_parse: tdbp pointer NULL\n");
		SENDERR(EINVAL);
	}

	satoa(tdbp->tdb_said, 0, sa, SATOA_BUF);

	tdbq = gettdb(&(tdbp->tdb_said));
	if (tdbq) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_add_parse: found an old Tunnel Descriptor Block\n"
			    "klips_debug:                for SA: %s, delete it first.\n", sa);
		SENDERR(EEXIST);
	}

	if(ip_chk_addr((unsigned long)tdbp->tdb_dst.s_addr) == IS_MYADDR) {
		tdbp->tdb_flags |= EMT_INBOUND;
	}

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_add_parse: existing Tunnel Descriptor Block not found (this\n"
		    "klips_debug:                is good) for SA: %s, %s-bound, allocating.\n",
		    sa, tdbp->tdb_flags & EMT_INBOUND ? "in" : "out");
	
	/* XXX tdbp->tdb_rcvif = &(enc_softc[em->em_if].enc_if);*/
	tdbp->tdb_rcvif = NULL;
	
	if ((error = pfkey_tdb_init(tdbp, extensions))) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_add_parse: not successful for SA: %s, deleting.\n", sa);
		ipsec_tdbwipe(tdbp);
		SENDERR(-error);
	}

	tdbp->tdb_lifetime_addtime_c = jiffies / HZ;
	if(!tdbp->tdb_lifetime_allocations_c) {
		tdbp->tdb_lifetime_allocations_c += 1;
	}

	puttdb(tdbp);
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_add_parse: successful for SA: %s\n", sa);
	
#if 0
	if ((error = ipsec_makeroute(&(em->em_eaddr), &(em->em_emask), em->em_ersaid)))
		SENDERR(-error);
#endif
 errlab:
	/* XXX send a message up about success/failure */
	return error;
}

int
pfkey_delete_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdbq, struct tdb *tdb2)
{
	struct tdb *tdbp;
	char sa[SATOA_BUF];
	int error = 0;
	unsigned long flags; /* save irq state for spinlock */

	satoa(tdbq->tdb_said, 0, sa, SATOA_BUF);

	spin_lock_irqsave(&tdb_lock, flags);

	tdbp = gettdb(&(tdbq->tdb_said));
	if (tdbp == NULL) {
		spin_unlock_irqrestore(&tdb_lock, flags);
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_delete_parse: Tunnel Descriptor Block not found for SA:\n"
			    "klips_debug:                %s, could not delete.\n", sa);
		SENDERR(ENXIO);  /* XXX -- wrong error message... */
	} else {
		if((error = deltdbchain(tdbp))) {
			spin_unlock_irqrestore(&tdb_lock, flags);
			SENDERR(-error);
		}
		spin_unlock_irqrestore(&tdb_lock, flags);
	}
 errlab:
	ipsec_tdbwipe(tdbq);
	return error;
}

int
pfkey_get_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;
	struct tdb *tdbp;
	char sa[SATOA_BUF];
	unsigned long flags; /* save irq state for spinlock */

	satoa(tdb1->tdb_said, 0, sa, SATOA_BUF);

	spin_lock_irqsave(&tdb_lock, flags);

	tdbp = gettdb(&(tdb1->tdb_said));
	if (tdbp == NULL) {
		spin_unlock_irqrestore(&tdb_lock, flags);
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_get_parse: Tunnel Descriptor Block not found for SA:\n"
			    "klips_debug:                %s, could not get.\n", sa);
		SENDERR(ENXIO);  /* XXX -- wrong error message... */
	} else {
		/* build message to send up */
#if 0
		pfkey_upmsg(sk->sock /* sockp->socket */, new_pfkey_msg); */
#endif
		spin_unlock_irqrestore(&tdb_lock, flags);
	}

	SENDERR(ENOSYS);
 errlab:
	return error;
}

int
pfkey_acquire_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;

	/* ++pfkey_msg_seq */

	SENDERR(ENOSYS);
 errlab:
	return error;
}

int
pfkey_register_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;

	pfkey_list_insert_socket(sk->socket, pfkey_open_sockets);

	switch(((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_satype) {
	case SADB_SATYPE_AH:
		/* send up register msg with supported AH algos */
		break;
	case SADB_SATYPE_ESP:
		/* send up register msg with supported ESP algos */
		break;
	default:
		/* send up register msg with *all* supported algos */
	}

	SENDERR(ENOSYS);
 errlab:
	return error;
}

int
pfkey_expire_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;

	SENDERR(ENOSYS);
 errlab:
	return error;
}

int
pfkey_flush_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;

	KLIPS_PRINT(debug_pfkey,
		    "klips_debug: pfkey_flush_parse: flushing type %d SAs\n",
		    ((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_satype);
	if ((error = ipsec_tdbcleanup(((struct sadb_msg*)extensions[SADB_EXT_RESERVED])->sadb_msg_satype)))
		SENDERR(-error);
 errlab:
	return error;
}

int
pfkey_dump_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;

	SENDERR(ENOSYS);
 errlab:
	return error;
}

int
pfkey_x_promisc_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;

	SENDERR(ENOSYS);
 errlab:
	return error;
}

int
pfkey_x_pchange_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	int error = 0;

	SENDERR(ENOSYS);
 errlab:
	return error;
}

int
pfkey_x_grpsa_parse(struct sock *sk, struct sadb_ext **extensions, struct tdb *tdb1, struct tdb *tdb2)
{
	struct tdb *tdb1p, *tdb2p, *tdbp;
	char sa1[SATOA_BUF], sa2[SATOA_BUF];
	int error = 0;
	unsigned long flags; /* save irq state for spinlock */

	spin_lock_irqsave(&tdb_lock, flags);

	if(!tdb1 || !tdb2) {
		SENDERR(EINVAL);
	}

	satoa(tdb1->tdb_said, 0, sa1, SATOA_BUF);
	satoa(tdb2->tdb_said, 0, sa2, SATOA_BUF);

	if(!(tdb1p = gettdb(&(tdb1->tdb_said)))) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug: pfkey_x_grpsa_parse: reserved Tunnel Descriptor Block\n"
			    "klips_debug:            for SA: %s not found.  Call SADB_ADD/UPDATE first.\n", sa1);
		SENDERR(EEXIST);
	}
	if(!(tdb2p = gettdb(&(tdb2->tdb_said)))) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug: pfkey_x_grpsa_parse: reserved Tunnel Descriptor Block\n"
			    "klips_debug:            for SA: %s not found.  Call SADB_ADD/UPDATE first.\n", sa2);
		SENDERR(EEXIST);
	}

	/* Is tdb1 already linked to tdb2? */
	tdbp = tdb2p;
	while(tdbp) {
		if(tdbp == tdb1p) {
			SENDERR(EEXIST);
		}
		tdbp = tdb2p->tdb_onext;
	}

	/* Is either one already linked? */
	if(tdb1p->tdb_onext) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug: pfkey_x_grpsa_parse: Tunnel Descriptor Block\n"
			    "klips_debug:                for SA: %s is already linked.\n", sa1);
		SENDERR(EEXIST);
	}
	if(tdb2p->tdb_inext) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug: pfkey_x_grpsa_parse: Tunnel Descriptor Block\n"
			    "klips_debug:                for SA: %s is already linked.\n", sa2);
		SENDERR(EEXIST);
	}

	/* link 'em */
	tdb1p->tdb_onext = tdb2p;
	tdb2p->tdb_inext = tdb1p;

 errlab:
	spin_unlock_irqrestore(&tdb_lock, flags);

	ipsec_tdbwipe(tdb1);
	memset((caddr_t)tdb1, 0, sizeof(*tdb1));
	kfree_s(tdb1, sizeof(*tdb1));
	tdb1 = NULL;

	ipsec_tdbwipe(tdb2);
	memset((caddr_t)tdb2, 0, sizeof(*tdb2));
	kfree_s(tdb2, sizeof(*tdb2));
	tdb1 = NULL;

	return error;
}


int (*ext_processors[SADB_EXT_MAX+1])(struct sadb_ext *pfkey_ext, struct tdb *tdb1, struct tdb *tdb2) =
{
  NULL, /* pfkey_msg_process, */
        pfkey_sa_process,
        pfkey_lifetime_process,
        pfkey_lifetime_process,
        pfkey_lifetime_process,
        pfkey_address_process,
        pfkey_address_process,
        pfkey_address_process,
        pfkey_key_process,
        pfkey_key_process,
        pfkey_ident_process,
        pfkey_ident_process,
        pfkey_sens_process,
        pfkey_prop_process,
        pfkey_supported_process,
        pfkey_supported_process,
        pfkey_spirange_process,
        pfkey_x_kmprivate_process,
        pfkey_x_satype_process,
        pfkey_sa_process,
        pfkey_address_process

};


int (*msg_parsers[SADB_MAX +1])(struct sock *sk, struct sadb_ext *extensions[], struct tdb *tdb1, struct tdb *tdb2)
 =
{
	NULL, /* RESERVED */
	pfkey_getspi_parse,
	pfkey_update_parse,
	pfkey_add_parse,
	pfkey_delete_parse,
	pfkey_get_parse,
	pfkey_acquire_parse,
	pfkey_register_parse,
	pfkey_expire_parse,
	pfkey_flush_parse,
	pfkey_dump_parse,
	pfkey_x_promisc_parse,
	pfkey_x_pchange_parse,
	pfkey_x_grpsa_parse
};

int
pfkey_msg_interp(struct sock *sk, struct sadb_msg *pfkey_msg)
{
	int error = 0;
	int i;
	struct sadb_ext *extensions[SADB_EXT_MAX+1];
	struct tdb *tdb1, *tdb2;
	
	pfkey_extensions_init(extensions);
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_msg_interp: parsing message "
		    "ver=%d, type=%d, errno=%d, satype=%d, len=%d, res=%d, seq=%d, pid=%d.\n", 
		    pfkey_msg->sadb_msg_version,
		    pfkey_msg->sadb_msg_type,
		    pfkey_msg->sadb_msg_errno,
		    pfkey_msg->sadb_msg_satype,
		    pfkey_msg->sadb_msg_len,
		    pfkey_msg->sadb_msg_reserved,
		    pfkey_msg->sadb_msg_seq,
		    pfkey_msg->sadb_msg_pid);
	
#if 0
	if(pfkey_msg->sadb_msg_type == SADB_ADD) {
#endif
		/* check for and use tdbp from previous ACQUIRE or GETSPI call? XXX */
		if((tdb1 = kmalloc(sizeof(*tdb1), GFP_ATOMIC) ) == NULL) {
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_msg_interp: memory allocation error\n");
			SENDERR(ENOMEM);
		}
		memset((caddr_t)tdb1, 0, sizeof(*tdb1));
		if((tdb2 = kmalloc(sizeof(*tdb2), GFP_ATOMIC) ) == NULL) {
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_msg_interp: memory allocation error\n");
			SENDERR(ENOMEM);
		}
		memset((caddr_t)tdb2, 0, sizeof(*tdb2));
#if 0
	}
#endif

	if(pfkey_msg->sadb_msg_satype > SADB_SATYPE_MAX) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_msg_interp: satype %d > max %d\n", 
			    pfkey_msg->sadb_msg_satype, SADB_SATYPE_MAX);
		SENDERR(EINVAL);
	}
	
	switch(pfkey_msg->sadb_msg_type) {
	case SADB_GETSPI:
	case SADB_UPDATE:
	case SADB_ADD:
	case SADB_DELETE:
	case SADB_X_GRPSA:
		if(!(tdb1->tdb_proto = sadb_satype2proto[pfkey_msg->sadb_msg_satype])) {
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_msg_interp: satype %d lookup failed.\n", 
				    pfkey_msg->sadb_msg_satype);
			SENDERR(EINVAL);
		}
		break;
	default:
	}
	
	/* The NULL below causes the default extension parsers to be used */
	/* Parse the extensions */
	if((error = pfkey_msg_parse(pfkey_msg, NULL, extensions, EXT_BITS_IN)))
	{
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_msg_interp: message parsing failed with error %d.\n",
			    error); 
		SENDERR(-error);
	}
	
	/* Process the extensions */
	for(i=1; i <= SADB_EXT_MAX;i++)	{
		if(extensions[i] != NULL) {
			KLIPS_PRINT(debug_pfkey,
				    "klips_debug:pfkey_msg_interp: "
				    "processing ext %d %p with processor %p.\n", 
				    i, extensions[i], ext_processors[i]);
			if((error = ext_processors[i](extensions[i],tdb1,tdb2))) {
				KLIPS_PRINT(debug_pfkey,
					    "klips_debug:pfkey_msg_interp: extension processing for type %d failed with error %d.\n",
					    i,error); 
				SENDERR(-error);
			}
			
		}
		
	}
	
	/* Parse the message types */
	KLIPS_PRINT(debug_pfkey,
		    "klips_debug:pfkey_msg_interp: "
		    "parsing message type %d with msg_parser %p.\n",
		    pfkey_msg->sadb_msg_type,
		    msg_parsers[pfkey_msg->sadb_msg_type]); 
	if((error = msg_parsers[pfkey_msg->sadb_msg_type](sk, extensions, tdb1, tdb2))) {
		KLIPS_PRINT(debug_pfkey,
			    "klips_debug:pfkey_msg_interp: message parsing failed with error %d.\n",
			    error); 
		SENDERR(-error);
	}
	
 errlab:
	return(error);
}

/*
 * $Log: pfkey_v2_parser.c,v $
 * Revision 1.24  1999/12/09 23:23:00  rgb
 * Added check to pfkey_sa_process() to do eroutes.
 * Converted to DIVUP() macro.
 * Converted if() to switch() in pfkey_register_parse().
 * Use new pfkey_extensions_init() instead of memset().
 *
 * Revision 1.23  1999/12/01 22:18:13  rgb
 * Preset minspi and maxspi values in case and spirange extension is not
 * included and check for the presence of an spirange extension before
 * using it.  Initialise tdb_sastate to LARVAL.
 * Fixed debugging output typo.
 * Fixed authentication context initialisation bugs (4 places).
 *
 * Revision 1.22  1999/11/27 11:53:08  rgb
 * Moved pfkey_msg_parse prototype to pfkey.h
 * Moved exts_permitted/required prototype to pfkey.h.
 * Moved sadb_satype2proto protocol lookup table to lib/pfkey_v2_parse.c.
 * Deleted SADB_X_EXT_SA2 code from pfkey_sa_process() since it will never
 * be called.
 * Moved protocol/algorithm checks to lib/pfkey_v2_parse.c
 * Debugging error messages added.
 * Enable lifetime_current checking.
 * Remove illegal requirement for SA extension to be present in an
 * originating GETSPI call.
 * Re-instate requirement for UPDATE or ADD message to be MATURE.
 * Add argument to pfkey_msg_parse() for direction.
 * Fixed IPIP dst address bug and purged redundant, leaky code.
 *
 * Revision 1.21  1999/11/24 05:24:20  rgb
 * hanged 'void*extensions' to 'struct sadb_ext*extensions'.
 * Fixed indention.
 * Ditched redundant replay check.
 * Fixed debug message text from 'parse' to 'process'.
 * Added more debug output.
 * Forgot to zero extensions array causing bug, fixed.
 *
 * Revision 1.20  1999/11/23 23:08:13  rgb
 * Move all common parsing code to lib/pfkey_v2_parse.c and rename
 * remaining bits to *_process. (PJO)
 * Add macros for dealing with alignment and rounding up more opaquely.
 * Use provided macro ADDRTOA_BUF instead of hardcoded value.
 * Sort out pfkey and freeswan headers, putting them in a library path.
 * Corrected a couple of bugs in as-yet-inactive code.
 *
 * Revision 1.19  1999/11/20 22:01:10  rgb
 * Add more descriptive error messages for non-zero reserved fields.
 * Add more descriptive error message for spirange parsing.
 * Start on supported extension parsing.
 * Start on register and get message parsing.
 *
 * Revision 1.18  1999/11/18 04:09:20  rgb
 * Replaced all kernel version macros to shorter, readable form.
 *
 * Revision 1.17  1999/11/17 15:53:41  rgb
 * Changed all occurrences of #include "../../../lib/freeswan.h"
 * to #include <freeswan.h> which works due to -Ilibfreeswan in the
 * klips/net/ipsec/Makefile.
 *
 * Revision 1.16  1999/10/26 16:57:43  rgb
 * Add shorter macros for compiler directives to visually clean-up.
 * Give ipv6 code meaningful compiler directive.
 * Add comments to other #if 0 debug code.
 * Remove unused *_bh_atomic() calls.
 * Fix mis-placed spinlock.
 *
 * Revision 1.15  1999/10/16 18:27:10  rgb
 * Clean-up unused cruft.
 * Fix-up lifetime_allocations_c and lifetime_addtime_c initialisations.
 *
 * Revision 1.14  1999/10/08 18:37:34  rgb
 * Fix end-of-line spacing to sate whining PHMs.
 *
 * Revision 1.13  1999/10/03 18:49:12  rgb
 * Spinlock fixes for 2.0.xx and 2.3.xx.
 *
 * Revision 1.12  1999/10/01 15:44:54  rgb
 * Move spinlock header include to 2.1> scope.
 *
 * Revision 1.11  1999/10/01 00:05:45  rgb
 * Added tdb structure locking.
 * Use 'jiffies' instead of do_get_timeofday().
 * Fix lifetime assignments.
 *
 * Revision 1.10  1999/09/21 15:24:45  rgb
 * Rework spirange code to save entropy and prevent endless loops.
 *
 * Revision 1.9  1999/09/16 12:10:21  rgb
 * Minor fixes to random spi selection for correctness and entropy conservation.
 *
 * Revision 1.8  1999/05/25 22:54:46  rgb
 * Fix comparison that should be an assignment in an if.
 *
 * Revision 1.7  1999/05/09 03:25:37  rgb
 * Fix bug introduced by 2.2 quick-and-dirty patch.
 *
 * Revision 1.6  1999/05/08 21:32:30  rgb
 * Fix error return reporting.
 *
 * Revision 1.5  1999/05/05 22:02:33  rgb
 * Add a quick and dirty port to 2.2 kernels by Marc Boucher <marc@mbsi.ca>.
 *
 * Revision 1.4  1999/04/29 15:22:40  rgb
 * Standardise an error return method.
 * Add debugging instrumentation.
 * Add check for existence of macros min/max.
 * Add extensions permitted/required in/out filters.
 * Add satype-to-protocol table.
 * Add a second tdb pointer to each parser to accomodate GRPSA.
 * Move AH & no_algo_set to GETSPI, UPDATE and ADD.
 * Add OOO window check.
 * Add support for IPPROTO_IPIP and hooks for IPPROTO_COMP.
 * Add timestamp to lifetime parse.
 * Fix address structure length checking bug.
 * Fix address structure allocation bug (forgot to kmalloc!).
 * Add checks for extension lengths.
 * Add checks for extension reserved illegal values.
 * Add check for spirange legal values.
 * Add an extension type for parsing a second satype, SA and
 * DST_ADDRESS.
 * Make changes to tdb_init() template to get pfkey_tdb_init(),
 * eliminating any mention of xformsw.
 * Implement getspi, update and grpsa (not tested).
 * Add stubs for as yet unimplemented message types.
 * Add table of message parsers to substitute for msg_parse switch.
 *
 * Revision 1.3  1999/04/15 17:58:07  rgb
 * Add RCSID labels.
 *
 * Revision 1.2  1999/04/15 15:37:26  rgb
 * Forward check changes from POST1_00 branch.
 *
 * Revision 1.1.2.1  1999/03/26 20:58:56  rgb
 * Add pfkeyv2 support to KLIPS.
 *
 */
