//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : bigint_matrix.h
// Author      : Patrick Theobald (PT)
// Last change : PT, Jun 28, 1995, initial version (1.8)
//

/*
$Id: bigint_matrix.h,v 1.5 1995/08/01 15:33:15 lidiaadm Exp $
*/

#ifndef LIDIA_BIGINT_MATRIX_H
#define LIDIA_BIGINT_MATRIX_H

#ifndef LIDIA_PRIMELIST_NAME
#define LIDIA_PRIMELIST_NAME "LIDIA_PRIMES"
#endif

#ifndef LIDIA_PRIMELIST_SIZE
#define LIDIA_PRIMELIST_SIZE 300
#endif

#include <LiDIA/bigint.h>

/* PRINT MODE SETTINGS */

#define BEAUTY_MODE      0
#define LIDIA_MODE       1
#define GP_MODE          2
#define MAPLE_MODE       3
#define MATHEMATICA_MODE 4
#define KANT_MODE	 5

class bigint_matrix
{
 protected:	

  long rows;
  long columns;
  bigint **value;
  
  int print_mode;
  
 public:
  
  bigint *mgcd(const bigint *, long);
  bigint *mgcd1(const bigint *, long, bigint_matrix &);
  void mgcd2(bigint &, const bigint *, long);
  bigint *mgcd2(const bigint *, long);
  bigint *mgcd2(const bigint *, long, bigint_matrix &);
  bigint *mgcd3(const bigint *, long, bigint_matrix &);
  bigint *mgcd4(const bigint *, long, bigint_matrix &);
  
  /**
   ** constructors
   **/
  
  bigint_matrix();
  bigint_matrix(long);
  bigint_matrix(long, long);
  bigint_matrix(long, long, long **);
  bigint_matrix(long, long, bigint **);
  bigint_matrix(const bigint_matrix &);
  
  /**
   ** destructor
   **/
  
  ~bigint_matrix();
  
  /**
   ** Input / Output
   **/
  
  friend ostream & operator << (ostream &, const bigint_matrix &);
  friend istream & operator >> (istream &, bigint_matrix &);
  
  /**
   ** set print_mode / get print_mode
   **/
  
  void set_print_mode(int);
  int get_print_mode();

  /**
   ** stream handling - LiDIA--MODE
   **/
  
  void write_to_stream(ostream &) const;
  friend void write_to_stream(const bigint_matrix &, ostream &);

  void read_from_stream(istream &);
  friend void read_from_stream(bigint_matrix &, istream &);

  /**
   ** access functions
   **/

  void sto(long, long, const bigint &);
  void sto_column(const bigint *, long, long);
  void sto_row(const bigint *, long, long);

  bigint operator() (long, long) const;
  bigint member(long, long) const;
  bigint *operator() (long) const;	/* columnvector */
  bigint *column(long) const;
  bigint *operator[] (long) const;	/* rowvector  */
  bigint *row(long) const;

  /**
   ** divide functions
   **/

  void divide(bigint_matrix &, bigint_matrix &, bigint_matrix &, bigint_matrix &) const;
  void divide_h(bigint_matrix &, bigint_matrix &) const;
  void divide_v(bigint_matrix &, bigint_matrix &) const;

  /**
   ** compose functions
   **/

  void compose(const bigint_matrix &, const bigint_matrix &, const bigint_matrix &, const bigint_matrix &);
  void compose_h(const bigint_matrix &, const bigint_matrix &);
  void compose_v(const bigint_matrix &, const bigint_matrix &);

  /**
   ** exchange functions / swap functions
   **/

  friend void swap(bigint_matrix &, bigint_matrix &);
  void swap_columns(long, long);
  void swap_rows(long, long);

  /**
   ** structur functions
   **/

  long get_no_of_rows() const;		/* number of rows  */
  long get_no_of_columns() const;	/* number of columns */

  void set_no_of_rows(long);	/* increase / decrease number of rows    */
  void set_no_of_columns(long);	/* increase / decrease number of columns */

  /**
   ** assignment
   **/

  bigint_matrix & operator = (const bigint_matrix &);
  void assign(const bigint_matrix &);
  friend void assign(bigint_matrix &, const bigint_matrix &);

  /**
   ** BEGIN: matrix arithmetic
   **/

  /**
   ** procedures
   **/

  friend void add(bigint_matrix &, const bigint_matrix &, const bigint_matrix &);
  friend void add(bigint_matrix &, const bigint_matrix &, const bigint &);
  friend void subtract(bigint_matrix &, const bigint_matrix &, const bigint_matrix &);
  friend void subtract(bigint_matrix &, const bigint_matrix &, const bigint &);
  friend void multiply(bigint_matrix &, const bigint_matrix &, const bigint_matrix &);
  friend void multiply(bigint_matrix &, const bigint_matrix &, const bigint &);
  friend void multiply(bigint *, const bigint_matrix &, const bigint *);
  friend void remainder(bigint_matrix &, const bigint_matrix &, const bigint &);
  friend void negate(bigint_matrix &, const bigint_matrix &);

  /**
   ** operators
   **/

  bigint_matrix operator - () const;
  bigint_matrix operator + (const bigint_matrix &) const;
  bigint_matrix operator + (const bigint &) const;
  bigint_matrix operator - (const bigint_matrix &) const;
  bigint_matrix operator - (const bigint &) const;
  bigint_matrix operator * (const bigint_matrix &) const;
  bigint_matrix operator * (const bigint &) const;
  bigint *operator * (const bigint *) const;
  bigint_matrix operator % (const bigint &) const;

  bigint_matrix & operator += (const bigint_matrix &);
  bigint_matrix & operator += (const bigint &);
  bigint_matrix & operator -= (const bigint_matrix &);
  bigint_matrix & operator -= (const bigint &);
  bigint_matrix & operator *= (const bigint_matrix &);
  bigint_matrix & operator *= (const bigint &);
  bigint_matrix & operator %= (const bigint &);

  /**
   ** boolean operators
   **/

  int operator == (const bigint_matrix &) const;
  int equal(const bigint_matrix &) const;
  friend int equal(const bigint_matrix &, const bigint_matrix &);

  int operator != (const bigint_matrix &) const;
  int unequal(const bigint_matrix &) const;
  friend int unequal(const bigint_matrix &, const bigint_matrix &);

  int is_column_zero(long) const;
  friend int is_column_zero(const bigint_matrix &, long);

  int is_row_zero(long) const;
  friend int is_row_zero(const bigint_matrix &, long);

  /**
   ** END: matrix arithmetic
   **/
  
  /**
   ** norms and bounds
   **/
  
  void max(bigint &) const;
  bigint max() const;
  friend bigint max(const bigint_matrix &);

  void max_abs(bigint &) const;
  bigint max_abs() const;
  friend bigint max_abs(const bigint_matrix &);

  void min(bigint &) const;
  bigint min() const;
  friend bigint min(const bigint_matrix &);

  void min_abs(bigint &) const;
  bigint min_abs() const;
  friend bigint min_abs(const bigint_matrix &);

  void hadamard(bigint &) const;
  bigint hadamard() const;
  friend bigint hadamard(const bigint_matrix &);

  void row_norm(bigint &, long, long);
  bigint row_norm(long, long);
  friend bigint row_norm(const bigint_matrix &, long, long);

  void column_norm(bigint &, long, long);
  bigint column_norm(long, long);
  friend bigint column_norm(const bigint_matrix &, long, long);

  /**
   ** randomize
   **/

  void randomize(const bigint &);
  friend bigint_matrix randomize(long, long, const bigint &);

  /**
   ** Chinese remaindering theorem
   **/

  friend bigint chinrest(const bigint *, const bigint *);
  friend void chinrest(bigint &, const bigint *, const bigint *);

  void chinrest(const bigint_matrix *, const bigint *);
  friend bigint_matrix chinrest(const bigint_matrix *, const bigint *);

  /**
   ** List of primes
   **/

  friend bigint *get_primes(const bigint &, const bigint &);

  /**
   ** BEGIN: Linear algebra
   ** PART 1
   **/

  /**
   ** trace
   **/

  void trace(bigint &) const;
  bigint trace() const;
  friend bigint trace(const bigint_matrix &);

  /**
   ** diagonal function
   **/

  void diag(const bigint &, const bigint &);
  friend void diag(bigint_matrix &, const bigint &, const bigint &);

  /**
   ** transpose function
   **/

  void trans(const bigint_matrix &);
  bigint_matrix trans() const;
  friend void trans(bigint_matrix &, const bigint_matrix &);
  friend bigint_matrix trans(const bigint_matrix &);

  /**
   ** regular expansion
   **/

  void regexpansion(const long *);
  friend void regexpansion(bigint_matrix &, const long *);

  /**
   ** rank
   **/

  long rank() const;
  friend long rank(const bigint_matrix &);

  /**
   ** rank and linearly independent rows
   **/

  long *lininr() const;
  friend long *lininr(const bigint_matrix &);

  /**
   ** rank and linearly independent columns
   **/

  long *lininc() const;
  friend long *lininc(const bigint_matrix &);

  /**
   ** adjoint matrix
   **/

  void adj(const bigint_matrix &);
  friend bigint_matrix adj(const bigint_matrix &);

  /**
   ** lattice determinant
   **/

  void latticedet(bigint &) const;
  void latticedet2(bigint &) const;
  void latticedet3(bigint &) const;
  void latticedet4(bigint &) const;
  bigint latticedet() const;
  bigint latticedet2() const;
  friend bigint latticedet(const bigint_matrix &);
  friend bigint latticedet2(const bigint_matrix &);

  /**
   ** determinant
   **/

  void det(bigint &) const;
  bigint det() const;
  friend bigint det(const bigint_matrix &);

  /**
   ** characteristic polynomial
   **/

  bigint *charpoly() const;
  friend bigint *charpoly(const bigint_matrix &);

  /**
   ** END: Linear algebra
   ** PART 1
   **/

  /**
   ** stream handling - other modes
   **/

  void write_to_gp(ostream &) const;
  void read_from_gp(istream &);

  void write_to_maple(ostream &) const;
  void read_from_maple(istream &);

  void write_to_mathematica(ostream &) const;
  void read_from_mathematica(istream &);

  void write_to_kash(ostream &) const;
  void read_from_kash(istream &);

  /**
   ** BEGIN: Linear algebra
   ** PART 2
   **/

  /**
   ** Hermite normal form
   **/

  /* BEGIN: INTERFACE */
  inline void hnf() { hnf_havas(); }
  inline friend void hnf(bigint_matrix &A) { A.hnf_havas(); }
  inline void hnf(bigint_matrix &A) { hnf_havas(A); }
  inline friend void hnf(bigint_matrix &A, bigint_matrix &TR) { A.hnf_havas(TR); }
  /* END: INTERFACE */

  void hnf_simple();
  void hnf_simple(bigint_matrix &);
  friend void hnf_simple(bigint_matrix &, bigint_matrix &);
  
  void hnf_havas();
  void hnf_havas2();
  void hnf_havas3();
  void hnf_havas(bigint_matrix &);
  friend void hnf_havas(bigint_matrix &, bigint_matrix &);

  void hnfmod_dkt();
  void hnfmod_dkt(const bigint &);
  friend void hnfmod_dkt(bigint_matrix &);
  friend void hnfmod_dkt(bigint_matrix &, const bigint &);

  void hnfmod_cohen();
  void hnfmod_cohen(const bigint &);
  friend void hnfmod_cohen(bigint_matrix &);
  friend void hnfmod_cohen(bigint_matrix &, const bigint &);

  void hnfmod_mueller(bigint_matrix &);
  friend void hnfmod_mueller(bigint_matrix &, bigint_matrix &);

  /**
   ** Kernel
   **/

  void kernel(const bigint_matrix &);
  friend bigint_matrix kernel(const bigint_matrix &);

  void kernel2(const bigint_matrix &);
  friend bigint_matrix kernel2(const bigint_matrix &);

  /**
   ** regular Invimage
   **/

  void reginvimage(const bigint_matrix &, const bigint_matrix &);
  friend bigint_matrix reginvimage(const bigint_matrix &, const bigint_matrix &);

  void reginvimage2(const bigint_matrix &, const bigint_matrix &);
  friend bigint_matrix reginvimage2(const bigint_matrix &, const bigint_matrix &);

  /**
   ** Image
   **/

  void image(const bigint_matrix &);
  friend bigint_matrix image(const bigint_matrix &);

  void image2(const bigint_matrix &);
  friend bigint_matrix image2(const bigint_matrix &);

  /**
   ** InvImage
   **/

  void invimage(const bigint_matrix &, const bigint *);
  friend bigint_matrix invimage(const bigint_matrix &, const bigint *);

  /**
   ** solve
   **/

  void solve(const bigint_matrix &, const bigint *);
  friend bigint_matrix solve(const bigint_matrix &, const bigint *);

  /**
   ** Smith normal form
   **/

  /* BEGIN: INTERFACE */
  inline void snf() { snf_havas(); }
  inline friend void snf(bigint_matrix &A) { A.snf_havas(); }
  inline void snf(bigint_matrix &A, bigint_matrix &B) { snf_havas(A,B); }
  inline friend void snf(bigint_matrix &A, bigint_matrix &B, bigint_matrix &C) { A.snf_havas(B,C); }
  /* END: INTERFACE */

  void snf_hartley();
  void snf_hartley(bigint_matrix &, bigint_matrix &);
  friend void snf_hartley(bigint_matrix &);
  friend void snf_hartley(bigint_matrix &, bigint_matrix &, bigint_matrix &);

  void snf_simple();
  void snf_simple(bigint_matrix &, bigint_matrix &);
  friend void snf_simple(bigint_matrix &);
  friend void snf_simple(bigint_matrix &, bigint_matrix &, bigint_matrix &);

  void snf_havas();
  void snf_havas(bigint_matrix &, bigint_matrix &);
  friend void snf_havas(bigint_matrix &);
  friend void snf_havas(bigint_matrix &, bigint_matrix &, bigint_matrix &); 

  void snf_mult(long art = 1);
  void snf_mult(bigint_matrix &, bigint_matrix &, long art = 1);
  friend void snf_mult(bigint_matrix &, long art = 1);
  friend void snf_mult(bigint_matrix &, bigint_matrix &, bigint_matrix &, long art = 1);

  void snf_add(long art = 1);
  void snf_add(bigint_matrix &, bigint_matrix &, long art =1);
  friend void snf_add(bigint_matrix &, long art = 1);
  friend void snf_add(bigint_matrix &, bigint_matrix &, bigint_matrix &, long art = 1);
 
  void snf_new(long art = 1);
  void snf_new(bigint_matrix &, bigint_matrix &, long art = 1);
  friend void snf_new(bigint_matrix &, long art = 1);
  friend void snf_new(bigint_matrix &, bigint_matrix &, bigint_matrix &, long art = 1);

  void snfmod_dkt();
  void snfmod_dkt(const bigint &);
  friend void snfmod_dkt(bigint_matrix &);
  friend void snfmod_dkt(bigint_matrix &, const bigint &);

  void snfmod_cohen();
  void snfmod_cohen(const bigint &);
  friend void snfmod_cohen(bigint_matrix &);
  friend void snfmod_cohen(bigint_matrix &, const bigint &);

  /**
   ** END: Linear algebra
   ** PART 2
   **/

  void gauss();
};

#endif

