//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : bigmod.h 
// Author      : Thomas Papanikolaou (TP)
// Last change : TP, Jan 29 1995, initial version 
//               VM, Feb 7 1995, modified invert() to invert(int)
//

#ifndef LIDIA_BIGMOD_H
#define LIDIA_BIGMOD_H

#include <LiDIA/bigint.h>

class bigmod
{
  /**
  ** the C type we use to represent a bigmod
  **/

  bigint I;
  static bigint M;

  void normalize();

            public:


  /**
  ** constructors and destructor; we could leave out some of these
  **/

            bigmod();
            bigmod(int i);
            bigmod(long l);
            bigmod(unsigned long ul);
            bigmod(double d);
            bigmod(const bigint & i);
            bigmod(const bigmod & a);
           ~bigmod();

  /**
  ** inline member functions
  **/

  int length() const;
  int bit_length() const;
  int is_zero() const;
  int is_one() const;
  int intify(int &i) const;
  int longify(long &i) const;

  bigint invert(int = 0);
  void negate();
  void assign_zero();
  void assign_one();
  void assign(int i);
  void assign(long i);
  void assign(unsigned long ui);
  void assign(const bigint & a);
  void assign(const bigmod & a);
  void multiply_by_2();
  void divide_by_2();

  static void set_modulus(const bigint &m);
  const bigint & mantissa() const;
  static const bigint & modulus();

  /**
  ** type checking
  **/

  friend int is_char(const bigmod & a);
  friend int is_uchar(const bigmod & a);
  friend int is_short(const bigmod & a);
  friend int is_ushort(const bigmod & a);
  friend int is_int(const bigmod & a);
  friend int is_uint(const bigmod & a);
  friend int is_long(const bigmod & a);
  friend int is_ulong(const bigmod & a);

	
  /**
  ** assignments
  **/

  int operator = (int i);
  long operator = (long l);
  unsigned long operator = (unsigned long ul);
  double operator = (double d);
  bigint operator = (const bigint & a);
  bigmod & operator = (const bigmod & a);

  /**
  ** comparisons
  **/

  friend int operator == (const bigmod & a, const bigmod & b);
  friend int operator != (const bigmod & a, const bigmod & b);

  /**
  ** operator overloading
  **/

  friend bigmod operator - (const bigmod & a);
  friend bigmod operator + (const bigmod & a, const bigmod & b);
  friend bigmod operator - (const bigmod & a, const bigmod & b);
  friend bigmod operator * (const bigmod & a, const bigmod & b);
  friend bigmod operator / (const bigmod & a, const bigmod & b);

  bigmod & operator += (const bigmod & a);
  bigmod & operator -= (const bigmod & a);
  bigmod & operator *= (const bigmod & a);
  bigmod & operator /= (const bigmod & a);

  bigmod & operator++ ();
  bigmod & operator-- ();
  int operator ! ();

  /**
  ** Procedural versions
  **/

  friend void invert(bigmod & a, const bigmod & b);
  friend void negate(bigmod & a, const bigmod & b);
  friend void multiply_by_2(bigmod & a, const bigmod & b);
  friend void divide_by_2(bigmod & a, const bigmod & b);
  friend void add(bigmod & c, const bigmod & a, const bigmod & b);
  friend void subtract(bigmod & c, const bigmod & a, const bigmod & b);
  friend void multiply(bigmod & c, const bigmod & a, const bigmod & b);
  friend void divide(bigmod & c, const bigmod & a, const bigmod & b);
  friend void power(bigmod & c, const bigmod & a, const bigint & b);
  friend void power(bigmod & c, const bigmod & a, long i);
  friend void inc(bigmod & c);
  friend void dec(bigmod & c);

  friend void add(bigmod & c, const bigmod & a, long i);
  friend void subtract(bigmod & c, const bigmod & a, long i);
  friend void multiply(bigmod & c, const bigmod & a, long i);
  friend void divide(bigmod & c, const bigmod & a, long i);

  friend void add(bigmod & c, long i, const bigmod & a);
  friend void subtract(bigmod & c, long i, const bigmod & a);
  friend void multiply(bigmod & c, long i, const bigmod & a);
  friend void divide(bigmod & c, long i, const bigmod & a);

  /**
  ** functions
  **/

  friend bigint mantissa(const bigmod & a);
  friend bigmod inverse(const bigmod & a);
  friend void seed(const bigmod & a);
  friend bigmod randomize();
  friend bigmod randomize(const bigmod & a);
  friend double dbl(const bigmod & a);

  friend void normalize(bigmod & a, const bigmod & b);
  friend void square(bigmod & a, const bigmod & b);
  friend void swap(bigmod & a, bigmod & b);

  /**
  ** input / output
  **/

  friend istream & operator >> (istream & in, bigmod & a);
  friend ostream & operator << (ostream & out, const bigmod & a);

  friend int string_to_bigmod(char *s, bigmod & a);
  friend int bigmod_to_string(const bigmod & a, char *s);

  /**
  ** using fread/fwrite
  **/

  void read_from_file(FILE * fp);
  void write_to_file(FILE * fp);

  /**
  ** using fscanf/fprintf
  **/

  void scan_from_file(FILE * fp);
  void print_to_file(FILE * fp);

};

#endif
