

//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : file_vector.h 
// Author      : Frank Lehmann (FL), Markus Maurer (MM),
//               Thomas Papanikolaou (TP)
// Last change : FL/MM, Feb 15 1995, initial version
//               FL/MM, May 10 1995, clean-up
//		 FL/MM, May 15 1995, changed type of size variables 
//				     into base_vector<T>::size_type
//		 FL/MM, Jul  6 1995, replaced base_vector<T>::size_type by lidia_size_t
//               FL/MM, Jul 24 1995, added 'const' to T* parameters of constructors



// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// *
// *    File        :  file_vector.h
// *
// *    Description :  -) definition of template class 'file_vector<T>'
// *                      to allow use of vectors with file-I/O-functions
// *
// *                   -) virtual base class is 'base_vector<T>'
// *
// *                   -) uses template class 'file_io_class<T>'
// *
// *		       -) requirements on type T are given by those
// *		          of the base class and the file_io_class
// *
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *



#ifndef LIDIA_FILE_VECTOR_H
#define LIDIA_FILE_VECTOR_H


#include <unistd.h>
#include <LiDIA/lidia.h>
#include <LiDIA/base_vector.h>
#include <LiDIA/file_io_class.h>


template <class T> class file_vector : virtual public base_vector < T >

 {

   /*

   public :

   // *****  constructor  functions  *****

   file_vector () ;
   file_vector (  char md ) ;
    
   file_vector (  lidia_size_t all ) ;
   file_vector (  lidia_size_t all,  char md ) ;
    
   file_vector (  lidia_size_t all,  lidia_size_t len ) ;
   file_vector (  lidia_size_t all,  lidia_size_t len,  char md ) ;
    
   file_vector ( const base_vector < T > & v ) ;
   file_vector ( const base_vector < T > & v,  char md ) ;
    
   file_vector ( const T *v,  lidia_size_t len ) ;
   file_vector ( const T *v,  lidia_size_t len,  char md ) ;


   // ***** destructor *****

   LiDIA_VEC_VIRTUAL ~file_vector() ;


   // ***** assignment *****

   file_vector<T> & operator= ( const file_vector<T> & v ) ;


   // ***** reading from / writing to file *****

   int write_to_file ( FILE *fp )  const ;         // writes the vector to file in binary-format

   int read_from_file ( FILE *fp ) ;               // reads a vector from file in binary-format 

   int append_b ( FILE *fp )  const ;              // appends the vector to file in binary format

   int append_b ( FILE *fp, lidia_size_t n )  const ;	   // appends the (n+1)-st element of the vector to file

   int print_to_file ( FILE *fp )  const ;         // writes the vector to file in ASCII-format  

   int scan_from_file ( FILE *fp ) ;               // reads a vector from file in ASCII-format  

   int append_a ( FILE *fp )  const ;              // appends the vector to file in ASCII-format

   int append_a ( FILE *fp, lidia_size_t n )  const ;	   // appends the (n+1)-st element of the vector to file


   */




   /************************************************************************************/
   /*      		 declaration and definition of functions 		       */
   /************************************************************************************/

   public :

   // *****  constructor  functions  *****

   file_vector () : base_vector< T > ( FIXED )     
    {
       debug_handler ( "file_vector", "constructor ()" ) ;
    }
    
   file_vector ( char md ) : base_vector< T > ( md )     
    {
       debug_handler ( "file_vector", "constructor ( char )" ) ;
    }

    
   file_vector ( lidia_size_t all ) : base_vector< T > ( all, FIXED ) 
    {
       debug_handler ( "file_vector", "constructor ( lidia_size_t )" ) ;
    }
    
   file_vector ( lidia_size_t all, char md ) : base_vector< T > ( all, md ) 
    {
       debug_handler ( "file_vector", "constructor ( lidia_size_t ,  char )" ) ;
    }

    
   file_vector ( lidia_size_t all, lidia_size_t len ) : base_vector< T > ( all, len, FIXED )  
    {
       debug_handler ( "file_vector", "constructor ( lidia_size_t, lidia_size_t )" ) ;
    }

   file_vector ( lidia_size_t all, lidia_size_t len, char md ) : base_vector< T > ( all, len, md )  
    {
       debug_handler ( "file_vector", "constructor ( lidia_size_t, lidia_size_t, char )" ) ;
    }

    
   file_vector ( const base_vector < T > & v ) : base_vector< T > ( v, FIXED )  
    {
       debug_handler ( "file_vector", "constructor ( const base_vector < T > & )" ) ;
    }
    
   file_vector ( const base_vector < T > & v, char md ) : base_vector< T > ( v, md )  
    {
       debug_handler ( "file_vector", "constructor ( const base_vector < T > &, char  )" ) ;
    }

    
   file_vector ( const T *v, lidia_size_t len ) : base_vector< T > ( v, len, FIXED )  
    {
       debug_handler ( "file_vector", "constructor ( const T *, lidia_size_t )" ) ;
    }
    
   file_vector ( const T *v, lidia_size_t len, char md ) : base_vector< T > ( v, len, md )  
    {
       debug_handler ( "file_vector", "constructor ( const T *, lidia_size_t, char )" ) ;
    }


   // ***** destructor *****

   LiDIA_VEC_VIRTUAL ~file_vector ()
    {
       debug_handler ( "file_vector", "destructor()" ) ;

       // ***   memory deallocation is done by
       // ***   base_vector - destructor
    }


   // ***** assignment *****

   file_vector<T> & operator= ( const file_vector<T> & v )
    {
       debug_handler ( "file_vector", "operator=" ) ;

       if ( &v != this )

          base_vector<T>::operator= ( v ) ;

       return ( *this ) ;
    }




   // *****  writing to file in binary-format *****

   int write_to_file ( FILE *fp ) const
    {
       debug_handler ( "file_vector", "write_to_file (FILE*)" ) ;

       lidia_size_t i ;

       fwrite ( &length, sizeof (lidia_size_t), 1, fp ) ;

       for ( i = 0 ; i < length ; i++ )
        {
           file_io_class< T >::write_to_file ( data[i], fp ) ;
        }

       return (0) ;
    }


   // ***** reading from file in binary-format *****

   int read_from_file ( FILE *fp )
    {
       debug_handler ( "file_vector", "read_from_file (FILE*)" ) ;

       lidia_size_t l, i ;

       int rc = 0 ;


       fread ( &l, sizeof (lidia_size_t), 1, fp ) ;

       if ( l > 0 )
        {
          if ( set_capacity ( l ) )

               rc = 1 ;
          else
           {
              set_size ( l ) ;

              for ( i = 0 ; i < l && ! feof(fp) ; i++ )
               {
                   file_io_class< T >::read_from_file ( data[i], fp ) ;
               }

              if ( i < l )
               {
                  lidia_error_handler ( "file_vector::read_from_file",
			          "Wrong file-format." ) ;
		  rc = 1 ;
	       }
           }
        }
       else
        {
           set_size ( 0 ) ;
        }

       return (rc) ;
    }



    // ***** appending one element to file in binary-format  *****

    // assumes a vector to be the last element in the file ;
    // if there is no vector to append to, a new vector containing the
    // (n+1)-st element will be appended to the file.

    int append_b ( FILE *fp, lidia_size_t n ) const
     {
        debug_handler ( "file_vector", "append_b (FILE*, lidia_size_t)" ) ;

        lidia_size_t  l ;
	long old_fpos ;

        if ( n >= length )
         {
           lidia_error_handler ( "file_vector", "append_b (FILE*, lidia_size_t)::index out of range" ) ;
           return (1) ;
         }
	else
	 {
	   old_fpos = ftell ( fp ) ;

	   fread ( &l, sizeof (lidia_size_t), 1, fp ) ;


	   // no vector was found

	   if ( feof ( fp ) )
	    {
	       l = 1 ;
	       fwrite ( &l, sizeof (lidia_size_t), 1, fp ) ;
	       file_io_class< T >::write_to_file ( data[n], fp ) ;
	    }

	   // append to existing vector

	   else
	    {
	       // increment length

	       l += 1 ;

	       // store new length 

	       fseek  ( fp, - sizeof (lidia_size_t), SEEK_CUR ) ;
	       fwrite ( &l, sizeof (lidia_size_t), 1 , fp ) ;

	       // append the element to the end of the vector

	       fseek ( fp, 0, SEEK_END ) ;
	       file_io_class< T >::write_to_file ( data[n], fp ) ;
	     }

	   // reset file-pointer

	   fseek ( fp, old_fpos, SEEK_SET ) ;

	   return (0) ;
         }
     }



    // ***** appending the vector to file in binary-format  *****

    // assumes a vector to be the last element in the file ;
    // if there is no vector to append to, the appending-operation
    // works like a write_to_file()-operation.

    int append_b ( FILE *fp ) const
     {
        debug_handler ( "file_vector", "append_b (FILE*)" ) ;

        lidia_size_t  l ;
        lidia_size_t  i ;
	long old_fpos ;

	
	old_fpos = ftell ( fp ) ;

	fread  ( &l, sizeof (lidia_size_t), 1, fp ) ;


	// no vector to append to

	if ( feof ( fp ) ) 
	 {
	    fwrite ( &length, sizeof (lidia_size_t), 1, fp ) ;

	    for ( i = 0 ; i < length ; i++ )
	     {
	       file_io_class< T >::write_to_file ( data[i], fp ) ;
	     }	   
	 }

	// append to the existing vector

	else
	 {
	    // increment the length

	    l += length ;

	    // store the new length

	    fseek  ( fp, - sizeof (lidia_size_t), SEEK_CUR ) ;
	    fwrite ( &l, sizeof (lidia_size_t), 1 , fp ) ;

	    // append the vector to the end of the vector in the file

	    fseek ( fp, 0, SEEK_END ) ;

	    for ( i = 0 ; i < length ; i++ )

                file_io_class< T >::write_to_file ( data[i], fp ) ;
	  }


	// reset file-pointer

	fseek ( fp, old_fpos, SEEK_SET ) ;

	return (0) ;
     }




   // *****  printing to file in ASCII-format  *****

   int print_to_file ( FILE *fp ) const
    {
       debug_handler ( "file_vector", "print_to_file (FILE*)" ) ;

       lidia_size_t i ;

       fprintf ( fp, "[ " ) ;

       for ( i = 0 ; i < length ; i++ )
        {
           file_io_class< T >::print_to_file ( data[i], fp ) ;
	   fprintf ( fp, " " ) ;
        }

       fprintf ( fp, "]" ) ;

       return (0) ;
    }



   // *****  scanning from file in ASCII-format  *****

   int scan_from_file ( FILE *fp ) 
    {
       debug_handler ( "file_vector", "scan_from_file (FILE*)" ) ;

       lidia_size_t i ;
       int  end	   ;
       long fpos   ;
       char c      ;

       base_vector<T> v(EXPAND) ;

       c = ' ' ;

       while ( !feof ( fp ) && c != '[' )
        {
             fscanf ( fp, "%c", &c ) ;
	}

       if ( feof ( fp ) )
	{
          return (1) ;
	}
       else
        {
	   i   = 0 ;
	   end = 0 ;

	   while ( !end )
	    {
	       c = ' ' ;
	       
	       while ( !feof ( fp ) && c == ' ' )
		{
		   fpos = ftell ( fp ) ;
		   fscanf ( fp, "%c", &c ) ;
		}

	       if ( !feof ( fp ) && c != ']' )
		{
		   fseek ( fp, fpos, SEEK_SET ) ;
		   file_io_class< T >::scan_from_file ( v[i], fp ) ;
		   i++ ;
		}
	       else
		{
		   end = 1 ;
		}
            }

           if ( c == ']' )
	    {
	      *this = v  ;
              return (0) ;
	    }
           else
	    {
              return (1) ;
	    }

        } // end of if feof ( fp )

    } // end of scan_from_file()



   // ***** appending one element to file in ASCII-format  *****

   // assumes a vector to be the last element in the file ;
   // if there is no vector to append to, a new vector containing the
   // (n+1)-st element will be appended to the file.


   int append_a ( FILE *fp, lidia_size_t n ) const
    {
       debug_handler ( "file_vector", "append_a (FILE*, lidia_size_t)" ) ;

       char c    ;
       int  end  ;
       int  ret_val ;

       long fpos ;
       long old_fpos ;

       T    tmp  ;


       // data[n] not accessible

       if ( n >= length )
	{
	   return (1) ;
	}


       old_fpos = ftell ( fp ) ;

       fscanf ( fp, "%c", &c ) ;


       // no vector was found

       if ( feof ( fp ) )
	{
	   fprintf ( fp, "[ " ) ;
	   file_io_class< T >::print_to_file ( data[n], fp ) ;
	   fprintf ( fp, " ]" ) ;

	   // reset file-pointer

	   fseek ( fp, old_fpos, SEEK_SET ) ;

	   return (0) ;
	}

       // append to existing vector

       // read blanks

       while ( !feof ( fp ) && c != '[' )
        {
             fscanf ( fp, "%c", &c ) ;
	}

       if ( feof ( fp ) )
	{
	  ret_val = 1 ;
	}


       // read vector

       else
        {
	   end = 0 ;

	   while ( !end )
	    {
	       c = ' ' ;
	       
	       while ( !feof ( fp ) && c == ' ' )
		{
		   fpos = ftell ( fp ) ;
		   fscanf ( fp, "%c", &c ) ;
		}

	       if ( !feof ( fp ) && c != ']' )
		{
		   fseek ( fp, fpos, SEEK_SET ) ;
		   file_io_class< T >::scan_from_file ( tmp, fp ) ;
		}
	       else
		{
		   end = 1 ;
		}
            }

           if ( c == ']' )
	    {
	       fseek ( fp, fpos, SEEK_SET ) ;
	       file_io_class< T >::print_to_file ( data[n], fp ) ;
	       fprintf ( fp, " ]" ) ;

	       ret_val = 0 ;
	    }
	   else
	    {
	       ret_val = 1 ;
	    }

        } // end of if ( feof ( fp ) ) 


	// reset file-pointer

	fseek ( fp, old_fpos, SEEK_SET ) ;

        return ( ret_val ) ;

     } // end of append_a




   // ***** appending the vector to file in binary-format  *****

   // assumes a vector to be the last element in the file ;
   // if there is no vector to append to, the appending-operation
   // works like a print_to_file()-operation.

   int append_a ( FILE *fp ) const
    {
       debug_handler ( "file_vector", "append_a (FILE*)" ) ;

       char c    ;
       lidia_size_t i ;
       int  end  ;
       int  ret_val ;

       long old_fpos ;
       long fpos ;

       T    tmp  ;


       old_fpos = ftell ( fp ) ;

       fscanf ( fp, "%c", &c ) ;


       // no vector was found

       if ( feof ( fp ) ) 
	{
 	   fprintf ( fp, "[ " ) ;

	   for ( i = 0 ; i < length ; i++ )
	    {
	      file_io_class< T >::print_to_file ( data[i], fp ) ;
	      fprintf ( fp, " " ) ;
	    }

	   fprintf ( fp, "]" ) ;
	   
	   // reset file-pointer

	   fseek ( fp, old_fpos, SEEK_SET ) ;

	   return (0) ;
	}

       // append to existing vector


       // read blanks

       while ( !feof ( fp ) && c != '[' )
        {
             fscanf ( fp, "%c", &c ) ;
	}

       if ( feof ( fp ) )
	{
          ret_val = 1 ;
	}


       // read vector

       else
        {
	   end = 0 ;

	   while ( !end )
	    {
	       c = ' ' ;
	       
	       while ( !feof ( fp ) && c == ' ' )
		{
		   fpos = ftell ( fp ) ;
		   fscanf ( fp, "%c", &c ) ;
		}

	       if ( !feof ( fp ) && c != ']' )
		{
		   fseek ( fp, fpos, SEEK_SET ) ;
		   file_io_class< T >::scan_from_file ( tmp, fp ) ;
		}
	       else
		{
		   end = 1 ;
		}
            }

           if ( c == ']' )
	    {
	       fseek ( fp, fpos, SEEK_SET ) ;

	       for ( i = 0 ; i < length ; i++ )
		{
		   file_io_class< T >::print_to_file ( data[i], fp ) ;
		   fprintf ( fp, " " ) ;
		}

	       fprintf ( fp, "]" ) ;

	       ret_val = 0 ;
	    }
	   else
	    {
	       ret_val = 1 ;
	    }

        } // end of if ( feof ( fp ) ) 


	// reset file-pointer

	fseek ( fp, old_fpos, SEEK_SET ) ;

        return ( ret_val ) ;

     } // end of append_a
     

 } ;  // end of class file_vector<T>

#endif


