//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : lattice_gensys.h 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 14 1995, initial version
//

#ifndef _lattice_gensys_h
#define _lattice_gensys_h

//
// include files needed
//

class lattice_basis;

#include <LiDIA/double_lattice.h>
#include <LiDIA/bigint_lattice.h>
#include <LiDIA/bigfloat_lattice.h>


//
// computing mode defines
//

#define double_mode          0x00000001
#define bigint_mode          0x00000010
#define bigfloat_mode        0x00000100

// 
// IO mode defines
//

#define own_io_mode          0x00000001
#define pari_io_mode         0x00000010
#define maple_io_mode        0x00000100
#define mathematica_io_mode  0x00001000

// 
// Class lattice_gensys
//

class lattice_gensys
{
  public :
    friend            lattice_basis;
    bool              lattice_read;

    long               lattice_rows;        // dimension of
    long               lattice_cols;        // the lattice

    long               prec;

    double_lattice   *ldo;
    bigint_lattice   *lbi;
    bigfloat_lattice *lbf;

    long              comp_mode;
    long              io_mode;

    unsigned long     red_count;

    timer             T;                // time needed for reduction

    double            y_param;         // quality parameter for reduction
    long              y_zaehler;
    long              y_nenner;

  public :

//
// Constructors / Destructor
//

    lattice_gensys(long, long);
    lattice_gensys(long, long, double**);
    lattice_gensys(long, long, bigint**);
    lattice_gensys(long, long, bigfloat**);
    lattice_gensys(const lattice_gensys&);
    lattice_gensys(const bigint_matrix&);
    virtual ~lattice_gensys();

//
// Input / Output 
//

    friend istream& operator >> (istream&, lattice_gensys&);
    friend ostream& operator << (ostream&, lattice_gensys);

//
// Assignments
//

    lattice_gensys& operator = (const lattice_gensys&);
    void assign(const lattice_gensys&);
   
    lattice_gensys& operator = (const bigint_matrix&);
    void assign(const bigint_matrix&);

//
// Element Operations
//

    void member(long, long, double&);
    void member(long, long, bigint&);
    void member(long, long, bigfloat&);

    void sto(long, long, const double&);
    void sto(long, long, const bigint&);
    void sto(long, long, const bigfloat&);

    void member(long&, long&, double**&);
    void member(long&, long&, bigint**&);
    void member(long&, long&, bigfloat**&);

    void sto(long, long, double**);
    void sto(long, long, bigint**);
    void sto(long, long, bigfloat**);

    long get_no_of_rows();
    long get_no_of_columns();

//
// Type Checking
//

   int check_double();
   int check_bigint();

//
// Modes Operations
//

    void set_computing_mode(long);
    long get_computing_mode(); 
 
    void set_io_mode(long);
    long get_io_mode();
  
//
// Quality of Reduction
//
  
    void set_computing_precision(long);
    long get_computing_precision();

    long get_read_precision();
     
    void set_reduction_parameter(double);
    void set_reduction_parameter(long, long);
    double get_reduction_parameter();
    void get_reduction_parameter(long&, long&);

    unsigned long get_redsteps();
//
// Time needed
//

    timer& get_computing_time();

//
// Algorithms
//

   void lin_gen_system(lattice_gensys&, long&);
   friend void lin_gen_system(bigint_matrix&, const bigint_matrix&, long& r);
   void mlll();
   void mlll(const lattice_gensys&);

   void compute_basis(const lattice_gensys&, const lattice_gensys&);
   
  protected :
// Precision needed for computation with floating polong numbers
// by Jutta Bartholomes
    virtual void compute_precision();
    long compute_read_precision();

// Conversion GP
    void read_from_Pari(char*);
    void write_to_Pari(char*&);
// Conversion Maple
    void read_from_Maple(char*);
    void write_to_Maple(char*&);
// Conversion Mathematica
    void read_from_Mathematica(char*);    
    void write_to_Mathematica(char*&);

    virtual void check_basis() {};
};

#endif
