//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : bm_arith.c
// Author      : Patrick Theobald (PT)
// Last change : PT, Jun 28 1995, initial version
//

/*
$Id: bm_arith.c,v 1.3 1995/08/07 08:30:51 lidiaadm Exp $
*/

#include <LiDIA/bigint_matrix.h>

/**
 ** BEGIN: modular arithmetic
 **/

/**
 ** base operations
 **/

inline void 
pos_div_rem(bigint &q, bigint &r, const bigint &a, const bigint &b)
{
  debug_handler("bigint_matrix","in inline pos_div_rem()");

  div_rem(q,r,a,b);
  if (r.is_lt_zero())
    if (b.is_lt_zero()) 
      {
	subtract(r,r,b);
	inc(q);
      }
    else 
      {
	add(r,r,b);
	dec(q);
      }
}

inline void 
pos_div_rem_long(long &q, long &r, long a, long b) 
{
  debug_handler("bigint_matrix","in inline pos_div_rem_long()");

  q = a / b; 
  r = a - q * b;
  if (r < 0)
    if (b < 0) 
      {
	r -= b;
	q++;
      }
    else 
      {
	r += b;
	q--;
      }
}

inline void 
best_remainder(bigint &a, const bigint &b, const bigint &mod) 
{
  debug_handler("bigint_matrix","in inline best_remainder()");

  if (mod.is_one() || b.is_zero()) 
    a.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      remainder(a,b,mod);
      if (a <= -mod2) 
	add(a, a, mod);
      if (a > mod2) 
	subtract(a, a, mod);
    }
}

inline void 
best_remainder_long(long &a, const bigint &b, long mod) 
{
  debug_handler("bigint_matrix","in inline best_remainder_long()");

  if (mod==1 || b.is_zero()) 
    a = 0;
  else 
    {
      register long mod2 = mod/2;
      remainder(a, b, mod);
      if (a <= -mod2) 
	a += mod;
      if (a > mod2) 
	a -= mod;
    }
}

inline void 
add_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline add_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      add(c, a, b);
      if (c <= -mod2) 
	add(c, c, mod);
      if (c > mod2) 
	subtract(c, c, mod);
    }
}

inline void 
add_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline add_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod / 2;
      double c = (double) a + (double) b;
      if (c <= (double) -mod2) 
	c += (double) mod;
      if (c > (double) mod2) 
	c -= (double) mod;
      d = (long) c;
    }
}

inline void 
sub_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline sub_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      subtract(c, a, b);
      if (c <= -mod2) 
	add(c, c, mod);
      if (c > mod2) 
	subtract(c, c, mod);
    }
}

inline void 
sub_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline sub_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod/2;
      double c = (double) a - (double) b;
      if (c <= (double) -mod2)
	c += (double) mod;
      if (c > (double) mod2) 
	c -= (double) mod;
      d = (long) c;
    }
}

inline void 
mult_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline mult_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      multiply(c, a, b);
      best_remainder(c, c, mod);
    }
}

inline void 
mult_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline mult_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod/2;
      double ab = ((double) a) * ((double) b);
      register long q = (long) (ab / ((double) mod));
      register long res = (long) (ab - (((double) q) * ((double) mod)));
      if (res > mod2) 
	res -= mod;
      if (res <= -mod2) 
	res += mod;
      d = res;
    }
}

inline void 
div_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline div_mod()");

  bigint u, v;
  bigint d = xgcd(u, v, b, mod);
  if (!d.is_one())
    lidia_error_handler("bigint_matrix", "div_mod - Version bigint :: "
		  "Inverse undefined");
  mult_mod(c, a, u, mod);
}

inline void 
div_mod_long(long &de, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline div_mod_long()");

  long u, v;
  register long d = xgcd(u, v, b, mod);
  if (d != 1)
    lidia_error_handler("bigint_matrix", "div_mod_long - Version long :: "
		  "Inverse undefined");
  mult_mod_long(de,a,u,mod);
}

inline void 
inv_mod(bigint &c, const bigint &a, const bigint &mod) 
{
  debug_handler("bigint_matrix","in inline inv_mod()");

  bigint d, v, mod2;
  shift_right(mod2,mod,1);
  d = xgcd(c, v, a, mod);
  if (!d.is_one())
    lidia_error_handler("bigint_matrix", "inv_mod - Version bigint :: "
		  "Inverse undefined");
  if (c <= -mod2) 
    add(c, c, mod);
  if (c > mod2) 
    subtract(c, c, mod);
}

inline void 
inv_mod_long(long &e, long a, long mod)
{
  debug_handler("bigint_matrix","in inline inv_mod_long()");

  long t, mod2 = mod/2;
  register long d = xgcd(e, t, a, mod);
  if (d != 1)
    lidia_error_handler("bigint_matrix", "inv_mod_long - Version long :: "
		  "Inverse undefined");
  if (e <= -mod2) 
    e += mod;
  if (e > mod2) 
    e -= mod;
}

/**
 ** END: Modular arithmetic
 **/

/**
 ** BEGIN: matrix arithmetic
 **/

/**
 ** procedures
 **/

void
add(bigint_matrix & RES, const bigint_matrix & M, const bigint_matrix & N)
{
  /**
   ** DESCRIPTION: RES = M + N,
   ** ERROR: M.rows != N.rows or M.columns != N.columns 
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in function "
		  "add(bigint_matrix &, const bigint_matrix &,const bigint_matrix &)",3);
  
  register long i, j;
  if (M.rows != N.rows || M.columns != N.columns)
    lidia_error_handler("bigint_matrix", "add :: Error in matrix dimensions");
  if (RES.rows != N.rows)
    RES.set_no_of_rows(N.rows);
  if (RES.columns != N.columns)
    RES.set_no_of_columns(N.columns);
  bigint *Mtmp, *Ntmp, *REStmp;
  for (i = 0; i < N.rows; i++)
    {
      Ntmp = N.value[i];
      Mtmp = M.value[i];
      REStmp = RES.value[i];
      for (j = 0; j < N.columns; j++)
	add(REStmp[j], Ntmp[j], Mtmp[j]);
    }
}

void
add(bigint_matrix & RES, const bigint_matrix & M, const bigint & a)
{
  /**
   ** DESCRIPTION: RES.value[x][y] = M.value[x][y] + a,
   **              x=0,...,M.rows-1, y=0,...,M.columns-1
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in function "
		  "add(bigint_matrix &, const bigint_matrix &, const bigint &)",3);

  if (RES.rows != M.rows)
    RES.set_no_of_rows(M.rows);
  if (RES.columns != M.columns)
    RES.set_no_of_columns(M.columns);
  register long i, j;
  bigint *REStmp, *Mtmp;
  for (i = 0; i < RES.rows; i++)
    {
      REStmp = RES.value[i];
      Mtmp = M.value[i];
      for (j = 0; j < RES.columns; j++)
	add(REStmp[j], Mtmp[j], a);
    }
}

void
subtract(bigint_matrix & RES, const bigint_matrix & M, const bigint_matrix & N)
{
  /**
   ** DESCRIPTION: RES = M - N
   ** ERROR: M.rows != N.rows or M.columns != N.columns
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in function "
		  "subtract(bigint_matrix &, const bigint_matrix &, "
		  "const bigint_matrix &)",3);

  register long i, j;
  if (M.rows != N.rows || M.columns != N.columns)
    lidia_error_handler("bigint_matrix", "subtract :: Error in matrix dimensions !");
  if (RES.rows != N.rows)
    RES.set_no_of_rows(N.rows);
  if (RES.columns != N.columns)
    RES.set_no_of_columns(N.columns);
  
  bigint *REStmp, *Mtmp, *Ntmp;
  for (i = 0; i < N.rows; i++)
    {
      REStmp = RES.value[i];
      Mtmp = M.value[i];
      Ntmp = N.value[i];
      for (j = 0; j < N.columns; j++)
	subtract(REStmp[j], Mtmp[j], Ntmp[j]);
    }
}

void
subtract(bigint_matrix & RES, const bigint_matrix & M, const bigint & a)
{
  /**
   ** DESCRIPTION: RES.value[x][y] = M.value[x][y] - a,
   **              x=0,...,M.rows-1, y=0,...,M.columns-1
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in function "
		  "subtract(bigint_matrix &, const bigint_matrix &, "
		  "const bigint &)",3);

  if (RES.rows != M.rows)
    RES.set_no_of_rows(M.rows);
  if (RES.columns != M.columns)
    RES.set_no_of_columns(M.columns);

  register long i, j;
  bigint *REStmp, *Mtmp;
  for (i = 0; i < RES.rows; i++)
    {
      REStmp = RES.value[i];
      Mtmp = M.value[i];
      for (j = 0; j < RES.columns; j++)
	subtract(REStmp[j], Mtmp[j], a);
    }
}

void
multiply(bigint_matrix & RES, const bigint_matrix & A, const bigint_matrix & B)
{
  /**
   ** DESCRIPTION: RES = A * B
   ** ERROR: A.columns != B.rows 
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in function "
		  "multiply(bigint_matrix &, const bigint_matrix &, "
		  "const bigint_matrix &)",3);

  if (A.columns != B.rows)
    lidia_error_handler("bigint_matrix", "multiply :: Error in matrix dimensions ");

  register long j, i, z;
  bigint TMP, TMP1, *Atmp, *REStmp;
  
  if (A.value != RES.value && B.value != RES.value)
    {
      if (RES.rows != A.rows)
	RES.set_no_of_rows(A.rows);
      if (RES.columns != B.columns)
	RES.set_no_of_columns(B.columns);
      for (j = 0; j < A.rows; j++)
	{
	  Atmp = A.value[j];
	  REStmp = RES.value[j];
	  for (i = 0; i < B.columns; i++)
	    {
	      TMP.assign_zero();
	      for (z = 0; z < B.rows; z++)
		{
		  multiply(TMP1, Atmp[z], B.value[z][i]);
		  add(TMP, TMP, TMP1);
		}
	      REStmp[i].assign(TMP);
	    }
	}
    }
  else
    {
      bigint_matrix RES1(A.rows, B.columns);
      for (j = 0; j < A.rows; j++)
	{
	  Atmp = A.value[j];
	  REStmp = RES1.value[j];
	  for (i = 0; i < B.columns; i++)
	    {
	      TMP.assign_zero();
	      for (z = 0; z < B.rows; z++)
		{
		  multiply(TMP1, Atmp[z], B.value[z][i]);
		  add(TMP, TMP, TMP1);
		}
	      REStmp[i].assign(TMP);
	    }
	}
      RES.assign(RES1);
    }
}

void
multiply(bigint_matrix & RES, const bigint_matrix & A, const bigint & k)
{
  /**
   ** DESCRIPTION: RES.value[x][y] = A.value[x][y]*k,
   **              x=0,...,A.rows-1, y=0,...,A.columns-1
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in function "
		  "multiply(bigint_matrix &, const bigint_matrix &, bigint &)",3);
  
  if (RES.rows != A.rows)
    RES.set_no_of_rows(A.rows);
  if (RES.columns != A.columns)
    RES.set_no_of_columns(A.columns);
  
  register long j, i;
  bigint *REStmp, *Atmp;
  for (j = 0; j < A.rows; j++)
    {
      REStmp = RES.value[j];
      Atmp = A.value[j];
      for (i = 0; i < A.columns; i++)
	multiply(REStmp[i], Atmp[i], k);
    }
}

void
multiply(bigint * c, const bigint_matrix & A, const bigint * v)
{
  /**
   ** DESCRIPTION: c[x] = A.value[x][0]*v[0]+...
   **                              +A.value[x][A.columns-1]*v[A.columns-1],
   **              x=0,...,A.rows-1
   ** ERROR: v == NULL or c == NULL
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in function "
		  "multiply(const bigint_matrix &, bigint *, const bigint *)",3);
  
  if (c==NULL || v==NULL)
    lidia_error_handler("bigint_matrix",
		  "multiply :: No memory allocated for parameters !");
  register long i, j;
  bigint TMP, TMP1, *tmp;
  for (i = 0; i < A.rows; i++)
    {
      TMP.assign_zero();
      tmp = A.value[i];
      for (j = 0; j < A.columns; j++)
	{
	  multiply(TMP1, tmp[j], v[j]);
	  add(TMP, TMP, TMP1);
	}
      c[i].assign(TMP);
    }
}

void
remainder(bigint_matrix & RES, const bigint_matrix & M, const bigint & mod)
{
  /**
   ** DESCRIPTION: RES = M % mod,
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in function "
		  "remainder(bigint_matrix &, const bigint_matrix &, "
		  "const bigint &)",3);

  register long i, j;  
  if (RES.rows != M.rows)
    RES.set_no_of_rows(M.rows);
  if (RES.columns != M.columns)
    RES.set_no_of_columns(M.columns);
  
  bigint *REStmp, *Mtmp;
  for (i = 0; i < M.rows; i++)
    {
      REStmp = RES.value[i];
      Mtmp = M.value[i];
      for (j = 0; j < M.columns; j++)
	best_remainder(REStmp[j], Mtmp[j], mod);
    }
}

void 
negate(bigint_matrix & A, const bigint_matrix & B)
{
  /**
   ** DESCRIPTION: A.value[x][y] = - B.value[x][y],
   **              x=0,...,rows-1, y=0,...,columns-1
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in function negate(bigint_matrix &, "
		  "const bigint_matrix &)",3);

  register long i, j;  
  if (A.rows != B.rows)
    A.set_no_of_rows(B.rows);
  if (A.columns != B.columns)
    A.set_no_of_columns(B.columns);
  
  bigint *Btmp, *Atmp;
  for (i = 0; i < B.rows; i++)
    {
      Btmp = B.value[i];
      Atmp = A.value[i];
      for (j = 0; j < B.columns; j++)
	::negate(Atmp[j],Btmp[j]);
    }
}

/**
 ** operators
 **/

bigint_matrix & bigint_matrix::
operator += (const bigint_matrix & M)
{
  debug_handler_l("bigint_matrix", "in operator +=(const bigint_matrix &)",3);
  
  add(*this, *this, M);
  return *this;
}

bigint_matrix & bigint_matrix::
operator += (const bigint & a)
{
  debug_handler_l("bigint_matrix", "in operator +=(const bigint &)",3);
  
  add(*this, *this, a);
  return *this;
}

bigint_matrix bigint_matrix:: 
operator + (const bigint_matrix & M) const
{
  debug_handler_l("bigint_matrix", "in operator + (const bigint_matrix&)",3);

  bigint_matrix RES(rows, columns);
  add(RES, *this, M);
  return RES;
}

bigint_matrix bigint_matrix::
operator + (const bigint & a) const
{
  debug_handler_l("bigint_matrix", "in operator + (const bigint &)",3);
  
  bigint_matrix RES(rows, columns);
  add(RES, *this, a);
  return RES;
}

bigint_matrix bigint_matrix::
operator - () const
{
  debug_handler_l("bigint_matrix", "in operator - ()",3);
  
  bigint_matrix A(rows,columns);
  negate(A,*this);
  return A;
}

bigint_matrix & bigint_matrix::
operator -= (const bigint_matrix & M)
{
  debug_handler_l("bigint_matrix", "in operator -=(const bigint_matrix &)",3);
  
  subtract(*this, *this, M);
  return *this;
}

bigint_matrix & bigint_matrix::
operator -= (const bigint & a)
{
  debug_handler_l("bigint_matrix", "in operator -= (const bigint &)",3);
  
  subtract(*this, *this, a);
  return *this;
}

bigint_matrix bigint_matrix:: 
operator - (const bigint_matrix & M) const
{
  debug_handler_l("bigint_matrix", "in operator - (const bigint_matrix &)",3);
  
  bigint_matrix RES(rows, columns);
  subtract(RES, *this, M);
  return RES;
}

bigint_matrix bigint_matrix:: 
operator - (const bigint & a) const
{
  debug_handler_l("bigint_matrix", "in operator - (const bigint &)",3);
  
  bigint_matrix RES(rows, columns);
  subtract(RES, *this, a);
  return RES;
}

bigint_matrix & bigint_matrix::
operator *= (const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in operator *= (const bigint_matrix &)",3);
  
  multiply(*this, *this, A);
  return *this;
}

bigint_matrix & bigint_matrix::
operator *= (const bigint & m)
{
  debug_handler_l("bigint_matrix", "in operator *= (const bigint &)",3);

  multiply(*this, *this, m);
  return *this;
}

bigint_matrix bigint_matrix::
operator * (const bigint_matrix & M) const
{
  debug_handler_l("bigint_matrix", "in operator * (const bigint_matrix &)",3);
  
  bigint_matrix RES(rows, M.columns);
  multiply(RES, *this, M);
  return RES;
}

bigint_matrix bigint_matrix:: 
operator * (const bigint &m) const
{
  debug_handler_l("bigint_matrix", "in operator * (const bigint &)",3);
  
  bigint_matrix RES(rows, columns);
  multiply(RES, *this, m);
  return RES;
}

bigint *bigint_matrix::
operator * (const bigint * v) const
{
  debug_handler_l("bigint_matrix", "in operator * (const bigint *)",3);
  
  bigint *b = new bigint[rows];
  memory_handler(b, "bigint_matrix", "operator * :: "
		 "Error in memory allocation (b)");
  multiply(b, *this, v);
  return b;
}

bigint_matrix & bigint_matrix::
operator %= (const bigint & mod)
{
  debug_handler_l("bigint_matrix", "in operator %= (const bigint &)",3);
  
  remainder(*this, *this, mod);
  return *this;
}

bigint_matrix bigint_matrix::
operator % (const bigint & mod) const
{
  debug_handler_l("bigint_matrix", "in operator % (const bigint &)",3);
  
  bigint_matrix RES(rows, columns);
  remainder(RES, *this, mod);
  return RES;
}

/**
 ** boolean operators
 **/

int bigint_matrix::
operator == (const bigint_matrix & N) const
{
  debug_handler_l("bigint_matrix", "in operator == (const bigint_matrix &)",3);
  
  return equal(N);
}

int bigint_matrix::
equal(const bigint_matrix & N) const
{
  /**
   ** DESCRIPTION: A.equal(B) returns 1, if A.value[x][y] == B.value[x][y],
   **              x=0,...,A.rows-1, y=0,...,A.columns-1
   **              and A.columns == B.columns and A.rows == B.rows
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "equal(const bigint_matrix &)",3);
  
  register long i, j;
  if (rows != N.rows)
    return 0;
  if (columns != N.columns)
    return 0;
  bigint *tmp, *Ntmp;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      Ntmp = N.value[i];
      for (j = 0; j < columns; j++)
	if (tmp[j] != Ntmp[j])
	  return 0;
    }
  return 1;
}

int
equal(const bigint_matrix & A, const bigint_matrix & B)
{
  debug_handler_l("bigint_matrix", "in function "
		  "equal(const bigint_matrix &, const bigint_matrix &)",3);
  
  return A.equal(B);
}

int bigint_matrix::
operator != (const bigint_matrix & N) const
{
  debug_handler_l("bigint_matrix", "in operator != (const bigint_matrix &)",3);

  return !equal(N);
}

int bigint_matrix::
unequal(const bigint_matrix & A) const
{
  debug_handler_l("bigint_matrix", "in member - function "
		  "unequal(const bigint_matrix &)",3);
  
  return (!equal(A));
}

int
unequal(const bigint_matrix & A, const bigint_matrix & B)
{
  debug_handler_l("bigint_matrix", "in function "
		  "unequal(const bigint_matrix &, const bigint_matrix &)",3);
  
  return !A.equal(B);
}

int bigint_matrix::
is_column_zero(long r) const
{
  /**
   ** DESCRIPTION: 1 == A.is_column_zero(r) <=> A.value[x][r] == 0, 
   **              x=0,...,rows-1
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "is_column_zero(long)",3);
  
  register long i = 0;
  while (i < rows)
    {
      if (value[i][r].is_zero())
	i++;
      else
	return 0;
    }
  return 1;
}

int
is_column_zero(const bigint_matrix & A, long i)
{
  debug_handler_l("bigint_matrix", "in function "
		  "is_column_zero(const bigint_matrix &, long)",3);

  return A.is_column_zero(i);
}

int bigint_matrix::
is_row_zero(long r) const
{
  /**
   ** DESCRIPTION: 1 == A.is_row_zero(r) <=> A.value[r][x] == 0, 
   **              x=0,...,columns-1
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "is_row_zero(long)",3);
  
  register long i = 0;
  bigint *tmp = value[r];
  while (i < columns)
    {
      if (tmp[i].is_zero())
	i++;
      else
	return 0;
    }
  return 1;
}

int
is_row_zero(const bigint_matrix & A, long i)
{
  debug_handler_l("bigint_matrix", "in function "
		  "is_row_zero(const bigint_matrix &, long)",3);
  
  return A.is_row_zero(i);
}

/**
 ** END: matrix arithmetic
 **/

/**
 ** norms and bounds
 **/

bigint bigint_matrix::
max() const
{
  debug_handler_l("bigint_matrix", "in member - function max()",3);
  
  bigint MAX;
  max(MAX);
  return MAX;
}

void bigint_matrix::
max(bigint &MAX) const
{ 
  /**
   ** DESCRIPTION: A.max(res);
   **              => res = maximum of all members of matrix A
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function max(bigint &)",3);
  
  MAX.assign(value[0][0]);
  register long i, j;
  bigint *tmp;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	if (MAX < tmp[j])
	  MAX.assign(tmp[j]);
    }
}

bigint
max(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function max(const bigint_matrix &)",3);
  
  return A.max();
}

bigint bigint_matrix::
max_abs() const
{
  debug_handler_l("bigint_matrix", "in member - function max_abs()",3);
  bigint RES;
  max_abs(RES);
  return RES;
}

void bigint_matrix::
max_abs(bigint &MAX) const
{
  /**
   ** DESCRIPTION: A.max_abs(res); 
   **              => res = absolute maximum of all members of matrix A
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function max_abs(bigint &)",3);
  
  MAX.assign(abs(value[0][0]));
  register long i, j;
  bigint *tmp;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	if (MAX < abs(tmp[j]))
	  MAX.assign(abs(tmp[j]));
    }
}

bigint
max_abs(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function max_abs(const bigint_matrix &)",3);
  
  return A.max_abs();
}

bigint bigint_matrix::
min() const
{
  debug_handler_l("bigint_matrix", "in member - function min()",3);
  
  bigint MIN;
  min(MIN);
  return MIN;
}

void bigint_matrix::
min(bigint &MIN) const
{
  /**
   ** DESCRIPTION: A.min(MIN); 
   **              => MIN = minimum of all members of matrix A
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function min(bigint &)",3);
  
  MIN.assign(value[0][0]);
  register long i, j;
  bigint *tmp;
  for (i = 0; i < rows; i++)
    {
      tmp = value[i];
      for (j = 0; j < columns; j++)
	if (MIN > tmp[j])
	  MIN.assign(tmp[j]);
    }
}

bigint
min(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function min(const bigint_matrix &)",3);
  
  return A.min();
}

bigint bigint_matrix::
min_abs() const
{
  debug_handler_l("bigint_matrix", "in member - function min_abs()",3);

  bigint MIN;
  min_abs(MIN);
  return MIN;
}

void bigint_matrix::
min_abs(bigint &MIN) const
{
  /**
   ** DESCRIPTION: A.min_abs(MIN); 
   **              => MIN = absolute minimum of all members of matrix A
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function min_abs(bigint &)",3);
  
  MIN.assign(abs(value[0][0]));
  register long i, j;
  bigint *tmp;
  for (i = 0; i < rows; i++)
  {
    tmp = value[i];
    for (j = 0; j < columns; j++)
      if (MIN > abs(tmp[j]))
	MIN.assign(abs(tmp[j]));
  }
}

bigint
min_abs(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function min_abs(const bigint_matrix &)",3);
  
  return A.min_abs();
}

void bigint_matrix::
hadamard(bigint & H) const
{
  /**
   ** DESCRIPTION: A.hadamard(H);
   **              => H = hadamard bound of matrix A
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "hadamard(bigint &)",3);
  
  long min = (columns<rows) ? columns : rows;
  long max = (columns<rows) ? rows : columns;
  register long i, j;
  bigint tmp, *Atmp;

  /* Step 1,2 */
  bigint_matrix A(*this);

  /* Step 3 - 11 */
  bigint *h = new bigint[max];
  memory_handler(h, "bigint_matrix", "hadamard :: "
		 "Error in memory allocation (h)");

  for (j = 0; j < columns; j++)
    {
      h[j].assign_zero();
      for (i = 0; i < rows; i++)
	{
	  multiply(tmp, A.value[i][j], A.value[i][j]);
	  add(h[j], h[j], tmp);
	}
    }
  
  /* Step 12 - 15 */
  j = 0;
  if (rows < columns)
    while (j < columns - 1)
      if (h[j] < h[j + 1])	
	{
	  swap(h[j], h[j + 1]);
	  j = 0;
	}				// h.sort(DOWN);
      else
	j++;
  
  /* Step 16 - 20 */
  bigint COLUMNS = 1;
  for (j = 0; j < min; j++)
    multiply(COLUMNS, COLUMNS, h[j]);
  
  /* Step 21 - 34 */
  for (i = 0; i < rows; i++)
    {
      h[i].assign_zero();
      Atmp = A.value[i];
      for (j = 0; j < columns; j++)
	{
	  multiply(tmp, Atmp[j], Atmp[j]);
	  add(h[i], h[i], tmp);
	}
    }

  j = 0;
  if (rows > columns)
    while (j < rows - 1)
      if (h[j] < h[j + 1])   
	{
	  swap(h[j], h[j + 1]);
	  j = 0;
	}				// h.sort(DOWN);
      else
	j++;
  
  /* Step 35 - 39 */
  bigint ROWS = bigint(1);
  for (i = 0; i < min; i++)
    multiply(ROWS, ROWS, h[i]);
  
  delete[] h;
  
  /* Step 40 - 45 */
  bigint M;
  if (COLUMNS < ROWS)
    M.assign(COLUMNS);
  else
    M.assign(ROWS);

  register long B = M.bit_length() - 1;
  bigint E = (B / bigint(2)) + bigint(2);
  power(H, bigint(2), E);
  dec(H);
}

bigint bigint_matrix::
hadamard() const
{
  debug_handler_l("bigint_matrix", "in member - function hadamard()",3);

  bigint H;
  hadamard(H);
  return H;
}

bigint
hadamard(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "hadamard(const bigint_matrix &)",3);

  bigint H;
  A.hadamard(H);
  return H;
}

void bigint_matrix::
row_norm(bigint & RES, long pos, long art)
{
  debug_handler_l("bigint_matrix", "in member - function "
		  "row_norm(bigint &, long, long)",3);

  RES.assign_zero();
  bigint TMP;
  bigint *tmp = value[pos];
  if (pos < 0 || pos >= rows)
    lidia_error_handler("bigint_matrix", "row_norm :: Error in argument size !");
  register long i;
  for (i = 0; i < columns; i++)
    {
      power(TMP, tmp[i], art);
      add(RES, RES, abs(TMP));
    }
}

void bigint_matrix::
column_norm(bigint & RES, long pos, long art)
{
  debug_handler_l("bigint_matrix", "in member - function "
		  "column_norm(bigint &, long, long)",3);
  
  RES.assign_zero();
  bigint TMP;
  if (pos < 0 || pos >= columns)
    lidia_error_handler("bigint_matrix", "column_norm :: Error in argument size !");
  register long i;
  for (i = 0; i < rows; i++)
    {
      power(TMP, value[i][pos], art);
      add(RES, RES, abs(TMP));
    }
}

















