//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : bm_linalg2.c
// Author      : Patrick Theobald (PT)
// Last change : PT, Jun 25 1995, initial version
//

/*
$Id: bm_linalg2.c,v 1.7 1995/08/10 08:50:28 theobald Exp $
*/

#include <LiDIA/bigint_matrix.h>

/**
 ** BEGIN: modular arithmetic
 **/

/**
 ** base operations
 **/

inline void 
pos_div_rem(bigint &q, bigint &r, const bigint &a, const bigint &b)
{
  debug_handler("bigint_matrix","in inline pos_div_rem()");

  div_rem(q,r,a,b);
  if (r.is_lt_zero())
    if (b.is_lt_zero()) 
      {
	subtract(r,r,b);
	inc(q);
      }
    else 
      {
	add(r,r,b);
	dec(q);
      }
}

inline void 
pos_div_rem_long(long &q, long &r, long a, long b) 
{
  debug_handler("bigint_matrix","in inline pos_div_rem_long()");

  q = a / b; 
  r = a - q * b;
  if (r < 0)
    if (b < 0) 
      {
	r -= b;
	q++;
      }
    else 
      {
	r += b;
	q--;
      }
}

inline void 
best_remainder(bigint &a, const bigint &b, const bigint &mod) 
{
  debug_handler("bigint_matrix","in inline best_remainder()");

  if (mod.is_one() || b.is_zero()) 
    a.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      remainder(a,b,mod);
      if (a <= -mod2) 
	add(a, a, mod);
      if (a > mod2) 
	subtract(a, a, mod);
    }
}

inline void 
best_remainder_long(long &a, const bigint &b, long mod) 
{
  debug_handler("bigint_matrix","in inline best_remainder_long()");

  if (mod==1 || b.is_zero()) 
    a = 0;
  else 
    {
      register long mod2 = mod/2;
      remainder(a, b, mod);
      if (a <= -mod2) 
	a += mod;
      if (a > mod2) 
	a -= mod;
    }
}

inline void 
add_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline add_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      add(c, a, b);
      if (c <= -mod2) 
	add(c, c, mod);
      if (c > mod2) 
	subtract(c, c, mod);
    }
}

inline void 
add_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline add_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod / 2;
      double c = (double) a + (double) b;
      if (c <= (double) -mod2) 
	c += (double) mod;
      if (c > (double) mod2) 
	c -= (double) mod;
      d = (long) c;
    }
}

inline void 
sub_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline sub_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      subtract(c, a, b);
      if (c <= -mod2) 
	add(c, c, mod);
      if (c > mod2) 
	subtract(c, c, mod);
    }
}

inline void 
sub_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline sub_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod/2;
      double c = (double) a - (double) b;
      if (c <= (double) -mod2)
	c += (double) mod;
      if (c > (double) mod2) 
	c -= (double) mod;
      d = (long) c;
    }
}

inline void 
mult_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline mult_mod()");

  if (mod.is_one()) 
    c.assign_zero();
  else 
    {
      bigint mod2;
      shift_right(mod2,mod,1);
      multiply(c, a, b);
      best_remainder(c, c, mod);
    }
}

inline void 
mult_mod_long(long &d, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline mult_mod_long()");

  if (mod==1) 
    d=0;
  else 
    {
      register long mod2 = mod/2;
      double ab = ((double) a) * ((double) b);
      register long q = (long) (ab / ((double) mod));
      register long res = (long) (ab - (((double) q) * ((double) mod)));
      if (res > mod2) 
	res -= mod;
      if (res <= -mod2) 
	res += mod;
      d = res;
    }
}

inline void 
div_mod(bigint &c, const bigint &a, const bigint &b, const bigint &mod)
{
  debug_handler("bigint_matrix","in inline div_mod()");

  bigint u, v;
  bigint d = xgcd(u, v, b, mod);
  if (!d.is_one())
    lidia_error_handler("bigint_matrix", "div_mod - Version bigint :: "
		  "Inverse undefined");
  mult_mod(c, a, u, mod);
}

inline void 
div_mod_long(long &de, long a, long b, long mod)
{
  debug_handler("bigint_matrix","in inline div_mod_long()");

  long u, v;
  register long d = xgcd(u, v, b, mod);
  if (d != 1)
    lidia_error_handler("bigint_matrix", "div_mod_long - Version long :: "
		  "Inverse undefined");
  mult_mod_long(de,a,u,mod);
}

inline void 
inv_mod(bigint &c, const bigint &a, const bigint &mod) 
{
  debug_handler("bigint_matrix","in inline inv_mod()");

  bigint d, v, mod2;
  shift_right(mod2,mod,1);
  d = xgcd(c, v, a, mod);
  if (!d.is_one())
    lidia_error_handler("bigint_matrix", "inv_mod - Version bigint :: "
		  "Inverse undefined");
  if (c <= -mod2) 
    add(c, c, mod);
  if (c > mod2) 
    subtract(c, c, mod);
}

inline void 
inv_mod_long(long &e, long a, long mod)
{
  debug_handler("bigint_matrix","in inline inv_mod_long()");

  long t, mod2 = mod/2;
  register long d = xgcd(e, t, a, mod);
  if (d != 1)
    lidia_error_handler("bigint_matrix", "inv_mod_long - Version long :: "
		  "Inverse undefined");
  if (e <= -mod2) 
    e += mod;
  if (e > mod2) 
    e -= mod;
}

/**
 ** END: Modular arithmetic
 **/

/**
 ** BEGIN: Linear algebra
 ** PART 2
 **/

/**
 ** Hermite normal form
 **/

void bigint_matrix::
hnfmod_dkt()
{
  debug_handler_l("bigint_matrix", "in member - function "
		  "hnfmod_dkt()",5);
  
  hnfmod_dkt(latticedet2());
}

void
hnfmod_dkt(bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "hnfmod_dkt(bigint_matrix &)",5);
  
  A.hnfmod_dkt(A.latticedet2());
}

void bigint_matrix::
hnfmod_dkt(const bigint &mod)
{
  /**
   ** DESCRIPTION: A.hnfmod_dkt(mod);
   **              => A in Hermite normal form
   **              => h = lattice determinant of lattice formed
   **              by the columns of matrix A
   ** ERROR: rank != rows
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "hnfmod_dkt(const bigint &)",5);
  
  if (rank() != rows)
    lidia_error_handler("bigint_matrix", "hnfmod_dkt :: "
		  "number of rows != rank");
  
  /* bigint part */
  register long i, j, z, diff = columns - rows;
  
  bigint RES0, RES1, RES2, RES3;  /* 0=lggT,1=rggt,2=ggt */
  bigint x, y;
  bigint TMP, TMP1, TMP2, TMP3;
  bigint *Atmp, *Atmp1 = NULL;
  
  /* Step 1,2 */
  for (i = 0; i < rows; i++)
    {
      Atmp = value[i];
      for (j = 0; j < columns; j++)
	best_remainder(Atmp[j], Atmp[j], mod);
    }
  
  /* Step 3 - 5 */
  for (i = rows - 1; i >= 0; i--)
    {
      Atmp = value[i];
      
      /* Step 6 -8 */
      for (j = diff + i - 1; j >= 0; j--)
	{
	  if (!Atmp[j].is_zero())
	    {
	      /* Step 9 - 12 */
	      if (Atmp[diff + i].is_zero())
		Atmp[diff + i].assign(mod);
	      
	      /* Step 13 - 18 */
	      RES2 = xgcd(RES0, RES1, Atmp[j], Atmp[diff + i]);
	      div_rem(x, RES3, Atmp[diff + i], RES2);
	      div_rem(y, RES3, Atmp[j], RES2);
	      
	      /* Step 19 - 28 */
	      for (z = 0; z <= i; z++)
		{
		  Atmp1 = value[z];
		  TMP.assign(Atmp1[j]);
		  TMP1.assign(Atmp1[diff + i]);
		  
		  /* Atmp1[j] = ((TMP * x) + (TMP1 * y)) % h; */
		  mult_mod(TMP2,TMP,x,mod);
		  mult_mod(TMP3,TMP1,y,mod);
		  sub_mod(Atmp1[j],TMP2,TMP3,mod);
		  
		  /* Atmp1[n-m+i] = ((TMP * RES0) + (TMP1 * RES1)) % h; */
		  mult_mod(TMP2,TMP,RES0,mod);
		  mult_mod(TMP3,TMP1,RES1,mod);
		  add_mod(Atmp1[diff+i],TMP2,TMP3,mod);
		}
	    }
	}
    }
  
  /* Step 29 - 32 */
  bigint D = mod;
  for (i = rows - 1; i >= 0; i--)
    {
      Atmp = value[i];
      
      /* Step 33 - 36 */
      if (Atmp[diff + i].is_zero())
	Atmp[diff + i].assign(D);
      
      /* Step 37 - 39 */
      RES2 = xgcd(RES0, RES1, Atmp[diff + i], D);
      
      /* Step 40 - 46 */
      for (z = 0; z < rows; z++)
	{
	  Atmp1 = value[z];
	  mult_mod(Atmp1[diff + i], Atmp1[diff + i], RES0, D);
	  if (Atmp1[diff+i].is_lt_zero())
	    add(Atmp1[diff+i],Atmp1[diff+i],D);
	}
      Atmp[columns - rows + i].assign(RES2);
      
      /* Step 47 - 49 */
      div_rem(D, TMP, D, RES2);
    }
  
  /* Step 50 - 52 */
  for (i = rows - 1; i >= 0; i--)
    {
      Atmp = value[i];
      
      /* Step 53 - 56 */
      if (Atmp[diff + i].is_zero())
	Atmp[diff + i].assign(mod);
      
      /* Step 57 - 59 */
      for (j = diff + i + 1; j < columns; j++)
	{
	  pos_div_rem(TMP, TMP1, Atmp[j], Atmp[diff + i]);
	      
	  /* Step 60 - 66 */
	  for (z = 0; z <= i; z++)
	    {
	      Atmp1 = value[z];
	      multiply(TMP1, Atmp1[diff + i], TMP);
	      subtract(Atmp1[j], Atmp1[j], TMP1);
	    }
	}
    }
}

void
hnfmod_dkt(bigint_matrix & A, const bigint & h)
{
  debug_handler_l("bigint_matrix", "in function "
		  "hnfmod_dkt(bigint_matrix &, const bigint &)",5);
  
  A.hnfmod_dkt(h);
}

void bigint_matrix::
hnfmod_cohen()
{
  debug_handler_l("bigint_matrix", "in member - function "
		  "hnfmod_cohen()",5);
  
  hnfmod_cohen(latticedet2());
}

void 
hnfmod_cohen(bigint_matrix &A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "hnfmod_cohen(bigint_matrix &)",5);
  
  A.hnfmod_cohen(A.latticedet2());
}

void bigint_matrix::
hnfmod_cohen(const bigint & D)
{
  debug_handler_l("bigint_matrix", "in member - function "
		  "hnfmod_cohen(const bigint &)",5);
  
  if (rank() != rows)
    lidia_error_handler("bigint_matrix", "in hnfmod_cohen - Error !! rang != rows ");
  bigint_matrix W(rows, columns);
  long i = rows - 1;
  long j = columns - 1;
  long k = columns - 1;
  long z;

  bigint R = D;
  bigint u, v, d, q;
  bigint TMP, TMP1, TMP2;
  bigint *tmp, *tmp1;

  for (i = rows - 1; i >= 0; i--)
  {
    tmp = value[i];
    if (tmp[k].is_zero())
      tmp[k].assign(R);
    while (j != 0)
    {
      j--;
      if (!tmp[j].is_zero())
      {
	d = xgcd(u, v, tmp[k], tmp[j]);
	div_rem(TMP1, TMP, tmp[j], d);
	div_rem(TMP2, TMP, tmp[k], d);
	for (z = 0; z < rows; z++)
	{
	  tmp1 = value[z];
	  TMP.assign(u * tmp1[k] + v * tmp1[j]);
	  best_remainder(tmp1[j], TMP2 * tmp1[j] - TMP1 * tmp1[k], R);
	  best_remainder(tmp1[k], TMP, R);
	}
      }
    }
    d = xgcd(u, v, tmp[k], R);
    for (z = 0; z < rows; z++)
    {
      best_remainder(W.value[z][k], u * value[z][k], R);
      if (W.value[z][k] < 0)
	add(W.value[z][k], W.value[z][k], R);
    }
    if (W.value[i][k].is_zero())
      W.value[i][k].assign(R);
    for (j = k + 1; j < columns; j++)
    {
      pos_div_rem(q, TMP, W.value[i][j], W.value[i][k]);
      for (z = 0; z < rows; z++)
	subtract(W.value[z][j], W.value[z][j], q * W.value[z][k]);
    }
    div_rem(R, TMP, R, d);
    k--;
    j = k;
  }
  assign(W);
}

void 
hnfmod_cohen(bigint_matrix &A, const bigint &DET)
{
  debug_handler_l("bigint_matrix", "in function "
		  "hnfmod_cohen(bigint_matrix &, const bigint &)",5);
  
  A.hnfmod_cohen(DET);
}

void bigint_matrix::
hnfmod_mueller(bigint_matrix & TRANS)
{
  /**
   ** DESCRIPTION: A.hnfmod_mueller(TRANS);
   **              => A in Hermite normal form
   **              => TRANS = transformtion matrix A*TRANS = HNFmod(A)
   ** ERROR: TRANS.rows != TRANS.columns or TRANS.rows != columns or
   **        rank != rows
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "hnfmod_mueller(bigint_matrix &, const bigint &)",5);
  
  register long i, j;
  bigint TMP, *TRANStmp;
  
  /* Step 1 */
  long *linuz = lininc();
  if (linuz[0] != rows)
    lidia_error_handler("bigint_matrix", "hnfmod_mueller :: rank not correct");

  /* Step 2,3 */
  if (rows == columns)
    {
      bigint_matrix A1(*this);
      bigint DET = det();
      if (DET.is_lt_zero())
	hnfmod_dkt(-DET);
      else
	hnfmod_dkt(DET);
      multiply(TRANS,::adj(A1), *this);
      
      for (i = 0; i < rows; i++)
	{
	  TRANStmp = TRANS.value[i];
	  for (j = 0; j < rows; j++)
	    div_rem(TRANStmp[j], TMP, TRANStmp[j], DET);
	}
    }
  else
    {
      long diff = columns - rows;

      bigint_matrix EAR(*this);
      EAR.regexpansion(linuz);
      bigint_matrix R(rows, rows), F(rows, diff);
      bigint_matrix E(columns, columns), P(columns, columns);
      E.diag(bigint(1), bigint(0));
      
      long k = linuz[0], l1 = 0, l2 = 0;
      for (i = 0; i < columns; i++)
	{
	  if (linuz[k] == i)
	    {
	      k--;
	      // R.sto_column(column(i), rows, l1);
              for (j=0;j<rows;j++)
		R.value[j][l1].assign(value[j][i]);
	      // P.sto_column(E.column(i), columns, l1);
              for (j=0;j<columns;j++)
		P.value[j][l1].assign(E.value[j][i]);
	      l1++;
	    }
	  else
	    {
	      //F.sto_column(column(i), rows, l2);
              for (j=0;j<rows;j++)
		F.value[j][l2].assign(value[j][i]);
	      //P.sto_column(E.column(i), columns, rows + l2);
              for (j=0;j<columns;j++)
		P.value[j][rows+l2].assign(E.value[j][i]);
	      l2++;
	    }
	}
      delete[] linuz;
      
      /* Step 4,5 */
      EAR.hnfmod_dkt();
      bigint_matrix ADJ =::adj(R);
      bigint DET = R.det();
      bigint_matrix U(diff, diff), V(diff, rows);
      bigint_matrix EHNF(rows, rows), Null(rows, diff);

      EAR.divide(U, V, Null, EHNF);

      bigint_matrix M(columns, columns);

      /* Step 6 */
      bigint_matrix T1 = (-ADJ) * (F * U);
      bigint_matrix T2 = (-ADJ) * (F * V) + (ADJ * EHNF);
      bigint_matrix T3 = (U * DET);
      bigint_matrix T4 = (V * DET);
      M.compose(T1, T2, T3, T4);
      
      /* Step 7 */
      multiply(TRANS, P, M);
      for (i = 0; i < columns; i++)
	{
	  TRANStmp = TRANS.value[i];
	  for (j = 0; j < columns; j++)
	    div_rem(TRANStmp[j], TMP, TRANStmp[j], DET);
	}
      multiply(*this, *this, TRANS);
    }
}

void
hnfmod_mueller(bigint_matrix & A, bigint_matrix & TRANS)
{
  debug_handler_l("bigint_matrix", "in function "
		  "hnfmod_mueller(const bigint_matrix &, bigint_matrix &)",5);
  
  A.hnfmod_mueller(TRANS);
}

void bigint_matrix::
hnf_simple()
{
  /**
   ** DESCRIPTION: HNF Computation
   ** ALGORITHM: Gauss with reduction
   ** IMPROVEMENTS: Theory of Havas / best reaminder
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function hnf_simple()",5);
  
  bigint_matrix TR(columns, columns);
  bigint *REM;
  register long startr = 0, startc = 0, i, j;
  
  for (startc=columns-1,startr=rows-1;startr>=0 && startc>=0;startr--,startc--)
    {
      bigint *ZU = row(startr);
      for (i = startc + 1; i < columns; ZU[i].assign_zero(), i++);
      REM = mgcd2(ZU, columns, TR);
      delete[] REM;
      delete[] ZU;
      
      TR = TR.trans();
      multiply(*this, *this, TR);
      for (i = 0; value[startr][i].is_zero() && i <= startc; i++);
      if (i > startc)
	return;
      swap_columns(startc, i);
    }
  
  bigint TMP, q;
  bigint *tmp, *tmp1;
  for (startc=columns-2,startr=rows-2;startr>=0 && startc>=0;startr--,startc--)
    {
      tmp = value[startr];
      for (i = startc + 1; i < columns; i++)
	{
	  pos_div_rem(q, TMP, tmp[i], tmp[startc]);
	  for (j = 0; j <= startr; j++)
	    {
	      tmp1 = value[j];
	      multiply(TMP,q,tmp1[startc]);
	      subtract(tmp1[i], tmp1[i], TMP);
	    }
	}
    }
}

void bigint_matrix::
hnf_simple(bigint_matrix & T)
{
  /**
   ** DESCRIPTION: HNF Computation
   ** ALGORITHM: Gauss with reduction
   ** IMPROVEMENTS: Theory of Havas / best reaminder
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "hnf_simple(bigint_matrix &)",5);
  
  if (T.columns != columns)
    T.set_no_of_columns(columns);
  if (T.rows != columns)
    T.set_no_of_rows(columns);
  
  T.diag(1, 0);
  
  bigint *REM;
  bigint_matrix TR = T;
  register long startr, startc, i, j;
  
  for (startc=columns-1,startr=rows-1;startr>=0 && startc>=0;startr--,startc--)
    {
      bigint *ZU = row(startr);
      for (i = startc + 1; i < columns; ZU[i].assign_zero(), i++);
      REM=mgcd2(ZU, columns, TR);
      delete[] REM;
      delete[] ZU;
      TR = TR.trans();
      multiply(*this, *this, TR);
      multiply(T, T, TR);
      for (i = 0; value[startr][i].is_zero() && i <= startc; i++);
      if (i > startc)
	return;
      swap_columns(startc, i);
      T.swap_columns(startc, i);
    }
  
  bigint TMP, q;
  bigint *tmp, *tmp1;
  for (startc=columns-2,startr=rows-2;startr>=0 && startc>=0;startr--,startc--)
    {
      tmp = value[startr];
      for (i = startc + 1; i < columns; i++)
	{
	  pos_div_rem(q, TMP, tmp[i], tmp[startc]);
	  for (j = 0; j <= startr; j++)
	    {
	      tmp1 = value[j];
	      multiply(TMP,q,tmp1[startc]);
	      subtract(tmp1[i], tmp1[i], TMP);
	    }
	  for (j = 0; j < columns; j++)
	    {
	      tmp1 = T.value[j];
	      multiply(TMP,q,tmp1[startc]);
	      subtract(tmp1[i], tmp1[i], TMP);
	    }
	}
    }
}

void
hnf_simple(bigint_matrix & A, bigint_matrix & TRANS)
{
  debug_handler_l("bigint_matrix", "in member - function "
		  "hnf_simple(bigint_matrix &, bigint_matrix &)",5);
  
  A.hnf_simple(TRANS);
}

void bigint_matrix::
hnf_havas()
{
  /**
   ** DESCRIPTION: HNF Computation
   ** ALGORITHM: Gauss with reduction
   ** IMPROVEMENTS: Theory of Havas / best reaminder
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function hnf_havas()",5);
  
  bigint q, res, TMP;
  bigint *tmp, *tmp1;
  
  register long startr, startc, i, j, SW;
  long index;
  
  /* Elimination */
  for (startc=columns-1,startr=rows-1;startr>=0 && startc>=0;startc--,startr--)
    {
      tmp = value[startr];
      
      /* init */
      for (index=0; index<=startc && tmp[index].is_zero();index++);
      
      if (index >startc)
	return;
      
      do
	{
	  SW=0;
	  for (i = 0; i <= startc; i++)
	    if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
	      index = i;
	  
	  for (i = 0; i <= startc; i++)
	    if ((i != index) && !tmp[i].is_zero())
	      {
		SW=1;
		div_rem(q, res, tmp[i], tmp[index]);
		for (j = 0; j < rows; j++)
		  {
		    tmp1 = value[j];
		    multiply(TMP,q,tmp1[index]);
		    subtract(tmp1[i], tmp1[i], TMP);
		  }
	      }
	}
      while (SW==1);
      
      if (tmp[index].is_lt_zero())
	for (i=0;i<=startr;i++)
	  value[i][index].negate();
      if (index != startc)
	swap_columns(startc, index);
    }
  
  /* column reduction */
  for (startc=columns-2,startr=rows-2;startr>=0 && startc>=0;startr--,startc--)
    {
      tmp = value[startr];
      for (i = startc + 1; i < columns; i++)
	{
	  pos_div_rem(q, TMP, tmp[i], tmp[startc]);
	  for (j = 0; j <= startr; j++)
	    {
	      tmp1 = value[j];
	      multiply(TMP,q,tmp1[startc]);
	      subtract(tmp1[i], tmp1[i], TMP);
	    }
	}
      
    }
}

void bigint_matrix::
hnf_havas2()
{
  /**
   ** DESCRIPTION: HNF Computation
   ** ALGORITHM: Gauss with reduction
   ** IMPROVEMENTS: Theory of Havas / best reaminder
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function hnf_havas()",2);

  bigint q, res, TMP;
  bigint *tmp, *tmp1;
  
  register long startr, startc, i, j, SW;
  register long index, index2;
  
  /* Elimination */
  for (startc=columns-1,startr=rows-1;startr>=0 && startc>=0;startc--,startr--)
    {
      tmp=value[startr];
      for (index2=0;index2<=startc && tmp[index2].is_zero();index2++);
      
      if (index2 > startc)
	{
	  startc++;
	  continue;
	}
      
      index = index2;       
      
      do
	{
	  SW=0;
	  for (i = index2; i <= startc; i++)
	    if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
	      index = i;
	  
	  for (i = index2; i <= startc; i++)
	    {
	      if ((i != index) && !tmp[i].is_zero())
		{
		  SW=1;
		  div_rem(q, res, tmp[i], tmp[index]);
		  for (j = 0; j <= startr; j++)
		    {
		      tmp1 = value[j];
		      multiply(TMP,q,tmp1[index]);
		      subtract(tmp1[i], tmp1[i], TMP);
		    }
		}
	    }
	}
      while (SW==1);
      
      if (tmp[index].is_lt_zero())
	for (i=0;i<=startr;i++)
	  value[i][index].negate();
      if (index != startc)
	swap_columns(startc, index);
    }
  
  /* column reduction */
  for (startc=columns-2,startr=rows-2;startr>=0 && startc>=0;startr--,startc--)
    {
      tmp = value[startr];
      for (i = startc + 1; i < columns; i++)
	{
	  pos_div_rem(q, TMP, tmp[i], tmp[startc]);
	  for (j = 0; j <= startr; j++)
	    {
	      tmp1 = value[j];
	      multiply(TMP,q,tmp1[startc]);
	      subtract(tmp1[i], tmp1[i], TMP);
	    }
	}
    }
}

void bigint_matrix::
hnf_havas3()
{
  /**
   ** DESCRIPTION: HNF Computation
   ** ALGORITHM: Gauss with reduction
   ** IMPROVEMENTS: Theory of Havas / best reaminder
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function hnf_havas()",5);

  bigint q, res, TMP;
  bigint *tmp, *tmp1;

  register long startr, startc, i, j, SW;
  register long index2, index;
  
  /* Elimination */
  for (startc=columns-1,startr=rows-1;startr>=0 && startc>=0;startc--,startr--)
    {
      tmp=value[startr];
      for (index2=0;index2<=startc && tmp[index2].is_zero();index2++);
      
      if (index2 > startc)
	{
	  startc++;
	  continue;
	}
      
      index = index2;       
      
      do
	{
	  SW=0;
	  for (i = index2; i <= startc; i++)
	    if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
	      index = i;
	  
	  for (i = index2; i <= startc; i++)
	    {
	      if ((i != index) && !tmp[i].is_zero())
		{
		  SW=1;
		  div_rem(q, res, tmp[i], tmp[index]);
		  for (j = 0; j <= startr; j++)
		    {
		      tmp1 = value[j];
		      multiply(TMP,q,tmp1[index]);
		      subtract(tmp1[i], tmp1[i], TMP);
		    }
		}
	    }
	}
      while (SW==1);
      
      if (tmp[index].is_lt_zero())
	for (i=0;i<=startr;i++)
	  value[i][index].negate();
      if (index != startc)
	swap_columns(startc, index);

      /* column reduction */
      for (i = startc+1;i<columns;i++)
	if (tmp[i] >= tmp[startc] || tmp[i] < 0)
	  {
	    pos_div_rem(q, TMP, tmp[i], tmp[startc]);
	    for (j=0; j <= startr; j++)
	      {
		tmp1 = value[j];
		multiply(TMP,q,tmp1[startc]);
		subtract(tmp1[i],tmp1[i],TMP);
	      }
	  }
    }     
}

void bigint_matrix::
hnf_havas(bigint_matrix & T)
{
  /**
   ** DESCRIPTION: HNF Computation
   ** ALGORITHM: Gauss with reduction
   ** IMPROVEMENTS: Theory of Havas / best reaminder
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "hnf_havas(bigint_matrix &)",5);
  
  bigint q, res, TMP;
  bigint *tmp, *tmp1;
  
  if (T.columns != columns)
    T.set_no_of_columns(columns);
  if (T.rows != columns)
    T.set_no_of_rows(columns);
  
  T.diag(1, 0);
  
  long startr, startc;
  long i, j, index, SW;

  /* Elimination */
  for (startc=columns-1,startr=rows;startr>=0 && startc>=0;startc--)
    {

      do
	{
	  startr--;
	  if (startr<0)
	    return;
	  tmp = value[startr];
	  for (index=0; index<=startc && tmp[index].is_zero();index++);
	}
      while (index>startc);

      do
	{
	  SW=0;
	  for (i = 0; i <= startc; i++)
	    if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
	      index = i;
	  
	  for (i = 0; i <= startc; i++)
	    if ((i != index) && !tmp[i].is_zero())
	      {
		SW=1;
		div_rem(q, res, tmp[i], tmp[index]);
		for (j = 0; j < rows; j++)
		  {
		    tmp1 = value[j];
		    multiply(TMP,q,tmp1[index]);
		    subtract(tmp1[i], tmp1[i], TMP);
		  }
		for (j = 0; j < columns; j++)
		  {
		    tmp1 = T.value[j];
		    multiply(TMP,q,tmp1[index]);
		    subtract(tmp1[i], tmp1[i], TMP);
		  }
	      }
	}
      while (SW==1);
      
      if (tmp[index].is_lt_zero())
	{
	  for (i=0;i<=startr;i++)
	    value[i][index].negate();
	  for (i=0;i<columns;i++)
	    T.value[i][index].negate();
	}
      if (index !=startc)
	{
	  swap_columns(startc, index);
	  T.swap_columns(startc, index);
	}
    }
  
  /* column reduction */
  for (startc=columns-2,startr=rows-2;startr>=0 && startc>=0;startr--,startc--)
    {
      tmp = value[startr];
      for (i = startc + 1; i < columns; i++)
	{
	  pos_div_rem(q, TMP, tmp[i], tmp[startc]);
	  for (j = 0; j <= startr; j++)
	    {
	      tmp1 = value[j];
	      multiply(TMP,q,tmp1[startc]);
	      subtract(tmp1[i], tmp1[i], TMP);
	    }
	  for (j = 0; j < columns; j++)
	    {
	      tmp1 = T.value[j];
	      multiply(TMP,q,tmp1[startc]);
	      subtract(tmp1[i], tmp1[i], TMP);
	    }
	}
    }
}

void
hnf_havas(bigint_matrix & A, bigint_matrix & TRANS)
{
  debug_handler_l("bigint_matrix", "in function "
		  "hnf_havas(bigint_matrix &, bigint_matrix &)",5);
  
  A.hnf_havas(TRANS);
}

/**
 ** Kernel
 **/

void bigint_matrix::
kernel(const bigint_matrix & A)
{
  /**
   ** DESCRIPTION: B.kernel(A);
   **              => The columns of matrix B form a basis
   **              of the kernel of matrix A.
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "kernel(const bigint_matrix &)",5);
  
  bigint *ZBAtmp, *Atmp;
  register long i, j;
  long c = A.columns;

  /* Step 1 */
  long *linuz = A.lininr();
  long r = linuz[0];
  if (r == c)
  {
    bigint_matrix RES(c, 1);
    assign(RES);
    delete[] linuz;
    return;
  }
  
  /* Step 2 */
  bigint_matrix ZBA(r, c);
  for (i = 1; i <= r; i++)
  {
    ZBAtmp = ZBA.value[i - 1];
    Atmp = A.value[linuz[r - i + 1]];
    for (j = 0; j < c; j++)
      ZBAtmp[j].assign(Atmp[j]);
  }
  delete[] linuz;
  
  /* Step 3 */
  bigint_matrix TRANS(c, c);
  ZBA.hnf(TRANS);
  
  /* Step 4 */
  bigint_matrix PART2(c, r);
  if (rows != c)
    set_no_of_rows(c);
  if (columns != c - r)
    set_no_of_columns(c - r);
  TRANS.divide_h(*this, PART2);
}

bigint_matrix
kernel(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "kernel(const bigint_matrix &)",5);

  bigint_matrix RET;
  RET.kernel(A);
  return RET;
}

void bigint_matrix::
kernel2(const bigint_matrix & A)
{
  /**
   ** DESCRIPTION: B.kernel2(A);
   **              => The columns of matrix B form a basis
   **              of the kernel of matrix A.
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "kernel2(const bigint_matrix &)",5);

  register long i;
  bigint_matrix B = A;
  B.hnf_havas(*this);

  for (i=0;i<A.columns && B.is_column_zero(i);i++);
  
  if (i==0)
    {
      bigint_matrix C(A.rows,1);
      assign(C);
    }
  else
    set_no_of_columns(i);
}

bigint_matrix
kernel2(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "kernel2(const bigint_matrix &)",5);
  
  bigint_matrix RET;
  RET.kernel2(A);
  return RET;
}

/**
 ** regular InvImage
 **/

void bigint_matrix::
reginvimage(const bigint_matrix & A, const bigint_matrix & B)
{
  /**
   ** DESCRIPTION: C.reginvimage(A,B);
   **              => A * C.column(j) = g(j)*B.column(j), j=0,...,B.columns
   **              => g(j) minimal
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "reginvimage(const bigint_matrix &, const bigint_matrix &",5);
  
  register long i, j;
  bigint TMP, TMP1;

  /* Step 1 */
  bigint DET = A.det();
  if (DET == 0)
  {
    lidia_error_handler("bigint_matrix", "reginvimage :: non regular matrix");
    return;
  }
  bigint_matrix ADJ = ::adj(A);

  /* Step 2 */
  bigint_matrix PROD = ADJ * B;
  bigint *u = new bigint[B.rows];
  memory_handler(u,"bigint_matrix","reginvimage :: "
		 "Error in memory allocation (u)");
  bigint *g, phi;

  /* Step 3 */
  if (rows != B.rows + 1)
    set_no_of_rows(B.rows + 1);
  if (columns != B.columns)
    set_no_of_columns(B.columns);
  for (i = 0; i < B.columns; i++)
  {
    for (j=0;j<PROD.rows;j++)
      u[j].assign(PROD.value[j][i]);
    g = mgcd2(u, PROD.rows);
    div_rem(phi, TMP, DET, gcd(g[0], DET));
    if (phi.is_lt_zero())
      phi.negate();

    /* Step 4 */
    for (j = 0; j < PROD.rows; j++)
    {
      multiply(TMP, phi, u[j]);
      div_rem(value[j][i], TMP1, TMP, DET);
    }
    value[PROD.rows][i].assign(phi);
    delete[] g;
  }
 delete[] u;
}

bigint_matrix
reginvimage(const bigint_matrix & A, const bigint_matrix & B)
{
  debug_handler_l("bigint_matrix", "in function "
		  "reginvimage(const bigint_matrix &, const bigint_matrix &",5);
  
  bigint_matrix X;
  X.reginvimage(A,B);
  return X;
}

void bigint_matrix::
reginvimage2(const bigint_matrix & A, const bigint_matrix & B)
{
  /**
   ** DESCRIPTION: C.reginvimage2(A,B);
   **              => A * C.column(j) = g(j)*B.column(j), j=0,...,B.columns
   **              => g(j) minimal
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "reginvimage2(const bigint_matrix &, const bigint_matrix &",5);
  
  register long i, j, len, oldlen;
  bigint TMP, TMP1;
  
  /* Step 1 */
  bigint DET = A.det();
  if (DET == 0)
    {
      lidia_error_handler("bigint_matrix", "reginvimage :: non regular matrix");
      return;
    }
  
  /* Step 2 */
  oldlen = B.rows;
  len = B.rows + 1;
  multiply(*this,::adj(A),B);

  bigint *u = new bigint[len];
  memory_handler(u,"bigint_matrix","reginvimage :: "
		 "Error in memory allocation (u)");
  bigint phi;
  
  /* Step 3 */
  set_no_of_rows(len);
  for (i = 0; i < B.columns; i++)
    {
      for (j=0;j<oldlen;j++)
	u[j].assign(value[j][i]);
      u[oldlen].assign(DET);
      mgcd2(TMP1, u, len);
      div_rem(phi, TMP, DET, TMP1);
      if (phi.is_lt_zero())
	phi.negate();
      
      /* Step 4 */
      for (j = 0; j < oldlen; j++)
	{
	  multiply(TMP, phi, u[j]);
	  div_rem(value[j][i], TMP1, TMP, DET);
	}
      value[oldlen][i].assign(phi);
    }
  delete[] u;
}

bigint_matrix
reginvimage2(const bigint_matrix & A, const bigint_matrix & B)
{
  debug_handler_l("bigint_matrix", "in function "
		  "reginvimage2(const bigint_matrix &, const bigint_matrix &",5);

  bigint_matrix X;
  X.reginvimage(A,B);
  return X;
}

/**
 ** Image
 **/

void bigint_matrix::
image(const bigint_matrix & A)
{
  /**
   ** DESCRIPTION: B.image(A);
   **              => The columns of matrix B form a basis
   **                 of the image of matrix A.
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "image(const bigint_matrix &)",5);
  
  bigint *ZBAtmp, *Atmp;
  register long i, j;

  /* Step 1 */
  long *v = A.lininr();
  long RANG = v[0];

  /* Step 2,3 */
  bigint_matrix ZBA(RANG, A.columns);
  for (i = 1; i <= RANG; i++)
  {
    ZBAtmp = ZBA.value[i - 1];
    Atmp = A.value[v[RANG - i + 1]];
    for (j = 0; j < A.columns; j++)
      ZBAtmp[j].assign(Atmp[j]);
  }
  delete[] v;

  /* Step 4 */
  bigint_matrix TRANS(A.columns, A.columns);
  ZBA.hnf(TRANS);

  /* Step 5 */
 if (rows != A.rows)
    set_no_of_rows(A.rows);
  if (columns != A.columns)
    set_no_of_columns(A.columns);

  if (A.columns == RANG)
    multiply(*this, A, TRANS);
  else
  {
    bigint_matrix M(A.rows, A.columns);
    multiply(M, A, TRANS);
    bigint_matrix PART1(RANG, A.columns - RANG);
    M.divide_h(PART1, *this);
  }
}

bigint_matrix
image(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "image(const bigint_matrix &)",5);
  
  bigint_matrix BILD;
  BILD.image(A);
  return BILD;
}

void bigint_matrix::
image2(const bigint_matrix & A)
{
  /**
   ** DESCRIPTION: B.image2(A);
   **              => The columns of matrix B form a basis
   **                 of the image of matrix A.
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "image2(const bigint_matrix &)",5);
  
  register long i;
  hnf_havas();

  for (i=0;i<columns && is_column_zero(i); i++);

  if (i != 0)
    if (i == columns)
      set_no_of_columns(1);
    else
      {
	bigint_matrix M(*this);
	bigint_matrix PART1(rows,i);
	set_no_of_columns(rows-i);
	M.divide_h(PART1, *this);
  }
}

bigint_matrix
image2(const bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "image(const bigint_matrix &)",5);
  
  bigint_matrix BILD;
  BILD.image(A);
  return BILD;
}

/**
 ** InvImage
 **/

void bigint_matrix::
invimage(const bigint_matrix & B, const bigint * b)
{
  /**
   ** DESCRIPTION: v = invimage(B,b);
   **              => A is a basis of the solution of B*x=b
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "invimage(const bigint_matrix &, const bigint *)",5);
  
  register long i;
  bigint *tmp;

  /* Step 1 */
  bigint_matrix A = B;
  A.set_no_of_columns(B.columns + 1);
  for (i = 0; i < B.rows; i++)
    A.value[i][B.columns].assign(-b[i]);
  kernel(A);

  /* Step 2 */
  if (is_column_zero(0) || is_row_zero(B.columns))
  {
    bigint_matrix C(B.rows, 1);
    set_no_of_columns(1);
    assign(C);
    return;
  }
  bigint *g = mgcd2(tmp=row(B.columns), columns);
  delete[] tmp;
  if (g[0] > 1)
  {
    bigint_matrix C(B.rows, 1);
    set_no_of_columns(1);
    assign(C);
    return;
  }

  /* Step 3,4  */
  bigint *x = (*this) * &(g[1]);
  delete[] g;

  /* Step 5 */
  kernel(B);
  set_no_of_columns(columns + 1);
  for (i = 0; i < rows; i++)
    value[i][columns-1].assign(x[i]);
  delete[] x;
}

bigint_matrix
invimage(const bigint_matrix & A, const bigint * b)
{
  debug_handler_l("bigint_matrix", "in function "
		  "invimage(const bigint_matrix &, const bigint *)",5);
  
  bigint_matrix B;
  B.invimage(A, b);
  return B;
}

/**
 ** solve
 **/

void bigint_matrix::
solve(const bigint_matrix & A, const bigint * b)
{
  /**
   ** DESCRIPTION: A.solve(B,b);
   **              => A is a basis of the solution of B*x=b
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "solve(const bigint_matrix &, const bigint *)",5);

  invimage(A, b);
}

bigint_matrix
solve(const bigint_matrix & A, const bigint * b)
{
  debug_handler_l("bigint_matrix", "in function "
		  "solve(const bigint_matrix &, const bigint *)",5);
  
  bigint_matrix B;
  B.invimage(A, b);
  return B;
}

/**
 ** Smith normal form
 **/

void bigint_matrix::
snf_hartley()
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** IMPROVEMENTS: Havas, Majewski
   ** PAPER: Recognizing badly represented Z-modules, Havas
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "snf_hartley()",5);
  
  register long startr, startc, TEILBARKEIT;
  bigint TMP1, TMP2;
  bigint *tmp;
  register long xpivot, ypivot, i, j, z;

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
    {

      /* pivot: first non zero */

      xpivot = -1;
      ypivot = -1;
      for (i = startr; i < rows; i++)
	{
	  tmp = value[i];
	  for (j = startc; j < columns; j++)
	  if (!tmp[j].is_zero())
	    {
	      xpivot = i;
	      ypivot = j;
	      i = rows;
	      j = columns;
	    }
	}

      if (xpivot != -1)
	{
	  /* swap to diagonalposition */
	  swap_rows(startr, xpivot);
	  swap_columns(startc, ypivot);
	  
	  TEILBARKEIT=0;
	  
	  while(TEILBARKEIT==0)
	    {
	      TEILBARKEIT=1;
	      
	      /* mgcd computation for row */
	      for (i = startc + 1; i < columns; i++)
		if (value[startr][i] % value[startr][startc] != 0)
		  {
		    div_rem(TMP1, TMP2, value[startr][i], value[startr][startc]);
		    for (j = startr; j < rows; j++)
		      {
			tmp = value[j];
			multiply(TMP2, tmp[startc], TMP1);
			subtract(tmp[i], tmp[i], TMP2);
		      }
		    swap_columns(startc, i);
		    i = startc;
		  }
	      
	      /* mgcd computation for column */
	      for (i = startr + 1; i < rows; i++)
		if (value[i][startc] % value[startr][startc] != 0)
		  {
		    div_rem(TMP1, TMP2, value[i][startc], value[startr][startc]);
		    for (j = startc; j < columns; j++)
		      {
			multiply(TMP2, value[startr][j], TMP1);
			subtract(value[i][j], value[i][j], TMP2);
		      }
		    TEILBARKEIT=0; //perhaps
		      swap_rows(i, startr);
		    i = startr;
		  }
	    }

	  /* row elimination */
	  for (i = startc + 1; i < columns; i++)
	    {
	      div_rem(TMP1, TMP2, value[startr][i], value[startr][startc]);
	      for (j = startr; j < rows; j++)
		{
		  tmp = value[j];
		  multiply(TMP2, tmp[startc], TMP1);
		  subtract(tmp[i], tmp[i], TMP2);
		}
	    }

	  /* column elimination */
	  for (i = startr + 1; i < rows; i++)
	    {
	      div_rem(TMP1, TMP2, value[i][startc], value[startr][startc]);
	      for (j = startc; j < columns; j++)
		{
		  multiply(TMP2, value[startr][j], TMP1);
		  subtract(value[i][j], value[i][j], TMP2);
		}
	    }
      
	  /* modulo test */
	  for (i = startr + 1; i < rows; i++)
	    for (j = startc + 1; j < columns; j++)
	      if (value[i][j] % value[startr][startc] != 0)
		{
		  for (z = 0; z < columns; z++)
		    add(value[startr][z], value[startr][z], value[i][z]);
		  i = rows;
		  j = columns;
		  startc = 0;
		  startr = 0;
		}
	}
    }
  
  /* diagonal >= 0 */
  for (i = 0; i < columns && i < rows; i++)
    if (value[i][i].is_lt_zero())
      value[i][i].negate();
}

void 
snf_hartley(bigint_matrix & A)
{
  debug_handler_l("bigint_matrix", "in function "
		  "snf_hartley(bigint_matrix &)",5);
  
  A.snf_hartley();
}

void bigint_matrix::
snf_hartley(bigint_matrix & T1, bigint_matrix & T2)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules
   ** VERSION: 1.8
   **/
  
  debug_handler_l("bigint_matrix", "in member - function "
		  "snf_hartley(bigint_matrix &, bigint_matrix &)",5);
  
  if (T1.columns != rows)
    T1.set_no_of_columns(rows);
  if (T1.rows != rows)
    T1.set_no_of_rows(rows);
  
  if (T2.columns != columns)
    T2.set_no_of_columns(columns);
  if (T2.rows != columns)
    T2.set_no_of_rows(columns);
  
  T1.diag(1, 0);
  T2.diag(1, 0);
  
  register long startr, startc, TEILBARKEIT;
  bigint TMP1, TMP2;
  long xpivot, ypivot;
  register long i, j, z;
  
  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
    {
      
      /* pivot: first non zero */
      xpivot = -1;
      ypivot = -1;
      for (i = startr; i < rows; i++)
	for (j = startc; j < columns; j++)
	  if (value[i][j] != 0)
	    {
	      xpivot = i;
	      ypivot = j;
	      i = rows;
	      j = columns;
	    }
      if (xpivot != -1)
	{
	  /* swap to diagonalposition */
	  swap_rows(startr, xpivot);
	  T1.swap_rows(startr, xpivot);
	  swap_columns(startc, ypivot);
	  T2.swap_columns(startc, ypivot);

	  TEILBARKEIT=0;
	  
	  while(TEILBARKEIT==0)
	    {
	      TEILBARKEIT=1;
	      
	      /* mgcd computation for row */
	      for (i = startc + 1; i < columns; i++)
		if (value[startr][i] % value[startr][startc] != 0)
		  {
		    div_rem(TMP1, TMP2, value[startr][i], value[startr][startc]);
		    for (j = startr; j < rows; j++)
		      {
			multiply(TMP2, value[j][startc], TMP1);
			subtract(value[j][i], value[j][i], TMP2);
		      }
		    for (j = 0; j < columns; j++)
		      {
			multiply(TMP2, T2.value[j][startc], TMP1);
			subtract(T2.value[j][i], T2.value[j][i], TMP2);
		      }
		    swap_columns(startc, i);
		    T2.swap_columns(startc, i);
		    i = startc;
		  }
	      
	      /* mgcd computation for column */
	      for (i = startr + 1; i < rows; i++)
		if (value[i][startc] % value[startr][startc] != 0)
		  {
		    div_rem(TMP1, TMP2, value[i][startc], value[startr][startc]);
		    for (j = startc; j < columns; j++)
		      {
			multiply(TMP2, value[startr][j], TMP1);
			subtract(value[i][j], value[i][j], TMP2);
		      }
		    for (j = 0; j < rows; j++)
		      {
			multiply(TMP2, T1.value[startr][j], TMP1);
			subtract(T1.value[i][j], T1.value[i][j], TMP2);
		      }
		    TEILBARKEIT=0;
		    swap_rows(i, startr);
		    T1.swap_rows(i, startr);
		    i = startr;
		  }
	    }
	  
	  /* row elimination */
	  for (i = startc + 1; i < columns; i++)
	    {
	      div_rem(TMP1, TMP2, value[startr][i], value[startr][startc]);
	      for (j = startr; j < rows; j++)
		{
		  multiply(TMP2, value[j][startc], TMP1);
		  subtract(value[j][i], value[j][i], TMP2);
		}
	      for (j = 0; j < columns; j++)
		{
		  multiply(TMP2, T2.value[j][startc], TMP1);
		  subtract(T2.value[j][i], T2.value[j][i], TMP2);
		}
	    }
	  
	  /* column elimination */
	  for (i = startr + 1; i < rows; i++)
	    {
	      div_rem(TMP1, TMP2, value[i][startc], value[startr][startc]);
	      for (j = startc; j < columns; j++)
		{
		  multiply(TMP2, value[startr][j], TMP1);
		  subtract(value[i][j], value[i][j], TMP2);
		}
	      for (j = 0; j < rows; j++)
		{
		  multiply(TMP2, T1.value[startr][j], TMP1);
		  subtract(T1.value[i][j], T1.value[i][j], TMP2);
		}
	    }
	  
	  /* modulo test */
	  for (i = startr + 1; i < rows; i++)
	    for (j = startc + 1; j < columns; j++)
	      if (value[i][j] % value[startr][startc] != 0)
		{
		  for (z = 0; z < columns; z++)
		    add(value[startr][z], value[startr][z], value[i][z]);
		  for (z = 0; z < rows; z++)
		    add(T1.value[startr][z], T1.value[startr][z], T1.value[i][z]);
		  i = rows;
		  j = columns;
		  startc = 0;
		  startr = 0;
		}
	}
    }
  
  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      {
	value[i][i].negate();
	for (z = 0; z < columns; z++)
	  T2.value[z][i].negate();
      }
}

void 
snf_hartley(bigint_matrix & A, bigint_matrix & T1, bigint_matrix & T2)
{
  debug_handler("bigint_matrix", "in function "
		"snf_hartley(bigint_matrix &, bigint_matrix &, bigint_matrix &)");
  A.snf_hartley(T1, T2);
}

void bigint_matrix::
snf_simple()
{
  /**
   ** DESCRIPTION: SNF Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** IMPROVEMENT: Havas
   ** PAPER: Recognizing badly represented Z-modules
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"snf_simple()");
  bigint PIVOT, TMP1, TMP2;
  bigint *tmp=NULL, *deltmp;

  bigint_matrix TR1(rows,rows);
  bigint_matrix TR2(columns,columns);
  bigint *REM;
  register long startr, startc, pivot, i, j, z, TEILBARKEIT;
  
  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
    {

      /* pivot: first non zero */
      pivot = -1;
      for (i = startr; i < rows; i++)
	for (j = startc; j < columns; j++)
	  if (value[i][j] != 0)
	    {
	      pivot = i;
	      i = rows;
	      j = columns;
	    }

      if (pivot != -1)
	{
	  /* swap pivot in actual row */
	  swap_rows(startr, pivot);
	      
	  TEILBARKEIT=0;

	  while(TEILBARKEIT==0)
	    {
	      TEILBARKEIT=1;
	      
	      /* mgcd computation and row elimination */
	      REM=mgcd2(deltmp=row(startr), columns, TR2);
  	      delete[] deltmp;

              delete[] REM;
	      TR2 = TR2.trans();
	      multiply(*this, *this, TR2);
	   	      
	      tmp = value[startr];
	      for (i = 0; tmp[i].is_zero() && i < columns; i++);
	      swap_columns(startc, i);
	   	  
	      /* mgcd computation and column elimination */
	      REM=mgcd2(deltmp=column(startc), rows, TR1);
              delete[] deltmp;

              delete[] REM;
	      multiply(*this, TR1, *this);
	   	      
	      for (i = 0; value[i][startc].is_zero() && i < rows; i++);
	      swap_rows(startr, i);
	   
	      /* control: row == 0 */
	      tmp = value[startr];
	      for (i = startc+1; tmp[i].is_zero() && i < columns; i++);
	      if (i!=columns)
		TEILBARKEIT=0;
	    } 
	  
	  /* modulo test */
	  for (i = startr; i < rows; i++)
	    for (j = startc + 1; j < columns; j++)
	      if (value[i][j] % value[startr][startc] != 0)
		{
		  if (i != startr)
		    for (z = 0; z < columns; z++)
		      add(tmp[z], tmp[z], value[i][z]);
		  i = rows;
		  j = columns;
		  startc--;
		  startr--;
		}
	}
    }
  
  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      value[i][i].negate();
}

void 
snf_simple(bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"snf_simple(bigint_matrix &)");
  A.snf_simple();
}


void bigint_matrix::
snf_simple(bigint_matrix & T1, bigint_matrix & T2)
{
  /**
   ** DESCRIPTION: SNF Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** IMPROVEMENT: Havas
   ** PAPER: Recognizing badly represented Z-modules
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"snf_simple(bigint_matrix &, bigint_matrix &)");
  bigint PIVOT, TMP1, TMP2;
  bigint *tmp=NULL, *deltmp;

  if (T1.columns != rows)
    T1.set_no_of_columns(rows);
  if (T1.rows != rows)
    T1.set_no_of_rows(rows);

  if (T2.columns != columns)
    T2.set_no_of_columns(columns);
  if (T2.rows != columns)
    T2.set_no_of_rows(columns);

  T1.diag(1, 0);
  T2.diag(1, 0);

  bigint_matrix TR1 = T1;
  bigint_matrix TR2 = T2;
  bigint *REM;
  register long startr, startc, pivot, i, j, z, TEILBARKEIT;
  
  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
    {

      /* pivot: first non zero */
      pivot = -1;
      for (i = startr; i < rows; i++)
	for (j = startc; j < columns; j++)
	  if (value[i][j] != 0)
	    {
	      pivot = i;
	      i = rows;
	      j = columns;
	    }

      if (pivot != -1)
	{
	  /* swap pivot in actual row */
	  swap_rows(startr, pivot);
	  T1.swap_rows(startr, pivot);
      
	  TEILBARKEIT=0;

	  while(TEILBARKEIT==0)
	    {
	      TEILBARKEIT=1;
	      
	      /* mgcd computation and row elimination */
	      REM=mgcd2(deltmp=row(startr), columns, TR2);
              delete[] deltmp;

              delete[] REM;
	      TR2 = TR2.trans();
	      multiply(*this, *this, TR2);
	      multiply(T2, T2, TR2);
	      
	      tmp = value[startr];
	      for (i = 0; tmp[i].is_zero() && i < columns; i++);
	      swap_columns(startc, i);
	      T2.swap_columns(startc, i);
	  
	      /* mgcd computation and column elimination */
	      REM=mgcd2(deltmp=column(startc), rows, TR1);
              delete[] deltmp;

              delete[] REM;
	      multiply(*this, TR1, *this);
	      multiply(T1, TR1, T1);
	      
	      for (i = 0; value[i][startc].is_zero() && i < rows; i++);
	      swap_rows(startr, i);
	      T1.swap_rows(startr, i);

	      /* control: row == 0 */
	      tmp = value[startr];
	      for (i = startc+1; tmp[i].is_zero() && i < columns; i++);
	      if (i!=columns)
		TEILBARKEIT=0;
	    } 
	  
	  /* modulo test */
	  for (i = startr; i < rows; i++)
	    for (j = startc + 1; j < columns; j++)
	      if (value[i][j] % value[startr][startc] != 0)
		{
		  if (i != startr)
		    {
		      for (z = 0; z < columns; z++)
			add(tmp[z], tmp[z], value[i][z]);
		      for (z = 0; z < rows; z++)
			add(T1.value[startr][z], T1.value[startr][z], T1.value[i][z]);
		    }
		  i = rows;
		  j = columns;
		  startc--;
		  startr--;
		}
	}
    }
  
  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      {
	value[i][i].negate();
	for (z = 0; z < columns; z++)
	  T2.value[z][i].negate();
      }
}

void 
snf_simple(bigint_matrix & A, bigint_matrix & T1, bigint_matrix & T2)
{
  debug_handler("bigint_matrix", "in function "
		"snf_simple(bigint_matrix &, bigint_matrix &, bigint_matrix &)");
  A.snf_simple(T1, T2);
}

void bigint_matrix::
snf_havas()
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules
   ** IMPROVEMENTS: Havas, best reaminder include mgcd
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"snf_havas()");
  register long i, j, z, index;
  bigint PIVOT;
  bigint *tmp=NULL;

  register long startr, startc, xpivot, ypivot, SW, TEILBARKEIT;
  bigint TMP1, TMP2;

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
    {
      
      /* pivot: first non zero */
      xpivot = -1;
      ypivot = -1;
      for (i = startr; i < rows; i++)
	for (j = startc; j < columns; j++)
	  if (!value[i][j].is_zero())
	    {
	      xpivot = i;
	      ypivot = j;
	      i = rows;
	      j = columns;
	    }
      
      if (xpivot != -1)
	{
	  /* swap to actual row */
	  swap_rows(startr, xpivot);

	  index = ypivot;

	  TEILBARKEIT=0;

	  while(TEILBARKEIT==0)
	    {
	      TEILBARKEIT=1;

	      /* gcd2(row(startr),columns,TR2); */
	      tmp = value[startr];
	      do
		{
		  SW=0;
		  for (i = 0; i < columns; i++)
		    if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		      index = i;
	      
		  for (i = 0; i < columns; i++)
		    if (i != index && !tmp[i].is_zero())
		      {
			SW=1;
			div_rem(TMP1, TMP2, tmp[i], tmp[index]);
			for (j = 0; j < rows; j++)
			  {
			    multiply(TMP2, value[j][index], TMP1);
			    subtract(value[j][i], value[j][i], TMP2);
			  }
		      }
		}
	      while (SW== 1);
	  
	      for (i = 0; value[startr][i].is_zero() && i < columns; i++);
	      swap_columns(startc, i);
      
	      /* mgcd2(column(startc),rows,TR1); */
	      index = startr; /* no index search */
	      do
		{
		  SW=0;
		  for (i = 0; i < rows; i++)
		    if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		      index = i;
		  
		  for (i = 0; i < rows; i++)
		    if ((i != index) && !value[i][startc].is_zero())
		      {
			SW=1;
			tmp = value[i];
			div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
			for (j = 0; j < columns; j++)
			  {
			    multiply(TMP2, value[index][j], TMP1);
			    subtract(tmp[j], tmp[j], TMP2);
			  }
		      }
		}
	      while (SW==1);

	      for (i = 0; value[i][startc].is_zero() && i < rows; i++);
	      swap_rows(startr, i);

	      for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	      if(index!=columns)
		TEILBARKEIT=0;
	      index = startc;
	    }
	  
	  /* modulo test */
	  for (i = startr; i < rows; i++)
	    for (j = startc + 1; j < columns; j++)
	      if (value[i][j] % value[startr][startc] != 0)
		{
		  if (i != startr)
		    for (z = 0; z < columns; z++)
		      add(tmp[z], tmp[z], value[i][z]);
		  i = rows;
		  j = columns;
		  startc--;
		  startr--;
		}
	}
    }
  
  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      value[i][i].negate();
}

void 
snf_havas(bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"snf_havas(bigint_matrix &)");
  A.snf_havas();
}


void bigint_matrix::
snf_havas(bigint_matrix & T1, bigint_matrix & T2)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules
   ** IMPROVEMENTS: Havas, best reaminder include mgcd
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"snf_havas(bigint_matrix &, bigint_matrix &)");
  register long i, j, z, index;
  bigint PIVOT;
  bigint *tmp=NULL;

  if (T1.columns != rows)
    T1.set_no_of_columns(rows);
  if (T1.rows != rows)
    T1.set_no_of_rows(rows);

  if (T2.columns != columns)
    T2.set_no_of_columns(columns);
  if (T2.rows != columns)
    T2.set_no_of_rows(columns);

  T1.diag(1, 0);
  T2.diag(1, 0);

  register long startr, startc, xpivot, ypivot, SW, TEILBARKEIT;
  bigint TMP1, TMP2;

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
    {
      
      /* pivot: first non zero */
      xpivot = -1;
      ypivot = -1;
      for (i = startr; i < rows; i++)
	for (j = startc; j < columns; j++)
	  if (!value[i][j].is_zero())
	    {
	      xpivot = i;
	      ypivot = j;
	      i = rows;
	      j = columns;
	    }
      
      if (xpivot != -1)
	{
	  /* swap to actual row */
	  swap_rows(startr, xpivot);
	  T1.swap_rows(startr, xpivot);

	  index = ypivot;

	  TEILBARKEIT=0;

	  while(TEILBARKEIT==0)
	    {
	      TEILBARKEIT=1;

	      /* gcd2(row(startr),columns,TR2); */
	      tmp = value[startr];
	      do
		{
		  SW=0;
		  for (i = 0; i < columns; i++)
		    if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		      index = i;
	      
		  for (i = 0; i < columns; i++)
		    if (i != index && !tmp[i].is_zero())
		      {
			SW=1;
			div_rem(TMP1, TMP2, tmp[i], tmp[index]);
			for (j = 0; j < rows; j++)
			  {
			    multiply(TMP2, value[j][index], TMP1);
			    subtract(value[j][i], value[j][i], TMP2);
			  }
			for (j = 0; j < columns; j++)
			  {
			    multiply(TMP2, T2.value[j][index], TMP1);
			    subtract(T2.value[j][i], T2.value[j][i], TMP2);
			  }
		      }
		}
	      while (SW== 1);
	  
	      for (i = 0; value[startr][i].is_zero() && i < columns; i++);
	      swap_columns(startc, i);
	      T2.swap_columns(startc, i);
      
	      /* mgcd2(column(startc),rows,TR1); */
	      index = startr; /* no index search */
	      do
		{
		  SW=0;
		  for (i = 0; i < rows; i++)
		    if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		      index = i;
		  
		  for (i = 0; i < rows; i++)
		    if ((i != index) && !value[i][startc].is_zero())
		      {
			SW=1;
			tmp = value[i];
			div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
			for (j = 0; j < columns; j++)
			  {
			    multiply(TMP2, value[index][j], TMP1);
			    subtract(tmp[j], tmp[j], TMP2);
			  }
			for (j = 0; j < rows; j++)
			  {
			    multiply(TMP2, T1.value[index][j], TMP1);
			    subtract(T1.value[i][j], T1.value[i][j], TMP2);
			  }
		      }
		}
	      while (SW==1);

	      for (i = 0; value[i][startc].is_zero() && i < rows; i++);
	      swap_rows(startr, i);
	      T1.swap_rows(startr, i);

	      for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	      if(index!=columns)
		TEILBARKEIT=0;
	      index = startc;
	    }
	  
	  /* modulo test */
	  for (i = startr; i < rows; i++)
	    for (j = startc + 1; j < columns; j++)
	      if (value[i][j] % value[startr][startc] != 0)
		{
		  if (i != startr)
		    {
		      for (z = 0; z < columns; z++)
			add(tmp[z], tmp[z], value[i][z]);
		      for (z = 0; z < rows; z++)
			add(T1.value[startr][z], T1.value[startr][z], T1.value[i][z]);
		    }
		  i = rows;
		  j = columns;
		  startc--;
		  startr--;
		}
	}
    }
  
  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      {
	value[i][i].negate();
	for (z = 0; z < columns; z++)
	  T2.value[z][i].negate();
      }
}

void 
snf_havas(bigint_matrix & A, bigint_matrix & T1, bigint_matrix & T2)
{
  debug_handler("bigint_matrix", "in function "
		"snf_havas(bigint_matrix &, bigint_matrix &, bigint_matrix &)");
  A.snf_havas(T1, T2);
}

void bigint_matrix::
snf_mult(long art)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules + pivot selection
   ** VERSION: 1.8
   **/
  
  debug_handler("bigint_matrix", "in member - function "
		"snf_mult(bigint_matrix &, bigint_matrix &)");
  register long i, j, z, index, SW;
  bigint TMP1, TMP2;
  bigint *tmp=NULL;

  register long startr, startc, xpivot, ypivot, TEILBARKEIT;
  bigint ROW, COLUMN, PIVOT, NORM;

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
  {
    
    /* pivotselection: minimale C * R norm */
    xpivot = -1;
    ypivot = -1;
    PIVOT.assign_zero();
    for (i = startr; i < rows; i++)
      for (j = startc; j < columns; j++)
	{
	  if (PIVOT == abs(value[i][j]))
	    {
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      multiply(TMP1, ROW, COLUMN);
	      if (TMP1 < NORM)
		{
		  NORM.assign(TMP1);
		  PIVOT.assign(abs(value[i][j]));
		  xpivot = i;
		  ypivot = j;
		}
	    }

	  if ((PIVOT > abs(value[i][j]) && !value[i][j].is_zero()) || PIVOT.is_zero())
	    {
	      PIVOT.assign(abs(value[i][j]));
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      multiply(NORM, ROW, COLUMN);
	      xpivot = i;
	      ypivot = j;
	    }
	}

    if (!PIVOT.is_zero())
      {
	
	/* swap to actual row */
	swap_rows(startr, xpivot);
	
	index = ypivot;

	TEILBARKEIT=0;

	while(TEILBARKEIT==0)
	  {
	    TEILBARKEIT=1;
	    
	    /* gcd2(row(startr),columns,TR2); */
	    tmp = value[startr];
	    do
	      {
		SW=0;
		for (i = 0; i < columns; i++)
		  if ((i != index) && !tmp[i].is_zero())
		    {
		      SW=1;
		      div_rem(TMP1, TMP2, tmp[i], tmp[index]);
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, value[j][index], TMP1);
			  subtract(value[j][i], value[j][i], TMP2);
			}
		    }
		
		for (i = 0; i < columns; i++)
		  if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		    index = i;
	      }
	    while (SW==1);

	    for (i = 0; value[startr][i].is_zero(); i++);
	    swap_columns(startc, i);
	
	    /* mgcd2(column(startc),rows,TR1); */
	    index = startr;
	    do
	      {
		SW=0;
		for (i = 0; i < rows; i++)
		  if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		    index = i;
		
		for (i = 0; i < rows; i++)
		  if ((i != index) && !value[i][startc].is_zero())
		    {
		      SW=1;
		      tmp = value[i];
		      div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, value[index][j], TMP1);
			  subtract(tmp[j], tmp[j], TMP2);
			}
		    }
	      }
	    while (SW==1);

	    for (i = 0; value[i][startc].is_zero(); i++);
	    swap_rows(startr, i);

	    tmp = value[startr];
	    for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	    if (index!=columns)
	      TEILBARKEIT=0;

	    index=startr;
	  }

	/* modulo test */
	for (i = startr; i < rows; i++)
	  for (j = startc + 1; j < columns; j++)
	    if (value[i][j] % value[startr][startc] != 0)
	      {
		if (i != startr)
		  for (z = 0; z < columns; z++)
		    add(tmp[z], tmp[z], value[i][z]);
		i = rows;
		j = columns;
		startc--;
		startr--;
	      }
      }
  }

  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      value[i][i].negate();
}

void 
snf_mult(bigint_matrix & A, long art)
{
  debug_handler("bigint_matrix", "in function "
		"snf_mult(bigint_matrix &, long)");
  A.snf_mult(art);
}

void bigint_matrix::
snf_mult(bigint_matrix & T1, bigint_matrix & T2, long art)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules + pivot selection
   ** VERSION: 1.8
   **/
  
  debug_handler("bigint_matrix", "in member - function "
		"snf_mult(bigint_matrix &, bigint_matrix &)");
  register long i, j, z, index, SW;
  bigint TMP1, TMP2;
  bigint *tmp=NULL;

  if (T1.columns != rows)
    T1.set_no_of_columns(rows);
  if (T1.rows != rows)
    T1.set_no_of_rows(rows);

  if (T2.columns != columns)
    T2.set_no_of_columns(columns);
  if (T2.rows != columns)
    T2.set_no_of_rows(columns);

  T1.diag(1, 0);
  T2.diag(1, 0);

  register long startr, startc, xpivot, ypivot, TEILBARKEIT;
  bigint ROW, COLUMN, PIVOT, NORM;

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
  {
    
    /* pivotselection: minimale C * R norm */
    xpivot = -1;
    ypivot = -1;
    PIVOT.assign_zero();
    for (i = startr; i < rows; i++)
      for (j = startc; j < columns; j++)
	{
	  if (PIVOT == abs(value[i][j]))
	    {
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      multiply(TMP1, ROW, COLUMN);
	      if (TMP1 < NORM)
		{
		  NORM.assign(TMP1);
		  PIVOT.assign(abs(value[i][j]));
		  xpivot = i;
		  ypivot = j;
		}
	    }

	  if ((PIVOT > abs(value[i][j]) && !value[i][j].is_zero()) || PIVOT.is_zero())
	    {
	      PIVOT.assign(abs(value[i][j]));
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      multiply(NORM, ROW, COLUMN);
	      xpivot = i;
	      ypivot = j;
	    }
	}

    if (!PIVOT.is_zero())
      {
	
	/* swap to actual row */
	swap_rows(startr, xpivot);
	T1.swap_rows(startr, xpivot);
	
	index = ypivot;

	TEILBARKEIT=0;

	while(TEILBARKEIT==0)
	  {
	    TEILBARKEIT=1;
	    
	    /* gcd2(row(startr),columns,TR2); */
	    tmp = value[startr];
	    do
	      {
		SW=0;
		for (i = 0; i < columns; i++)
		  if ((i != index) && !tmp[i].is_zero())
		    {
		      SW=1;
		      div_rem(TMP1, TMP2, tmp[i], tmp[index]);
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, value[j][index], TMP1);
			  subtract(value[j][i], value[j][i], TMP2);
			}
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, T2.value[j][index], TMP1);
			  subtract(T2.value[j][i], T2.value[j][i], TMP2);
			}
		    }
		
		for (i = 0; i < columns; i++)
		  if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		    index = i;
	      }
	    while (SW==1);

	    for (i = 0; value[startr][i].is_zero(); i++);
	    swap_columns(startc, i);
	    T2.swap_columns(startc, i);
	
	    /* mgcd2(column(startc),rows,TR1); */
	    index = startr;
	    do
	      {
		SW=0;
		for (i = 0; i < rows; i++)
		  if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		    index = i;
		
		for (i = 0; i < rows; i++)
		  if ((i != index) && !value[i][startc].is_zero())
		    {
		      SW=1;
		      tmp = value[i];
		      div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, value[index][j], TMP1);
			  subtract(tmp[j], tmp[j], TMP2);
			}
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, T1.value[index][j], TMP1);
			  subtract(T1.value[i][j], T1.value[i][j], TMP2);
			}
		    }
	      }
	    while (SW==1);

	    for (i = 0; value[i][startc].is_zero(); i++);
	    swap_rows(startr, i);
	    T1.swap_rows(startr, i);

	    tmp = value[startr];
	    for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	    if (index!=columns)
	      TEILBARKEIT=0;

	    index=startr;
	  }

	/* modulo test */
	for (i = startr; i < rows; i++)
	  for (j = startc + 1; j < columns; j++)
	    if (value[i][j] % value[startr][startc] != 0)
	      {
		if (i != startr)
		  {
		    for (z = 0; z < columns; z++)
		      add(tmp[z], tmp[z], value[i][z]);
		    for (z = 0; z < rows; z++)
		      add(T1.value[startr][z], T1.value[startr][z], T1.value[i][z]);
		  }
		i = rows;
		j = columns;
		startc--;
		startr--;
	      }
      }
  }

  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      {
	value[i][i].negate();
	for (z = 0; z < columns; z++)
	  T2.value[z][i].negate();
      }
}

void 
snf_mult(bigint_matrix & A, bigint_matrix & T1, bigint_matrix & T2, long art)
{
  debug_handler("bigint_matrix", "in function "
		"snf_mult(bigint_matrix &, bigint_matrix &, bigint_matrix &, long)");
  A.snf_mult(T1, T2, art);
}

void bigint_matrix::
snf_add(long art)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules + pivot selection
   ** VERSION: 1.8
   **/
  
  debug_handler("bigint_matrix", "in member - function "
		"snf_add(long)");
  register long i, j, z, index, SW;
  bigint TMP1, TMP2;
  bigint *tmp=NULL;

  register long startr, startc, xpivot, ypivot, TEILBARKEIT;
  bigint ROW, COLUMN, PIVOT, NORM;

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
  {
    
    /* pivotselection: minimale C * R norm */
    xpivot = -1;
    ypivot = -1;
    PIVOT.assign_zero();
    for (i = startr; i < rows; i++)
      for (j = startc; j < columns; j++)
	{
	  if (PIVOT == abs(value[i][j]))
	    {
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      add(TMP1, ROW, COLUMN);
	      if (TMP1 < NORM)
		{
		  NORM.assign(TMP1);
		  PIVOT.assign(abs(value[i][j]));
		  xpivot = i;
		  ypivot = j;
		}
	    }

	  if ((PIVOT > abs(value[i][j]) && !value[i][j].is_zero()) || PIVOT.is_zero())
	    {
	      PIVOT.assign(abs(value[i][j]));
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      add(NORM, ROW, COLUMN);
	      xpivot = i;
	      ypivot = j;
	    }
	}

    if (!PIVOT.is_zero())
      {
	
	/* swap to actual row */
	swap_rows(startr, xpivot);
	
	index = ypivot;

	TEILBARKEIT=0;

	while(TEILBARKEIT==0)
	  {
	    TEILBARKEIT=1;
	    
	    /* gcd2(row(startr),columns,TR2); */
	    tmp = value[startr];
	    do
	      {
		SW=0;
		for (i = 0; i < columns; i++)
		  if ((i != index) && !tmp[i].is_zero())
		    {
		      SW=1;
		      div_rem(TMP1, TMP2, tmp[i], tmp[index]);
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, value[j][index], TMP1);
			  subtract(value[j][i], value[j][i], TMP2);
			}
		    }
		
		for (i = 0; i < columns; i++)
		  if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		    index = i;
	      }
	    while (SW==1);

	    for (i = 0; value[startr][i].is_zero(); i++);
	    swap_columns(startc, i);
	
	    /* mgcd2(column(startc),rows,TR1); */
	    index = startr;
	    do
	      {
		SW=0;
		for (i = 0; i < rows; i++)
		  if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		    index = i;
		
		for (i = 0; i < rows; i++)
		  if ((i != index) && !value[i][startc].is_zero())
		    {
		      SW=1;
		      tmp = value[i];
		      div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, value[index][j], TMP1);
			  subtract(tmp[j], tmp[j], TMP2);
			}
		    }
	      }
	    while (SW==1);

	    for (i = 0; value[i][startc].is_zero(); i++);
	    swap_rows(startr, i);

	    tmp = value[startr];
	    for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	    if (index!=columns)
	      TEILBARKEIT=0;

	    index=startr;
	  }

	/* modulo test */
	for (i = startr; i < rows; i++)
	  for (j = startc + 1; j < columns; j++)
	    if (value[i][j] % value[startr][startc] != 0)
	      {
		if (i != startr)
		  for (z = 0; z < columns; z++)
		    add(tmp[z], tmp[z], value[i][z]);
		i = rows;
		j = columns;
		startc--;
		startr--;
	      }
      }
  }

  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      value[i][i].negate();
}

void 
snf_add(bigint_matrix & A, long art)
{
  debug_handler("bigint_matrix", "in function "
		"snf_add(bigint_matrix &, long)");
  A.snf_add(art);
}

void bigint_matrix::
snf_add(bigint_matrix & T1, bigint_matrix & T2, long art)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules + pivot selection
   ** VERSION: 1.8
   **/
  
  debug_handler("bigint_matrix", "in member - function "
		"snf_add(bigint_matrix &, bigint_matrix &, long)");
  register long i, j, z, index, SW;
  bigint TMP1, TMP2;
  bigint *tmp=NULL;

  if (T1.columns != rows)
    T1.set_no_of_columns(rows);
  if (T1.rows != rows)
    T1.set_no_of_rows(rows);

  if (T2.columns != columns)
    T2.set_no_of_columns(columns);
  if (T2.rows != columns)
    T2.set_no_of_rows(columns);

  T1.diag(1, 0);
  T2.diag(1, 0);

  register long startr, startc, xpivot, ypivot, TEILBARKEIT;
  bigint ROW, COLUMN, PIVOT, NORM;

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
  {
    
    /* pivotselection: minimale C * R norm */
    xpivot = -1;
    ypivot = -1;
    PIVOT.assign_zero();
    for (i = startr; i < rows; i++)
      for (j = startc; j < columns; j++)
	{
	  if (PIVOT == abs(value[i][j]))
	    {
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      add(TMP1, ROW, COLUMN);
	      if (TMP1 < NORM)
		{
		  NORM.assign(TMP1);
		  PIVOT.assign(abs(value[i][j]));
		  xpivot = i;
		  ypivot = j;
		}
	    }

	  if ((PIVOT > abs(value[i][j]) && !value[i][j].is_zero()) || PIVOT.is_zero())
	    {
	      PIVOT.assign(abs(value[i][j]));
	      row_norm(ROW, i, art);
	      column_norm(COLUMN, j, art);
	      add(NORM, ROW, COLUMN);
	      xpivot = i;
	      ypivot = j;
	    }
	}

    if (!PIVOT.is_zero())
      {
	
	/* swap to actual row */
	swap_rows(startr, xpivot);
	T1.swap_rows(startr, xpivot);
	
	index = ypivot;

	TEILBARKEIT=0;

	while(TEILBARKEIT==0)
	  {
	    TEILBARKEIT=1;
	    
	    /* gcd2(row(startr),columns,TR2); */
	    tmp = value[startr];
	    do
	      {
		SW=0;
		for (i = 0; i < columns; i++)
		  if ((i != index) && !tmp[i].is_zero())
		    {
		      SW=1;
		      div_rem(TMP1, TMP2, tmp[i], tmp[index]);
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, value[j][index], TMP1);
			  subtract(value[j][i], value[j][i], TMP2);
			}
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, T2.value[j][index], TMP1);
			  subtract(T2.value[j][i], T2.value[j][i], TMP2);
			}
		    }
		
		for (i = 0; i < columns; i++)
		  if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		    index = i;
	      }
	    while (SW==1);

	    for (i = 0; value[startr][i].is_zero(); i++);
	    swap_columns(startc, i);
	    T2.swap_columns(startc, i);
	
	    /* mgcd2(column(startc),rows,TR1); */
	    index = startr;
	    do
	      {
		SW=0;
		for (i = 0; i < rows; i++)
		  if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		    index = i;
		
		for (i = 0; i < rows; i++)
		  if ((i != index) && !value[i][startc].is_zero())
		    {
		      SW=1;
		      tmp = value[i];
		      div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, value[index][j], TMP1);
			  subtract(tmp[j], tmp[j], TMP2);
			}
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, T1.value[index][j], TMP1);
			  subtract(T1.value[i][j], T1.value[i][j], TMP2);
			}
		    }
	      }
	    while (SW==1);

	    for (i = 0; value[i][startc].is_zero(); i++);
	    swap_rows(startr, i);
	    T1.swap_rows(startr, i);

	    tmp = value[startr];
	    for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	    if (index!=columns)
	      TEILBARKEIT=0;

	    index=startr;
	  }

	/* modulo test */
	for (i = startr; i < rows; i++)
	  for (j = startc + 1; j < columns; j++)
	    if (value[i][j] % value[startr][startc] != 0)
	      {
		if (i != startr)
		  {
		    for (z = 0; z < columns; z++)
		      add(tmp[z], tmp[z], value[i][z]);
		    for (z = 0; z < rows; z++)
		      add(T1.value[startr][z], T1.value[startr][z], T1.value[i][z]);
		  }
		i = rows;
		j = columns;
		startc--;
		startr--;
	      }
      }
  }

  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      {
	value[i][i].negate();
	for (z = 0; z < columns; z++)
	  T2.value[z][i].negate();
      }
}

void 
snf_add(bigint_matrix & A, bigint_matrix & T1, bigint_matrix & T2, long art)
{
  debug_handler("bigint_matrix", "in function "
		"snf_add(bigint_matrix &, bigint_matrix &, bigint_matrix &, long)");
  A.snf_add(T1, T2, art);
}

void bigint_matrix::
snf_new(long art)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules + pivot selection
   ** VERSION: 1.8
   **/
  
  debug_handler("bigint_matrix", "in member - function "
		"snf_new(long)");
  register long i, j, z, index, SW;
  bigint TMP1, TMP2;
  bigint *tmp=NULL;

  register long startr, startc, xpivot, ypivot, TEILBARKEIT;
  bigint ROW, COLUMN, PIVOT, NORM;

  bigint *RO = new bigint[rows];
  bigint *CO = new bigint[columns];

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
  {
    /* norm computation */
    for(i=0;i<rows;i++)
      row_norm(RO[i],i,art);
    for(i=0;i<columns;i++)
      column_norm(CO[i],i,art);

    /* pivotselection: new minimale C * R norm */
    xpivot = -1;
    ypivot = -1;
    PIVOT.assign_zero();
    NORM.assign_zero();
    for (i = startr; i < rows; i++)
      for (j = startc; j < columns; j++)
	{
	  multiply(TMP1,RO[i],CO[j]);
	  
	  if (!value[i][j].is_zero() && (NORM > TMP1 || PIVOT.is_zero()))
	    {
	      NORM.assign(TMP1);
	      PIVOT.assign(abs(value[i][j]));
	      xpivot = i;
	      ypivot = j;
	    }
	    
	  if (NORM==TMP1 && !PIVOT.is_zero() && !value[i][j].is_zero())
	    {
	      if (PIVOT > abs(value[i][j]))
		{
		  PIVOT.assign(abs(value[i][j]));
		  NORM.assign(TMP1);
		  xpivot = i;
		  ypivot = j;
		}
	    }
	}

    if (!PIVOT.is_zero())
      {
	
	/* swap to actual row */
	swap_rows(startr, xpivot);
	
	index = ypivot;

	TEILBARKEIT=0;

	while(TEILBARKEIT==0)
	  {
	    TEILBARKEIT=1;
	    
	    /* gcd2(row(startr),columns,TR2); */
	    tmp = value[startr];
	    do
	      {
		SW=0;
		for (i = 0; i < columns; i++)
		  if ((i != index) && !tmp[i].is_zero())
		    {
		      SW=1;
		      div_rem(TMP1, TMP2, tmp[i], tmp[index]);
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, value[j][index], TMP1);
			  subtract(value[j][i], value[j][i], TMP2);
			}
		    }
		
		for (i = 0; i < columns; i++)
		  if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		    index = i;
	      }
	    while (SW==1);

	    for (i = 0; value[startr][i].is_zero(); i++);
	    swap_columns(startc, i);
	
	    /* mgcd2(column(startc),rows,TR1); */
	    index = startr;
	    do
	      {
		SW=0;
		for (i = 0; i < rows; i++)
		  if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		    index = i;
		
		for (i = 0; i < rows; i++)
		  if ((i != index) && !value[i][startc].is_zero())
		    {
		      SW=1;
		      tmp = value[i];
		      div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, value[index][j], TMP1);
			  subtract(tmp[j], tmp[j], TMP2);
			}
		    }
	      }
	    while (SW==1);

	    for (i = 0; value[i][startc].is_zero(); i++);
	    swap_rows(startr, i);

	    tmp = value[startr];
	    for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	    if (index!=columns)
	      TEILBARKEIT=0;

	    index=startr;
	  }

	/* modulo test */
	for (i = startr; i < rows; i++)
	  for (j = startc + 1; j < columns; j++)
	    if (value[i][j] % value[startr][startc] != 0)
	      {
		if (i != startr)
		  for (z = 0; z < columns; z++)
		    add(tmp[z], tmp[z], value[i][z]);
		i = rows;
		j = columns;
		startc--;
		startr--;
	      }
      }
  }

  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      value[i][i].negate();
}

void 
snf_new(bigint_matrix & A, long art)
{
  debug_handler("bigint_matrix", "in function "
		"snf_new(bigint_matrix &, long)");
  A.snf_new(art);
}

void bigint_matrix::
snf_new(bigint_matrix & T1, bigint_matrix & T2, long art)
{
  /**
   ** DESCRIPTION: snf Computation
   ** ALGORITHM: given in Hartley and Hawkes
   ** PAPER: Recognizing badly represented Z-modules + pivot selection
   ** VERSION: 1.8
   **/
  
  debug_handler("bigint_matrix", "in member - function "
		"snf_new(bigint_matrix &, bigint_matrix &, long)");
  register long i, j, z, index, SW;
  bigint TMP1, TMP2;
  bigint *tmp=NULL;

  if (T1.columns != rows)
    T1.set_no_of_columns(rows);
  if (T1.rows != rows)
    T1.set_no_of_rows(rows);

  if (T2.columns != columns)
    T2.set_no_of_columns(columns);
  if (T2.rows != columns)
    T2.set_no_of_rows(columns);

  T1.diag(1, 0);
  T2.diag(1, 0);

  register long startr, startc, xpivot, ypivot, TEILBARKEIT;
  bigint ROW, COLUMN, PIVOT, NORM;

  bigint *RO = new bigint[rows];
  bigint *CO = new bigint[columns];

  for (startc = 0, startr = 0; startr < rows && startc < columns; startr++, startc++)
  {
    /* norm computation */
    for(i=0;i<rows;i++)
      row_norm(RO[i],i,art);
    for(i=0;i<columns;i++)
      column_norm(CO[i],i,art);

    /* pivotselection: new minimale C * R norm */
    xpivot = -1;
    ypivot = -1;
    PIVOT.assign_zero();
    NORM.assign_zero();
    for (i = startr; i < rows; i++)
      for (j = startc; j < columns; j++)
	{
	  /*row_norm(ROW,i,art);
	  column_norm(COLUMN,j,art);*/
	  multiply(TMP1,RO[i],CO[j]);
	  
	  if (!value[i][j].is_zero() && (NORM > TMP1 || PIVOT.is_zero()))
	    {
	      NORM.assign(TMP1);
	      PIVOT.assign(abs(value[i][j]));
	      xpivot = i;
	      ypivot = j;
	    }
	    
	  if (NORM==TMP1 && !PIVOT.is_zero() && !value[i][j].is_zero())
	    {
	      if (PIVOT > abs(value[i][j]))
		{
		  PIVOT.assign(abs(value[i][j]));
		  NORM.assign(TMP1);
		  xpivot = i;
		  ypivot = j;
		}
	    }
	}

    if (!PIVOT.is_zero())
      {
	
	/* swap to actual row */
	swap_rows(startr, xpivot);
	T1.swap_rows(startr, xpivot);
	
	index = ypivot;

	TEILBARKEIT=0;

	while(TEILBARKEIT==0)
	  {
	    TEILBARKEIT=1;
	    
	    /* gcd2(row(startr),columns,TR2); */
	    tmp = value[startr];
	    do
	      {
		SW=0;
		for (i = 0; i < columns; i++)
		  if ((i != index) && !tmp[i].is_zero())
		    {
		      SW=1;
		      div_rem(TMP1, TMP2, tmp[i], tmp[index]);
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, value[j][index], TMP1);
			  subtract(value[j][i], value[j][i], TMP2);
			}
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, T2.value[j][index], TMP1);
			  subtract(T2.value[j][i], T2.value[j][i], TMP2);
			}
		    }
		
		for (i = 0; i < columns; i++)
		  if ((abs(tmp[index]) > abs(tmp[i])) && !tmp[i].is_zero())
		    index = i;
	      }
	    while (SW==1);

	    for (i = 0; value[startr][i].is_zero(); i++);
	    swap_columns(startc, i);
	    T2.swap_columns(startc, i);
	
	    /* mgcd2(column(startc),rows,TR1); */
	    index = startr;
	    do
	      {
		SW=0;
		for (i = 0; i < rows; i++)
		  if ((abs(value[index][startc]) > abs(value[i][startc])) && !value[i][startc].is_zero())
		    index = i;
		
		for (i = 0; i < rows; i++)
		  if ((i != index) && !value[i][startc].is_zero())
		    {
		      SW=1;
		      tmp = value[i];
		      div_rem(TMP1, TMP2, tmp[startc], value[index][startc]);
		      for (j = 0; j < columns; j++)
			{
			  multiply(TMP2, value[index][j], TMP1);
			  subtract(tmp[j], tmp[j], TMP2);
			}
		      for (j = 0; j < rows; j++)
			{
			  multiply(TMP2, T1.value[index][j], TMP1);
			  subtract(T1.value[i][j], T1.value[i][j], TMP2);
			}
		    }
	      }
	    while (SW==1);

	    for (i = 0; value[i][startc].is_zero(); i++);
	    swap_rows(startr, i);
	    T1.swap_rows(startr, i);

	    tmp = value[startr];
	    for (index=startc+1;index<columns && tmp[index].is_zero();index++);
	    if (index!=columns)
	      TEILBARKEIT=0;

	    index=startr;
	  }

	/* modulo test */
	for (i = startr; i < rows; i++)
	  for (j = startc + 1; j < columns; j++)
	    if (value[i][j] % value[startr][startc] != 0)
	      {
		if (i != startr)
		  {
		    for (z = 0; z < columns; z++)
		      add(tmp[z], tmp[z], value[i][z]);
		    for (z = 0; z < rows; z++)
		      add(T1.value[startr][z], T1.value[startr][z], T1.value[i][z]);
		  }
		i = rows;
		j = columns;
		startc--;
		startr--;
	      }
      }
  }

  /* diagonal >= 0 */
  for (i = 0; i < rows && i < columns; i++)
    if (value[i][i] < 0)
      {
	value[i][i].negate();
	for (z = 0; z < columns; z++)
	  T2.value[z][i].negate();
      }
}

void 
snf_new(bigint_matrix & A, bigint_matrix & T1, bigint_matrix & T2, long art)
{
  debug_handler("bigint_matrix", "in function "
		"snf_new(bigint_matrix &, bigint_matrix &, bigint_matrix &, long)");
  A.snf_new(T1, T2, art);
}

void bigint_matrix::
snfmod_dkt()
{
  debug_handler("bigint_matrix", "in member - function "
		"snfmod_dkt()");
  snfmod_dkt(latticedet2());
}

void
snfmod_dkt(bigint_matrix & A)
{
  debug_handler("bigint_matrix", "in function "
		"snfmod_dkt(bigint_matrix &)");
  A.snfmod_dkt(A.latticedet2());
}

void bigint_matrix::
snfmod_dkt(const bigint &mod)
{
  /**
   ** DESCRIPTION: A.snfmod_dkt(mod);
   **              => A in Smith normal form
   **              => h = lattice determinant of lattice formed
   **              by the columns of matrix A
   ** PAPER: Asymptotically fast triangulazation of matrices over rings
   ** IMPROVEMENT: Hafner, McCurly
   ** ERROR: rank != rows
   ** VERSION: 1.8
   **/

  debug_handler_l("bigint_matrix", "in member - function "
		  "snfmod_dkt(const bigint &)",5);
  
  if (rank() != rows)
    lidia_error_handler("bigint_matrix", "snfmod_dkt :: "
		  "number of rows != rank");
  
  register long diagindex, j, z, l;
  
  bigint RES0, RES1, RES2, RES3;  /* 0=lggT,1=rggt,2=ggt */
  bigint x, y;
  bigint TMP, TMP1, TMP2, TMP3;
  bigint *Atmp, *Atmp1 = NULL;
  
  /* A = A % mod */
  for (z = 0; z < rows; z++)
    {
      Atmp = value[z];
      for (j = 0; j < columns; j++)
	best_remainder(Atmp[j], Atmp[j], mod);
    }
  
  /* Step 3 - 5 */
  for (diagindex = 0; diagindex < rows; diagindex++)
    {
      Atmp = value[diagindex];
      
      /* if diagonalelement == 0 then diagonalelement = mod */
      if (Atmp[diagindex].is_zero())
	Atmp[diagindex].assign(mod);
      
      /* Step 6 -8 */
      for (j = diagindex+1; j < columns; j++)
	if (!Atmp[j].is_zero())
	  {
	    /* columns reduction */
	    RES2 = xgcd(RES0, RES1, Atmp[j], Atmp[diagindex]);
	    div_rem(x, RES3, Atmp[diagindex], RES2);
	    div_rem(y, RES3, Atmp[j], RES2);
	    
	    for (z = 0; z < rows; z++)
	      {
		Atmp1 = value[z];
		TMP.assign(Atmp1[j]);
		TMP1.assign(Atmp1[diagindex]);
		
		mult_mod(TMP2,TMP,x,mod);
		mult_mod(TMP3,TMP1,y,mod);
		sub_mod(Atmp1[j],TMP2,TMP3,mod);
		
		mult_mod(TMP2,TMP,RES0,mod);
		mult_mod(TMP3,TMP1,RES1,mod);
		add_mod(Atmp1[diagindex],TMP2,TMP3,mod);
	      }
	  }
      
      for (j = diagindex+1; j < rows; j++)
	if (!value[j][diagindex].is_zero())
	  {
	    /* row reduction */
	    RES2 = xgcd(RES0, RES1, value[j][diagindex], Atmp[diagindex]);
	    div_rem(x, RES3, Atmp[diagindex], RES2);
	    div_rem(y, RES3, value[j][diagindex], RES2);
	    
	    for (z = 0; z < columns; z++)
	      {
		TMP.assign(value[j][z]);
		TMP1.assign(value[diagindex][z]);
		
		mult_mod(TMP2,TMP,x,mod);
		mult_mod(TMP3,TMP1,y,mod);
		sub_mod(value[j][z],TMP2,TMP3,mod);
		
		mult_mod(TMP2,TMP,RES0,mod);
		mult_mod(TMP3,TMP1,RES1,mod);
		add_mod(value[diagindex][z],TMP2,TMP3,mod);
	      }
	  }
      
      /* value[diagindex][diagindex] | value[i][j] ??? */
      TMP = Atmp[diagindex];
      for (j=diagindex+1;j<rows;j++)
	for (z=diagindex+1;z<columns;z++)
	  {
	    if (value[j][z] % TMP != 0)
	      {
		if (j != diagindex)
		  for (l = diagindex; l<columns; l++)
		    add_mod(Atmp[l],Atmp[l],value[j][l],mod);
		j = rows;
		z = columns;
	      }
	  }     
      
      for (z=diagindex+1; z<columns && Atmp[z].is_zero(); z++);
      if (z != columns)
	diagindex--;     
    }
  
  /* Step 29 - 32 */
  bigint D = mod;
  for (j = 0; j < rows; j++)
    {
      Atmp = value[j];
      Atmp[j].assign(xgcd(RES0, RES1, Atmp[j], D));       
      div_rem(D, TMP, D, Atmp[j]);
    }
}

void
snfmod_dkt(bigint_matrix & A, const bigint & h)
{
  debug_handler("bigint_matrix", "in function "
		"snfmod_dkt(bigint_matrix &, const bigint &)");
  A.snfmod_dkt(h);
}

void bigint_matrix::
snfmod_cohen()
{
  debug_handler("bigint_matrix", "in member - function "
		"snfmod_cohen()");
   snfmod_cohen(latticedet2());
}

void 
snfmod_cohen(bigint_matrix &A)
{
  debug_handler("bigint_matrix", "in function "
		"snfmod_cohen(bigint_matrix &)");
  A.snfmod_cohen(A.latticedet2());
}

void bigint_matrix::
snfmod_cohen(const bigint & mod)
{
  /**
   ** DESCRIPTION: A.snfmod_cohen(mod);
   **              => A in Smith normal form
   **              => mod = multiple of lattice determinant of lattice formed
   **              by the columns of matrix A
   ** ERROR: rank != rows
   ** VERSION: 1.8
   **/

  debug_handler("bigint_matrix", "in member - function "
		"snfmod_cohen(const bigint &)");
  if (rank() != rows )
    lidia_error_handler("bigint_matrix", "snfmod_cohen :: "
		  "number of rows != rang");
  
  register long diagindex, j, z, l;
  
  bigint RES0, RES1, RES2, RES3;  /* 0=lggT,1=rggt,2=ggt */
  bigint x, y;
  bigint TMP, TMP1, TMP2, TMP3;
  bigint *Atmp, *Atmp1 = NULL;
  bigint D = mod;

  /* Step 1 */
  for (diagindex = 0; diagindex < rows; diagindex++)
    {
      Atmp = value[diagindex];
      
      if (Atmp[diagindex].is_zero())
	Atmp[diagindex].assign(mod);
      
      /* Step 2 - 4 */
      for (j = diagindex+1; j < columns; j++)
	if (!Atmp[j].is_zero())
	  {
	    /* columns reduction */
	    RES2 = xgcd(RES0, RES1, Atmp[j], Atmp[diagindex]);
	    div_rem(x, RES3, Atmp[diagindex], RES2);
	    div_rem(y, RES3, Atmp[j], RES2);
	    
	    for (z = 0; z < rows; z++)
	      {
		Atmp1 = value[z];
		TMP.assign(Atmp1[j]);
		TMP1.assign(Atmp1[diagindex]);
		
		mult_mod(TMP2,TMP,x,mod);
		mult_mod(TMP3,TMP1,y,mod);
		sub_mod(Atmp1[j],TMP2,TMP3,mod);
		
		mult_mod(TMP2,TMP,RES0,mod);
		mult_mod(TMP3,TMP1,RES1,mod);
		add_mod(Atmp1[diagindex],TMP2,TMP3,mod);
	      }
	  }
      
      /* Step 5 - 7 */
      for (j = diagindex+1; j < rows; j++)
	if (!value[j][diagindex].is_zero())
	  {
	    /* row reduction */
	    RES2 = xgcd(RES0, RES1, value[j][diagindex], Atmp[diagindex]);
	    div_rem(x, RES3, Atmp[diagindex], RES2);
	    div_rem(y, RES3, value[j][diagindex], RES2);
	    
	    for (z = 0; z < columns; z++)
	      {
		TMP.assign(value[j][z]);
		TMP1.assign(value[diagindex][z]);
		
		mult_mod(TMP2,TMP,x,mod);
		mult_mod(TMP3,TMP1,y,mod);
		sub_mod(value[j][z],TMP2,TMP3,mod);
		
		mult_mod(TMP2,TMP,RES0,mod);
		mult_mod(TMP3,TMP1,RES1,mod);
		add_mod(value[diagindex][z],TMP2,TMP3,mod);
	      }
	  }
      
      /* Step 8,9 */
      TMP = Atmp[diagindex];
      
      for (j=diagindex+1;j<rows;j++)
	for (z=diagindex+1;z<columns;z++)
	  {
	    if (value[j][z] % TMP != 0)
	      {
		if (j != diagindex)
		  for (l = diagindex; l<columns; l++)
		    add_mod(Atmp[l],Atmp[l],value[j][l],mod);
		j = rows;
		z = columns;
	      }
	  }     
      
      for (z=diagindex+1; z<columns && Atmp[z].is_zero(); z++);
      if (z != columns)
	diagindex--;
      else
	{
	  /* Step 10 */
	  Atmp[diagindex]=xgcd(TMP1,TMP2,TMP,D);
	  div_rem(D,TMP1,D,Atmp[diagindex]);
	}
    }
}

void 
snfmod_cohen(bigint_matrix &A, const bigint &DET)
{
  debug_handler("bigint_matrix", "in function "
		"snfmod_cohen(bigint_matrix &, const bigint &)");
  A.snfmod_cohen(DET);
}

/**
 ** END: Linear algebra
 ** PART 2
 **/

void bigint_matrix::
gauss()
{
  debug_handler("bigint_matrix", "in member - function gauss()");

  bigint_matrix TR(columns, columns);
 bigint *REM;
  register long startr = 0, startc = 0, i;

  for (startc = columns - 1, startr = rows - 1; startr >= 0 && startc >= 0; startr--, startc--)
  {
    bigint *ZU = row(startr);
    for (i = startc + 1; i < columns; ZU[i].assign_zero(), i++);
    REM=mgcd1(ZU, columns, TR);
    delete[] REM;
    delete[] ZU;
    
    TR = TR.trans();
    multiply(*this, *this, TR);
    for (i = 0; value[startr][i].is_zero() && i <= startc; i++);
    if (i > startc)
      return;
    swap_columns(startc, i);

  }
}
















