//
// LiDIA - a library for computational number theory
//   Copyright (c) 1995 by the LiDIA Group
//
// File        : lattice_gensys.c 
// Author      : Werner Backes (WB), Thorsten Lauer (TL) 
// Last change : WB/TL, Feb 14 1995, initial version
//

#include <LiDIA/lattice_gensys.h>

//
// Constructors / Destructors
//

lattice_gensys::lattice_gensys(long n, long m)
{
  debug_handler("lattice_gensys","lattice_gensys(n, m)");
  if ((n > 0) && (m > 0))
    {
      lattice_rows=n;
      lattice_cols=m;
      comp_mode=bigfloat_mode;
      io_mode=own_io_mode;
      lbf=new bigfloat_lattice(lattice_rows, lattice_cols);
      memory_handler(lbf,"lattice_gensys","lattice_gensys(n, m) :: "
                         "not enough memory");
      lattice_read=false;
      prec=0;
      red_count=0;
      y_param=1.0;
      y_zaehler=1;
      y_nenner=1;
      return;
    }
  lidia_error_handler("lattice_gensys","lattice_gensys(n, m) :: "
                "illegal lattice size !");
}

lattice_gensys::lattice_gensys(long n, long m, double** ado)
{
  debug_handler("lattice_gensys","lattice_gensys(n, m, ado)");
  if ((n > 0) && (m > 0))
    {
      lattice_rows=n;
      lattice_cols=m;
      comp_mode=double_mode;
      io_mode=own_io_mode;
      ldo=new double_lattice(lattice_rows, lattice_cols);
      memory_handler(ldo,"lattice_gensys","lattice_gensys(n, m, ado) :: "
                         "not enough memory");
      sto(lattice_rows, lattice_cols, ado);
      lattice_read=true;
      prec=0;
      red_count=0;
      y_param=1.0;
      y_zaehler=1;
      y_nenner=1;
      return;
    }
  lidia_error_handler("lattice_gensys","lattice_gensys(n, m, ado) :: "
                "illegal lattice size !");
}

lattice_gensys::lattice_gensys(long n, long m, bigint** abi)
{
  debug_handler("lattice_gensys","lattice_gensys(n, m, abi)");
  if ((n > 0) && (m > 0))
    {
      lattice_rows=n;
      lattice_cols=m;
      comp_mode=bigint_mode;
      io_mode=own_io_mode;
      lbi=new bigint_lattice(lattice_rows, lattice_cols);
      memory_handler(lbi,"lattice_gensys","lattice_gensys(n, m, abi) :: "
                         "not enough memory");
      sto(lattice_rows, lattice_cols, abi);
      lattice_read=true;
      prec=0;
      red_count=0;
      y_param=1.0;
      y_zaehler=1;
      y_nenner=1;
      return;
    }
  lidia_error_handler("lattice_gensys","lattice_gensys(n, m, abi) :: "
                "illegal lattice size !");
}

lattice_gensys::lattice_gensys(long n, long m, bigfloat** abf)
{
  debug_handler("lattice_gensys","lattice_gensys(n, m, abf)");
  if ((n > 0) && (m > 0))
    {
      lattice_rows=n;
      lattice_cols=m;
      comp_mode=bigfloat_mode;
      io_mode=own_io_mode;
      lbf=new bigfloat_lattice(lattice_rows, lattice_cols);
      memory_handler(lbf,"lattice_gensys","lattice_gensys(n, m, abf) :: "
                         "not enough memory");
      sto(lattice_rows, lattice_cols, abf);
      lattice_read=false;
      prec=0;
      red_count=0;
      y_param=1.0;
      y_zaehler=1;
      y_nenner=1;
      return;
    }
  lidia_error_handler("lattice_gensys","lattice_gensys(n, m, abf) :: "
                "illegal lattice size !");
}

lattice_gensys::lattice_gensys(const lattice_gensys& L)
{
  debug_handler("lattice_gensys","lattice_gensys(L)");
  lattice_rows=L.lattice_rows;
  lattice_cols=L.lattice_cols;
  comp_mode=L.comp_mode;
  io_mode=L.io_mode;
  switch (comp_mode) {
    case double_mode : {
      ldo=new double_lattice(*L.ldo);
      memory_handler(ldo,"lattice_gensys","lattice_gensys(L) :: "
                         "not enough memory");
    } break;
    case bigint_mode : {
      lbi=new bigint_lattice(*L.lbi); 
      memory_handler(lbi,"lattice_gensys","lattice_gensys(L) :: "
                         "not enough memory");
    } break;
    case bigfloat_mode : {
      lbf=new bigfloat_lattice(*L.lbf); 
      memory_handler(lbf,"lattice_gensys","lattice_gensys(L) :: "
                         "not enough memory");
    } break;
    default : {
      lidia_error_handler("lattice_gensys","lattice_gensys(L) :: "
                    "illegal computing mode"); 
    } break;
  }
  prec=L.prec;
  red_count=L.red_count;
  lattice_read=L.lattice_read;
  y_param=L.y_param;
  y_zaehler=L.y_zaehler;
  y_nenner=L.y_nenner;
}

lattice_gensys::lattice_gensys(const bigint_matrix& A)
{
  debug_handler("lattice_gensys","lattice_gensys(A)");
  long i, j;
  
  red_count=0;
  lbi=new bigint_lattice(A);
  memory_handler(lbi,"lattice_gensys","lattice_gensys(A) :: "
                     "not enough memory");
  comp_mode=bigint_mode;
  io_mode=own_io_mode;
  lattice_read=true;
  lattice_cols=A.get_no_of_columns();
  lattice_rows=A.get_no_of_rows();
  for (i=0;i<lattice_rows;i++)
    for (j=0;j<lattice_cols;j++)
      lbi->value[j][i].assign(A.member(i,j));
  y_param=1.0;
  y_zaehler=1;
  y_nenner=1;
  prec=0;
}

lattice_gensys::~lattice_gensys()
{
  debug_handler("lattice_gensys","~lattice_gensys()");
  switch (comp_mode) {
    case double_mode : {
      delete ldo;
    } break;
    case bigint_mode : {
      delete lbi;
    } break;
    case bigfloat_mode : {
      delete lbf;
    } break;
    default : {
      lidia_error_handler("lattice_gensys","~lattice_gensys() :: "
                    "illegal computing mode");
    } break;
  }
}

//
// Assignments
//

void lattice_gensys::assign(const lattice_gensys& L)
{
  debug_handler("lattice_gensys","assign(L)");
  if ((lattice_rows == L.lattice_rows) && (lattice_cols == L.lattice_cols))
    {
      red_count=L.red_count;
      y_param=L.y_param; 
      y_zaehler=L.y_zaehler;
      y_nenner=L.y_nenner;
      T=L.T;
      prec=L.prec;  
      lattice_read=false;
      set_computing_mode(L.comp_mode);
      comp_mode=L.comp_mode;
      lattice_read=L.lattice_read;
      lattice_cols=L.lattice_cols;
      lattice_rows=L.lattice_rows;
      switch (comp_mode) {
        case bigfloat_mode : {
          lbf->assign(*L.lbf);    
        } break;
        case bigint_mode   : {
          lbi->assign(*L.lbi); 
        } break;
        case double_mode   : {
          ldo->assign(*L.ldo); 
        } break;
        default : {
          lidia_error_handler("lattice_gensys","assign(L) :: "
                        "illegal computing mode");
        } break;
      }
      return;
    }
  lidia_error_handler("lattice_gensys","assign(L) :: "
                "assignment of two different size lattices !");
}
	
void lattice_gensys::assign(const bigint_matrix& M)
{
  debug_handler("lattice_gensys","assign(M)");

  if ((lattice_cols == M.get_no_of_columns()) &&
      (lattice_rows == M.get_no_of_rows()))
    {
      red_count=0;
      switch (comp_mode) {
        case bigfloat_mode : {
          delete lbf; 
          lbi=new bigint_lattice(M);
          memory_handler(lbi,"lattice_gensys","assign(M) :: "
                             "not enough memory !");
        } break;
        case bigint_mode : {
          lbi->assign(M);
        } break;
        case double_mode : { 
          delete ldo; 
          lbi=new bigint_lattice(M);
          memory_handler(lbi,"lattice_gensys","assign(M) :: "
                             "not enough memory !");
        } break;
        default : {
          lidia_error_handler("lattice_gensys","assign(M) :: "
                        "illegal computing mode");
        } break;
      }
      comp_mode=bigint_mode;
      y_param=1.0;
      y_zaehler=1;
      y_nenner=1;
      prec=0;
      lattice_read=true;
      return;
    }
  lidia_error_handler("lattice_gensys","assign(M) :: "
                "assignment of two different size lattices !");
}

lattice_gensys& lattice_gensys::operator =(const lattice_gensys& L)
{
  debug_handler("lattice_gensys","operator = (L)");
  assign(L);
  return(*this);
}

lattice_gensys& lattice_gensys::operator =(const bigint_matrix &M)
{
  debug_handler("lattice_gensys","operator = (M)");
  assign(M);
  return(*this);
}
 
//
// Input / Output 
//

istream& operator >>(istream& in, lattice_gensys& L)
{
  debug_handler("lattice_gensys","operator >> (in, L)");
  char *st,*old_st;
  long mem_size=100001;
  long anz,i,j;
  st=new char[1];

  switch (L.io_mode) {
    case pari_io_mode : {
      do {
        old_st=st;
        st=new char[mem_size];
        memory_handler(st, "lattice_gensys","operator >> (in, L) :: "
                           "not enough memory !");
        for (i=0;i<mem_size-100000;st[i]=old_st[i++]);
        in.get(st+(mem_size-100001),100001,']');
        mem_size+=100000;
        anz=in.gcount();
      } while (anz == 100000);
      for (j=0,i=0;i<mem_size-200001+anz;st[(((st[i]==' ')||(st[i]=='\t')||(st[i]=='\n'))?j:j++)]=st[i++]);
      st[j++]=']';
      st[j]=0;
      L.read_from_Pari(st);
      if (L.comp_mode == bigfloat_mode)
	L.compute_precision();
    } break;
    case maple_io_mode : {
      do {
        old_st=st;
        st=new char[mem_size];
        memory_handler(st, "lattice_gensys","operator >> (in, L) :: "
                           "not enough memory !");
        for (i=0;i<mem_size-100000;st[i]=old_st[i++]);
        in.get(st+(mem_size-100001),100001,';');
        mem_size+=100000;
        anz=in.gcount();
      } while (anz == 100000);
      for (j=0,i=0;i<mem_size-200001+anz;st[(((st[i]==' ')||(st[i]=='\t')||(st[i]=='\n'))?j:j++)]=st[i++]);
      st[j]=0;
      L.read_from_Maple(st);
      if (L.comp_mode == bigfloat_mode)
        L.compute_precision();
    } break;
    case mathematica_io_mode : {
      do {
        old_st=st;
        st=new char[mem_size];
        memory_handler(st, "lattice_gensys","operator >> (in, L) :: "
                           "not enough memory !");
        for (i=0;i<mem_size-100000;st[i]=old_st[i++]);
        in.getline(st+(mem_size-100001),100001,'}');
        anz=in.gcount();
        if (anz != 100000)
          {
            st[(mem_size-100002)+anz]='}';
            if (strchr(st,'{') == NULL)
              lidia_error_handler("lattice_gensys","operator >> (in, L) :: "
                            "illegal mathematica format !");
          }
        old_st=st+(mem_size-100001);
        mem_size+=anz;
      } while ((anz==100000) || (strchr(old_st,'{')!=NULL));
      for (j=0,i=0;i<strlen(st);st[(((st[i]==' ')||(st[i]=='\t')||(st[i]=='\n'))?j:j++)]=st[i++]);
      st[j]=0; 
      L.read_from_Mathematica(st);
      if (L.comp_mode == bigfloat_mode)
        L.compute_precision();
    } break;
    case own_io_mode : {
      switch (L.comp_mode) {
        case double_mode : {
          in >> *L.ldo; 
        } break; 
        case bigint_mode : {
          in >> *L.lbi;
        } break;
        case bigfloat_mode : {
          in >> *L.lbf; 
          L.compute_precision();
        } break;
        default : {
          lidia_error_handler("lattice_gensys","operator >> (in, L) :: "
                        "illegal computing mode !");
        } break;
      }
    } break;
    default : {
      delete st;
      lidia_error_handler("lattice_gensys","operator >> (in, L) :: "
                    "illegal IO mode !");
    } break;
  }
  L.lattice_read=true;
  delete st;
  return (in);
}   

ostream& operator <<(ostream& out, lattice_gensys L)
{
  debug_handler("lattice_gensys","operator << (out, L)");
  char *st;
  switch (L.io_mode) {
    case pari_io_mode : {
      L.write_to_Pari(st);
      out << st;
    } break;
    case maple_io_mode : {
      L.write_to_Maple(st);
      out << st;
    } break;
    case mathematica_io_mode : {
      L.write_to_Mathematica(st);
      out << st;
    } break;
    case own_io_mode : {
      switch (L.comp_mode) {
        case double_mode : {
          out << *L.ldo; 
        } break; 
        case bigint_mode : {
          out << *L.lbi;
        } break;
        case bigfloat_mode : {
          out << *L.lbf; 
        } break;
        default : {
          lidia_error_handler("lattice_gensys","operator << (out, L)"
                        "illegal computing mode !");
        } break;
      }
    } break;
    default : {
      lidia_error_handler("lattice_gensys","operator << (out, L) :: "
                    "illegal IO Mode !");
    } break;
  }
  return (out);
}

//
// Element Operations
//

void lattice_gensys::member(long x, long y, double& d)
{
  debug_handler("lattice_gensys","member(x, y, d)"); 
  if ((x < 0) || (x >= lattice_rows) ||
      (y < 0) || (y >= lattice_cols))
    lidia_error_handler("lattice_gensys","member(x, y, d) :: "
                  "illegal lattice index !");
  else
    {
      switch (comp_mode) {
        case double_mode : {
          d=ldo->value[y][x]; 
        } break;
        case bigint_mode : {
          bigfloat temp(lbi->value[y][x]);  
          temp.doublify(d);
        } break;
        case bigfloat_mode : {
          lbf->value[y][x].doublify(d); 
        } break;
        default : {
          lidia_error_handler("lattice_gensys","member(x, y, d) :: "
                        "illegal computing mode");
        } break;
      }
    }
}

void lattice_gensys::member(long x, long y, bigint& bi)
{
  debug_handler("lattice_gensys","member(x, y, bi)"); 
  bigfloat temp;
  if ((x < 0) || (x >= lattice_rows) ||
      (y < 0) || (y >= lattice_cols))
    lidia_error_handler("lattice_gensys","member(x, y, bi) :: "
                  "illegal lattice index !");
  else
    {
      switch (comp_mode) {
        case double_mode : {
          bigint temp(ldo->value[y][x]); 
          bi.assign(temp);
        } break;
        case bigint_mode : {
          bi.assign(lbi->value[y][x]); 
        } break;
        case bigfloat_mode : {
          lbf->value[y][x].bigintify(bi); 
        } break;
        default : {
          lidia_error_handler("lattice_gensys","member(x, y, bi) :: "
                        "illegal computing mode");
        } break;
      }
    }
}

void lattice_gensys::member(long x, long y, bigfloat& bf)
{
  debug_handler("lattice_gensys","member(x, y, bf)"); 
  if ((x < 0) || (x >= lattice_rows) ||
      (y < 0) || (y >= lattice_cols))
    lidia_error_handler("lattice_gensys","member(x, y, bf) :: "
                  "illegal lattice index !");
  else
    {
      switch (comp_mode) {
        case double_mode : { 
          bf.assign(ldo->value[y][x]); 
        } break;
        case bigint_mode : { 
          bf.assign(lbi->value[y][x]); 
        } break;  
        case bigfloat_mode : {
          bf.assign(lbf->value[y][x]); 
        } break;
        default : {
          lidia_error_handler("lattice_gensys","member(x, y, bf) :: "
                        "illegal computing mode");
        } break;
      }
    }
}

void lattice_gensys::sto(long x, long y, const double& d)
{
  debug_handler("lattice_gensys","sto(x, y, d)"); 
  if ((x < 0) || (x >= lattice_rows) ||
      (y < 0) || (y >= lattice_cols))
    lidia_error_handler("lattice_gensys","sto(x, y, d) :: "
                  "illegal lattice index !");
  else
    {
      switch (comp_mode) {
        case double_mode : { 
          ldo->value[y][x]=d; 
        } break;
        case bigint_mode : {
          bigint temp(d);  
          lbi->value[y][x].assign(temp);
        } break;
        case bigfloat_mode : {
          lbf->value[y][x].assign(d); 
          compute_precision();
        } break;
        default : {
          lidia_error_handler("lattice_gensys","sto(x, y, d) :: "
                        "illegal computing mode");
        } break;
      }
      lattice_read=true;
    }
}

void lattice_gensys::sto(long x, long y, const bigint& bi)
{
  debug_handler("lattice_gensys","sto(x, y, bi)"); 
  if ((x < 0) || (x >= lattice_rows) ||
      (y < 0) || (y >= lattice_cols))
    lidia_error_handler("lattice_gensys","sto(x, y, bi) :: "
                  "illegal lattice index !");
  else
    {
      switch (comp_mode) {
        case double_mode : {
          bigfloat temp(bi);
          temp.doublify(ldo->value[y][x]); 
        } break;
        case bigint_mode : { 
          lbi->value[y][x].assign(bi); 
        } break;
        case bigfloat_mode : {
          lbf->value[y][x].assign(bi); 
          compute_precision();
        } break;
        default : {
          lidia_error_handler("lattice_gensys","sto(x, y, bi) :: "
                        "illegal computing mode");
        } break;
      }
      lattice_read=true;
    }
}

void lattice_gensys::sto(long x, long y, const bigfloat& bf)
{
  debug_handler("lattice_gensys","sto(x, y, bf)"); 
  bigfloat temp(bf);
  if ((x < 0) || (x >= lattice_rows) ||
      (y < 0) || (y >= lattice_cols))
    lidia_error_handler("lattice_gensys","member(x, y, bf) :: "
                  "illegal lattice index !");
  else
    {
      switch (comp_mode) {
        case double_mode : { 
          temp.doublify(ldo->value[y][x]); 
        } break;
        case bigint_mode : { 
          temp.bigintify(lbi->value[y][x]); 
        } break;  
        case bigfloat_mode : {
          lbf->value[y][x].assign(temp); 
          compute_precision();
        } break;
        default : {
          lidia_error_handler("lattice_gensys","sto(x, y, bf) :: "
                        "illegal computing mode");
        } break;
      }
      lattice_read=true;
    }
}

void lattice_gensys::member(long& n, long& m, double**& ado)
{
  debug_handler("lattice_gensys","member(n, m, ado)");
  long x,y;
  bigfloat temp;
  ado=new double*[lattice_rows];
  memory_handler(ado, "lattice_gensys","member(n, m, ado) :: "
                      "not enough memory !");
  for (x=0;x<lattice_rows;x++)
    {
      ado[x]=new double[lattice_cols];
      memory_handler(ado[x], "lattice_gensys","member(n, m, ado) :: "
                             "not enough memory !");
    }
  n=lattice_rows;
  m=lattice_cols;
  switch (comp_mode) {
    case double_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           ado[x][y]=ldo->value[y][x];
    } break;
    case bigint_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           {
             temp.assign(lbi->value[y][x]);
             temp.doublify(ado[x][y]);
           }
    } break;
    case bigfloat_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           lbf->value[y][x].doublify(ado[x][y]);
    } break;
    default : {
      lidia_error_handler("lattice_gensys","member(n, m, ado) :: "
                    "illegal computing mode");
    } break;
  }
}

void lattice_gensys::member(long& n, long& m, bigint**& abi)
{
  debug_handler("lattice_gensys","member(n, m, abi)");
  long x,y;
  bigfloat temp;
    
  abi=new bigint*[lattice_rows];
  memory_handler(abi, "lattice_gensys","member(n, m, abi) :: "
                      "not enough memory !");
  for (x=0;x<lattice_rows;x++)
    {
      abi[x]=new bigint[lattice_cols];
      memory_handler(abi[x], "lattice_gensys","member(n, m, abi) :: "
                             "not enough memory !");
    }
  n=lattice_rows;
  m=lattice_cols;
  switch (comp_mode) {
    case double_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           {
             temp.assign(ldo->value[y][x]);
             temp.bigintify(abi[x][y]);
           }
    } break;
    case bigint_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           abi[x][y].assign(lbi->value[y][x]);
    } break;
    case bigfloat_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           lbf->value[y][x].bigintify(abi[x][y]);
    } break;
    default : {
      lidia_error_handler("lattice_gensys","member(n, m, abi) :: "
                    "illegal computing mode");
    } break;
  }
}

void lattice_gensys::member(long& n, long& m, bigfloat**& abf)
{
  debug_handler("lattice_gensys","member(n, m, abf)");
  long x,y;
  abf=new bigfloat*[lattice_rows];
  memory_handler(abf, "lattice_gensys","member(n, m, abf) :: "
                      "not enough memory !");
  for (x=0;x<lattice_rows;x++)
    {
      abf[x]=new bigfloat[lattice_cols];
      memory_handler(abf[x], "lattice_gensys","member(n, m, abf) :: "
                             "not enough memory !");
    }
  n=lattice_rows;
  m=lattice_cols;
  switch (comp_mode) {
    case double_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           abf[x][y].assign(ldo->value[y][x]);
    } break;
    case bigint_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           abf[x][y].assign(lbi->value[y][x]);
    } break;
    case bigfloat_mode : {
      for (x=0;x<lattice_rows;++x)
        for (y=0;y<lattice_cols;++y)
           abf[x][y].assign(lbf->value[y][x]);
    } break;
    default : {
      lidia_error_handler("lattice_gensys","member(n, m, abf) :: "
                    "illegal computing mode");
    } break;
  }
}

void lattice_gensys::sto(long n, long m, double** ado)
{
  debug_handler("lattice_gensys","sto(n, m, ado)");
  long x,y;
  bigfloat temp;
  if ((n == lattice_rows) && (m == lattice_cols))
    {
      switch (comp_mode) {
        case double_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
               ldo->value[y][x]=ado[x][y];
        } break;
        case bigint_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
               {
                 temp.assign(ado[x][y]);
                 temp.bigintify(lbi->value[y][x]);
               }
        } break;
        case bigfloat_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
               lbf->value[y][x].assign(ado[x][y]);
          compute_precision();
        } break;
        default : {
          lidia_error_handler("lattice_gensys","sto(n, m, ado) :: "
                        "illegal computing mode");
        } break;
      }
      lattice_read=true;
      return;
    }
  lidia_error_handler("lattice_gensys","sto(n, m, ado) :: "
                "illegal array size !");
}
 
void lattice_gensys::sto(long n, long m, bigint** abi)
{
  debug_handler("lattice_gensys","sto(n, m, abi)");
  long x,y;
  bigfloat temp;
  if ((n == lattice_rows) && (m == lattice_cols))
    {
      switch (comp_mode) {
        case double_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
              { 
                temp.assign(abi[x][y]);
                temp.doublify(ldo->value[y][x]);
              }
        } break;
        case bigint_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
              lbi->value[y][x].assign(abi[x][y]);
        } break;
        case bigfloat_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
              lbf->value[y][x].assign(abi[x][y]);
          compute_precision();
        } break;
        default : {
          lidia_error_handler("lattice_gensys","sto(n, m, abi) :: "
                        "illegal computing mode");
        } break;
      }
      lattice_read=true;
      return;
    }
  lidia_error_handler("lattice_gensys","sto(n, m, abi) :: "
                "illegal array size !");
}

void lattice_gensys::sto(long n, long m, bigfloat** abf)
{
  debug_handler("lattice_gensys","sto(n, m, abf)");
  long x,y;
  if ((n == lattice_rows) && (m == lattice_cols))
    {
      switch (comp_mode) {
        case double_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
              abf[x][y].doublify(ldo->value[y][x]);
        } break;
        case bigint_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
              abf[x][y].bigintify(lbi->value[y][x]);
        } break;
        case bigfloat_mode : {
          for (x=0;x<lattice_rows;++x)
            for (y=0;y<lattice_cols;++y)
              lbf->value[y][x].assign(abf[x][y]);
          compute_precision();
        } break;
        default : {
          lidia_error_handler("lattice_gensys","sto(n, m, abf) :: "
                        "illegal computing mode");
        } break;
      }
      lattice_read=true;
      return;
    }
  lidia_error_handler("lattice_gensys","sto(n, m, abf) :: "
                "illegal array size !");
}

long lattice_gensys::get_no_of_rows()
{
  debug_handler("lattiec_gensys","get_no_of_rows()");
  return (lattice_rows);
}

long lattice_gensys::get_no_of_columns()
{
  debug_handler("lattice_gensys","get_no_of_columns()");
  return (lattice_cols);
}

//
// Type Checking
//

// Checks if you can convert to double without losing significant digits
int lattice_gensys::check_double()
{
  debug_handler("lattice_gensys","check_double()");
  long x, y;
  int overflow=0;
  switch (comp_mode) {
    case bigfloat_mode : {
      for (x=0;x<lattice_rows;++x)
        {
          for (y=0;y<lattice_cols;++y)
            if ((overflow = is_double(lbf->value[y][x])) == 0)
              break;
          if (overflow == 0)
            break;
        }
    } break;
    case bigint_mode : {
      overflow=0; 
    } break;
    case double_mode : {
      overflow=1; 
    } break;
    default : {
      lidia_error_handler("lattice_gensys","check_double() :: "
                    "illegal computing mode");
    } break;
  } 
  return (overflow);   
}

// Checks if you can covert to bigint (no floating point)
int lattice_gensys::check_bigint()
{
  debug_handler("lattice_gensys","check_bigint()");
  int overflow=0;
  long x, y;
  bigfloat zero;
  switch (comp_mode) {
    case bigfloat_mode : {
      for (x=0;x<lattice_rows;++x)
        {
          for (y=0;y<lattice_cols;++y)
            {
              round(lbf->value[y][x],zero);
              subtract(zero,zero,lbf->value[y][x]);
               if ((overflow = zero.is_approx_zero()) == 0)
                 break;
            }
          if (overflow == 0)
            break;
        }
    } break;
    case bigint_mode : {
      overflow=1; 
    } break;
    case double_mode : { 
      overflow=0; 
    } break;
    default : {
      lidia_error_handler("lattice_gensys","check_bigint() :: "
                    "illegal computing mode");
    } break;
  } 
  return (overflow);   
}

void lattice_gensys::set_computing_mode(long m)
{
  debug_handler("lattice_gensys","set_computing_mode(m)");
  long x,y;
  switch (comp_mode) {
    case bigfloat_mode : {
      switch (m) {
        case bigfloat_mode : break;
        case bigint_mode : {
          lbi=new bigint_lattice(lattice_rows,lattice_cols);
          memory_handler(lbi,"lattice_gensys","set_computing_mode(m) :: "
                             "not enough memory !"); 
          if (lattice_read == true)
            {
              for (x=0;x<lattice_rows;++x)
                 for (y=0;y<lattice_cols;++y)
                   lbf->value[y][x].bigintify(lbi->value[y][x]);
            }
          comp_mode=bigint_mode;
          delete lbf;
        } break;
        case double_mode : {
          ldo=new double_lattice(lattice_rows,lattice_cols);
          memory_handler(ldo,"lattice_gensys","set_computing_mode(m) :: "
                             "not enough memory !"); 
          if (lattice_read == true)
            { 
              for (x=0;x<lattice_rows;++x)
                 for (y=0;y<lattice_cols;++y) 
                   lbf->value[y][x].doublify(ldo->value[y][x]);
            }
          comp_mode=double_mode;
          delete lbf;
        } break;
        default : {
          lidia_error_handler("lattice_gensys","set_computing_mode(m) :: "
                        "illegal computing mode !");
        } break;
      }
    } break;
    case bigint_mode : {
      switch (m) {
        case bigfloat_mode : {
          lbf=new bigfloat_lattice(lattice_rows,lattice_cols); 
          memory_handler(lbf,"lattice_gensys","set_computing_mode(m) :: "
                             "not enough memory !"); 
          if (lattice_read == true)
            {
              for (x=0;x<lattice_rows;++x)
                for (y=0;y<lattice_cols;++y)
                  lbf->value[y][x].assign(lbi->value[y][x]);
              compute_precision();
            }
          comp_mode=bigfloat_mode;
          delete lbi;
        } break;
        case bigint_mode : break;
        case double_mode : {
          bigfloat temp;
          ldo=new double_lattice(lattice_rows,lattice_cols); 
          memory_handler(ldo,"lattice_gensys","set_computing_mode(m) :: "
                             "not enough memory !"); 
          if (lattice_read == true)
            {
              for (x=0;x<lattice_rows;++x)
                for (y=0;y<lattice_cols;++y)
                  { 
                    temp.assign(lbi->value[y][x]);
                    temp.doublify(ldo->value[y][x]);
                  }
            }
          comp_mode=double_mode;
          delete lbi;
        } break;
        default : {
          lidia_error_handler("lattice_gensys","set_computing_mode(m) :: "
                        "illegal computing mode !");
        } break;
      }
    } break;
    case double_mode : {
      switch (m) {
        case bigfloat_mode : {
          lbf=new bigfloat_lattice(lattice_rows,lattice_cols); 
          memory_handler(lbf,"lattice_gensys","set_computing_mode(m) :: "
                             "not enough memory !"); 
          if (lattice_read == true)
            {
              for (x=0;x<lattice_rows;++x)
                for (y=0;y<lattice_cols;++y)
                  lbf->value[y][x].assign(ldo->value[y][x]);
              compute_precision();
            }
          comp_mode=bigfloat_mode;
          delete ldo;
        } break;
        case bigint_mode : {
          bigfloat temp;
          lbi=new bigint_lattice(lattice_rows,lattice_cols); 
          memory_handler(lbi,"lattice_gensys","set_computing_mode(m) :: "
                             "not enough memory !"); 
          if (lattice_read == true)
            {
              for (x=0;x<lattice_rows;++x)
                 for (y=0;y<lattice_cols;++y)
                   {
                     temp.assign(ldo->value[y][x]);
                     temp.bigintify(lbi->value[y][x]);
                   }
            }
          comp_mode=bigint_mode;
          delete ldo;
        } break;
        case double_mode : break;
        default : {
          lidia_error_handler("lattice_gensys","set_computing_mode(m) :: "
                        "illegal computing mode !");
        } break;
      }
    } break;
    default : {
      lidia_error_handler("lattice_gensys","set_computing_mode(m) :: "
                    "illegal computing mode !");
    } break;     
  }
}

long lattice_gensys::get_computing_mode()
{
  debug_handler("lattice_gensys","get_computing_mode()");
  return (comp_mode);
}

void lattice_gensys::set_io_mode(long m)
{
  debug_handler("lattice_gensys","set_io_mode(m)");
  if ((m == own_io_mode) ||
      (m == pari_io_mode) ||
      (m == maple_io_mode) ||
      (m == mathematica_io_mode))
    {
      io_mode=m;
      return;
    }
  lidia_error_handler("lattice_gensys","set_io_mode(m) :: "
                "illegal IO mode");
}

long lattice_gensys::get_io_mode()
{
  debug_handler("lattice_gensys","get_io_mode()");
  return (io_mode);
}

//
// Other Member Functions
//

void lattice_gensys::set_computing_precision(long p)
{
  debug_handler("lattice_gensys","set_computing_precision(p)");
  prec=p;
}

long lattice_gensys::get_computing_precision()
{
  debug_handler("lattice_gensys","get_computing_precision()");
  return (prec);
}

long lattice_gensys::get_read_precision()
{
  debug_handler("lattice_gensys","get_read_precision()");
  if (comp_mode == bigfloat_mode)
    return(compute_read_precision());
  return(0);
}
    
void lattice_gensys::set_reduction_parameter(double y)
{
  debug_handler("lattice_gensys","set_reduction_parameter(y)");
  y_param=((y<=0.25)?0.26:((y>1.0)?1.0:y));
  y_zaehler=(long )(y*100);
  y_nenner=100;
}

void lattice_gensys::set_reduction_parameter(long a, long b)
{
  debug_handler("lattice_gensys","set_reduction_parameter(a, b)");
  double d=(double)a*1.0/(b*1.0);
  y_param=((d<=0.25)?0.26:((d>1.0)?1.0:d));
  y_zaehler=((d<=0.25)?26:((d>1.0)?1:a));
  y_nenner=((d<=0.25)?100:((d>1.0)?1:b));
  
}
    
double lattice_gensys::get_reduction_parameter()
{
  debug_handler("lattice_gensys","get_reduction_parameter()");
  return (y_param);
}

void lattice_gensys::get_reduction_parameter(long& a, long& b)
{
  debug_handler("lattice_gensys","get_reduction_parameter(a, b)");
  a=y_zaehler;
  b=y_nenner;
}

unsigned long lattice_gensys::get_redsteps()
{
  debug_handler("lattice_gensys","get_redsteps()");
  return (red_count);
}

//
// Time needed
//

timer& lattice_gensys::get_computing_time()
{
  debug_handler("lattice_gensys","get_computing_time()");
  return (T);
}

//
// protected functions
//

long lattice_gensys::compute_read_precision()
{
  debug_handler("lattice_gensys","compute_read_precision()");
  long x, y;
  long new_prec;
  long read_prec=0;
  for (x=0;x<lattice_rows;++x)
    for (y=0;y<lattice_cols;++y)
      if (read_prec < (new_prec=(long )(lbf->value[y][x].bit_length()/log(10)+1)))
        read_prec=new_prec;
  return (read_prec);
}

void lattice_gensys::compute_precision()
{
  debug_handler("lattice_gensys","compute_precision()");
  bigfloat alpha,zweipotq;
  long read_prec=compute_read_precision();
  long n2=lattice_cols;
  lbf->alpha_compute(alpha);
  lbf->zwei_pot_q_compute(zweipotq,n2, alpha);
  read_prec=2*read_prec+lattice_cols-1;
  prec=(long )(2*(((zweipotq.exponent()+zweipotq.bit_length()+1)*
        log(2)/log(10)+1)+read_prec)+(lattice_cols+lattice_rows)-1);
}

void lattice_gensys::read_from_Pari(char* st)
{
  debug_handler("lattice_gensys","read_from_Pari(st) [1]");
  long x,y,cols=1,rows=1, cols_test;
  char *posem,*po,*po_old;

  if ((st[0] != '[') || (st[strlen(st)-1] != ']'))
    lidia_error_handler("lattice_gensys","read_from_Pari(st) :: "
                          "illegal Pari format !");
  posem=strchr(st,';');
  if (posem != NULL)
    posem[0]=0;
  po=st;
  while ((po=strchr(po+1,',')) != NULL)
    ++cols;
  if (posem != NULL)
    {
      posem[0]=';';
      ++rows;
      po=posem;
      po_old=po+1;
      while ((po=strchr(po+1,';')) != NULL)
        {
          cols_test=1;
          po[0]=0;
          while ((po_old=strchr(po_old+1,',')) != NULL)
            cols_test++;
          if (cols_test != cols)
            lidia_error_handler("lattice_gensys","read_from_Pari(st) :: "
                          "illegal Pari format !");
          po[0]=';';
          po_old=po+1;
         ++rows;
        }
      cols_test=1;
      while ((po_old=strchr(po_old+1,',')) != NULL)
        cols_test++;
      if (cols_test != cols)
        lidia_error_handler("lattice_gensys","read_from_Pari(st) :: "
                      "illegal Pari format !");
    }

  debug_handler("lattice_gensys","read_from_Pari(st) [2]");

  if ((rows != lattice_rows) || (cols != lattice_cols))
    {
      lattice_rows=rows;
      lattice_cols=cols; 
      switch (comp_mode) {
        case bigfloat_mode : {
          delete lbf;
          lbf=new bigfloat_lattice(lattice_rows, lattice_cols);
          memory_handler(lbf,"lattice_gensys","read_from_Pari(st) :: "
                             "not enough memory !");
        } break;
        case bigint_mode : {
          delete lbi;
          lbi=new bigint_lattice(lattice_rows, lattice_cols);
          memory_handler(lbi,"lattice_gensys","read_from_Pari(st) :: "
                             "not enough memory !");
         } break;
        case double_mode : {
          delete ldo;
          ldo=new double_lattice(lattice_rows, lattice_cols);
          memory_handler(ldo,"lattice_gensys","read_from_Pari(st) :: "
                             "not enough memory !");
         } break;
        default : {
          lidia_error_handler("lattice_gensys","read_for_Pari(st) :: "
                        "illegal computing mode !");
        } break;
      }
    }

  check_basis();

  debug_handler("lattice_gensys","read_from_Pari(st) [3]");

  char *azcp=st,*ezcp,c;
  bigfloat temp;
  
  azcp++;
  for (x=0;x<rows;x++)
    {
     for (y=0;y<cols;y++)
       {
         ezcp=azcp;
         if (x==(rows-1))
           {
             if (y==(cols-1))
               c=']';
             else
               c=',';
           }
         else
           {
             if (y == (cols-1))
               c=';';
             else
               c=',';
           }
         ezcp=strchr(ezcp+1,c);
         if (ezcp == NULL)
           lidia_error_handler("lattice_gensys","read_from_Pari(st) :: "
                         "illegal Pari format !");
         ezcp[0]=0;
         switch (comp_mode) {
           case bigfloat_mode : {
             string_to_bigfloat(azcp,lbf->value[y][x]);
           } break;
           case bigint_mode : {
             string_to_bigint(azcp,lbi->value[y][x]);
           } break;
           case double_mode : {
             ldo->value[y][x]=atof(azcp); 
           } break;
         }
         ezcp[0]=c;
         azcp=++ezcp;
       }   
    }  
}

void lattice_gensys::read_from_Maple(char* st)
{
  debug_handler("lattice_gensys","read_from_Maple(st) [1]");
  long cols=0,rows=0, ab_cols=0, ab_rows=0, pos_x=0, pos_y=0;
  char *poko,*popo,*po;

  po=strchr(st,'(');
  if (po == NULL)
    lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                  "illegal Maple format !");
  poko=strchr(po++,',');
  if (poko != NULL)
    {
      poko[0]=0;
      popo=strchr(po,'.');
      if (popo != NULL)
        {
          popo[0]=0;
          sscanf(po,"%ld",&ab_rows);
//          ab_rows=atol(po);
          popo[0]='.';
        }
      else
        lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                      "illegal Maple format !");
      popo=strchr(popo+1,'.');
      if (popo != NULL)
        {
          sscanf(popo+1,"%ld",&rows);
          ++rows;
//        rows=atol(popo+1)+1;
        }
      else
        lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                      "illegal Maple format !");      
      rows-=ab_rows;
    }
  else
    lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                  "illegal Maple format !");

  poko[0]=',';
  po=poko+1;
  poko=strchr(po,',');
  if (poko != NULL)
    {
      poko[0]=0;
      popo=strchr(po,'.');
      if (popo != NULL)
        {
          popo[0]=0;
          sscanf(po,"%ld",&ab_cols);
//          ab_cols=atol(po);
          popo[0]='.';
        }
      else
        lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                      "illegal Maple format !");
      popo=strchr(popo+1,'.');
      if (popo != NULL)
        {
          sscanf(popo+1,"%ld",&cols);
          ++cols;
//        cols=atol(popo+1)+1;
        }
      else
        lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                      "illegal Maple format !");      
      cols-=ab_cols;
    }
  else
    lidia_error_handler("lattice_gensys","read_from_Maple :: "
                  "illegal Maple format !");
  poko[0]=',';

  if ((rows != lattice_rows) || (cols != lattice_cols))
    {
      lattice_rows=rows;
      lattice_cols=cols; 
      switch (comp_mode) {
        case bigfloat_mode : {
          delete lbf;
          lbf=new bigfloat_lattice(lattice_rows, lattice_cols);
          memory_handler(lbf,"lattice_gensys","read_from_Maple(st) :: "
                             "not enough memory !");
        } break;
        case bigint_mode : {
          delete lbi;
          lbi=new bigint_lattice(lattice_rows, lattice_cols);
          memory_handler(lbi,"lattice_gensys","read_from_Maple(st) :: "
                             "not enough memory !");
         } break;
        case double_mode : {
          delete ldo;
          ldo=new double_lattice(lattice_rows, lattice_cols);
          memory_handler(ldo,"lattice_gensys","read_from_Maple(st) :: "
                             "not enough memory !");
         } break;
        default : {
          lidia_error_handler("lattice_gensys","read_for_Maple(st) :: "
                        "illegal computing mode !");
        } break;
      }
    }

  check_basis();

  po=strchr(poko+1,'[');
  po++;
  while (poko[0] != ']')
    {
      po++;
      poko=strchr(po,',');
      if (poko != NULL)
        {
          poko[0]=0;
          sscanf(po,"%ld",&pos_x);
//          pos_x=atol(po);
          poko[0]=',';
        }
      else
        lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                      "illegal Maple format ! ");
      po=poko+1;
      poko=strchr(po,')');
      if (poko != NULL)
        {
          poko[0]=0;
          sscanf(po,"%ld",&pos_y);
//          pos_y=atol(po);
          poko[0]=')';
        }
      else
        lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                      "illegal Maple format !");        
      po=poko+2;
      poko=strchr(po,',');
      if (poko == NULL)
        poko=strchr(po,']');
      if (poko != NULL)
        {
          poko[0]=0;
          if (((pos_y-ab_cols) >=0) && ((pos_y-ab_cols) < lattice_rows) &&
              ((pos_x-ab_rows) >=0) && ((pos_x-ab_rows) < lattice_cols))
            {
              switch (comp_mode) {
                case bigfloat_mode : {
                  string_to_bigfloat(po,lbf->value[pos_y-ab_cols][pos_x-ab_rows]);
                } break;
                case bigint_mode : {
                  string_to_bigint(po,lbi->value[pos_y-ab_cols][pos_x-ab_rows]);
                } break;
                case double_mode : {
                   ldo->value[pos_y-ab_cols][pos_x-ab_rows]=atof(po);
                } break;
                default : {
                  lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                                "illegal computing mode !");
                } break;
              }
            }
          else
            lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                          "illegal Maple format !");
          if (poko[1] == ')')
            poko[0]=']';
          else
            poko[0]=',';
        }
      else
        lidia_error_handler("lattice_gensys","read_from_Maple(st) :: "
                      "illegal Maple format !");
      po=poko+1;
    }
  debug_handler("lattice_gensys","read_from_Maple(st) [3]");
}

void lattice_gensys::read_from_Mathematica(char* st)
{
  debug_handler("lattice_gensys","read_from_Mathematica(st) [1]");
  long x,y,cols=1,rows=0, cols_test;
  char *pokl,*po,*po_old;

  if ((st[0] != '{') || (st[strlen(st)-1] != '}'))
    lidia_error_handler("lattice_gensys","read_from_Mathematica(st) :: "
                          "illegal Mathematica format !");

  po=st+2;
  pokl=strchr(st+2,'}');
  if (pokl != NULL)
    pokl[0]=0;
  while ((po=strchr(po+1,',')) != NULL)
    ++cols;
  if (pokl != NULL)
    {
      pokl[0]='}';
      ++rows;
      po=pokl;
      po_old=po+3;
      while ((po=strchr(po+2,'}')) != NULL)
        {
          cols_test=1;
          po[0]=0;
          while ((po_old=strchr(po_old+1,',')) != NULL)
            cols_test++;
          if (cols_test != cols)
            lidia_error_handler("lattice_gensys","read_from_Mathematica(st) :: "
                          "illegal Mathematica format !");
          po[0]='}';
          po_old=po+3;
          ++rows;
        }
    }

  debug_handler("lattice_gensys","read_from_Mathematica(st) [2]");
  if ((rows != lattice_rows) || (cols != lattice_cols))
    {
      lattice_rows=rows;
      lattice_cols=cols; 
      switch (comp_mode) {
        case bigfloat_mode : {
          delete lbf;
          lbf=new bigfloat_lattice(lattice_rows, lattice_cols);
          memory_handler(lbf,"lattice_gensys","read_from_Mathematica(st) :: "
                             "not enough memory !");
        } break;
        case bigint_mode : {
          delete lbi;
          lbi=new bigint_lattice(lattice_rows, lattice_cols);
          memory_handler(lbi,"lattice_gensys","read_from_Mathematica(st) :: "
                             "not enough memory !");
         } break;
        case double_mode : {
          delete ldo;
          ldo=new double_lattice(lattice_rows, lattice_cols);
          memory_handler(ldo,"lattice_gensys","read_from_Mathematica(st) :: "
                             "not enough memory !");
         } break;
        default : {
          lidia_error_handler("lattice_gensys","read_for_Mathematica(st) :: "
                        "illegal computing mode !");
        } break;
      }
    }

  check_basis();

  debug_handler("lattice_gensys","read_from_Mathematica(st) [3]");

  char *azcp=st+2,*ezcp,c;
  bigfloat temp;

  for (x=0;x<rows;x++)
    {
     for (y=0;y<cols;y++)
       {
         ezcp=azcp;
         if (y==(cols-1))
           c='}';
         else
           c=',';
         ezcp=strchr(ezcp+1,c);
         if (ezcp == NULL)
           lidia_error_handler("lattice_gensys","read_from_Mathematica(st) :: "
                         "illegal Mathematica format !");
         ezcp[0]=0;
         switch (comp_mode) {
           case bigfloat_mode : {
             string_to_bigfloat(azcp,lbf->value[y][x]);
           } break;
           case bigint_mode : {
             string_to_bigint(azcp,lbi->value[y][x]);             
           } break;
           case double_mode : {
             ldo->value[y][x]=atof(azcp); 
           } break;
         }
         ezcp[0]=c;
         if ((c == '}') && (y == (cols-1)))
           ezcp+=2; 
         azcp=++ezcp;
       }   
    }  
}

void lattice_gensys::write_to_Pari(char *&st)
{
  debug_handler("lattice_gensys","write_to_Pari(st) [1]");
  long x, y, i;
  char *temp=0;
  char *pn;
  long cc=0,size=0;
  switch (comp_mode) {
    case bigfloat_mode : { 
      for (x=0;x<lattice_rows;x++)
        for (y=0;y<lattice_cols;y++)
          size+=(long)((lbf->value[y][x].bit_length())*log(2)/log(10)+18);
    } break;
    case bigint_mode : {
      for (x=0;x<lattice_rows;x++)
        for (y=0;y<lattice_cols;y++)
          size+=(long)((lbi->value[y][x].bit_length())*log(2)/log(10)+3);
    } break;
    case double_mode : {
      size=lattice_rows*lattice_cols*26;
    } break;
    default : {
      lidia_error_handler("lattice_gensys","write_to_Pari(st) :: "
                    "illegal computing mode");
    } break;
  }
  debug_handler("lattice_gensys","write_to_Pari(st) [2]");
  st=new char[size+2];
  memory_handler (st, "lattice_gensys","write_to_Pari(st) :: "
                      "not enough memory !");
  pn=st+1;
  st[0]='[';
  st[1]=0;
  for (x=0;x<lattice_rows;x++)
    for (y=0;y<lattice_cols;y++)
      {
        switch (comp_mode) {
          case bigfloat_mode : {
            temp=new char[(long )((lbf->value[y][x].bit_length())*log(2)/log(10)+18)];
            memory_handler (temp,"lattice_gensys","write_to_Pari(st) :: "
                                 "not enough memory !");
            cc=bigfloat_to_string(lbf->value[y][x],temp);  
          } break;
          case bigint_mode : {
            temp=new char[(long )((lbi->value[y][x].bit_length())*log(2)/log(10)+3)];
            memory_handler (temp,"lattice_gensys","write_to_Pari(st) :: "
                                 "not enough memory !");
            cc=bigfloat_to_string(lbi->value[y][x],temp);  
          } break;
          case double_mode : {
            temp=new char[26];
            memory_handler (temp,"lattice_gensys","write_to_Pari(st) :: "
                                 "not enough memory !");
            sprintf(temp,"%f",ldo->value[y][x]);
            cc=strlen(temp)-1;
          } break;
          default : {
            lidia_error_handler("lattice_gensys","write_to_Pari(st) :: "
                          "illegal computing mode !");
          } break;
        } 
        for (i=0;i<cc;++i) // 0 auch kopieren
          pn[i]=temp[i];
        if (x==(lattice_rows-1))
          {
            if (y==(lattice_cols-1))
              pn[i]=']';
            else
              pn[i]=',';
          }
        else
          {
            if (y == (lattice_cols-1))
              pn[i]=';';
            else
              pn[i]=',';
          }
        pn[i+1]=0;
        delete temp;
        pn+=i+1;        
      }
}

void lattice_gensys::write_to_Maple(char *&st)
{
  debug_handler("lattice_gensys","write_to_Maple(st) [1]");
  long x, y;
  char *pn;
  long lest,size=0;
  switch (comp_mode) {
    case bigfloat_mode : { 
      for (x=0;x<lattice_rows;x++)
        for (y=0;y<lattice_cols;y++)
          size+=(long)((lbf->value[y][x].bit_length())*log(2)/log(10)+20);
    } break;
    case bigint_mode : {
      for (x=0;x<lattice_rows;x++)
        for (y=0;y<lattice_cols;y++)
          size+=(long)((lbi->value[y][x].bit_length())*log(2)/log(10)+5);
    } break;
    case double_mode : {
      size=lattice_rows*lattice_cols*28;
    } break;
    default : {
      lidia_error_handler("lattice_gensys","write_to_Maple(st) :: "
                    "illegal computing mode !");
    } break;
  }
  debug_handler("lattice_gensys","write_to_Maple(st) [2]");
  st=new char[size+50];
  memory_handler (st, "lattice_gensys","write_to_Maple(st) :: "
                      "not enough memory");
  sprintf(st,"a:=array(1 .. %ld, 1 .. %ld, [",lattice_rows, lattice_cols);
  lest=strlen(st);
  pn=st+lest;
  for (x=0;x<lattice_rows;x++)
    for (y=0;y<lattice_cols;y++)
      {
        sprintf(pn,"(%ld, %ld)=",x+1, y+1);
        pn=st+strlen(st);
        switch (comp_mode) {
          case bigfloat_mode : {
            bigfloat_to_string(lbf->value[y][x],pn);
          } break;
          case bigint_mode : {
            bigint_to_string(lbi->value[y][x],pn);
          } break;
          case double_mode : {
            sprintf(pn,"%f", ldo->value[y][x]);
          } break;
          default : {
            lidia_error_handler("lattice_gensys","write_to_Maple(st) :: "
                          "illegal computing mode !");
          } break;
        }
        pn=st+strlen(st); 
        if ((x==(lattice_rows-1)) && (y==(lattice_cols-1)))
          sprintf(pn,"]);");
        else
          pn++[0]=',';  
      }
}

void lattice_gensys::write_to_Mathematica(char *&st)
{
  debug_handler("lattice_gensys","write_to_Mathematica(st) [1]");
  long x, y, i;
  char *temp=0;
  char *pn;
  long cc=0,size=0;
  switch (comp_mode) {
    case bigfloat_mode : { 
      for (x=0;x<lattice_rows;x++)
        for (y=0;y<lattice_cols;y++)
          size+=(long)((lbf->value[y][x].bit_length())*log(2)/log(10)+20);
    } break;
    case bigint_mode : {
      for (x=0;x<lattice_rows;x++)
        for (y=0;y<lattice_cols;y++)
          size+=(long)((lbi->value[y][x].bit_length())*log(2)/log(10)+5);
    } break;
    case double_mode : {
      size=lattice_rows*lattice_cols*28;
    } break;
  }
  debug_handler("lattice_gensys","write_to_Mathematica(st) [2]");
  st=new char[size+3];
  memory_handler (st, "lattice_gensys","write_to_Mathematica(st) :: "
                      "not enough memory !");
  pn=st+2;
  st[0]='{';
  st[1]='{';
  st[2]=0;
  for (x=0;x<lattice_rows;x++)
    for (y=0;y<lattice_cols;y++)
      {
        switch (comp_mode) {
          case bigfloat_mode : {
            temp=new char[(long )((lbf->value[y][x].bit_length())*log(2)/log(10)+18)];
            memory_handler (temp, "lattice_gensys","write_to_Mathematica(st) :: "
                                  "not enough memory !");
            cc=bigfloat_to_string(lbf->value[y][x],temp);  
          } break;
          case bigint_mode : {
            temp=new char[(long )((lbi->value[y][x].bit_length())*log(2)/log(10)+3)];
            memory_handler (temp, "lattice_gensys","write_to_Mathematica(st) :: "
                                  "not enough memory !");
            cc=bigfloat_to_string(lbi->value[y][x],temp);  
          } break;
          case double_mode : {
            temp=new char[28];
            memory_handler (temp, "lattice_gensys","write_to_Mathematica(st) :: "
                                  "not enough memory !");
            sprintf(temp,"%f",ldo->value[y][x]);
            cc=strlen(temp)-1;
          } break;
        } 
        for (i=0;i<cc;++i) // 0 auch kopieren
          pn[i]=temp[i];
        if (x==(lattice_rows-1))
          {
            if (y==(lattice_cols-1))
              { 
                pn[i++]='}';
                pn[i]='}';
              }
            else
              pn[i]=',';
          }
        else
          {
            if (y == (lattice_cols-1))
              {
                pn[i++]='}';
                pn[i++]=',';
                pn[i]='{';
              }
            else
              pn[i]=',';
          }
        pn[i+1]=0;
        delete temp;
        pn+=i+1;        
      }
}

//
// Algorithms
//

// The Buchmann - Kessler algoritm on the identic matrix, result : transformation matrix
void lattice_gensys::lin_gen_system(lattice_gensys& Tr, long& rank)
{
  debug_handler("lattice_gensys","lin_gen_system(Tr, rank)");
  if (comp_mode == Tr.comp_mode)
     {
      if ((lattice_cols != Tr.lattice_cols) ||
          (lattice_cols != Tr.lattice_rows))
        lidia_error_handler("lattice_gensys","lin_gen_system(Tr, rank) :: "
                      "illegal size of transformation lattice !");
      switch (comp_mode) {
        case bigfloat_mode : {
          lbf->lin_gen_system(*Tr.lbf, prec, y_param, red_count, T, rank); 
          Tr.compute_precision();
        } break;
        case bigint_mode : {
          lbi->lin_gen_system(*Tr.lbi, y_zaehler, y_nenner, red_count, T, rank);
        } break;
        case double_mode : {
          ldo->lin_gen_system(*Tr.ldo, y_param, red_count, T, rank); 
        } break;
      }
      return;    
    }
  lidia_error_handler("lattice_gensys","lin_gen_system(Tr, rank) :: "
                "Two different computing modes !");
  
}

void lin_gen_system(bigint_matrix& Tr, const bigint_matrix& A, long& rank)
{
  debug_handler("lattice_gensys","lin_gen_system(Tr, A, rank)");
  lattice_gensys C(A), S(Tr.get_no_of_rows(), Tr.get_no_of_columns());
  bigint m;
  long i, j;
  
  C.lin_gen_system(S, rank);
  for (i=0; i<Tr.get_no_of_rows();i++)
    for (j=0; j<Tr.get_no_of_columns();j++) 
      {
        S.member(i,j,m);
        Tr.sto(i,j,m);
      }
}

// Multiplies the matrix with the transformation matrix, result : reduced matrix
void lattice_gensys::compute_basis(const lattice_gensys& A, const lattice_gensys& B)
{
  debug_handler("lattice_gensys","compute_basis(A, B)");
  if ((comp_mode==A.comp_mode) && (A.comp_mode==B.comp_mode))
    {
      if ((A.lattice_cols != B.lattice_cols) ||
          (A.lattice_cols != B.lattice_rows) ||
          (lattice_cols != A.lattice_cols) ||
          (lattice_rows != A.lattice_rows))
        lidia_error_handler("lattice_gensys","compute_basis(A, B) :: "
                      "illegal sized lattices !");
      switch (comp_mode) {
        case bigfloat_mode : {
          lbf->compute_basis(*A.lbf,*B.lbf);
        } break;
        case bigint_mode : {
          lbi->compute_basis(*A.lbi,*B.lbi); 
        } break;
        case double_mode : {
          ldo->compute_basis(*A.ldo,*B.ldo); 
        } break;
      }
      return;
    }
  lidia_error_handler("lattice_gensys","compute_basis(A,B) :: "
                "different computing modes !");
}

