//
// lmath.cxx
//
// Some algorithms for having fun with LInteger's 
// Leonard Janke
// August 1996

#include "lmath.hxx"

static const int AToT[256]={ 
    0, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    7, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    6, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    5, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0, 
    4, 0, 1, 0, 2, 0, 1, 0, 3, 0, 1, 0, 2, 0, 1, 0 };
  
 static const int AToB[256]={
    1,   1,   1,   3,   1,   5,   3,   7,   
    1,   9,   5,  11,   3,  13,   7,  15,
    1,  17,   9,  19,   5,  21,  11,  23,   
    3,  25,  13,  27,   7,  29,  15,  31,
    1,  33,  17,  35,   9,  37,  19,  39,
    5,  41,  21,  43,  11,  45,  23,  47,
    3,  49,  25,  51,  13,  53,  27,  55,
    7,  57,  29,  59,  15,  61,  31,  63,
    1,  65,  33,  67,  17,  69,  35,  71,
    9,  73,  37,  75,  19,  77,  39,  79,  
    5,  81,  41,  83,  21,  85,  43,  87,  
   11,  89,  45,  91,  23,  93,  47,  95,  
    3,  97,  49,  99,  25, 101,  51, 103,  
   13, 105,  53, 107,  27, 109,  55, 111,
    7, 113,  57, 115,  29, 117,  59, 119,  
   15, 121,  61, 123,  31, 125,  63, 127,
    1, 129,  65, 131,  33, 133,  67, 135,  
   17, 137,  69, 139,  35, 141,  71, 143,
    9, 145,  73, 147,  37, 149,  75, 151,  
   19, 153,  77, 155,  39, 157,  79, 159,  
    5, 161,  81, 163,  41, 165,  83, 167,  
   21, 169,  85, 171,  43, 173,  87, 175,
   11, 177,  89, 179,  45, 181,  91, 183,  
   23, 185,  93, 187,  47, 189,  95, 191,
    3, 193,  97, 195,  49, 197,  99, 199,  
   25, 201, 101, 203,  51, 205, 103, 207,
   13, 209, 105, 211,  53, 213, 107, 215,  
   27, 217, 109, 219,  55, 221, 111, 223,
    7, 225, 113, 227,  57, 229, 115, 231,  
   29, 233, 117, 235,  59, 237, 119, 239,
   15, 241, 121, 243,  61, 245, 123, 247,  
   31, 249, 125, 251,  63, 253, 127, 255 };

int LMath::LUByte(unsigned int* base, int index)
{
  unsigned int word=*(base+index/LMisc::bytesPerInt);
  int byteWithinWord=index%LMisc::bytesPerInt;
  int offset=(LMisc::bytesPerInt-1-byteWithinWord)*8;

  unsigned int mask= 0xff << offset; 
					
  return int((word&mask) >> offset);
}

int LMath::FirstNonZeroByte(const unsigned int a)
{
  // better not call this with *a==0 !!!!!!

  int byteNum=0;

  unsigned int andFactor=0xff << (LMisc::bytesPerInt-1)*8;

  while ( 1 )
    {
      if ( a & andFactor )
	return byteNum;
      byteNum++;
      andFactor>>=8;
    }
}
 
LInteger LMath::ExtendedEuclid(const LInteger& a, const LInteger& b,
			       LInteger& u, LInteger& v)
{
  // This algorithm is from Henri Cohen's 
  // _A_Course_in_Computational_Algebraic_Number_Theory_ 

  LInteger v1, t1;
  LInteger d, q, v3, t3;
  u=1;
  d=a;

  if ( !b )
    {
      v=0;
      return d;
    }

  v1=0;
  v3=b;

  while ( v3 )
    {
      LMath::LongDivide(v3,d,q,t3);
      t1=u-q*v1;
      u=v1;
      d=v3;
      v1=t1;
      v3=t3;
    }

  v=(d-a*u)/b;
  return d;
}

LInteger LMath::TwoToThe(const int power)
{
  assert( power >= 0 );

  int magSize=power/LMisc::bitsPerInt+1;

  unsigned int* mag=new unsigned int[magSize];

  LMisc::MemZero(mag,magSize);

  *mag=1<<power%LMisc::bitsPerInt;

  return LInteger(mag,magSize,0,0);
}

LInteger LMath::ModExp(const LInteger& g, const LInteger& x, const LInteger& n)
{
  // This algorithm is from Henri Cohen's book 
  // _A_Course_in_Computational_Algebraic_Number_Theory_ 

  LInteger y(1);
  LInteger X(x);
  LInteger z(g);

  while ( X )
    {
      if ( int(X) & 1 )
	y=(y*z)%n;

      X=X>>1;
      z=(z*z)%n;
    }
  return y;
}

LInteger LMath::SpecialModExp(const LInteger& g, const LInteger& x, 
	const LInteger& n)
{
  // This algorithm is from Henri Cohen's book 
  // _A_Course_in_Computational_Algebraic_Number_Theory_ 

  if ( x == 0 )
    return LInteger(1);

  Monty MontyRep(n);

  unsigned int* z=new unsigned int[n._size*128];
  unsigned int* zSquared=new unsigned int[n._size];
  unsigned int* y=new unsigned int[n._size];

  // make z[1]
  int z1Offset=n._size-g._size;

  LMisc::MemZero(z,z1Offset);
  LMisc::MemCopy(g._magnitude,z+z1Offset,g._size);

  MontyRep.ConvertToMonty(z);

  // make zSquared

  MontyRep.MontySquare(z,zSquared);

  // make z[3]  ... z[255]

  for (int i=3; i<256; i+=2)
    MontyRep.MontyMultiply(z+ZIndexToOffset(i-2,n._size),zSquared,
			   z+ZIndexToOffset(i,n._size));

  int start=FirstNonZeroByte(*x._magnitude);

  // deal with first non zero byte of x 
  
  unsigned char a=LUByte(x._magnitude,start);
  int b=AToB[a];
  LMisc::MemCopy(z+ZIndexToOffset(b,n._size),y,n._size);
  int t=AToT[a];
  for (int i=0; i<t; i++)
    MontyRep.MontySquare(y,y);

  // deal with remainder of x 

  for (int byte=start+1; byte<x._size*LMisc::bytesPerInt ; byte++)
    {
      a=LUByte(x._magnitude,byte);
      b=AToB[a];
      t=AToT[a];
      for (int i=0; i<8-t; i++)
	MontyRep.MontySquare(y,y);
      if ( a )
	MontyRep.MontyMultiply(y,z+ZIndexToOffset(b,n._size),y);
      for (int i=0; i<t; i++)
	MontyRep.MontySquare(y,y);
    }

  MontyRep.ConvertFromMonty(y);

  int ySize=LInteger::compress(y,n._size);

  delete[] z;
  delete[] zSquared;

  return LInteger(y,ySize,0,0);
}

void LMath::LongDivide(const LInteger& divisor, const LInteger& dividend, 
		       LInteger& Quotient, LInteger& Remainder)
{
  assert( divisor!=0 );

  if ( divisor._size > dividend._size )
    {
      if ( dividend >= 0 )
	{
	  Quotient=0;
	  Remainder=dividend;
	}
      else
	{
	  if ( divisor > 0 )
	    Quotient=LInteger::Negative(1);
	  else
	    Quotient=1;

	  Remainder=LInteger::AbsoluteValue(divisor)+dividend;
	}

      return;
    }

  unsigned int* q;
  unsigned int* r;
  int qSize, rSize;
    
  BMath::Divide(divisor._magnitude, divisor._size, 
		dividend._magnitude, dividend._size, q,r);

  qSize=LInteger::compress(q,dividend._size-divisor._size+1);
  rSize=LInteger::compress(r,dividend._size+1);

  if (  !dividend._sign && !divisor._sign )
    {
      Quotient=LInteger(q,qSize,0,0);
      Remainder=LInteger(r,rSize,0,0);
      return;
    }

  if ( !dividend._sign && divisor._sign )
    {
      Quotient=LInteger::Negative(LInteger(q,qSize,0,0));
      Remainder=LInteger(r,rSize,0,0);
      return;
    }

  if ( dividend._sign && !divisor._sign )
    {
      Remainder=LInteger(r,rSize,0,0);

      if ( Remainder == 0 )
	Quotient=LInteger::Negative(LInteger(q,qSize,0,0));
      else
	{
	  Quotient=LInteger::Negative(LInteger(q,qSize,0,0)+1);
	  Remainder=LInteger::AbsoluteValue(divisor)-Remainder;
	}

      return;
    }

   // dividend._sign && divisor._sign

  Remainder=LInteger(r,rSize,0,0);

  if ( Remainder == 0 )
    Quotient=LInteger(q,qSize,0,0);
  else
    {
      Quotient=LInteger::Negative(LInteger(q,qSize,0,0)+1);
      Remainder=LInteger::AbsoluteValue(divisor)-Remainder;
    }

  return;
}
