#!/bin/sh
#
# James M. Galvin					1/94
# 
# This script is intended to be a replacement for the specification of
# mhl as a user's showproc.  To use this script, the following line must
# be included in a user's .mh_profile (assuming this script is named
# "pemshow.sh"):
#
# 	showproc: pemshow.sh
# 
# The variable MHLPROC, which specifies the location of the MHL program,
# must be tailored.  See the subroutine that appears immediately after
# these comments.  A command line option of "-nomhl" will substitute
# "cat" for mhl.
# 
# Each message to be shown is examined to determine if it is a PEM
# message according to RFC 1421.  If so, depending on the command line
# arguments, which override .mh_profile arguments, it will de-enhance
# the message and show it, show the enhanced message, or query the user
# as to the message's disposition.  The default is to query the user as
# to the message's disposition.
# 
# Messages will always be displayed in the order in which they appear on
# the command line.  All PEM de-enhancements will be completed before
# displaying any messages.  For each PEM message, its PEM annotations
# are collected into a message which is displayed immediately prior to
# the de-enhanced version of the PEM message; the annotations are not
# saved after viewing.
# 
# 
# The following command line arguments are supported:
# 
# 	-mhl	-- set MHLPROC to location of mhl
#	-nomhl	-- set MHLPROC to cat
# 
# 	-pem	-- always perform PEM de-enhancements
# 	-nopem	-- never perform PEM de-enhancements
# 
# 	-pemq	-- always query as to message disposition
# 	-*	-- all other '-' arguments are passed to MHL
# 
# 	*	-- all other arguments are assumed to be message files
# 
# The presence of -nopem will cause the immediate invocation of MHLPROC.
#
# The presence of -pem will cause each message to be examined to
# determine if it is a PEM message.  If it is, the message will be
# de-enhanced and the de-enhanced message will be displayed in its
# place.  The PEM annotations will be placed in a message and displayed
# immediately prior to the de-enhanced message.
# 
# The presence of -pemq will cause each message to be examined to
# determine if it is a PEM message.  If it is, the user will be queried
# as to whether the message should be de-enhanced and displayed, or
# displayed in its enhanced form.  This option is useful for overriding
# the presence of a -nopem or -pem in a user's .mh_profile.
# 
# If there are multiple occurances of -pem, -nopem, or -pemq, the last
# occurance will override all other occurances.
# 
# A check is made to see if stdin is a terminal.  If it is not, -pemq is
# assumed to be a -pem; -nopem functions as specified.
# 
# ADVANCED COMMENTS
# 
# Some users might wonder why this script co-exists so effectively with
# MIME-PEM, since it only checks for RFC 1421 messages.  The reason why
# this works is because the "show" command checks to see if the message
# to be shown is a MIME message before it invokes a user's showproc.
# Thus, if a message is a MIME message, "mhn" will be invoked.  If you
# have installed the PEM enhancements to "mhn" the message will be
# de-enhanced according to the configuration of your MIME environment.
# Otherwise, "show" invokes your showproc, this script, and RFC 1421 are
# de-enhanced according to its configuration.
# 
# The following may have to be tailored appropriately.

domhl() {
	if [ -r /usr/lib/mh/mhl -a -x /usr/lib/mh/mhl ]; then
		MHLPROC=/usr/lib/mh/mhl
	elif [ -r /usr/local/lib/mh/mhl -a -x /usr/local/lib/mh/mhl ]; then
		MHLPROC=/usr/local/lib/mh/mhl
	else
		echo $0: no mhl found\; continuing with \"more\"... 1>&2
		MHLPROC=more
	fi
}

# Local variables; do not change
ARGS=
DOPEM=0
FILES=
OFILES=
RFILES=

trap 'rm -f $RFILES' 0 1 2 3 9 15

umask 077

domhl

while [ $# -gt 0 ]; do
	case $1 in
	-help)
	    echo Usage: $0 \[ -pem \] \[ -nopem \] \[ mhlargs \] files 1>&2
	    exit 1
	    ;;

	-mhl) domhl()
	    ;;
	-nomhl) MHLPROC=cat
	    ;;

	-pem) DOPEM=1
	    shift
	    ;;
	-nopem) DOPEM=-1
	    shift
	    ;;

	-pemq) DOPEM=0
	    shift
	    ;;

	-*) ARGS="$ARGS $1"
	    shift
	    ;;

	*)  FILES="$FILES $1"
	    shift
	    ;;

	esac
done

tty -s
if [ $? -ne 0 -a $DOPEM -eq 0 ]; then
	DOPEM=1
fi

if [ -z "$FILES" ]; then
	echo $0: no filename specified: unsupported function 1>&2
	exit 1
fi

if [ $DOPEM -eq -1 ]; then
	$MHLPROC $ARGS $FILES
	exit $?
fi

for f in $FILES ; do
	egrep -s -e "^-----BEGIN PRIVACY-ENHANCED MESSAGE-----$" $f

	if [ $? -eq 0 ]; then
		egrep -s -i -e '^proc-type:.*mic-' $f
		if [ $? -eq 0 ]; then
			TYPE=signed
		else
			TYPE=encrypted
		fi

		if [ $DOPEM -eq 0 ]; then
			echo De-enhance $TYPE message $f: \(y/\[n\]\)\?
			read ans
			case $ans in
				y|Y) ;;
				*)   OFILES="$OFILES $f"
				     continue;;
			esac
		fi

		if [ "$TYPE" = "encrypted" ]; then
			pemlogcheck > /dev/null 2>&1
			if [ $? -ne 0 ]; then
				echo "
You must login to PEM before viewing message $f.
Use \"pemlogin\" to do this.
"
				OFILES="$OFILES $f"
				sleep 2
				continue
			fi
		fi
		
		echo De-enhancing message $f...

		DATA="`basename $f`.d$$"
		ANNO="`basename $f`.a$$"

		RFILES="$RFILES $DATA $ANNO"

		(echo PEM-Annotations: ; echo "") >$ANNO

		catrcc $f >$DATA  2>>$ANNO

		if [ $? -eq 0 ]; then
			OFILES="$OFILES $ANNO $DATA"
		else
			OFILES="$OFILES $ANNO $f"
		fi
	else
		OFILES="$OFILES $f"
	fi
done

if [ ! -z "$OFILES" ]; then
	$MHLPROC $ARGS $OFILES
	exit $?
fi

exit 0
