/*
Copyright (C) 1992,1993,1994 Trusted Information Systems, Inc.

Export of this software from the United States of America or
Canada requires a specific license from the United States
Government.  This version of this software is not suitable for
export.

WITHIN THAT CONSTRAINT, the full text of the license agreement
that specifies the conditions under which this software may be
used is published in the file license.txt in the same directory
as that containing the TIS/PEM source.

Trusted Information Systems makes no representation about the
suitability of this software for any purpose.  It is provided
"as is" without express or implied warranty.
*/

/* validate_user - return a list of userlist structures forming a path
 * from the argument user to a root.
 * 
 * Root is defined to be those user records identified by the alias
 * "me".  One of these user records must have been used to create a
 * "trusted" tag for the subordinate user.  The root record must be
 * self-signed with a trusted line.
 * 
 * Certificates will be followed if they exist, although the "final"
 * certificate must contained a trusted tag created by "root".
 */

#include "config.h"
#include "general.h"

#include "bbuf.h"
#include "crypto.h"
#include "cryptocb.h"
#include "key.h"
#include "new.h"
#include "user_list.h"
#include "util.h"


static char **get_rootByAlias(ul, alias)
struct user_list *ul;
char             *alias;
{
    while (ul != NULLUL) {
	if (tagval_user(ul->user, "alias", alias) != NULLVP)
	    break;
	ul = ul->next;
    }

    return ul->user;
}


static int verify_trusted(user, issuer)
char  **user,
      **issuer;
{
    int             ret = NOTOK;
    char           *pubkey = NULLCP,
                   *micinfo_trusted = NULLCP,
                  **av,
                   *cp;
    struct bbuf    *signbb = NULLBB,
                   *mic = NULLBB,
                   *emic = NULLBB,
                   *key = NULLBB,
                   *dsigbb = NULLBB;
    struct bbuf     ekey,
                    esigbb;
    struct algent  *mic_ae = NULL_ALGENT;
    struct algid   *mic_aid = NULL_ALGID;

    /* get the trusted line and find the signature */

    if ((av = tag_user(user, "trusted")) == NULLVP)
	return NOTOK;
    if ((cp = INDEX(*(av+1), ',')) == NULLCP)
	return NOTOK;
    micinfo_trusted = add2cp(NULLCP, ++cp);

    /* flatten user record for signing */

    signbb = trustbb_user(user);

    /* MD5 hash the data */

    if ((mic_ae = getalgcode(algorithms, MD5, MIC)) == NULL_ALGENT)
	goto cleanup;
    (void) gen_md5(NULL_KEY, signbb, &mic);

    /* encode the MIC */

    mic_aid = alloc_algid();
    mic_aid->alg = mic_ae->code;
    if (encode_mic(mic, mic_aid, &emic) != OK)
	goto cleanup;

    /* get issuer's public key and setup for use */

    if ((pubkey = key_user(issuer)) == NULLCP)
	goto cleanup;

    ekey.data = (unsigned char *)pubkey;
    ekey.length = strlen(pubkey);
    (void) bdecode(&ekey, &key);

    if (set_key(key) != OK)
	goto cleanup;

    /* verify the signature */

    /* find just the sig in micinfo_trusted, place in bbuf, decode it, and
       pass that to verify */

    if ((cp = INDEX(micinfo_trusted, ',')) == NULLCP
	|| (cp = INDEX(++cp, ',')) == NULLCP)
	goto cleanup;
    esigbb.data = (unsigned char *)++cp;
    esigbb.length = strlen(cp);
    (void) bdecode(&esigbb, &dsigbb);

    if (verify(dsigbb, emic) != OK) /* XXX FIRST ARGUMENT WRONG */
	goto cleanup;

    ret = OK;

 cleanup:

    FREE(micinfo_trusted);

    FREE_ALGID(mic_aid);

    FREE_BBUF(dsigbb);
    FREE_BBUF(emic);
    FREE_BBUF(key);
    FREE_BBUF(mic);
    FREE_BBUF(signbb);

    return ret;
}


static int verify_cert(user, issuer)
char  **user,
      **issuer;
{
    int             ret = NOTOK;
    struct bbuf     ecert,
                   *dcert = NULLBB,
                    eicert,
                   *dicert = NULLBB,
                    ecrl,
                   *dcrl = NULLBB;
    char          **av;

    /* get the user's and issuer's certificate and the crl */

    if ((av = tag_user(user, "certificate")) == NULLVP)
	goto cleanup;
    ecert.length = strlen(*(av+1));
    ecert.data = alloc_uchar(ecert.length);
    BCOPY(*(av+1), ecert.data, ecert.length);
    (void) bdecode(&ecert, &dcert);

    if ((av = tag_user(issuer, "certificate")) == NULLVP)
	goto cleanup;
    eicert.length = strlen(*(av+1));
    eicert.data = alloc_uchar(eicert.length);
    BCOPY(*(av+1), eicert.data, eicert.length);
    (void) bdecode(&eicert, &dicert);

    if ((av = tag_user(issuer, "crl")) != NULLVP) {
	ecrl.length = strlen(*(av+1));
	ecrl.data = alloc_uchar(ecrl.length);
	BCOPY(*(av+1), ecrl.data, ecrl.length);
	(void) bdecode(&ecrl, &dcrl);
    }

    if (validate_cert(dcert, dicert, dcrl) != OK)
	goto cleanup;

    ret = dcert->status;

 cleanup:

    FREE_BBUF(dcert);
    FREE_BBUF(dcrl);
    FREE_BBUF(dicert);

    return ret;
}


struct user_list  *validate_user(user, roots)
char             **user;
struct user_list  *roots;
{
    char             **trusted_user,
                     **root_alias = NULLVP,
                     **issuer = NULLVP,
                     **dname_issuer,
                     **upk,
                     **ipk;
    char              *alias_issuer = NULLCP,
                      *cp;
    struct user_list  *ret = NULLUL;
    int                atroot = 0,
                       attop = 0;

    /* if we get the trusted line, find the issuer */

    if ((trusted_user = tag_user(user, "trusted")) != NULLVP) {
	if ((cp = INDEX(*(trusted_user+1), ',')) == NULLCP)
	    return NULLUL;
	*cp = NULLC;
	alias_issuer = add2cp(NULLCP, *(trusted_user+1));
	*cp = ',';
    }

    /* do we have a certificate to work with */

    else if (tag_user(user, "certificate") != NULLVP) {
	if ((dname_issuer = tag_user(user, "issuer-name")) == NULLVP)
	    return NULLUL;	/* break out certificate?? */
    }
    else
	return NULLUL;

    /* is this user trusted by root? */

    if (alias_issuer != NULLCP) {
	char             **root_aliases = NULLVP;
	struct user_list  *ul;

	/* collect the root aliases for easy comparisons */

	for (ul = roots; ul != NULLUL; ul = ul->next) {
	    char **user, **alias;

	    for (user = ul->user; user != NULLVP; user = alias + 2) {
		if ((alias = tag_user(user, "alias")) == NULLVP)
		    break;
		root_aliases = add2av(root_aliases, add2cp(NULLCP, "alias"));
		root_aliases = add2av(root_aliases,
				      add2cp(NULLCP, *(alias+1)));
	    }
	}

	if ((root_alias
	     = tagval_user(root_aliases, "alias", alias_issuer)) == NULLVP) {

	    /* no, so let's retrieve the issuer record */
	    rewind_indexfile();
	    if ((issuer = get_tv_user("alias", alias_issuer)) == NULLVP)
		goto cleanup;
	}
	else {
	    char **tmp;

	    /* yes, so let's find the user record in the given list */
	    tmp = get_rootByAlias(roots, *(root_alias+1));
	    while (*tmp != NULLCP)
		issuer = add2av(issuer, add2cp(NULLCP, *tmp++));
	    atroot = 1;
	}

	if (verify_trusted(user, issuer) != OK)
	    goto cleanup;

	FREE_AV(root_aliases);
    }
    else {			/* must have dname for issuer */
	int status;

	rewind_indexfile();
	if ((issuer = get_tv_user("subject-name", *(dname_issuer+1)))
	    == NULLVP)
	    goto cleanup;

	if ((status = verify_cert(user, issuer)) == NOTOK)
	    goto cleanup;

	/* XXX
	 * 
	 * well, we should do something better here, but for a first cut
	 * let's just make sure the signature is good.
	 */
	if (status & BBADSIG)
	    goto cleanup;
    }

    if ((upk = tag_user(user, "public-key")) == NULLVP)
	goto cleanup;
    if ((ipk = tag_user(issuer, "public-key")) == NULLVP)
	goto cleanup;
    attop = ! strcmp(*(upk+1), *(ipk+1));

    ret = alloc_ul();

    if (!atroot) {
	if (attop || (ret->next = validate_user(issuer, roots)) == NULLUL) {
	    FREE_UL(ret);
	    goto cleanup;
	}
	ret->user = issuer;
	issuer = NULLVP;
    }
    else {
	if (tag_user(issuer, "trusted") != NULLVP) {
	    ret->user = issuer;
	    if (!attop && validate_user(issuer, ret) == NULLUL) {
		issuer = NULLVP;
		FREE_UL(ret);
		goto cleanup;
	    }
	    issuer = NULLVP;
	}
	else {
	    FREE_UL(ret);
	    goto cleanup;
	}
    }

 cleanup:

    FREE(alias_issuer);

    FREE_AV(issuer);

    return ret;
}
