/*
 *  $Id: subr.c,v 1.1 1998/12/02 23:37:09 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <fist.h>
#include <cryptfs.h>


#define ENCRYPT_FILENAMES 1


unsigned char cbc_iv[8] = {0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10};


void *
fist_get_userpass(const super_block_t *super)
{
    fist_key_t *ht_entry;
    int pos = (current->uid & FIST_HASH_MASK);

    for (ht_entry = stopd(super)->s_user_key[pos];
	 ht_entry;
	 ht_entry = ht_entry->next) {
	if (ht_entry->uid == current->uid) {
	    fist_dprint(6, "userhash found entry for uid %d\n", current->uid);
	    return (void *) &ht_entry->key;
	}
    }
    fist_dprint(5, "HT did not find entry with uid %d\n", current->uid);
    return NULL;
}


void
fist_set_userpass(const super_block_t *super, unsigned char *key)
{
    fist_key_t *ht_entry;
    int pos = (current->uid & FIST_HASH_MASK);

    fist_dprint(6, "setting key for uid %d\n", current->uid);
    fist_dprint(7, "SET_USERPASS key is %x %x %x\n", key[0], key[1], key[15]);
    for (ht_entry = stopd(super)->s_user_key[pos];
	 ht_entry;
	 ht_entry = ht_entry->next) {
	if (ht_entry->uid == current->uid) {
	    /* key present just have to change it */
	    BF_set_key(&ht_entry->key, 16, key);
	    return;
	}
    }
    /* key not present */
    ht_entry = kmalloc(sizeof(fist_key_t), GFP_KERNEL);
    ht_entry->uid = current->uid;
    BF_set_key(&ht_entry->key, 16, key);
    ht_entry->next = stopd(super)->s_user_key[pos];
    stopd(super)->s_user_key[pos] = ht_entry;
}


void
fist_free_userpass(const super_block_t *super)
{
    fist_key_t *ht_entry, *next_ht;
    int bucket;

    for (bucket = 0; bucket < FIST_HASH_SIZE; ++bucket) {
	for (ht_entry = stopd(super)->s_user_key[bucket];
	     ht_entry;
	     ht_entry = next_ht) {
	    next_ht = ht_entry->next;
	    kfree_s(ht_entry, sizeof(fist_key_t));
	}
	stopd(super)->s_user_key[bucket] = NULL;
    }
}


int
cryptfs_encode_block(const char *from, char *to, int len, void *key)
{
    /* blowfish variables */
    unsigned char iv[8];
    int n;

    if (len > CRYPT_BLOCK_SIZE || len < 0)
	printk("CEB: from=0x%x, to=0x%x, len=%d\n", (int) from, (int) to, len);

    fist_dprint(8, "ENCODING: %d bytes: %x %x %x %x\n",
		len, from[0], from[1], from[2], from[3]);

    memcpy(iv, cbc_iv, 8);
    n = 0;			/* opaque variable for blowfish's internal
				 * stat */
    BF_cfb64_encrypt((char *) from,
		     to,
		     len, (BF_KEY *) key, iv, &n,
		     BF_ENCRYPT);
    fist_dprint(8, "AFTER ENCODING: %d bytes: %x %x %x %x\n",
		len, to[0], to[1], to[2], to[3]);
    return len;
}


int
cryptfs_decode_block(const char *from, char *to, int len, void *key)
{
    /* blowfish variables */
    unsigned char iv[8];
    int n;

    if (len > CRYPT_BLOCK_SIZE || len < 0)
	printk("CDB: from=%x, to=%x, len=%d\n", (int) from, (int) to, len);

    fist_dprint(8, "DECODING: %d bytes: %x %x %x %x\n",
		len, from[0], from[1], from[2], from[3]);

    memcpy(iv, cbc_iv, 8);
    n = 0;			/* opaque variable for blowfish's internal
				 * stat */
    BF_cfb64_encrypt((char *) from,
		     to,
		     len, (BF_KEY *) key, iv, &n,
		     BF_DECRYPT);
    fist_dprint(8, "AFTER DECODING: %d bytes: %x %x %x %x\n",
		len, to[0], to[1], to[2], to[3]);
    return len;
}


int
cryptfs_encode_filename(const char *name,
			int length,
			char **uuencoded_name,
			void *key,
			int skip_dots)
{
    char *crypted_name;
    const char *ptr;
    int rounded_length, uuencoded_length, n, i, j;
    unsigned char iv[8];
    short csum;

    fist_dprint(8, "ENCODEFILENAME: cleartext filename \"%s\"\n", name);

    if (!(ENCRYPT_FILENAMES) || (skip_dots && (name[0] == '.' &&
					       (length == 1 ||
						(name[1] == '.' && length == 2))))) {
	uuencoded_length = length + 1;
	*uuencoded_name = kmalloc(uuencoded_length, GFP_KERNEL);
	memcpy(*uuencoded_name, name, length);
	(*uuencoded_name)[length] = '\0';
	goto out;
    }
    for (csum = 0, i = 0, ptr = name; i < length; ptr++, i++)
	csum += *ptr;
    /*
     * rounded_length is an multiple of 3 rounded-up length
     * the uuencode algorithm processes 3 source bytes at a time
     * so we have to make sure we don't read past the memory
     * we have allocated
     *
     * it uses length + 3 to provide 2 bytes for the checksum
     * and one byte for the length
     */
    rounded_length = (((length + 3) + 2) / 3) * 3;
    crypted_name = kmalloc(rounded_length, GFP_KERNEL);

    memcpy(iv, cbc_iv, 8);
    n = 0;
    *(short *) crypted_name = csum;
    crypted_name[2] = length;
    BF_cfb64_encrypt((char *) name, crypted_name + 3,
		     length, (BF_KEY *) key, iv, &n,
		     BF_ENCRYPT);
    /*
     * clear the last few unused bytes
     * so that we get consistent results from uuencode
     */
    for (i = length + 3; i < rounded_length; i++)
	crypted_name[i] = 0;

    uuencoded_length = (((length + 3) + 2) / 3) * 4 + 1;
    *uuencoded_name = kmalloc(uuencoded_length, GFP_KERNEL);

    for (i = 0, j = 0; i < rounded_length; i += 3, j += 4) {
	(*uuencoded_name)[j] = 48 + ((crypted_name[i] >> 2) & 63);
	(*uuencoded_name)[j + 1] = 48 + (((crypted_name[i] << 4) & 48) | ((crypted_name[i + 1] >> 4) & 15));
	(*uuencoded_name)[j + 2] = 48 + (((crypted_name[i + 1] << 2) & 60) | ((crypted_name[i + 2] >> 6) & 3));
	(*uuencoded_name)[j + 3] = 48 + (crypted_name[i + 2] & 63);
    }
    (*uuencoded_name)[j] = '\0';

    kfree_s(crypted_name, rounded_length);
 out:
    fist_dprint(8, "ENCODEFILENAME: encoded filename \"%s\"\n", *uuencoded_name);
    return uuencoded_length;
}


int
cryptfs_decode_filename(const char *name,
			int length,
			char **decrypted_name,
			void *key,
			int skip_dots)
{
    int n, i, j, saved_length, saved_csum, csum;
    int uudecoded_length, error = 0;
    unsigned char iv[8];
    char *uudecoded_name;

    if (!(ENCRYPT_FILENAMES) || (skip_dots && (name[0] == '.' &&
					       (length == 1 ||
						(name[1] == '.' && length == 2))))) {
	*decrypted_name = kmalloc(length, GFP_KERNEL);
	for (i = 0; i < length; i++)
	    (*decrypted_name)[i] = name[i];
	error = length;
	goto out;
    }
    if (key == NULL) {
	error = -EACCES;
	goto out;
    }
    uudecoded_length = ((length + 3) / 4) * 3;
    uudecoded_name = kmalloc(uudecoded_length, GFP_KERNEL);

    for (i = 0, j = 0; i < length; i += 4, j += 3) {
	uudecoded_name[j] = ((name[i] - 48) <<2) | ((name[i + 1] - 48) >>4);
	uudecoded_name[j + 1] = (((name[i + 1] - 48) <<4) & 240) | ((name[i + 2] - 48) >>2);
	uudecoded_name[j + 2] = (((name[i + 2] - 48) <<6) & 192) | ((name[i + 3] - 48) &63);
    }
    saved_csum = *(short *) uudecoded_name;
    saved_length = uudecoded_name[2];
    if (saved_length > uudecoded_length) {
	fist_dprint(7, "Problems with the length - too big: %d", saved_length);
	error = -EACCES;
	goto out_free;
    }
    *decrypted_name = (char *) kmalloc(saved_length, GFP_KERNEL);
    memcpy(iv, cbc_iv, 8);
    n = 0;
    BF_cfb64_encrypt(uudecoded_name + 3, *decrypted_name,
		     saved_length, (BF_KEY *) key, iv, &n,
		     BF_DECRYPT);
    for (csum = 0, i = 0; i < saved_length; i++)
	csum += (*decrypted_name)[i];
    if (csum != saved_csum) {
	fist_dprint(7, "Checksum error\n");
#if 0
	/* XXX:EZK why +1? */
	kfree_s(*decrypted_name, saved_length + 1);
#else
	kfree_s(*decrypted_name, saved_length);
#endif
	error = -EACCES;
	goto out_free;
    }
    error = saved_length;
 out_free:
    kfree_s(uudecoded_name, uudecoded_length);
 out:
    return error;
}


/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
