#ifndef EC_INCL_ECASH_H
#define EC_INCL_ECASH_H
/* beginning of ecashlib.h */


/****************************************************************
*								*
* ecash.h							*
*								*
* Version history:						*
*  960123 -	Ian Goldberg <iang@cs.berkeley.edu>		*
*  960129 -	Felix Croes <felix@digicash.com>		*
*  960327 -	Bryce Wilcox <bryce@digicash.com>		*
*		Felix Croes <felix@digicash.com>		*
*		Marcel van der Peijl <bigmac@digicash.com>	*
*		Branko Lankester <branko@digicash.com>		*
*		Niels Ferguson <niels@digicash.com>		*
*  960606 -     Bryce Wilcox <bryce@digicash.com>               *
*								*
*								*
*								*
* Copyright (c) 1996 DigiCash					*
*								*
* "DigiCash" and "Ecash" are trademarks.			*
*								*
* This file is published for informational purposes only.	*
* DigiCash makes no claims about its suitability for any	*
* particular purpose.						*
*								*
****************************************************************/


#include <stddef.h>
#include "eecdef.h"


/*
 * About Classes:
 *
 * The ecashlib is best understood as being made up of a set
 * of object oriented classes.  Each class is a conceptual
 * unit designed to perform a specific function in the
 * ecashlib.  Each class has some private data members which
 * contain all of the data that the class uses.  Each class
 * also has some public methods which can be called by other
 * classes or by the Ecash application.
 * 
 * (In fact, many classes also have protected methods which 
 * are available only to ecashlib classes, and private
 * methods, which are available only to the class itself.
 * But Ecash application programmers will not need to know
 * about those methods.)
 *
 * The struct which contains a classes private data is named
 * "EC_Foo".  The methods of a class are named 
 * "EC_foo_func()".  Each method (except for the 
 * "EC_foo_new()" method) takes as its first argument a
 * "this pointer"-- a pointer to the instance of class Foo
 * which is to execute the method.  
 *	Objects (class instances) are created either by 
 * calling "EC_foo_new()" or by calling a "get" method which
 * gets an instance of the class.  For example calling 
 * "EC_info_get_coinage()" creates an instance of EC_Coinage
 * and passes back a pointer to it.
 *	Objects are destroyed by calling "EC_foo_free()".  This
 * is like a C++ destructor.  It performs clean-up 
 * operations, frees any memory allocations or other locks 
 * that the object holds and frees the structure which 
 * contains the object's private data.
 *
 * Although this object-oriented programming methodology may
 * be unfamiliar to you at first, it should become intuitive
 * and valuable as soon as you begin to use it.
 */


/*
 * Conventions:
 *
 *
 *	  Names
 *
 * All global symbols and data types are prefixed with "EC_" to 
 * prevent namespace conflicts.
 *
 * 
 * For functions:	EC_classabbreviation_function_name
 * For data types:	EC_DataTypeName
 * For constants:	EC_CONSTANT_NAME
 * For variables:	var_name
 *				 
 * Each function name includes the name of the class to which that 
 * function belongs.  Functions which have no class (the
 * initialization functions and a couple of others) use the name 
 * "main" for their class name.
 * 
 * Variable names have special conventions.  (Note that in this 
 * API variable names are only found as parameter names in function 
 * declarations.)
 *  1. Pointer variables have "_ptr" appended to their name if they are 
 * going to be assigned to a _new_ object or if they are going to have 
 * free called upon them.  Else, they have "_lnk" appended to their 
 * name.
 *  2. NULL-terminated Char * variables have "_str" in their name.
 *  3. If a parameter is a pointer pointer and it is used to pass data
 * _out_ of the function, then it has "_out" appended to its name.  Note
 * that anytime the data being passed out is a pointer, it is the caller's
 * responsibility to free that pointer once the caller is finished with it.
 *
 *
 *	 Parameters
 *
 * Functions which belong to a class (except for the 'new' functions)
 * always have as their first parameter a pointer to an instance of 
 * that class.
 *
 * Pointer parameters which are used to pass data out of the function
 * always occur after other parameters.
 *
 * On some platforms functions which are exported from the ecashlib
 * will need a keyword in the declaration (e.g. "far", "loadds", 
 * "pascal").  To handle this, each function declared in the API 
 * header files will be prefixed with a macro "EC_EXPORT" which will be
 * #define'd to the appropriate keywords for that platform.  Each
 * application function that is to be called by the ecashlib must be
 * pre-fixed by the macro "CALLBACK" which will be similarly
 * #define'd.
 */


/*
 * What version of the API is this?  The major number is the
 * version of the API.  The minor number represents a revision
 * of the API which is not a version change.
 * Format: major * 0x0100 + minor
 */
#define EC_API_VERSION 0x0100




		/* primitives */

	/* Int, Short, Long, Int32, Int16, Byte */

/*
 * eecdef.h defines the following:
 * 
 * Int -- a primitive datatype which is at least as long as a Short
 *	and at most as long as a Long.
 * UInt -- same as above, but unsigned.
 * Short -- a primitive datatype which is at least 16 bits.
 * UShort -- same as above, but unsigned.
 * Long -- a primitive datatype which is at least 32 bits and is at
 *	least as long as a Short.
 * ULong -- same as above, but unsigned.
 * Int32 -- a primitive datatype which is exactly 32 bits long.
 * UInt32 -- same as above but unsigned.
 * Int16 -- a primitive datatype which is exactly 16 bits long.
 * UInt16 -- same as above but unsigned.
 * Byte -- a primitive datatype which is exactly 8 bits long and
 *	unsigned.
 */



	/* ecashlib header files */
#include "epocktid.h"
#include "eaccid.h"
#include "eerrno.h"
#include "erecord.h"
#include "eacth.h"
#include "emsg.h"
#include "epocket.h"
#include "einfo.h"
#include "ecoinage.h"
#include "eaddr.h"
#include "eamount.h"
#include "emintid.h"
#include "eprotocl.h"
#include "eloglevl.h"



	/* General Ecashlib functions */

/*
 * Initialize the package by specifying ANSI C-style memory
 * allocation, reallocation and deallocation, as well as a 
 * log function and a yield function.  The log function is 
 * called with two parameters, a string to log and a logging
 * level.
 *
 * The malloc and realloc functions must conform to 
 * ANSI C3.159-1989.
 *
 * The realloc function must handle a NULL pointer 
 * "data_ptr" parameter by allocating a new block of the 
 * appropriate size and returning a pointer to it.
 *
 * If a NULL pointer is passed instead of the "malloc", 
 * "realloc" or "free" function pointer then ecashlib will 
 * try to use a standard function in place of each such 
 * function.  This will not work on all platforms.
 *! Why not?  malloc, realloc and free are ANSI C.
 *
 * The "yield" function will get called frequently when 
 * ecashlib is doing computation-intensive functions.  If
 * a NULL pointer is passed instead of a yield function 
 * pointer, it will not be called.
 *
 * return values:
 *	standard EC_Errno
 */
EC_EXPORT EC_Errno EC_main_init(void *malloc_cb_initt(size_t size), 
	void *realloc_cb_initt(void *data_ptr, size_t new_size), 
	void free_cb_initt(void *data_ptr), 
	void yield_cb_initt(int this_is_the_last_yield),
	void log_cb_initt(EC_LogLevel level, const Char *text_str));

/*
 * What version of the ecashlib is this?
 * The return value is of the format "xx.yyy.zzz {keywords ...}"
 * where xx is the major version number, yyy is the minor version
 * number (which involves a change to the API) and zzz is the
 * patchlevel.  Finally optional keywords may follow.
 */
EC_EXPORT Char *EC_main_get_libver(void);

/*
 * Clean_up will cause ecashlib to clean up its files and internal 
 * memory allocation.  It also unlocks any files or callbacks that
 * it may have locked.  Call this as the last thing, after you have 
 * freed all EC_ data-structures that you were responsible for.
 */
EC_EXPORT EC_Errno EC_main_cleanup(void);



	/* EC_Coinage string conversion functions. */

/* These are just for convenience.  You may safely ignore them. */

/*
 * Convert a string to a currency ID.
 *
 * parameters:
 *	name_str: a string identifying the currency.  This can be any of 
 *		several strings such as the 3-letter abbreviation (e.g. "USD"), 
 *		the full name (e.g. "United States Dollar") and perhaps some 
 *		alternate names (e.g. "U.S. Dollar" or "US$").  The list of 
 *		which alternate names are understood by this function is 
 *		maintained by DigiCash.
 *
 * return values:
 *	the currency ID on success, EC_CURRID_INVALID on failure
 */
EC_EXPORT EC_CurrencyID EC_main_str_to_curr(const Char *name_str);

/*
 * Convert a currency ID to the currency's abbreviation.
 *
 * parameters:
 *	currID: the currency
 *	abbr: the 3-letter abbreviation for that currency.  The list
 *		of currencies and their 3-letter abbreviations is maintained
 *		by DigiCash.
 */
EC_EXPORT EC_Errno EC_main_curr_to_abbr(EC_CurrencyID currID, Char abbr_buf[4]);

/*
 * Convert a currency ID to the currency's name.
 *
 * parameters:
 *	currID: the currency
 *
 * return values:
 *	the full name for that currency on success (the list of currencies 
 *		and their names is maintained by DigiCash.), NULL pointer on
 *		failure
 */
EC_EXPORT Char *EC_main_curr_to_name(EC_CurrencyID currID);

/*
 * Convert an EC_Amount and EC_Coinage to its string representation.
 *
 * parameters:
 *	amount: the amount
 *	coinage: the coinage of ecash that the amount is in
 *
 * return values:
 *	the string representation of that amount and coinage on success,
 *		NULL pointer on failure
 */
EC_EXPORT Char *EC_main_value_to_str(EC_Amount amount, 
	const EC_Coinage *coinage_lnk);



/* end of ecashlib.h */
#endif

