#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include "client.h"
#include "des.h"

/* Copy src to dest, stripping leading and trailing writespace */
void strcpy_strip(char *dest, char *src)
{
    char *		end;
    int			len = 0;

    while(*src && isspace(*src)) ++src;
    end = strchr(src, '\0');
    if(!end) return;
    if (end != src) {
	--end;
	while(end > src && isspace(*end)) --end;
	len = end - src + 1;
    }

    memmove(dest, src, len);

    dest[len] = 0;
}

typedef struct {
    char *name;
    char *addr;
    unsigned short port;
    EC_M_Currency currency;
} mintinfo;

static mintinfo knownmints[] = {
    { "Mark Twain Bank", "199.217.176.1", 5885, EC_M_CURRENCY_US_DOLLARS },
    { "EUnet", "193.64.218.250", 5885, EC_M_CURRENCY_FINNISH_MARKS },
    { "Beta Research Bank", "dc.digicash.com", 8666, EC_M_CURRENCY_BETA_RESEARCH_BUCKS },
    { "Other", "", 0, EC_M_CURRENCY_NONE },
    { NULL, NULL, 0, 0 }
};

int main(int argc, char **argv)
{
    char *		progname = argv[0];
    EC_Errno		err;
    int			done;
    EC_W_Wallet		wallet = NULL;
    char *		walletid = NULL;
    char		cmdbuf[1000];
    char *		cmdptr;
    int			res;
    int			amount;
    EC_M_Currency	curr;
    int			minpayments;
    char		shop[1000];
    char		descr[1000];
    char		mfilename[1000];
    char		amtstr[80];
    int			openaccount = 0;
    int		        fromseqno, toseqno;
    int			offset;

    /* Check the arguments */
    --argc; ++argv;
    while (argc) {
	if (!strcmp(argv[0], "-o")) {
	    /* Open an account */
	    openaccount = 1;
	    --argc; ++argv;
	} else if (!strcmp(argv[0], "-w") && argc > 1) {
	    walletid = argv[1];
	    --argc; ++argv;
	    --argc; ++argv;
	} else {
	    /* Error */
	    fprintf(stderr, "Usage: %s [-w walletid] [-o]\n", progname);
	    exit(1);
	}
    }

    err = EC_L_main_init(NULL, NULL, NULL, NULL, NULL);
    if (err) {
	fprintf(stderr, "Error initializing ecash!\n");
	exit(1);
    }

    printf("Sample -lucre client, version 0.1\n");
    printf("Using library version: %s\n", EC_L_main_get_libver());
    printf("-lucre internal version number 0x%08x\n", LUCRE_VERSION);
    printf("This program WILL%s blind withdrawn coins.\n",
	LUCRE_DOES_BLINDING ? "" : " NOT");

    if (openaccount) {
	char		mintnum[1000];
	char		accID[1000];
	char		accpw[1000];
	char		walletpp[1000];
	char		resp[1000];
	char *		walletname;
	struct stat	st;
	mintinfo	*whichmint;
	int		mintcount, mintchoice;
	char		temp_mintaddr[1000];
	char *		mintaddr;
	unsigned short	mintport;
	EC_M_Currency	mintcurr;

	/* See if the wallet exists yet */
	walletname = EC_W_wallet_getname(walletid);
	if (!walletname) {
	    fprintf(stderr, "Cannot determine wallet name!\n");
	    exit(1);
	}
	res = stat(walletname, &st);
	if (!res) {
	    /* There's already something there! */
	    printf("\nA wallet already exists at the pathname\n%s\n",
		    walletname);
	    free(walletname);
	    printf("It is unlikely you want to overwrite this wallet.\n");
	    printf("Do you really want to do so (y/N)? ");
	    fflush(stdout);
	    if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) exit(1);
	    strcpy_strip(resp, cmdbuf);
	    if (resp[0] != 'y' && resp[0] != 'Y') {
		printf("Account opening aborted.\n");
		exit(1);
	    }
	}

	/* Get the mint info */
	printf("\nAt which mint should the account be created:\n");
	mintcount=0;
	for(whichmint = knownmints; whichmint->name; whichmint++) {
	    ++mintcount;
	    printf("%4d. %s\n", mintcount, whichmint->name);
	}
	printf("[1-%d]: ", mintcount);
	fflush(stdout);
	if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) exit(1);
	strcpy_strip(mintnum, cmdbuf);
	mintchoice = atoi(mintnum);
	if (mintchoice < 1 || mintchoice > mintcount) {
	    printf("Invalid selection.  Aborting.\n");
	    exit(1);
	}
	--mintchoice;
	if (knownmints[mintchoice].addr[0] == '\0') {
	    /* They chose "Other"; ask for info */
	    printf("Enter hostname or IP address of mint: ");
	    fflush(stdout);
	    if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) exit(1);
	    strcpy_strip(temp_mintaddr, cmdbuf);
	    mintaddr = temp_mintaddr;
	    printf("Enter port number of mint: ");
	    fflush(stdout);
	    if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) exit(1);
	    mintport = atoi(cmdbuf);
	    printf(
		"Now enter the currency for the account you wish to open.\n"
		"You can enter either the currency number, or, in some cases,\n"
		"the currency code.  For example, US dollars could be entered\n"
		"as \"(#4)\" or as \"USD\" (without the quotes).\n"
	    );
	    printf("Enter currency id for mint: ");
	    fflush(stdout);
	    if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) exit(1);
	    mintcurr = EC_M_CURRENCY_NONE;
	    EC_U_str2curramt(cmdbuf, &mintcurr, NULL, NULL);
	} else {
	    /* We have info */
	    mintaddr = knownmints[mintchoice].addr;
	    mintport = knownmints[mintchoice].port;
	    mintcurr = knownmints[mintchoice].currency;
	}
	/* Get the accountID, account password, and wallet pass phrase */
	printf("\nEnter account name for new account: ");
	fflush(stdout);
	if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) exit(1);
	strcpy_strip(accID, cmdbuf);
	printf("Enter password for new account: ");
	fflush(stdout);
	if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) exit(1);
	strcpy_strip(accpw, cmdbuf);
	printf("\nYou should now enter a pass phrase to be used to protect "
		"your signature key.\n");
	res = des_read_pw_string(cmdbuf, sizeof(cmdbuf)-1, "pass phrase: ", 1);
	if (res) {
	    printf("Passwords don't match.  Aborting.\n");
	    exit(1);
	}
	strcpy(walletpp, cmdbuf);

	/* Open the account */
	res = lucre_openacc(walletid, walletpp, accID, accpw,
	    mintaddr, mintport, mintcurr);
	printf("Account opening %s.\n", res ? "failed" : "succeeded");
	if (res) {
	    exit(1);
	}
    }

    /* Try to open the wallet */
    wallet = EC_W_wallet_open(walletid);
    if (!wallet) {
	fprintf(stderr, "Cannot open wallet!  Set $ECWALLET to the location "
		"of your wallet, or use\n   %s -w walletid\n", progname);
	exit(1);
    }

    done = 0;
    while(!done) {
	/* Get a command */
	printf("\ns-lc) ");
	fflush(stdout);
	if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) break;

	/* Parse it */
	cmdptr = cmdbuf;
	while (*cmdptr == ' ' || *cmdptr == '\t') ++cmdptr;
	switch(*cmdptr) {
	case '\n':
	    /* blank line */
	    break;

	case 'w':
	    /* withdraw */
	    while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
	    amount = 0;
	    minpayments = 8;
	    curr = wallet->userrec->currency;
	    err = EC_U_str2curramt(cmdptr, &curr, &amount, &cmdptr);
	    if (err) {
		printf("Cannot understand amount.\n");
		break;
	    }
	    sscanf(cmdptr, "%d", &minpayments);
	    EC_U_curramt2str(wallet->userrec->currency, amount, amtstr,
		sizeof(amtstr));
	    printf("Withdrawing %s (%d payments):\n", amtstr, minpayments);
	    res = lucre_withdraw(wallet, amount, minpayments);
	    printf("Withdrawal %s.\n", res ? "failed" : "succeeded");
	    break;

	case 'd':
	    /* deposit */
	    while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
	    amount = 0;
	    curr = wallet->userrec->currency;
	    err = EC_U_str2curramt(cmdptr, &curr, &amount, NULL);
	    if (err) {
		printf("Cannot understand amount.\n");
		break;
	    }
	    EC_U_curramt2str(wallet->userrec->currency, amount, amtstr,
		sizeof(amtstr));
	    printf("Depositing %s:\n", amtstr);
	    res = lucre_deposit_cash(wallet, amount);
	    printf("Deposit %s.\n", res ? "failed" : "succeeded");
	    break;

	case 'p':
	    /* pay */
	    while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
	    amount = 0;
	    curr = wallet->userrec->currency;
	    err = EC_U_str2curramt(cmdptr, &curr, &amount, NULL);
	    if (err) {
		printf("Cannot understand amount.\n");
		break;
	    }
	    EC_U_curramt2str(wallet->userrec->currency, amount, amtstr,
		sizeof(amtstr));
	    printf("Paying %s:\n", amtstr);
	    printf("Enter name of shop [@]: ");
	    fflush(stdout);
	    if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) break;
	    strcpy_strip(shop, cmdbuf);
	    if (shop[0] == '\0') {
		strcpy(shop, "@");
	    }
	    printf("Enter description: ");
	    fflush(stdout);
	    if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) break;
	    strcpy_strip(descr, cmdbuf);
	    res = lucre_pay(wallet, amount, shop, descr);
	    printf("Payment %s.\n", res ? "failed" : "succeeded");
	    break;

	case 'g':
	    /* get payment */
	    res = lucre_getpay(wallet);
	    printf("Get payment %s.\n", res ? "failed" : "succeeded");
	    break;

	case 'r':
	    /* request payment */
	    while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
	    amount = 0;
	    curr = wallet->userrec->currency;
	    err = EC_U_str2curramt(cmdptr, &curr, &amount, NULL);
	    if (err) {
		printf("Cannot understand amount.\n");
		break;
	    }
	    EC_U_curramt2str(wallet->userrec->currency, amount, amtstr,
		sizeof(amtstr));
	    printf("Payment request for %s:\n", amtstr);
	    printf("Enter description: ");
	    fflush(stdout);
	    if (!fgets(cmdbuf, sizeof(cmdbuf)-1, stdin)) break;
	    strcpy_strip(descr, cmdbuf);
	    res = lucre_reqpay(wallet, amount, descr);
	    printf("Payment request %s.\n", res ? "failed" : "succeeded");
	    break;

	case 'e':
	    /* expect request */
	    res = lucre_expreq(wallet);
	    printf("Payment request receipt %s.\n", res ? "failed" :
		    "succeeded");
	    break;

	case 'b':
	    /* balance */
	    res = lucre_balance(wallet);
	    break;

	case 'l':
	    /* list payments */
	    while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
	    fromseqno = 0;
	    toseqno = 0;
	    offset = 0;
	    sscanf(cmdptr, "%d%n", &fromseqno, &offset);
	    if (fromseqno == 0) {
		toseqno = 0;
	    } else if (fromseqno < 0) {
		toseqno = -fromseqno;
		fromseqno = 1;
	    } else {
		cmdptr += offset;
		sscanf(cmdptr, "%d", &toseqno);
		/* This lets you write "2-8" */
		if (toseqno < 0) toseqno = -toseqno;
		if (!strcmp(cmdptr, "-\n")) toseqno = INT_MAX;
	    }
	    res = lucre_list_payments(wallet,fromseqno,toseqno);
	    break;

	case 's':
	    /* show contents */
	    res = lucre_show_contents(wallet);
	    break;

	case 'c':
	    /* cancel payment */
	    while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
	    fromseqno = 0;
	    sscanf(cmdptr, "%d", &fromseqno);
	    if (fromseqno < 1) {
		printf("You must specify the sequence number of the "
			"payment to cancel.\n");
		break;
	    }
	    res = lucre_cancel_payment(wallet, fromseqno);
	    printf("Cancel %s.\n", res ? "failed" : "succeeded");
	    break;

	case 'n':
	    /* new passphrase */
	    res = lucre_passwd(wallet);
	    printf("Pass phrase change %s.\n", res ? "failed" : "succeeded");
	    break;

	case 'q':
	    /* quit */
	    done = 1;
	    break;

	case 'M':
	    if (wallet->userrec->userID == 0) {
		while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
		mfilename[0] = '\0';
		sscanf(cmdptr, "%s", mfilename);
		res = lucre_moneychangers_block_create(wallet, mfilename);
		printf("Moneychager block creation %s.\n", res ? "failed" :
		    "succeeded");
		break;
	    }

	case 'm':
	    if (wallet->userrec->userID == 0) {
		while (*cmdptr && *cmdptr != ' ' && *cmdptr != '\t') ++cmdptr;
		mfilename[0] = '\0';
		sscanf(cmdptr, "%s", mfilename);
		if (mfilename[0] == '\0') {
		    printf("You must specify a file from which to read.\n");
		    break;
		}
		res = lucre_moneychangers_block_parse(wallet, mfilename);
		printf("Moneychager block reading %s.\n", res ? "failed" :
		    "succeeded");
		break;
	    }

	default:
	    printf("\nValid commands are:\n");
	    printf("withdraw [amount [minpayments]]\n");
	    printf("deposit [amount]\n");
	    printf("pay [amount]\n");
	    printf("get_pay\n");
	    printf("request_payment [amount]\n");
	    printf("expect_request\n");
	    printf("balance_check\n");
	    printf("list_payments [from-to]\n");
	    printf("show_coins\n");
	    printf("cancel_payment seqno\n");
	    printf("new_passphrase\n");
	    if (wallet->userrec->userID == 0) {
		printf("moneychanger_block_read filename\n");
		printf("Moneychanger_block_create [filename]\n");
	    }
	    printf("quit\n");
	    printf("\nCommands may be abbreviated to their first letter.\n");
	    printf("Unsupplied arguments default as follows:\n");
	    printf("    amount = 0\n    minpayments = 8\n");
	    if (wallet->userrec->userID == 0) {
		printf("    filename = stdout\n");
	    }
	    break;
	}
    }

    EC_W_wallet_close(wallet);
    EC_L_main_cleanup();

    printf("\n");
    return 0;
}
