/* Utilities to convert between (currency, amount) pairs and strings */

#include <string.h>
#include <ctype.h>
#include "lucre.h"

typedef struct {
    EC_M_Currency curr;
    char *label;
    char *decimal;
    int scale;
    int ssize;
} currency_data;

static currency_data currency_list[] = {
    { EC_M_CURRENCY_CYBERBUCKS,          "cb$", ".", 100, -2 },
    { EC_M_CURRENCY_US_DOLLARS,          "USD", ".", 100, -2 },
    { EC_M_CURRENCY_US_DOLLARS,          "$",   ".", 100, -2 },
    { EC_M_CURRENCY_US_DOLLARS,          "US$", ".", 100, -2 },
    { EC_M_CURRENCY_GERMAN_MARKS,        "DEM", ",", 100, -2 },
    { EC_M_CURRENCY_FINNISH_MARKS,       "FIM", ",", 100, -2 },
    { EC_M_CURRENCY_BETA_RESEARCH_BUCKS, "BRB", ".", 100, -2 },
    { EC_M_CURRENCY_NONE,                "",    ".",   1,  0 }
};

/* Convert a currency and amount (possibly negative) into a printable string */
void EC_U_curramt2str_opt(EC_M_Currency curr, Int32 amt, char *buf,
    size_t buflen, UInt32 show_curr, UInt32 show_amt)
{
    char *label = NULL, *neg = NULL, *decimal = NULL;
    int scale, ssize;
    currency_data *c;
    char templabel[20];
    char tempamt[80];
    char *tempdig;
    int res;
    int digtoleft;
    char commac,decimalc;

    /* Reserve one for the \0 */
    if (buflen == 0) return;
    *buf = '\0';
    --buflen;
    if (buflen == 0) return;

    /* If it's negative, record the sign and make it positive */
    if (amt < 0) {
        neg = " -";
        amt = -amt;
    } else {
        neg = " ";
    }

    /* Look through the list for a matching entry */
    for (c = currency_list; c->curr; ++c) {
        if (c->curr == curr) {
            label = c->label;
            decimal = c->decimal;
            scale = c->scale;
            ssize = c->ssize;
            break;
        }
    }

    /* Handle missing entires sanely */
    if (!label) {
        label = templabel;
        sprintf(label, "(#%d)", curr);
	decimal = c->decimal;
	scale = c->scale;
	ssize = c->ssize;
    }

    /* 10^ssize is our resolution; scale = 10^abs(ssize) */
    if (decimal[0] == ',') {
	commac = '.';
	decimalc = ',';
    } else {
	commac = ',';
	decimalc = '.';
    }

    if (show_curr) {
	strncpy(buf, label, buflen+1);
	buf[buflen] = '\0';
	res = strlen(label); if (res > buflen) {
	    buf += buflen;
	    buflen = 0;
	    if (!show_amt) {
		res = buflen;
		strcpy(buf-1, ">");
	    }
	} else {
	    buf += res;
	    buflen -= res;
	}
    }

    /* If we didn't want both the currency and the amount, remove the
	space in between */
    if (!show_curr || !show_amt) {
	++neg;
    }

    if (show_amt) {
	strncpy(buf, neg, buflen+1);
	buf[buflen] = '\0';
	res = strlen(neg) > buflen ? buflen : strlen(neg);
	buf += res;
	buflen -= res;

	sprintf(tempamt, "%d", amt);
	tempdig = tempamt;
	digtoleft = strlen(tempamt) + ssize;
	if (digtoleft <= 0) {
	    if (buflen > 0)  { *buf = '0'; ++buf; --buflen; }
	    if (buflen > 0 && (digtoleft < 0 || ssize < 0))
		{ *buf = decimalc; ++buf; --buflen; }
	    while(digtoleft < 0) {
		if (buflen > 0)  { *buf = '0'; ++buf; --buflen; }
		++digtoleft;
	    }
	    *buf = '\0';
	    strncat(buf, tempamt, buflen);
	    res = strlen(tempamt); if (res > buflen) {
		res = buflen;
		strcpy(buf+res-1, ">");
	    }
	    buf += res;
	    buflen -= res;
	} else {
	    int noroom = 0;
	    while (digtoleft > 0) {
		noroom = 1;
		if (buflen > 0) { *buf = *tempdig ? *tempdig : '0';
				    ++buf; --buflen; noroom = 0; }
		if (*tempdig) ++tempdig;
		--digtoleft;
		if (digtoleft > 0 && (digtoleft % 3 == 0)) {
		    noroom = 1;
		    if (buflen > 0) { *buf = commac;
					++buf; --buflen; noroom = 0; }
		}
	    }
	    if (*tempdig) {
		if (buflen > 0) { *buf = decimalc;
				    ++buf; --buflen; noroom = 0; }
		*buf = '\0';
		strncat(buf, tempdig, buflen);
		res = strlen(tempdig); if (res > buflen) {
		    res = buflen;
		    strcpy(buf+res-1, ">");
		}
		buf += res;
		buflen -= res;
	    } else if (noroom) {
		strcpy(buf-1, ">");
	    }
	}
    }
}

void EC_U_curramt2str(EC_M_Currency curr, Int32 amt, char *buf, size_t buflen)
{
    EC_U_curramt2str_opt(curr, amt, buf, buflen, 1, 1);
}

/* Convert a string into a currency and a (signed) amount.  Either of the
    pointers may be NULL to avoid assignment.  If pcurr is not NULL,
    and the initial value of *pcurr is not EC_M_CURRENCY_NONE, that currency
    will be "forced".  There are some heuristics here, so the program should
    canonicalize the amount for the user (using EC_U_curramt2str), and allow
    its verification. */
EC_Errno EC_U_str2curramt(char *str, EC_M_Currency *pcurr, Int32 *pamt,
    char **pend)
{
    currency_data *c, *maybebest;
    char *p;
    EC_M_Currency curr;
    int offset;
    Int32 neg = 1;
    unsigned long long amt = 0;
    char comma, decimal;
    char lastpunc;
    int digssincepunc;

    /* Start at the beginning of the string */
    p = str ? str : "";

    /* Skip leading whitespace */
    while (*p && isspace(*p)) ++p;

    /* Look for a currency indicator */
    offset = 0;
    if (sscanf(p, "(#%d)%n", (unsigned *)&curr, &offset) > 0) {
	/* We found a currency */
	p += offset;
	for (c = currency_list; c->curr; ++c) ;
    } else {
	/* Scan the list */
	maybebest = NULL;
	for (c = currency_list; c->curr; ++c) {
	    if (pcurr && *pcurr && (*pcurr != c->curr)) continue;
	    if (pcurr && *pcurr && !maybebest) maybebest = c;
	    if (!strncasecmp(p, c->label, strlen(c->label))) {
		/* Match! */
		p += strlen(c->label);
		break;
	    }
	}

	/* Now, if c->curr is not 0, we found a string match.  Otherwise,
	   if maybebest is not NULL, it's the best entry for the requested
	   currency.  Otherwise, we'll just use the defaults at the
	   end of the table. */
	if (!c->curr && maybebest) {
	    c = maybebest;
	}

	curr = c->curr;
    }

    /* p is now pointing at the beginning of the numbers. */

    /* Eat whitespace, and look for maybe a + or - */
    while (*p && isspace(*p)) ++p;
    if (*p == '+') ++p;
    else if (*p == '-') {
	neg = -1;
	++p;
    }
    /* We really should have numbers now; from now on we only read
	digits, commas, and decimals. */
    amt = 0;
    /* Set this properly, but we'll actually accept either one. */
    if (c->decimal[0] == ',') {
	comma = '.';
	decimal = ',';
    } else {
	comma = ',';
	decimal = '.';
    }
    lastpunc = '\0';
    digssincepunc = 0;
    while(*p) {
	if (isdigit(*p)) {
	    /* Shift in the digit */
	    amt = 10*amt + ((*p)-'0');
	    if (lastpunc) digssincepunc++;
	} else if (*p == decimal || *p == comma) {
	    lastpunc = *p;
	    digssincepunc = 0;
	} else if (isspace(*p)) {
	    break;
	} else {
	    /* Bad character; return with an error. */
	    return EC_ERR_BAD_VALUE;
	}
	++p;
    }
    /* Now we get to make a guess as to where the decimal goes. */
    if ((c->ssize != -3 && digssincepunc == 3) || digssincepunc == 0) {
	/* That was a comma, so amt is in units. */
	if (c->ssize < 0) amt *= c->scale;
	else if (c->ssize > 0) amt /= c->scale;
    } else if (c->ssize <= 0 && digssincepunc <= -(c->ssize)) {
	/* That was a decimal, so we're scaled */
	int pw = -(c->ssize)-digssincepunc;
	while(pw) { amt *= 10; --pw; }
    } else {
	/* If it wasn't one of the above cases, it's really weird. */
	return EC_ERR_BAD_VALUE;
    }

    amt *= neg;

    if (pcurr) *pcurr = curr;
    if (pamt) *pamt = amt;
    if (pend) *pend = p;

    return EC_ERR_NONE;
}
