/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"

#include <termios.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

/* global vars used for the string-history routines */
static char **Hist = NULL;
static short HistCur = 0;
static short HistLast = 0;

void mutt_init_history (void)
{
  safe_free ((void **)&Hist);
  if (HistSize)
  {
    Hist = (char **)safe_malloc (HistSize * sizeof (char *));
    memset (Hist, 0, HistSize * sizeof (char *));
  }
}

static void sh_add (char *s)
{
  safe_free ((void **)&Hist[HistLast]);
  Hist[HistLast++] = safe_strdup (s);
  if (HistLast > HistSize - 1) HistLast = 0;
}

static char *sh_next (void)
{
  int next = HistCur + 1;

  if (next > HistLast - 1) next = 0;
  if (Hist[next]) HistCur = next;
  return (Hist[HistCur] ? Hist[HistCur] : "");
}

static char *sh_prev (void)
{
  int prev = HistCur - 1;

  if (prev < 0)
  {
    prev = HistLast - 1;
    if (prev < 0)
      prev = 0;
  }
  if (Hist[prev]) HistCur = prev;
  return (Hist[HistCur] ? Hist[HistCur] : "");
}

/*
 * This is the routine for user input into Mutt.  It looks at the user's
 * Charset to see whether or not it is ISO-8859 compatible (superset of
 * US-ASCII).  If not, it is assumed to be Chinese/Japanese/Korean
 * two-byte charset.
 *
 * Returns 0 if input was given, or -1 if abort.
 */
int ci_enter_string (unsigned char *buf, size_t buflen, int y, int x, int flags)
{
  int ch, lastchar, i, len, j, beg = 0, ch8bit, ch8pair = 0;
  int pass = (flags == M_PASS);
  int iso8859 = mutt_compat_charset (Charset);

  len = COLS - x - 1;

redraw_it:

  move(y,x);
  lastchar = i = strlen (buf);
  if (!pass && i>0)
  {
    beg = lastchar - len;
    if (beg < 0)
      beg = 0;
    else if (beg > 0)
      clrtoeol ();
    for (j=beg; j<lastchar && j<beg+len; j++)
    {
      addch (buf[j]);
    }
  }
  clrtoeol ();
  refresh ();

  FOREVER
  {
    if ((ch = ci_getch ()) == ERR)
      return (-1);

    if (flags == M_FILE && (ch == ' ' || ch == '\t'))
    {
      buf[i]=0;
      mutt_complete (buf);
      goto redraw_it;
    }
    else if (ch == '\t' && (flags == M_ALIAS))
    {
      /* invoke the alias-menu to get more addresses */
      buf[i] = 0;
      alias_menu (buf, buflen);
      return (0);
    }
    else if (ch == KEY_UP)
    {
      if (!pass)
      {
	strfcpy(buf, sh_prev(), buflen);
	goto redraw_it;
      }
    }
    else if (ch == KEY_DOWN)
    {
      if (!pass)
      {
	strfcpy(buf, sh_next(), buflen);
	goto redraw_it;
      }
    }
    /* try to catch anything that looks like a delete key */
    else if (ch == KEY_BACKSPACE || ch == KEY_DC || ch == 127 || ch == ctrl('H'))
    {
      if (i == 0)
      {
	beep();
	refresh();
	continue;
      }
      if (i == lastchar)
      {
	lastchar--;
	i--;
	if ( ! iso8859 )
	{
	  if ((buf[i] & 0x80) & (buf[i-1] & 0x80))
	  {
	    lastchar--;
	    i--;
	  }
	}
	if (!pass)
	{
	  if (i < beg)
	  {
	    move(y,x);
	    clrtoeol();
	    beg -= len;
	    if (beg < 0) beg = 0;
	    for(j=beg; j<lastchar && j<beg+len; j++)
	    {
	      addch(buf[j]);
	    }
	  }
	  else
	  {
	    move(y,x+i-beg);
	    clrtoeol();
	    if ( ! iso8859 )
	    {
	      if ( ( buf[i-1] & 0x80 ) & ( buf[i-2] & 0x80 ) )
		delch();
	    }
	    if ( i > beg ) delch();
	  }
	}
      }
      else
      {
	if ( ! iso8859 )
	  ch8bit = ((buf[i-1] & 0x80) & (buf[i-2] & 0x80))? 1: 0; 
	else
	  ch8bit = 0;
	      
	for ( j = i ; j < lastchar ; j++ )
	{
	  if (ch8bit)
	    buf[j-2] = buf[j];
	  else
	    buf[j-1] = buf[j];
	}
	      
	i--;
	lastchar--;
	      
	if ( ch8bit )
	{
	  i--;
	  lastchar--;
	}
	      
	if (!pass)
	{
	  buf[lastchar] = 0;
	  move(y,x+i-beg);
	  clrtoeol();   
	  for(j=i; j<lastchar && j<beg+len; j++)
	  {
	    addch(buf[j]);
	  }
	  move(y,x+i-beg);
	}
      }
    }
    else if (ch == ctrl('A') || ch == KEY_HOME)
    {
      i = 0;
      if (!pass)
      {
	move(y,x);
	beg = 0;
	if (x + lastchar > len)
	{
	  clrtoeol();
	  for(j=0;j<len;j++)
	    addch(buf[j]);
	  move(y,x);
	}
      }
    }
    else if (ch == ctrl('E') || ch == KEY_END)
    {
      if (!pass)
      {
	if (lastchar < beg + len)
	  move(y,x+lastchar-beg);
	else
	{
	  move(y,x);
	  clrtoeol();
	  beg = lastchar-len/2;
	  for(j=beg; j<lastchar; j++)
	    addch(buf[j]);
	}
      }
      i = lastchar;
    }
    else if (ch == ctrl('U'))
    {
      lastchar = i = 0;
      buf[0] = 0;
      if (!pass)
      {
	move(y,x);
	clrtoeol();
	beg = 0;
      }
    }
    else if (ch == ctrl('K'))
    {
      lastchar = i;
      buf[i] = 0;
      if (!pass)
	clrtoeol();
    }
    else if (ch == ctrl('B') || ch == KEY_LEFT)
    {
      if (i == 0)
	beep();
      else {
	i--;
	
	if ( ! iso8859 )
	{
	  if ( ( buf[i] & 0x80 ) & ( buf[i-1] & 0x80 ) ) i--;
	}
	
	if (!pass)
	{
	  if (i < beg)
	  {
	    beg -= len;
	    if (beg < 0) beg = 0;
	    move(y,x);
	    clrtoeol();
	    for(j=beg;j<beg+len;j++)
	      addch(buf[j]);
	  }
	}
	move(y,x+i-beg);
      }
    }
    else if (ch == ctrl('F') || ch == KEY_RIGHT)
    {
      if (i == lastchar)
	beep();
      else
      {
	i++;
	
	if ( ! iso8859 )
	{
	  if ( ( buf[i] & 0x80 ) & ( buf[i+1] & 0x80 ) ) i++;
	}
	      
	if (!pass)
	{
	  if (i >= beg + len)
	  {
	    beg = i;
	    move(y,x);
	    clrtoeol();
	    for(j=beg; j<lastchar && j<beg+len; j++)
	      addch(buf[j]);
	    move(y,x);
	  }
	  else
	    move(y,x+i-beg);
	}
      }
    }
    else if (ch == ctrl('D'))
    {
      for (j=i;j<lastchar;j++)
	buf[j] = buf[j+1];
      lastchar--;
      if (!pass)
      {
	clrtoeol();
	for (j=i;j<lastchar && j<beg+len; j++)
	  addch(buf[j]);
	move(y,x+i-beg);
      }
    }
    else if (CI_is_return(ch))
    {
      buf[lastchar] = 0;
      if (!pass)
	sh_add (buf);

      /* reset the history to the last element */
      HistCur = HistLast;

      return(0);
    }
    else if (IsPrint(ch) && (lastchar+1 < buflen))
    {
      if ( ! iso8859 )
      {
	ch8pair = ( ch & 0x80 )? ci_getch(): 0;
	ch8bit = ( ch8pair && IsPrint(ch8pair) );
      }
      else
	ch8bit = 0;

      if (i == lastchar)
      {
	buf[i++] = ch;
	lastchar++;
	      
	if ( ch8bit )
	{
	  buf[i++] = ch8pair;
	  lastchar++;
	}
	      
	if (!pass)
	{
	  if (i > beg+len)
	  {
	    beg = i - len/2;
	    move(y,x);
	    clrtoeol();
	    for (j=beg;j<lastchar;j++)
	      addch(buf[j]);
	  }
	  else
	  {
	    addch(ch);
	    if ( ch8bit ) addch(ch8pair);
	  }
	}
      }
      else
      {
	for ( j = lastchar ; j > i ; j-- )
	{
	  if ( ch8bit )
	    buf[j+1] = buf[j-1];
	  else
	    buf[j] = buf[j-1];
	}
	buf[i++] = ch;
	lastchar++;

	if ( ch8bit )
	{
	  buf[i++] = ch8pair;
	  lastchar++;
	}

	if (!pass)
	{
	  if (i > beg + len)
	  {
	    move(y,x);
	    clrtoeol();
	    beg = ch8bit ? i-2 : i-1;
	    for(j=beg;j<lastchar && j<beg+len; j++)
	      addch(buf[j]);
	  }
	  else
	  {
	    clrtoeol();
	    j = ch8bit ? i-2 : i-1;
	    for ( ; j < lastchar && j < beg + len ; j++ )
	      addch(buf[j]);
	  }
	  move(y,x+i-beg);
	}
      }
      if ( ch8bit ) ch8pair = 0;
    }
    else
      beep();
    refresh();
  }
  /* not reached */
}

int ci_get_field (char *field, char *buf, size_t buflen, int complete)
{
  int ret;
  int len = strlen (field); /* in case field==buffer */

  CLEARLINE (LINES-1);
  addstr (field);
  refresh ();
  ret = ci_enter_string (buf, buflen, LINES-1, len, complete);
  CLEARLINE (LINES-1);
  return (ret);
}

void ci_clear_error (void)
{
  Errorbuf[0] = 0;
  CLEARLINE (LINES-1);
}

void ci_edit_file (const char *p)
{
  char cmd[STRING];

  endwin ();
  if (strstr (Editor, "%s"))
    snprintf (cmd, sizeof (cmd), Editor, p);
  else
    snprintf (cmd, sizeof (cmd), "%s %s", Editor, p);
  mutt_system (cmd);
  keypad (stdscr, TRUE);
}

int mutt_yesorno (const char *msg, int def)
{
  int ch;

  CLEARLINE(LINES-1);
  printw("%s: [%c] ", msg, def ? 'y' : 'n');
  FOREVER {
    refresh();
    ch = ci_getch();
    if (ch == ERR)
      return(-1);
    if (CI_is_return(ch)) break;
    else if (ch == 'y')
    {
      def = 1;
      break;
    }
    else if (ch == 'n')
    {
      def = 0;
      break;
    }
    else
      beep();
  }
  addstr (def ? "Yes" : "No");
  refresh ();
  return (def);
}

int ci_getch (void)
{
  int ch = getch ();

  return (ch == ctrl ('G') ? ERR : ch);
}

int mutt_get_password (char *msg, char *buf, size_t buflen)
{
  int rc;

  CLEARLINE (LINES-1);
  addstr (msg);
  rc = ci_enter_string (buf, buflen, LINES-1, strlen(msg), M_PASS);
  CLEARLINE (LINES-1);
  return (rc);
}

void mutt_error (const char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  vsprintf (Errorbuf, fmt, ap);
  va_end (ap);
  
  SETCOLOR (MT_COLOR_ERROR);
  mvaddstr (LINES-1, 0, Errorbuf);
  clrtoeol ();
  SETCOLOR (MT_COLOR_NORMAL);
  refresh ();
}

void ci_endwin (const char *msg)
{
  move (LINES-1, COLS-1);
  attrset (A_NORMAL);
  refresh ();
  fputc ('\n', stdout);
  endwin ();
  if (msg)
    puts (msg);
}

void ci_start_curses (void)
{
#ifdef USE_SLANG_CURSES
  SLtt_Ignore_Beep = 1; /* don't do that #*$@^! annoying visual beep! */
#endif
  initscr();
  ci_start_color();
  keypad(stdscr, TRUE);
  cbreak();
  noecho();
  ci_signal_init();
}

void mutt_perror (const char *s)
{
  char *p = strerror (errno);

  mutt_error ("%s: %s (errno = %d)", s, p ? p : "unknown error", errno);
}

int ci_any_key_to_continue (const char *s)
{
  struct termios t;
  struct termios old;
  int f, ch;

  f = open ("/dev/tty", O_RDONLY);
  tcgetattr (f, &t);
  memcpy ((void *)&old, (void *)&t, sizeof(struct termios)); /* save original state */
  t.c_lflag &= ~(ICANON | ECHO);
  t.c_cc[VMIN] = 1;
  t.c_cc[VTIME] = 0;
  tcsetattr (f, TCSADRAIN, &t);
  if (s)
    fputs (s, stdout);
  else
    fputs ("Press any key to continue...", stdout);
  fflush (stdout);
  ch = fgetc (stdin);
  fflush (stdin);
  tcsetattr (f, TCSADRAIN, &old);
  close (f);
  fputs ("\r\n", stdout);
  return (ch);
}

void mutt_do_pager (char *banner, char *tempfile)
{
  if (!strcmp (Pager, "builtin"))
    ci_simple_pager (banner, tempfile, NULL);
  else
  {
    char cmd[STRING];
    
    endwin ();
    snprintf (cmd, sizeof (cmd), "%s %s", Pager, tempfile);
    mutt_system (cmd);
    mutt_unlink (tempfile);
  }
}
