/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "keymap.h"

#include <stdlib.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>

#define M_FOLDERFMT "---Mutt: Folder Mode [Directory: %s]"

struct folder_file {
  mode_t mode;
  char *name;
  char *desc;
};

/*
 * Global variables for this package.
 */
static struct folder_file *Entries = NULL;
static short LastEntry = 0; /* number of real entries */
static short MaxEntry = 0;  /* max entry */
static short DescFieldLen = 0;
static short NameFieldLen = 0;
static char LastDir[_POSIX_PATH_MAX] = "";

/*
 * Frees up the memory allocated for the local-global variables.
 */
static void destroy_global (void)
{
  int c;

  for (c = 0; c < LastEntry; c++)
  {
    safe_free ((void **)&Entries[c].name);
    safe_free ((void **)&Entries[c].desc);
  }
  safe_free ((void **)&Entries);
}

int compare_entries (const void *a, const void *b)
{
  struct folder_file *pa = (struct folder_file *)a;
  struct folder_file *pb = (struct folder_file *)b;

  return (strcmp (pa->name, pb->name));
}

static int examine_directory (const char *d)
{
  struct stat s;
  DIR *dp;
  struct dirent *de;
  int l;
  char buffer[LONG_STRING];

  if (stat (d, &s) == -1)
  {
    mutt_perror (d);
    return (-1);
  }

  if (!S_ISDIR (s.st_mode))
  {
    mutt_error ("Warning: %s is not a directory", buffer);
    sleep (2);
    return (-1);
  }

  LastEntry = 0;
  NameFieldLen = 0;
  DescFieldLen = 0;
  MaxEntry = 256;
  Entries = (struct folder_file *)safe_malloc (sizeof (struct folder_file) * MaxEntry);

  if ((dp = opendir (d)) == NULL)
    return (-1);

  while ((de = readdir (dp)) != NULL)
  {
    if (strcmp (de->d_name, ".") == 0) continue;    /* we don't need . */
    
    sprintf (buffer, "%s/%s", d, de->d_name);
    if (stat (buffer, &s) == -1)                 /* get file-info */
      continue;
    
    if ((! S_ISREG (s.st_mode)) && (! S_ISDIR (s.st_mode))) continue;
    
    if (S_ISDIR (s.st_mode))
      sprintf (buffer, "Directory");
    else
      sprintf (buffer, "(%7d bytes)", (int)s.st_size);
    
    if (LastEntry == MaxEntry)
    {
      /* need to allocate more space */
      safe_realloc ((void **)&Entries, sizeof (struct folder_file) * (MaxEntry += 256));
    }

    Entries[LastEntry].mode = s.st_mode;
    Entries[LastEntry].name = safe_strdup (de->d_name);
    Entries[LastEntry].desc = safe_strdup (buffer);
    
    /* desclen and namelen is strlen of the longest strings */
    if ((l = strlen (de->d_name)) > NameFieldLen)
      NameFieldLen = l;

    if ((l = strlen (buffer)) > DescFieldLen)
      DescFieldLen = l;
    
    LastEntry++;
  }
  closedir (dp);  

  qsort (Entries, LastEntry, sizeof (struct folder_file), compare_entries);

  return 0;
}

int select_file_search (const char *pat, MUTTMENU *menu)
{
  int i;
  RE_TYPE (re);
  
  if (REGCOMP (re, pat, 0) != 0)
  {
    beep ();
    mutt_error ("Error in regexp!");
    return (-1);
  }
  
  for (i = menu->current + 1 ; i < menu->max ; i++)
  {
    if (REGEXEC (re, Entries[i].name) == 0)
    {
      REGFREE (re);
      return (i);
    }
  }

  beep ();
  mutt_error ("Not found.");
  
  REGFREE (re);

  return (-1);
}

void folder_makeEntry (char *s, int num, size_t slen)
{
  char fmt[SHORT_STRING];

  /*
   * the version of snprintf distributed with Mutt doesn't seem to understand
   * the format "%-*s", so I have to build the format string first...
   */
  snprintf (fmt, sizeof (fmt), "%%-%ds   %%-%ds", NameFieldLen, DescFieldLen);
  snprintf (s, slen, fmt, Entries[num].name, Entries[num].desc);
}

void mutt_select_file (char *f, size_t flen)
{
  char buffer[STRING];
  MUTTMENU *menu;
  
  if (!LastDir[0])
  {
    strfcpy (LastDir, Maildir, sizeof (LastDir));
    mutt_expand_path (LastDir, sizeof (LastDir));
  }

  *f = 0;

  if (examine_directory (LastDir) == -1) return;

  snprintf (buffer, sizeof (buffer), M_FOLDERFMT, LastDir);
  menu = mutt_newMenu (buffer);
  menu->menu = MENU_FOLDER;
  menu->makeEntry = folder_makeEntry;
  menu->search = select_file_search;
  menu->max = LastEntry;

  FOREVER
  {
    switch (mutt_menuLoop (menu))
    {
      case OP_GENERIC_SELECT_ENTRY:

	if (S_ISDIR (Entries[menu->current].mode))
	{
	  if (strcmp (Entries[menu->current].name, "..") == 0)
	  {
	    if (strcmp ("..", LastDir+strlen (LastDir)-2) == 0)
	      strcat (LastDir, "/..");
	    else
	    {
	      char *p = strrchr (LastDir, '/');

	      if (strcmp (LastDir, "/") == 0);
	      else if (p)
		*p = 0;
	      else
		strcpy (LastDir, "/..");
	    }
	  }
	  else
	    sprintf (LastDir + strlen (LastDir), "/%s", Entries[menu->current].name);

	  destroy_global ();

	  if (examine_directory (LastDir) == -1) return;

	  menu->current = 0;
	  menu->top = 0;
	  menu->max = LastEntry;
	  safe_free ((void **)&menu->title);
	  snprintf (buffer, sizeof (buffer), M_FOLDERFMT, LastDir);
	  menu->title = safe_strdup (buffer);
	  menu->redraw = REDRAW_FULL;
	  break;
	}
	snprintf (f, flen, "%s/%s", LastDir, Entries[menu->current].name);

	/*
	 * Fall through to OP_GENERIC_EXIT
	 */

      case OP_GENERIC_EXIT:

	destroy_global ();
	mutt_menuDestroy (&menu);
	return;
    }
  }
  /* not reached */
}
