/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

typedef void handler (BODY *, STATE *);
typedef handler *handler_t;

int Index_hex[128] = {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     0, 1, 2, 3,  4, 5, 6, 7,  8, 9,-1,-1, -1,-1,-1,-1,
    -1,10,11,12, 13,14,15,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,10,11,12, 13,14,15,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1
};

int Index_64[128] = {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
    52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,-1,-1,-1,
    -1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
    15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
    -1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
    41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1
};

void mutt_decode_xbit (STATE *s, long len)
{
  int linelen;
  char buffer[LONG_STRING];

  while (len > 0) {
    if (fgets(buffer, LONG_STRING, s->fpin) == 0) return;
    linelen = strlen(buffer);
    len -= linelen;
    if (buffer[linelen-2] == '\r') {
      buffer[linelen-2] = '\n';
      buffer[linelen-1] = 0;
    }
    if (s->prefix) state_puts(s->prefix, s);
    state_puts(buffer, s);
  }
}

void mutt_decode_quoted (STATE *s, long len, int istext)
{
  char *c, buffer[LONG_STRING];
  int ch, soft = 0;

  while (len > 0) {
    fgets(buffer, LONG_STRING, s->fpin);
    c = buffer;
    len -= strlen(buffer);
    if (s->prefix && !soft)
      state_puts(s->prefix, s);
    soft = 0;
    while (*c) {
      if (*c == '=') {
        if (c[1] == '\n' || c[1] == '\r' || c[1] == ' ' || c[1] == '\t') {
          /* Skip whitespace at the end of the line since BODY does not
           * allow for it */
          soft = 1;
          break;
        }
        ch = hexval((int)c[1]) << 4;
        ch |= hexval((int)c[2]);
        state_putc(ch, s);
        c += 3;
      }
      else if (istext && c[0] == '\r' && c[1] == '\n') {
        state_putc('\n', s);
        break;
      }
      else {
        state_putc(*c, s);
        c++;
      }
    }
  }
}

void mutt_decode_base64 (STATE *s, long len, int istext)
{
  char buffer[5];
  int c1, c2, c3, c4, ch, cr = 0;

  buffer[4] = 0;

  if (s->prefix)
    state_puts(s->prefix, s);
  while (len > 0) {
    buffer[0] = fgetc(s->fpin);
    len--;
    if (whitespace(buffer[0]))
      continue;
    buffer[1] = fgetc(s->fpin);
    buffer[2] = fgetc(s->fpin);
    buffer[3] = fgetc(s->fpin);
    len -= 3;

    dprint(1,(debugfile,"decode_base64(): buffer=\"%s\", len=%ld\n", buffer, len));

    c1 = base64val((int)buffer[0]);
    c2 = base64val((int)buffer[1]);
    ch = (c1 << 2) | (c2 >> 4);

    if (cr && ch != '\n')
      state_putc('\r', s);
    cr = 0;
      
    if (istext && ch == '\r')
      cr = 1;
    else {
      state_putc(ch, s);
      if (ch == '\n' && s->prefix)
        state_puts(s->prefix, s);
    }

    if (buffer[2] == '=')
      break;
    c3 = base64val((int)buffer[2]);
    ch = ((c2 & 0xf) << 4) | (c3 >> 2);

    if (cr && ch != '\n')
      state_putc('\r', s);
    cr = 0;

    if (istext && ch == '\r')
      cr = 1;
    else {
      state_putc(ch, s);
      if (ch == '\n' && s->prefix)
        state_puts(s->prefix, s);
    }

    if (buffer[3] == '=')
      break;
    c4 = base64val((int)buffer[3]);
    ch = ((c3 & 0x3) << 6) | c4;

    if (cr && ch != '\n')
      state_putc('\r', s);
    cr = 0;

    if (istext && ch == '\r')
      cr = 1;
    else {
      state_putc(ch, s);
      if (ch == '\n' && s->prefix)
        state_puts(s->prefix, s);
    }
  }
}

void mutt_text_handler (BODY *a, STATE *s)
{
  if (!a) {
    state_puts("[Error: mutt_text_handler() called with NULL argument.]\n", s);
    return;
  }
  if (a->encoding == ENCQUOTEDPRINTABLE)
    mutt_decode_quoted(s, a->length, 1);
  else if (a->encoding == ENCBASE64)
    mutt_decode_base64(s, a->length, 1);
  else
    mutt_decode_xbit(s, a->length);
}

void text_unsupported_handler (BODY *a, STATE *s)
{
  char buffer[STRING];

  if (s->displaying) {
    sprintf(buffer, "[%s/%s is not supported.  Treating as Text/Plain.]\n\n",
	    TYPE(a->type), a->subtype);
    state_puts(buffer, s);
  }
  mutt_text_handler(a, s);
}

/* A minimal implementation of RFC1563. */
void text_enriched_handler (BODY *a, STATE *s)
{
  long bytes = a->length;
  int lastchar = 0;
  char *p, buffer[LONG_STRING];
  char *q;

  dprint(3,(debugfile,"text_enriched_handler(): entering.\n"));
  while (bytes > 0) {
    if (fgets(buffer, sizeof(buffer), s->fpin) == 0) break;
    bytes -= strlen(buffer);
    p = buffer;
    while (*p) {
      if (*p == '<') {
        p++;
        if (*p == '<') {
          state_putc('<', s);
          p++;
          continue;
        }
        if (strncasecmp("param>", p, 6) == 0) {
          while (bytes > 0) {
            p = strstr(p, "</");
            if (!p) {
              if (fgets(buffer, sizeof(buffer), s->fpin) == NULL) break;
              bytes -= strlen(buffer);
              p = buffer;
            }
            else {
              p += 2;
              if (strncasecmp("param>", p, 6) == 0) {
                p += 6;
                break;
              }
            }
          }
          continue;
        }
        /* skip over the parameter. */
        if ((q = strchr(p, '>')) != 0) p = q;
        p++;
      }
      else if (*p == '\n') {
        if (lastchar == '\n')
          state_putc('\n', s);
        else
          state_putc(' ', s);
        p++;
      }
      else {
        state_putc(*p, s);
        p++;
      }
      lastchar = *p;
    }
  }
  dprint(3,(debugfile,"text_enriched_handler(): exiting.\n"));
}

#define TXTPLAIN    1
#define TXTENRICHED 2
#define TXTHTML     3

void alternative_handler (BODY *a, STATE *s)
{
  BODY *choice = 0;
  int type = 0;

  if (a && a->parts) a = a->parts;
  while (a) {
    if (a->type == TYPETEXT) {
      if (strcasecmp("plain", a->subtype) == 0) {
        choice = a;
        type = TXTPLAIN;
      }
      else if (strcasecmp("enriched", a->subtype) == 0) {
        if (type == 0 || type > TXTENRICHED) {
          choice = a;
          type = TXTENRICHED;
        }
      }
      else if (strcasecmp("html", a->subtype) == 0) {
        if (type == 0) {
          choice = a;
          type = TXTHTML;
        }
      }
    }
    a = a->next;
  }
  if (choice)
    mutt_body_handler(choice, s);
  else if (s->displaying) {
    /* didn't find anything that we could display! */
    state_puts("[Error!  Could not display any parts of Multipart/Alternative!]\n", s);
  }
}

void charset_unsupported_handler (BODY *a, STATE *s)
{
  char buffer[STRING], *charset;

  if (s->displaying) {
    charset = mutt_get_parameter("charset", a->parameter);
    sprintf(buffer, "[Warning!  %s charset is not compatible with your display.]\n\n", charset ? charset : "US-ASCII");
    state_puts(buffer, s);
  }
  mutt_text_handler(a, s);
}

void unsupported_handler (BODY *a, STATE *s)
{
  char buffer[STRING];

  if (s->displaying) {
    snprintf(buffer, sizeof(buffer),
	     "[%s/%s is unsupported.  Use 'v' to view this part.]\n",
	     TYPE(a->type),
	     a->subtype);
    state_puts(buffer, s);
  }
}

void message_handler (BODY *a, STATE *s)
{
  char buffer[LONG_STRING];

  FOREVER {
    fgets(buffer, LONG_STRING, s->fpin);
    if (buffer[0] == '\n' || (buffer[0] == '\r' && buffer[1] == '\n') ||
	(!whitespace(buffer[0]) && strchr(buffer, ':') == 0))
      break;
    if (s->prefix) state_puts(s->prefix, s);
    state_puts(buffer, s);
  }
  if (s->prefix) state_puts(s->prefix, s);
  state_putc('\n', s);
  mutt_body_handler(a->parts, s);
}

void multipart_handler (BODY *a, STATE *s)
{
  BODY *p = a->parts;
  char buffer[STRING];
  int count;

  for (p=a->parts, count=1; p; p=p->next, count++) {
    if (s->displaying) {
      sprintf(buffer, "[Attachment #%d: %s]\n", count, p->description ? p->description : (p->filename ? p->filename : ""));
      state_puts(buffer, s);
      
      sprintf(buffer, "[Type: %s/%s, Encoding: %s, Size: %ld]\n\n",
        TYPE(p->type), p->subtype, ENCODING(p->encoding),
        p->length);
      state_puts(buffer, s);
    }
    else if (a->description) {
      sprintf(buffer, "Content-Description: %s\n", a->description);
      state_puts(buffer, s);
    }
    mutt_body_handler(p, s);
    state_putc('\n', s);
  }
}

handler_t get_handler (BODY *a)
{
  if (a->type == TYPETEXT)
  {
    if (strcasecmp("plain", a->subtype) == 0)
    {
      char *charset = mutt_get_parameter("charset", a->parameter);
      if (charset)
      {
        if (strcasecmp(charset, Charset) == 0 ||
          (mutt_compat_charset(Charset) && mutt_compat_charset(charset)))
        return mutt_text_handler;
      }
      /* no charset was given, so it defaults to US-ASCII */
      else if (mutt_compat_charset(Charset))
        return mutt_text_handler;
      return charset_unsupported_handler;
    }
    if (strcasecmp("enriched", a->subtype) == 0)
      return text_enriched_handler;
    return(text_unsupported_handler);
  }
  if (a->type == TYPEMESSAGE)
  {
    if (strcasecmp("rfc822", a->subtype) == 0)
      return(message_handler);
    return(mutt_text_handler);
  }
  if (a->type == TYPEMULTIPART)
  {
    if (strcasecmp("alternative", a->subtype) == 0)
      return alternative_handler;
#ifdef _PGPPATH
    else if (strcasecmp("signed", a->subtype) == 0)
    {
      char *p = mutt_get_parameter("protocol", a->parameter);

      if (!p)
      {
        dprint (1, (debugfile, "get_handler(): no \"protocol\" parameter in Multipart/Signed.\n"));
        mutt_error ("Warning!  Multipart/Signed has no protocol.");
        sleep (1);
      }
      else if (strcasecmp ("application/pgp-signature", p) == 0)
      {
	if (option(OPTVERIFYSIG)) 
	  return (pgp_signed_handler);
      }
    }
    else if (strcasecmp ("encrypted", a->subtype) == 0)
    {
      char *p = mutt_get_parameter ("protocol", a->parameter);

      if (!p)
      {
        mutt_error("Error!  Multipart/Encrypted has no protocol parameter!");
        sleep (1);
      }
      else if (strcasecmp("application/pgp-encrypted", p) == 0)
        return (pgp_encrypted_handler);
    }
#endif /* _PGPPATH */
    return (multipart_handler);
  }
#ifdef _PGPPATH
  if (a->type == TYPEAPPLICATION)
  {
    if (strcasecmp ("pgp", a->subtype) == 0 ||
	strcasecmp ("x-pgp-message", a->subtype) == 0 ||
	strcasecmp ("pgp-signed", a->subtype) == 0)
      return (application_pgp_handler);
  }
#endif /* _PGPPATH */
  return (unsupported_handler);
}

void mutt_body_handler (BODY *a, STATE *s)
{
  handler_t h = get_handler (a);

  fseek (s->fpin, a->offset, 0);
  h (a, s);
}
