/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "muttlib.h"
#include <limits.h>
#include <string.h>
#include <stdlib.h>

typedef struct hook {
  char *pattern;
  char *command;
  struct hook *next;
  int type;
} HOOK;

static HOOK *Hooks = NULL;

void mutt_add_hook (int type, char *pat, char *val)
{
  HOOK *ptr = Hooks;

  while (ptr && ptr->next) ptr = ptr->next;
  if (ptr)
  {
    ptr->next = (HOOK *)safe_malloc(sizeof(HOOK));
    ptr = ptr->next;
  }
  else
    Hooks = ptr = (HOOK *)safe_malloc(sizeof(HOOK));
  ptr->next = NULL;
  ptr->pattern = safe_strdup (pat);
  ptr->command = safe_strdup (val);
  ptr->type = type;
}

void mutt_folder_hook (char *path)
{
  HOOK *tmp = Hooks;
  char buf[_POSIX_PATH_MAX];
  RE_TYPE (re);

  while (tmp)
  {
    if (tmp->type == M_FOLDERHOOK)
    {
      strfcpy (buf, tmp->pattern, sizeof(buf));
      mutt_expand_path (buf, sizeof (buf));
      if (REGCOMP (re, buf, 0) == 0)
      {
	if (REGEXEC (re, path) == 0)
	{
	  if (mutt_parse_rc_line (tmp->command, 1, buf, sizeof (buf)) == -1)
	  {
	    mutt_error ("%s", buf);
	    REGFREE (re);
	    return;
	  }
	}
	REGFREE(re);
      }
    }
    tmp = tmp->next;
  }
}

char *mutt_find_hook (int type, const char *pat)
{
  HOOK *tmp = Hooks;
  char buf[_POSIX_PATH_MAX];
  char *p = buf;
  int flag = (type == M_SAVEHOOK) ? REG_ICASE : 0;
  RE_TYPE (re);

  while (tmp)
  {
    if (tmp->type == type)
    {
      if (tmp->type == M_MBOXHOOK)
      {
	strfcpy (buf, tmp->pattern, sizeof (buf));
	mutt_expand_path (buf, sizeof (buf));
      }
      else
	p = tmp->pattern;

      if (REGCOMP (re, p, flag) == 0)
      {
	if (REGEXEC (re, pat) == 0)
	{
	  REGFREE (re);
	  return (tmp->command);
	}
	REGFREE (re);
      }
    }
    tmp = tmp->next;
  }
  return (NULL);
}

int mutt_save_hook (ADDRESS *adr, char *path, size_t pathlen)
{
  char buf[SHORT_STRING];
  char *p;

  while (adr)
  {
    buf[0] = 0;
    mutt_simple_address (buf, sizeof (buf), adr);
    if ((p = mutt_find_hook (M_SAVEHOOK, buf)))
    {
      strfcpy (path, p, pathlen);
      return 0;
    }
    adr = adr->next;
  }
  return (-1);
}

void mutt_default_save (char *path, size_t pathlen, ENVELOPE *env)
{
  char tmp[_POSIX_PATH_MAX];
  ADDRESS *adr;

  if (mutt_save_hook (env->from, path, pathlen) == 0 ||
      mutt_save_hook (env->to, path, pathlen) == 0 ||
      mutt_save_hook (env->cc, path, pathlen) == 0)
    return;

  /* see if this message is to/cc a mailing list */
  adr = env->to;
  while (adr)
  {
    if (mutt_is_mail_list (adr))
    {
      strfcpy (tmp, adr->mailbox, sizeof (tmp));
      goto done;
    }
    adr = adr->next;
  }
  adr = env->cc;
  while (adr)
  {
    if (mutt_is_mail_list (adr))
    {
      strfcpy (tmp, adr->mailbox, sizeof (tmp));
      goto done;
    }
    adr = adr->next;
  }

  if (env->reply_to)
    strfcpy (tmp, env->reply_to->mailbox, sizeof (tmp));
  else if (env->from)
    strfcpy (tmp, env->from->mailbox, sizeof (tmp));
  else if (env->to)
    strfcpy (tmp, env->to->mailbox, sizeof (tmp));
  else if (env->cc)
    strfcpy (tmp, env->cc->mailbox, sizeof (tmp));
  else
    *path = 0;

done:

  mutt_strlower (tmp);
  snprintf (path, pathlen, "=%s", tmp);
}
