/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#define MAIN_C 1

#include "mutt.h"
#include "mutt_curses.h"
#include "rfc822.h"
#include "copyright.h"

#include <string.h>
#include <stdlib.h>
#include <locale.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/utsname.h>

#include "version.h"

void mutt_exit (int code)
{
  ci_endwin (NULL);
  exit (code);
}

#define M_VERFMT "%s   Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>\n\
WWW: http://www.cs.hmc.edu/~me/mutt/\n\n"

static void mutt_help (void)
{
  printf(M_VERFMT, MuttVersion);
  puts("Usage:\n\
\tmutt [ -nz ] [ -F <muttrc> ] [ -f <mailbox> ]\n\
\tmutt [ -n ] [ -a <file> ] [ -F <muttrc> ] [ -i <file> ] [ -s <subj> ]\n\
\t\t[ -c <addr> ] <addr> [ ... ]\n\
\tmutt [ -n ] [ -F <muttrc> ] -p\n\
\tmutt -v\n\
\tmutt -h\n\n\
-a <file>\tattach a file to the message\n\
-c <address>\tspecify a carbon-copy (CC) address\n\
-f <mailbox>\tspecify which mailbox to read\n\
-F <muttrc>\tspecify an alternate muttrc file\n\
-i <file>\tspecify a file which Mutt should include in the reply\n\
-n\t\tcauses Mutt not to read the system Muttrc\n\
-p\t\trecall a postponed message\n\
-s <subj>\tspecify a subject (must be in quotes if it has spaces)\n\
-v\t\tshow version and compile-time definitions\n\
-z\t\texit immediately if there are no messages in the mailbox\n\
-h\t\tthis help message");

  exit(0);
}

static void show_version (void)
{
  struct utsname uts;

  puts (Copyright);

  printf (M_VERFMT, MuttVersion);
  uname (&uts);
#ifndef _AIX
  printf ("System info: %s %s", uts.sysname, uts.release);
#else
  printf ("System info: %s %s.%s", uts.sysname, uts.version, uts.release);
#endif /* _AIX */

#ifdef NCURSES_VERSION
  printf (" [using ncurses %s]", NCURSES_VERSION);
#elif defined(USE_SLANG_CURSES)
  printf (" [using slang %d]", SLANG_VERSION);
#endif

  puts ("\n\nCompile time definitions:");

#ifdef DOMAIN
  printf ("DOMAIN=\"%s\"\n", DOMAIN);
#else
  puts ("-DOMAIN");
#endif

#ifdef USE_DOMAIN
  fputs ("+USE_DOMAIN  ", stdout);
#else
  fputs("-USE_DOMAIN  ", stdout);
#endif

#ifdef HIDDEN_HOST
  fputs ("+HIDDEN_HOST  ", stdout);
#else
  fputs ("-HIDDEN_HOST  ", stdout);
#endif

#ifdef DONT_ADD_FROM
  puts ("+DONT_ADD_FROM");
#else
  puts ("-DONT_ADD_FROM");
#endif

#ifdef HOMESPOOL
  fputs ("+HOMESPOOL  ", stdout);
#else
  fputs ("-HOMESPOOL  ", stdout);
#endif
 
#ifdef WORLD_WRITABLE_SPOOL
  fputs ("+WORLD_WRITABLE_SPOOL  ", stdout);
#else
  fputs ("-WORLD_WRITABLE_SPOOL  ", stdout);
#endif

#ifdef USE_SETGID
  puts ("+USE_SETGID");
#else
  puts ("-USE_SETGID");
#endif

#ifdef USE_FCNTL
  fputs ("+USE_FCNTL  ", stdout);
#else
  fputs ("-USE_FCNTL  ", stdout);
#endif

#ifdef USE_FLOCK
  fputs ("+USE_FLOCK  ", stdout);
#else
  fputs ("-USE_FLOCK  ", stdout);
#endif

#ifdef USE_DOTLOCK
  puts ("+USE_DOTLOCK");
#else
  puts ("-USE_DOTLOCK");
#endif

#ifdef USE_DSN
  fputs ("+USE_DSN", stdout);
#else
  fputs ("-USE_DSN", stdout);
#endif

#ifdef USE_8BITMIME
  fputs ("  +USE_8BITMIME", stdout);
#else
  fputs ("  -USE_8BITMIME", stdout);
#endif

#ifdef USE_POP
  fputs ("  +USE_POP", stdout);
#else
  fputs ("  -USE_POP", stdout);
#endif

#ifdef USE_GNU_REGEX
  fputs ("  +USE_GNU_REGEX", stdout);
#else
  fputs ("  -USE_GNU_REGEX", stdout);
#endif

#ifdef HAVE_COLOR
  puts ("  +HAVE_COLOR");
#else
  puts ("  -HAVE_COLOR");
#endif

  printf ("SENDMAIL=\"%s\"\n", SENDMAIL);
  printf ("MAILPATH=\"%s\"\n", MAILPATH);
  printf ("SYSMUTTRC=\"%s\"\n", SYSMUTTRC);
  printf ("MIMETYPES=\"%s\"\n", MIMETYPES);

#ifdef ISPELL
  printf ("ISPELL=\"%s\"\n", ISPELL);
#else
  puts ("-ISPELL");
#endif

#ifdef _PGPPATH
  printf ("_PGPPATH=\"%s\"\n", _PGPPATH);
#endif

  exit (0);
}

int main (int argc, char **argv)
{
  char folder[_POSIX_PATH_MAX] = "";
  char *subject=NULL;
  char *includeFile=NULL;
  BODY *attach = NULL;
  ENVELOPE *env = NULL;
  int nosysrc=0;
  int sendflags=0;
  int ignore_empty=0;
  int i;
  extern char *optarg;
  extern int optind;

  SRAND (time (NULL));
  setlocale (LC_CTYPE, "");
  umask (077);

#ifdef USE_SETGID
  /*
   * Determine the user's default gid and the gid to use for locking the spool
   * mailbox on those systems which require setgid "mail" to write to the
   * directory.  This function also resets the gid to "normal" since the
   * effective gid will be "mail" when we start (Mutt attempts to run non-setgid
   * whenever possible to reduce the possibility of security holes).
   */

  /* Get the default gid for the user */
  UserGid = getgid ();

  /* it is assumed that we are setgid to the correct gid to begin with */
  MailGid = getegid ();

  /* reset the effective gid to the normal gid */
  if (SETEGID (UserGid) != 0)
  {
    perror ("setegid");
    exit (0);
  }
#endif

  strcpy (Muttrc, "~/.muttrc");

  while ((i = getopt (argc, argv, "a:F:f:c:d:s:i:hnpvz")) != EOF)
    switch(i)
    {
      case 'a':
	{
	  BODY *pattach = attach;

	  while (pattach && pattach->next)
	    pattach = pattach->next;
	  if (pattach)
	    pattach->next = mutt_make_attach (optarg);
	  else
	    attach = mutt_make_attach (optarg);
	}
	break;

      case 'F':
	strfcpy (Muttrc, optarg, sizeof (Muttrc));
	break;

      case 'f':
	strfcpy (folder, optarg, sizeof (folder));
	break;

      case 'c':
	if (!env)
	  env = mutt_new_envelope ();
	if (mutt_parse_adrlist (&env->cc, optarg, "@") == -1)
	{
	  fprintf (stderr, "%s: %s\n", optarg, ParseError);
	  exit (0);
	}
	break;

      case 'd':
#ifdef DEBUG
	debuglevel = atoi (optarg);
	printf ("Debugging at level %d.\n", debuglevel);
#else
	printf ("DEBUG was not defined during compilation.  Ignored.\n");
#endif
	break;

      case 'i':
	includeFile = optarg;
	break;

      case 'n':
	nosysrc = 1;
	break;

      case 'p':
	sendflags |= SENDPOSTPONED;
	break;

      case 's':
	subject = optarg;
	break;

      case 'v':
	show_version ();
	/* not reached */

      case 'z':
	ignore_empty = 1;
	break;

      default:
	mutt_help ();
    }

  memset (Options, 0, sizeof (Options));

  /* Check for a batch send. */
  if (!isatty (0))
    set_option (OPTBATCHMODE);
  else
  {
    keymap_init (); /* must come before mutt_init */
    ci_start_curses ();
  }

  /* set defaults and read init files */
  mutt_init (nosysrc);

  if (sendflags & SENDPOSTPONED)
  {
    ci_send_message (SENDPOSTPONED, NULL, NULL, NULL, NULL);
    ci_endwin (NULL);
  }
  else if (subject || attach || option (OPTBATCHMODE) || optind < argc)
  {
    if (!env)
      env = mutt_new_envelope ();

    for (i = optind; i < argc; i++)
    {
      if (mutt_parse_adrlist (&env->to, argv[i], "@") == -1)
      {
	printf ("%s: %s\n", argv[i], ParseError);
	exit (0);
      }
    }

    if (!env->to)
    {
      fputs ("No recipients specified.\n", stderr);
      exit (0);
    }

    env->subject = safe_strdup (subject);
    ci_send_message (sendflags, env, attach, includeFile, NULL);
    if (!option (OPTBATCHMODE))
      ci_endwin (NULL);
  }
  else
  {
    if (!folder[0])
      strfcpy (folder, Spoolfile, sizeof (folder));
    mutt_expand_path (folder, sizeof (folder));

    if (ignore_empty)
    {
      struct stat st;

      /* check to see if there are any messages in the folder */
      if (stat (folder, &st) != 0)
      {
	ci_endwin (strerror (errno));
	exit (0);
      }
      if (st.st_size == 0)
      {
	ci_endwin ("Mailbox is empty.");
	exit (0);
      }
    }

    mutt_folder_hook (folder);
    if ((Context = mutt_open_mailbox (folder, NULL, 0)) != NULL)
    {
      ci_main ();
      ci_endwin (NULL);
    }
    else
      ci_endwin ("Unable to open mailbox.");
  }
  exit (0);
}
