/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "keymap.h"

#include <string.h>

#if 0
#define OFFSET (option (OPTSTATUSONTOP) ? 1 : 0)
#endif
#define OFFSET 1

MUTTMENU *mutt_newMenu (const char *title)
{
  MUTTMENU *p = (MUTTMENU *)safe_malloc(sizeof(MUTTMENU));

  memset (p, 0, sizeof (MUTTMENU));
  p->title = safe_strdup (title);
  p->redraw = REDRAW_FULL;
  return (p);
}

void mutt_menuDestroy (MUTTMENU **p)
{
  safe_free ((void **)&(*p)->title);
  safe_free ((void **)p);
}

int mutt_menuLoop (MUTTMENU *menu)
{
  int i;
  char buffer[STRING];

  FOREVER
  {
    /*
     * See if all or part of the screen needs to be updated.
     */
    switch (menu->redraw)
    {
      case REDRAW_FULL:

	clear ();
	SETCOLOR (MT_COLOR_STATUS);
	mvprintw (option (OPTSTATUSONTOP) ? 0 : LINES-2, 0, "%-*.*s", COLS, COLS, menu->title);
	SETCOLOR (MT_COLOR_NORMAL);

	/* fall through to REDRAW_INDEX */

      case REDRAW_INDEX:

	for (i = menu->top; i < menu->top + PAGELEN; i++)
	{
	  if (i < menu->max)
	  {
	    menu->makeEntry (buffer, i, sizeof(buffer));

	    CLEARLINE (i - menu->top + OFFSET);

	    if (option (OPTARROWCURSOR))
	    {
	      if (i == menu->current)
	      {
		SETCOLOR (MT_COLOR_INDICATOR);
		addstr ("->");
		SETCOLOR (MT_COLOR_NORMAL);
	      }
	      buffer[COLS-3] = 0;
	      move (i - menu->top + OFFSET, 3);
	      addstr (buffer);
	    }
	    else
	    {
	      buffer[COLS-1] = 0;

	      if (i == menu->current)
		SETCOLOR (MT_COLOR_INDICATOR);
	      printw ("%-*.*s", COLS, COLS, buffer);
	      if ( i == menu->current)
		SETCOLOR (MT_COLOR_NORMAL);
	    }
	  }
	  else
	    CLEARLINE (i-menu->top+OFFSET);
	}
	menu->redraw = 0;
	break;

      case REDRAW_MOTION:

	move (menu->oldcurrent + OFFSET - menu->top, 0);
	if (option (OPTARROWCURSOR))
	{
	  /* clear the pointer */
	  addstr ("  ");

	  /* now draw it in the new location */
	  move (menu->current + OFFSET - menu->top, 0);
	  SETCOLOR (MT_COLOR_INDICATOR);
	  addstr ("->");
	  SETCOLOR (MT_COLOR_NORMAL);
	}
	else
	{
	  /* erase the current indicator */
	  clrtoeol ();
	  menu->makeEntry (buffer, menu->oldcurrent, sizeof (buffer));
	  buffer[COLS-1] = 0;
	  printw ("%-*.*s", COLS, COLS, buffer);

	  /* now draw the new one to reflect the change */
	  menu->makeEntry (buffer, menu->current, sizeof (buffer));
	  buffer[COLS-1] = 0;
	  move (menu->current - menu->top + OFFSET, 0);
	  SETCOLOR (MT_COLOR_INDICATOR);
	  printw ("%-*.*s", COLS, COLS, buffer);
	  SETCOLOR (MT_COLOR_NORMAL);
	}
	menu->redraw = 0;
	break;

      case REDRAW_CURRENT:

	move (menu->current + OFFSET - menu->top, 0);
	clrtoeol ();
	menu->makeEntry (buffer, menu->current, sizeof (buffer));
	if (option (OPTARROWCURSOR))
	{
	  SETCOLOR (MT_COLOR_INDICATOR);
	  addstr ("->");
	  SETCOLOR (MT_COLOR_NORMAL);
	  buffer[COLS-3] = 0;
	  mvaddstr (menu->current + OFFSET - menu->top, 3, buffer);
	}
	else
	{
	  SETCOLOR (MT_COLOR_INDICATOR);
	  printw ("%-*.*s", COLS, COLS, buffer);
	  SETCOLOR (MT_COLOR_NORMAL);
	}
	menu->redraw = 0;
	break;
    }

    /* move the cursor out of the way */
    if (option (OPTARROWCURSOR))
    {
      move (menu->current - menu->top + OFFSET, 2);
    }
    else
    {
      move (menu->current - menu->top + OFFSET, COLS-1);
    }

    refresh ();
    i = dokey (menu->menu);
    ci_clear_error ();
    if (i == ERR)
    {
#ifdef USE_SLANG_CURSES
      if (Sigwinch)
      {
	mutt_resize_screen ();
	menu->redraw = REDRAW_FULL;
	Sigwinch = 0;
      }
#endif
      continue;
    }

    switch (i)
    {
      case OP_GENERIC_NEXT_ENTRY:

	if (menu->current < menu->max - 1)
	{
	  menu->oldcurrent = menu->current;
	  menu->current++;
	  if (menu->current >= menu->top + PAGELEN)
	  {
	    menu->top = menu->current;
	    menu->redraw = REDRAW_INDEX;
	  }
	  else
	    menu->redraw = REDRAW_MOTION;
	}
	else
	{
	  beep ();
	  mutt_error ("You are on the last entry.");
	}
	break;

      case OP_GENERIC_PREVIOUS_ENTRY:

	if (menu->current > 0)
	{
	  menu->oldcurrent = menu->current;
	  if (--menu->current < menu->top)
	  {
	    if ((menu->top = menu->current - PAGELEN + 1) < 0)
	      menu->top = 0;
	    menu->redraw = REDRAW_INDEX;
	  }
	  else
	    menu->redraw = REDRAW_MOTION;
	}
	else
	{
	  beep ();
	  mutt_error("You are on the first entry.");
	}
	break;

      case OP_NEXT_PAGE:

	if (menu->top + PAGELEN <= menu->max - 1)
	{
	  menu->top += PAGELEN;
	  menu->current = menu->top;
	  menu->redraw = REDRAW_INDEX;
	}
	else
	{
	  beep ();
	  mutt_error ("You are on the last page.");
	}
	break;

      case OP_PREVIOUS_PAGE:

	if (menu->top > 0)
	{
	  if ((menu->top -= PAGELEN) < 0)
	    menu->top = 0;
	  menu->current = menu->top + PAGELEN - 1;
	  menu->redraw = REDRAW_INDEX;
	}
	else
	{
	  beep ();
	  mutt_error("You are on the first page.");
	}
	break;

      case OP_GENERIC_FIRST_ENTRY:

	menu->oldcurrent = menu->current;
	if (menu->top > 0)
	  menu->redraw = REDRAW_INDEX;
	else
	  menu->redraw = REDRAW_MOTION;
	menu->top = menu->current = 0;
	break;

      case OP_GENERIC_LAST_ENTRY:

	menu->oldcurrent = menu->current;
	menu->current = menu->max - 1;
	if (menu->current > menu->top + PAGELEN - 1)
	{
	  while (menu->top + PAGELEN  - 1 < menu->current)
	    menu->top += PAGELEN;
	  menu->redraw = REDRAW_INDEX;
	}
	else
	  menu->redraw = REDRAW_MOTION;
	break;

      case OP_TOP_OF_PAGE:

	menu->oldcurrent = menu->current;
	menu->current = menu->top;
	menu->redraw = REDRAW_MOTION;
	break;

      case OP_MIDDLE_OF_PAGE:

	menu->oldcurrent = menu->current;
	i = menu->top + PAGELEN;
	if (i > menu->max - 1)
	  i = menu->max - 1;
	menu->current = menu->top + (i - menu->top) / 2;
	menu->redraw = REDRAW_MOTION;
	break;

      case OP_BOTTOM_OF_PAGE:

	menu->oldcurrent = menu->current;
	menu->current = menu->top + PAGELEN - 1;
	if (menu->current > menu->max - 1)
	  menu->current = menu->max - 1;
	menu->redraw = REDRAW_MOTION;
	break;

      case OP_SEARCH:
	
	if (menu->search)
	{
	  buffer[0] = 0;
	  if (ci_get_field ("Search for: ", buffer, sizeof (buffer), 0) == 0 && buffer[0])
	  {
	    menu->oldcurrent = menu->current;
	    if ((menu->current = menu->search (buffer, menu)) != -1)
	    {
	      if (menu->top + PAGELEN - 1 < menu->current)
	      {
		while (menu->top + PAGELEN - 1 < menu->current)
		  menu->top += PAGELEN;
		menu->redraw = REDRAW_INDEX;
	      }
	      else
		menu->redraw = REDRAW_MOTION;
	    }
	    else
	      menu->current = menu->oldcurrent;
	  }
	}
	else
	{
	  beep ();
	  mutt_error ("Search is not implemented for this menu.");
	}
	break;

      case OP_REDRAW:

	clearok (stdscr, TRUE);
	menu->redraw = REDRAW_FULL;
	break;

      case OP_HELP:

	ci_help (menu->menu);
	menu->redraw = REDRAW_FULL;
	break;

      case OP_NULL:

	beep ();
	mutt_error ("Key is not bound.  Press '?' for help.");
	break;

      default:
	return (i);
    }
  }
  /* not reached */
}
