/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "config.h"

#include <stdio.h>
#include <sys/types.h>
#include <time.h>
#include <limits.h>
#include <stdarg.h>

#define TRUE 1
#define FALSE 0

#define EPOCH 1971

#ifdef HAVE_SRAND48
#define LRAND lrand48
#define SRAND srand48
#define DRAND drand48
#else
#define LRAND rand
#define SRAND srand
#define DRAND (double)rand
#endif /* HAVE_SRAND48 */

#ifdef HAVE_SETEGID
#define SETEGID setegid
#else
#define SETEGID setgid
#endif

/*
 * A (more) generic interface to regular expression matching
 */
#ifdef USE_GNU_REGEX
#define HAVE_REGCOMP
#endif

#ifdef HAVE_REGCOMP
#include "regex.h"
#define REGCOMP(X,Y,Z) regcomp(&X, Y, REG_NOSUB|(Z))
#define REGEXEC(X,Y) regexec(&X, Y, (size_t)0, (regmatch_t *)0, (int)0)
#define REGFREE(X) regfree(&X)
#define RE_TYPE(X) regex_t X
#elif defined(HAVE_REGCMP)
#ifdef HAVE_LIBGEN_H
#include <libgen.h>
#endif
#define REGCOMP(X,Y,Z) ((X = regcmp(Y, (char *)0)) == NULL)
#define REGEXEC(X,Y) regex(X,Y) == NULL
#define REGFREE(X) free(X)
#define REG_ICASE 0
#define RE_TYPE(X) char *X
#else
#define REGCOMP(X,Y,Z) (re_comp(Y) != NULL)
#define REGEXEC(X,Y) !re_exec(Y)
#define REGFREE(X) 
#define REG_ICASE 0
#define RE_TYPE(X) 
#endif

#define whitespace(c) ((c) == ' ' || (c) == '\t' || (c) == '\r' || (c) == '\n')

#define FOREVER while (1)

#if DEBUG
#define dprint(N,X) if(debuglevel>=N) fprintf X
#else
#define dprint(N,X) 
#endif

#define strfcpy(A,B,C) strncpy(A,B,C), *(A+(C)-1)=0
#define MSGLEN(x) ((x)->content->length + ((x)->content->offset - (x)->offset))

#define HUGE_STRING	5120
#define LONG_STRING     1024
#define STRING          256
#define SHORT_STRING    128

/* Content-Disposition values */
#define DISPINLINE 0
#define DISPATTACH 1

#define TYPE(X) body_types[(X)]
#define ENCODING(X) body_encodings[(X)]

#define is_multipart(x) ((x)->type == TYPEMULTIPART || ((x)->type == TYPEMESSAGE && !strcasecmp((x)->subtype, "rfc822")))

/* Content-Type */
enum {
  TYPEOTHER,
  TYPEAUDIO,
  TYPEAPPLICATION,
  TYPEIMAGE,
  TYPEMESSAGE,
  TYPEMULTIPART,
  TYPETEXT,
  TYPEVIDEO
};

/* Content-Transfer-Encoding */
enum {
  ENCOTHER,
  ENC7BIT,
  ENC8BIT,
  ENCQUOTEDPRINTABLE,
  ENCBASE64,
  ENCBINARY
};

/* MIME encoding/decoding global vars */
extern int Index_hex[];
extern int Index_64[];
extern char Base64_chars[];

#define hexval(c) Index_hex[(int)(c)]
#define base64val(c) Index_64[(int)(c)]

/* flags for mutt_copy_header() */
#define CH_UPDATE 1      /* update the status and x-status fields? */
#define CH_WEED   (1<<1) /* weed the headers? */
#define CH_DECODE (1<<2) /* do RFC1522 decoding? */
#define CH_XMIT   (1<<3) /* transmitting this message? */
#define CH_FROM   (1<<4) /* retain the "From " message separator? */
#define CH_PREFIX (1<<5) /* use Prefix string? */

typedef int sort_t (const void *, const void *);
sort_t *mutt_get_sort_func (int);

#ifdef MAIN_C
const char *Sort_methods[] = {
  "date-sent",
  "size",
  "subject",
  "from",
  "mailbox-order",
  "threads",
  "date-received",
  0
};
#else
extern const char *Sort_methods[];
#endif

/* These must be in the same order as in the global var "Sort_methods" */
#define SORTDATE     1   /* the date the mail was sent. */
#define SORTSIZE     2
#define SORTSUBJECT  3
#define SORTFROM     4
#define SORTORDER    5   /* the order the messages appear in the mailbox. */
#define SORTTHREADS  6
#define SORTRECEIVED 7   /* when the message were delivered locally */
#define SORTMAX      7   /* FOO - needs to be changed if more methods */
#define SORTMASK     0x7
#define SORTREVERSE  8

enum {
  /* flags for mutt_open_mailbox() */
  M_NOSORT = 1,/* do not sort the mailbox after opening it */

  /* flags for _mutt_make_string() */
  M_FORCESUBJ, /* print the subject even if it didn't change */
  M_NOTREE, /* don't draw the thread tree */

  /* flags for ci_enter_string() */
  M_ALIAS, /* do alias "completion" by calling up the alias-menu */
  M_FILE,  /* do file completion */
  M_PASS,  /* password mode (no echo) */

  /* types for mutt_add_hook() */
  M_FOLDERHOOK,
  M_SAVEHOOK,
  M_MBOXHOOK,

  /* modes for mutt_display_attachment() */
  M_REGULAR,
  M_MAILCAP,
  M_AS_TEXT,

  /* action codes used by mutt_set_flag() and ci_pattern_function() */
  M_ALL,
  M_NONE,
  M_NEW,
  M_OLD,
  M_REPLIED,
  M_READ,
  M_UNREAD,
  M_DELETE,
  M_UNDELETE,
  M_FLAG,
  M_TAG,
  M_UNTAG,
  M_LIMIT
};

/* possible arguments to set_trioption() */
enum {
  M_NO,
  M_YES,
  M_ASK
};

enum {
  M_VERIFYSIG, /* verify PGP signatures */
  M_USEMAILCAP
};

/* flags to ci_send_message() */
#define SENDREPLY      1
#define SENDGROUPREPLY 2
#define SENDLISTREPLY  4
#define SENDFORWARD    8
#define SENDPOSTPONED  16

enum {
  OPTPROMPTAFTER,
  OPTSTATUSONTOP,
  OPTALLOW8BIT,
  OPTASKCOPY,
  OPTHOLD,
  OPTMETOO,
  OPTEDITHDRS,
  OPTARROWCURSOR,
  OPTASKCC,
  OPTHEADER,
  OPTPOINTNEW,
  OPTASKDEL,
  OPTASKMOVE,
  OPTREVALIAS,
  OPTFORCENAME,
  OPTKEEPEMPTY,
  OPTPAGERSTOP,
  OPTSIGDASHES,
  OPTASKPOSTPONE,
  OPTASKREPLYTO,
  OPTASKBCC,
  OPTAUTOINCLUDE,
  OPTAUTOEDIT,
  OPTASKPRINT,
  OPTPGPAUTOSIGN,
  OPTMARKOLD,
  OPTCONFIRMFILES,
  OPTCONFIRMCREATE,
  OPTCONFIRMAPPEND,
  OPTCONFIRMFOLDERS,
  OPTASKRECALL,
  OPTPOPDELETE,
  OPTSAVENAME,
  OPTTILDE,
  OPTMIMEFWD,
  OPTFCCATTACH,
  OPTRESOLVE,
  OPTSTRICTTHREADS,
  OPTVERIFYSIG, /* pseudo option controlled by M_VERIFYSIG */
  OPTBATCHMODE,  /* pseudo option when sending in batch mode */
  OPTNEEDREDRAW, /* pseudo option to notify caller of a submenu */
  OPTMAX
};

#define set_option(x) Options[(x)/8] |= (1 << ((x) % 8))
#define unset_option(x) Options[(x)/8] &= ~(1 << ((x) % 8))
#define option(x) (Options[(x)/8] & (1 << ((x) % 8)))

/* A dynamic array */
typedef struct {
  char **data;
  short len;
  short max;
} ARRAY;

typedef struct {
  FILE *fpin;
  FILE *fpout;
  char *prefix;
  short displaying; /* TRUE only if we are sending the output to a pager. */
} STATE;

typedef struct address {
  char *personal;
  char *mailbox;
  char *adl;
  char *host;
  char *error;
  struct address *next;
} ADDRESS;

typedef struct alias {
  char *name;
  ADDRESS *addr;
  struct alias *next;
} ALIAS;

typedef struct list_t {
  char *data;
  struct list_t *next;
} LIST;

typedef struct envelope {
  ADDRESS *from;
  ADDRESS *to;
  ADDRESS *cc;
  ADDRESS *bcc;
  ADDRESS *sender;
  ADDRESS *reply_to;
  char *subject;
  char *message_id;
  LIST *references;  /* message references (in reverse order) */
  LIST *userhdrs;    /* user defined headers */
} ENVELOPE;

typedef struct userheader {
  char *key;
  char *val;
  struct userheader *next;
} USERHEADER;

typedef struct parameter {
  char *attribute;
  char *value;
  struct parameter *next;
} PARAMETER;

/* Information that helps in determing the Content-* of an attachment */
typedef struct content {
  unsigned int space : 1;  /* whitespace at the end of lines? */
  unsigned int binary : 1; /* long lines, or CR not in CRLF pair */
  unsigned int from : 1;   /* has a line beginning with "From "? */
  unsigned int dot : 1;    /* has a line consisting of a single dot? */
  long hibin;              /* 8-bit characters */
  long lobin;              /* unprintable 7-bit chars (eg., control chars) */
  long ascii;              /* number of ascii chars */
  long linemax;            /* length of the longest line in the file */
} CONTENT;

typedef struct body {
  unsigned int type : 3;        /* content-type primary type */
  unsigned int encoding : 3;    /* content-transfer-encoding */
  unsigned int disposition : 1; /* content-disposition */
  unsigned int use_disp : 1;    /* Content-Disposition field printed? */
  unsigned int unlink : 1;      /* flag to indicate the the file named by
				 * "filename" should be unlink()ed before
				 * free()ing this structure
				 */
  char *subtype;                /* content-type subtype */
  PARAMETER *parameter;         /* parameters of the content-type */
  char *description;            /* content-description */
  long hdr_offset;              /* offset in stream where the headers begin.
				 * this info is used when invoking metamail,
				 * where we need to send the headers of the
				 * attachment
				 */
  long offset;                  /* offset where the actual data begins */
  long length;                  /* length (in bytes) of attachment */
  char *filename;               /* when sending a message, this is the file
				 * to which this structure refers
				 */
  CONTENT *content;             /* structure used to store detailed info about
				 * the content of the attachment.  this is used
				 * to determine what content-transfer-encoding
				 * is required when sending mail.
				 */
  struct body *next;            /* next attachment in the list */
  struct body *parts;           /* parts of a multipart or message/rfc822 */
} BODY;

typedef struct header {
  unsigned int mime : 1;    /* has a Mime-Version header? */
  unsigned int mailcap : 1; /* requires mailcap to display? */
  unsigned int flagged : 1; /* marked important? */
  unsigned int tagged : 1;
  unsigned int deleted : 1;
  unsigned int changed : 1;
  unsigned int old : 1;
  unsigned int read : 1;

  unsigned int replied : 1;
#ifdef _PGPPATH
  unsigned int pgp : 2;
#endif
  unsigned int subject_changed : 1; /* used for threading */
  unsigned int xxx : 4;             /* unused */

  /* Components from the "DATE:" field.  Use "date_sent" for quick compare. */
  unsigned int year : 7;
  unsigned int month : 5;
  unsigned int day : 6;
  unsigned int hours : 5;
  unsigned int minutes : 6;
  unsigned int seconds : 6;
  unsigned int zhours : 5;
  unsigned int zminutes : 6;
  unsigned int zoccident : 1;

  time_t date_sent;     /* time when the message was sent */
  time_t received;      /* time when the message was placed in the mailbox */
  long offset;          /* where in the stream does this message begin? */
  unsigned short lines; /* how many lines in the body of this message? */
  short index;          /* the absolute (unsorted) message number */
  short msgno;		/* number displayed to the user */
  short virtual;        /* virtual message number */
  ENVELOPE *env;	/* envelope information */
  BODY *content;	/* list of MIME parts */
  
  /* the following are used for threading support */
  struct header *parent;
  struct header *child;  /* decendants of this message */
  struct header *next;   /* next message in this thread */
  struct header *prev;   /* previous message in thread */
  char *tree;            /* character string to print thread tree */

} HEADER;

typedef struct {
  char *name;
  FILE *fp;
  time_t mtime; /* last modification date */
  long size;    /* size of the mailbox  */
} FOLDER;

typedef struct {
  FOLDER *folder;
  HEADER **hdrs;
  short *v2r;              /* mapping from virtual to real msgno */
  short hdrmax;            /* number of pointers in hdrs */
  short msgcount;          /* number of messages in the mailbox */
  short vcount;            /* the number of virtual messages */
  short tagged;            /* how many messages are tagged? */
  short new;               /* how many new messages? */
  unsigned int weed : 1;   /* is header weeding turned on? */
  unsigned int locked : 1; /* is the mailbox locked? */
} CONTEXT;

/* ----------------------------------------------------------------------------
 * Function prototypes.
 */

#define mutt_make_string(A,B,C,D) _mutt_make_string(A,B,C,D,M_NOTREE)
void _mutt_make_string (char *, size_t, char *, HEADER *, int);

int trioption (int, int);

FOLDER *folder_open (const char *, const char *mode);
void folder_close (FOLDER **);

int lock_file (FOLDER *, int);
void unlock_file (FOLDER *);

ADDRESS *mutt_default_from (void);
ADDRESS *mutt_expand_aliases (ADDRESS *);
ADDRESS *mutt_new_address (void);
ADDRESS *mutt_remove_duplicates (ADDRESS *);
BODY *mutt_make_attach (const char *);
BODY *mutt_new_body (void);
BODY *mutt_read_mime_header (FILE *);
BODY *mutt_make_attach (const char *);
BODY *mutt_make_multipart (BODY *);
ENVELOPE *mutt_new_envelope (void);
ENVELOPE *read_rfc822_header (FILE *, HEADER *);
HEADER *mutt_new_header (void);
LIST *mutt_new_list (void);
PARAMETER *mutt_new_parameter (void);
STATE *mutt_new_state (void);

char *mutt_expand_path (char *, size_t);
char *mutt_find_hook (int, const char *);
char *mutt_generate_boundary (void);
char *mutt_get_parameter (const char *, PARAMETER *);
char *mutt_strlower (char *);

CONTEXT *mutt_open_mailbox (char *, CONTEXT *, int);
int mutt_check_mailbox (CONTEXT *);
int mutt_parse_mailbox (CONTEXT *);
int mutt_sync_mailbox (CONTEXT *);
int mutt_close_mailbox (CONTEXT *);
void mutt_sort_headers (CONTEXT *);
void mutt_fastclose_mailbox (CONTEXT *);

void mutt_add_hook (int, char *, char *);
void mutt_add_to_list (LIST **, const char *);
void mutt_block_signals (void);
void mutt_body_handler (BODY *, STATE *);
void mutt_bounce_message (HEADER *, char *);
void mutt_complete (char *);
void mutt_copy_header (FILE *, HEADER *, FILE *, int);
void mutt_decode_base64 (STATE *, long, int);
void mutt_decode_quoted (STATE *, long, int);
void mutt_decode_xbit (STATE *, long);
void mutt_default_save (char *, size_t, ENVELOPE *);
void mutt_display_attachment (BODY *, int);
void mutt_do_pager (char *, char *);
void mutt_do_thread (HEADER *, int);
void mutt_error (const char *, ...);
void mutt_exit (int);
void mutt_fetchPopMail (void);
void mutt_folder_cc (ENVELOPE *, BODY *, const char *);
void mutt_folder_hook (char *);
void mutt_free_address (ADDRESS **);
void mutt_free_alias (ALIAS **);
void mutt_free_body (BODY **);
void mutt_free_envelope (ENVELOPE **);
void mutt_free_header (HEADER **);
void mutt_free_list (LIST **);
void mutt_free_state (STATE **);
void mutt_free_parameter (PARAMETER **);
void mutt_generate_header (char *, size_t, HEADER *, int);
void mutt_init (int);
void mutt_init_history (void);
void mutt_mktemp (char *);
void mutt_parse_mime_message (HEADER *);
void mutt_perror (const char *);
void mutt_pipe_message (HEADER *, const char *);
void mutt_postpone_message (const char *, ENVELOPE *, const char *, BODY *);
void mutt_print_message (HEADER *);
void mutt_set_flag (HEADER *, int, int);
void mutt_show_version (void);
void mutt_simple_address (char *, size_t, ADDRESS *);
void mutt_sort_threads (CONTEXT *);
void mutt_status_line (char *, size_t, char *);
void mutt_tabs_to_spaces (char *);
void mutt_text_handler (BODY *, STATE *);
void mutt_unblock_signals (void);
void mutt_unlink (const char *);
void mutt_write_address (char *, size_t, ADDRESS *);

int is_local_site (const char *);
int mutt_addr_is_user (ADDRESS *);
int mutt_body_search (int, const char *);
int mutt_check_encoding (const char *);
int mutt_check_key (const char *);
int mutt_check_menu (const char *);
int mutt_check_mime_type (const char *);
int mutt_check_month (const char *);
int mutt_check_pgp (HEADER *h);
int mutt_compat_charset (const char *);
int mutt_copy_bytes (FILE *, FILE *, long);
int mutt_create_filter (const char *, FILE **, FILE **);
int mutt_get_password (char *, char *, size_t);
int mutt_is_mail_list (ADDRESS *);
int mutt_is_text_type (int, char *);
int mutt_is_valid_mailbox (const char *);
int mutt_matches_ignore (const char *, LIST *);
int mutt_needs_mailcap (BODY *);
int mutt_next_thread (HEADER *);
int mutt_num_postponed (void);
int mutt_parse_adrlist (ADDRESS **, char *, char *);
int mutt_parse_macro (char *, char *, size_t);
int mutt_parse_rc_line (char *, int, char *, int);
int mutt_previous_thread (HEADER *);
int mutt_save_attachment (BODY *, char *);
int mutt_save_hook (ADDRESS *, char *, size_t);
int mutt_save_message (HEADER *, const char *, int);
int mutt_send_message (ENVELOPE *, BODY *, const char *);
int mutt_system (const char *);
int mutt_was_modified (CONTEXT *);
int mutt_write_mime_body (BODY *, FILE *);
int mutt_write_mime_header (BODY *, FILE *);
int mutt_write_rfc822_header (FILE *, ENVELOPE *, BODY *, int);
int mutt_yesorno (const char *, int);

time_t mutt_gmsecs (int, int, int, int, int, int);
time_t mutt_gmtime (HEADER *);

int state_puts (char *, STATE *);
int state_putc (int, STATE *);

char *safe_strdup (const char *);
void *safe_malloc (unsigned int);
void safe_realloc (void **, unsigned int);
void safe_free (void **);

char *strtokq (char *, char *);
char *skip_whitespace (char *);

/* from.c */
time_t is_from (const char *, char *, size_t);

/* aliaslib.c */
ADDRESS *alias_reverse_lookup (ADDRESS *);

/* parse.c */
BODY *parse_multipart (FILE *, char *);
BODY *parse_messageRFC822 (FILE *, BODY *);
BODY *read_mime_header (FILE *);

/* pgp.c */
#ifdef _PGPPATH
#define PGPENCRYPT 1
#define PGPSIGN    2
#define PGPKEY     3

int pgp_valid_passphrase (void);
void application_pgp_handler (BODY *, STATE *);
void pgp_signed_handler (BODY *, STATE *);
void pgp_encrypted_handler (BODY *, STATE *);
BODY *pgp_signMessage (BODY *);
BODY *pgp_encryptMessage (BODY *, char *, int);
char *pgp_findKeys (ADDRESS *, ADDRESS *, ADDRESS *);
int pgp_sendMenu (void);
int pgp_query (BODY *);
void pgp_void_passphrase (void);
#endif /* _PGPPATH */

/* ----------------------------------------------------------------------------
 * Prototypes for broken systems
 */

#ifndef HAVE_SNPRINTF
extern int snprintf (char *, size_t, const char *, ...);
#endif

#ifndef HAVE_STRERROR
#ifndef STDC_HEADERS
extern int sys_nerr;
extern char *sys_errlist[];
#endif

#define strerror(x) ((x) > 0 && (x) < sys_nerr) ? sys_errlist[(x)] : 0
#endif /* !HAVE_STRERROR */

/*
 * AIX doesn't define these in any headers (sigh).
 */
int strcasecmp (const char *, const char *);
int strncasecmp (const char *, const char *, size_t);

#ifdef _AIX
int setegid (gid_t);
#endif /* _AIX */

#ifndef STDC_HEADERS
extern FILE *fdopen();
extern int system();
extern int puts();
extern int fputs();
extern int fputc();
extern int fseek();
extern char *strchr();
extern int getopt();
extern int fputs();
extern int fputc();
extern int fclose();
extern int fprintf();
extern int printf();
extern int fgetc();
extern int tolower();
extern int toupper();
extern int sscanf();
extern size_t fread();
extern size_t fwrite();
extern int system();
extern int rename();
extern time_t time();
extern struct tm *localtime();
extern char *asctime();
extern char *strpbrk();
extern int fflush();
extern long lrand48();
extern void srand48();
extern time_t mktime();
extern int vsprintf();
extern int ungetc();
extern char *mktemp();
extern int ftruncate();
extern char *re_comp();
extern int re_exec();
extern void *memset();
extern int pclose ();

/*
 * IRIX barfs on empty var decls because the system include file uses elipsis
 * in the declaration.  So declare all the args to avoid compiler errors.  This
 * should be harmless on other systems.
 */
int ioctl (int, int, ...);

#endif

/* ----------------------------------------------------------------------------
 * Global variables - these are only used for configuration or static data
 */

/* nifty trick I stole from ELM 2.5alpha. */
#ifdef MAIN_C
#define WHERE 
#define INITVAL(x) = x
#else
#define WHERE extern
#define INITVAL(x) 
#endif

WHERE CONTEXT *Context;

WHERE char Errorbuf[SHORT_STRING];
WHERE char Init_error[SHORT_STRING];

WHERE char AliasFile[_POSIX_PATH_MAX];
WHERE char Attribution[SHORT_STRING];
WHERE char Charset[SHORT_STRING];
WHERE char Domain[SHORT_STRING];
#ifdef USE_DSN
WHERE char DsnNotify[SHORT_STRING] INITVAL("");
WHERE char DsnReturn[SHORT_STRING] INITVAL("");
#endif
WHERE char Editor[SHORT_STRING];
WHERE char ForwFmt[SHORT_STRING];
WHERE char Fqdn[SHORT_STRING];
WHERE char Hdr_format[SHORT_STRING];
WHERE char Homedir[_POSIX_PATH_MAX];
WHERE char Hostname[SHORT_STRING];
WHERE char In_reply_to[SHORT_STRING];
WHERE char Inbox[_POSIX_PATH_MAX];
WHERE char Local_sig[_POSIX_PATH_MAX] INITVAL({0});
WHERE char Maildir[_POSIX_PATH_MAX];
WHERE char Metamail[_POSIX_PATH_MAX];
WHERE char Muttrc[_POSIX_PATH_MAX];
WHERE char Outbox[_POSIX_PATH_MAX] INITVAL({0});
WHERE char Pager[_POSIX_PATH_MAX];
WHERE char PagerFmt[SHORT_STRING];
WHERE char ParseError[SHORT_STRING] INITVAL({0});
WHERE char Pgp[_POSIX_PATH_MAX];
WHERE char PopHost[SHORT_STRING] INITVAL({0});
WHERE char PopUser[SHORT_STRING] INITVAL({0});
WHERE char PopPass[SHORT_STRING] INITVAL({0});
WHERE char PostIndentString[SHORT_STRING] INITVAL({0});
WHERE char Postponed[_POSIX_PATH_MAX];
WHERE char Prefix[SHORT_STRING];
WHERE char Print[_POSIX_PATH_MAX];
WHERE char QuoteString[SHORT_STRING];
WHERE char Realname[SHORT_STRING];
WHERE char Remote_sig[_POSIX_PATH_MAX] INITVAL({0});
WHERE char Sendmail[_POSIX_PATH_MAX];
WHERE char Shell[_POSIX_PATH_MAX];
WHERE char Signature[_POSIX_PATH_MAX];
WHERE char Spoolfile[_POSIX_PATH_MAX];
WHERE char Status[SHORT_STRING];
WHERE char Tochars[SHORT_STRING];
WHERE char Tempdir[_POSIX_PATH_MAX];
WHERE char Username[SHORT_STRING];
WHERE char Web_browser[_POSIX_PATH_MAX] INITVAL({0});
WHERE char Web_Xbrowser[_POSIX_PATH_MAX] INITVAL({0});

WHERE LIST *Ignore INITVAL(0);
WHERE LIST *UnIgnore INITVAL(0);
WHERE LIST *Mailing_lists INITVAL(0);

/* bit vector for boolean variables */
#ifdef MAIN_C
unsigned char Options[(OPTMAX + 7)/8];
#else
extern unsigned char Options[];
#endif

/* bit vector for the yes/no/ask variable type */
WHERE unsigned long TriOptions INITVAL(0);

WHERE short HistSize INITVAL(10);
WHERE short PagerContext INITVAL(0);
WHERE short PopPort INITVAL(110);
WHERE short ReadMsginc INITVAL(10);
WHERE short Sleep_time INITVAL(1);
WHERE short Sort;
WHERE short Timeout INITVAL(600); /* seconds */
WHERE short WriteMsginc INITVAL(10);

WHERE ALIAS *Aliases INITVAL(0);
WHERE ADDRESS *Alternates INITVAL(0);
WHERE USERHEADER *User_defined_headers INITVAL(0);
WHERE LIST *Localsites INITVAL(0);

#if DEBUG
WHERE FILE *debugfile INITVAL(0);
WHERE int debuglevel INITVAL(0);
#endif

#ifdef USE_SETGID
WHERE gid_t MailGid;
WHERE gid_t UserGid;
#endif /* USE_SETGID */

#ifdef MAIN_C
const char *Weekdays[] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
const char *Months[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", "ERR" };

const char *body_types[] = { "x-unknown", "audio", "application", "image", "message", "multipart", "text", "video" };
const char *body_encodings[] = { "x-unknown", "7bit", "8bit", "quoted-printable", "base64", "binary" };
#else
extern const char *Weekdays[];
extern const char *Months[];
extern const char *body_types[];
extern const char *body_encodings[];
extern char MuttVersion[];
#endif
