/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"

#include <sys/stat.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#undef PAGELEN
#define PAGELEN (LINES-2)

static short InHelp = 0;

#define ISHEADER(x) ((x) == MT_COLOR_HEADER || (x) == MT_COLOR_SUBJECT || (x) == MT_COLOR_FROM)

struct line_t {
  long offset;
  short type;
  short continuation;
};

static int upNLines (int nlines, struct line_t *info, int cur, int hiding)
{
  while (cur > 0 && nlines > 0)
  {
    cur--;
    if (!hiding || info[cur].type != MT_COLOR_QUOTED) nlines--;
  }

  return cur;
}

/*
 * Args:
 *	show	if 0, don't show characters
 *		   1, show characters (used for displaying help)
 *		   2, show characters in color
 *
 * returns -1 when EOF is reached
 */

static int
display_line (FILE *f, struct line_t **lineInfo, int n, int *last, int *max,
	      int show, int hiding)
{
  int ch, t, col = 0;

  if (n == *last) (*last)++;

  if (*last == *max)
  {
    safe_realloc ((void **)lineInfo, sizeof (struct line_t) * (*max += LINES));
    for (ch = *last; ch < *max ; ch++)
    {
      (*lineInfo)[ch].type = -1;
      (*lineInfo)[ch].continuation = 0;
    }
  }

  fseek (f, (*lineInfo)[n].offset, 0);

  /* only do color hiliting if we are viewing a message */
  if (show == 2)
  {
    if ((*lineInfo)[n].type == -1)
    {
      char buf[LONG_STRING];
      RE_TYPE (re);

      /* determine the line class */

      REGCOMP (re, QuoteString, REG_ICASE);

      buf[sizeof (buf) - 1] = 0;
      if (fgets (buf, sizeof (buf) - 1, f) == NULL)
      {
	(*lineInfo)[n].type = 0;
	return (-1);
      }

      if (n == 0 || ISHEADER((*lineInfo)[n-1].type))
      {
	if (buf[0] == '\n')
	  (*lineInfo)[n].type = MT_COLOR_NORMAL;
	else if (strncasecmp ("subject:", buf, 8) == 0)
	  (*lineInfo)[n].type = MT_COLOR_SUBJECT;
	else if (strncasecmp ("from:", buf, 5) == 0)
	  (*lineInfo)[n].type = MT_COLOR_FROM;
	else if (n > 0 && (buf[0] == ' ' || buf[0] == '\t'))
	  (*lineInfo)[n].type = (*lineInfo)[n-1].type; /* wrapped line */
	else
	  (*lineInfo)[n].type = MT_COLOR_HEADER;
      }
      else if (strncmp (buf, "[Attachment #", 13) == 0)
	(*lineInfo)[n].type = MT_COLOR_ATTACHMENT;
      else if (strcmp ("-- \n", buf) == 0)
	(*lineInfo)[n].type = MT_COLOR_SIGNATURE;
      else if (n > 0 && (*lineInfo)[n-1].type == MT_COLOR_SIGNATURE)
	(*lineInfo)[n].type = MT_COLOR_SIGNATURE;
      else if (REGEXEC (re, buf) == 0)
	(*lineInfo)[n].type = MT_COLOR_QUOTED;
      else
	(*lineInfo)[n].type = MT_COLOR_NORMAL;

      REGFREE (re);

      fseek (f, (*lineInfo)[n].offset, 0);

      if (hiding && (*lineInfo)[n].type == MT_COLOR_QUOTED) show = 0;
    }

    SETCOLOR ((*lineInfo)[n].type);
  }

  while ((ch = fgetc (f)) != EOF)
  {
    if (ch == '\n')
    {
      if (show) addch ('\n');
      goto finish;
    }
    else if (ch == '\t')
    {
      if ((t = (col/8 + 1) * 8) >= COLS)
      {
	/* tab stop is the beginning of the next line */
	(*lineInfo)[n+1].type = (*lineInfo)[n].type;
	(*lineInfo)[n+1].continuation = 1;
	if (show) addch ('\n');
	goto finish;
      }
      else
      {
	while (col < t)
	{
	  if (show) addch (' ');
	  col++;
	}
      }
    }
    else if (IsPrint (ch))
    {
      if (col < COLS)
      {
	if (show) addch (ch);
	col++;
      }
      else
      {
	fseek (f, -1, 1);
	(*lineInfo)[n+1].type = (*lineInfo)[n].type;
	(*lineInfo)[n+1].continuation = 1;
	if (show) addch ('\n');
	goto finish;
      }
    }
    else if (iscntrl (ch))
    {
      if (col < COLS - 1)
      {
	if (show)
	{
	  addch ('^');
	  addch (ch + '@');
	}
	col += 2;
      }
      else
      {
	(*lineInfo)[n+1].type = (*lineInfo)[n].type;
	(*lineInfo)[n+1].continuation = 1;
	fseek (f, -1, 1);
	if (show) addch ('\n');
	goto finish;
      }
    }
  }

  if (!col) return (-1);

finish:

  (*lineInfo)[n+1].offset = ftell (f);

  return 0;
}

int ci_simple_pager (const char *banner, const char *fname, HEADER *hdr)
{
  static char searchbuf[STRING];
  char buffer[LONG_STRING];
  int maxLine, lastLine = 0;
  struct line_t *lineInfo;
  int i, ch = 0, rc = -1;
  int lines = 0, curline = 0, topline = 0, oldtopline = 0, err, redraw = 1;
  int hideQuoted = 0; /* reap quoted text? */
  FILE *fp = NULL;
  struct stat sb;
  int do_color = hdr ? 2 : 1;
  RE_TYPE (re);

  if ((fp = fopen (fname, "r")) == NULL)
  {
    mutt_error ("Could not open file %s!", fname);
    return (-1);
  }

  if (stat (fname, &sb) != 0)
  {
    fclose (fp);
    return (-1);
  }
  unlink (fname);

  /* Initialize variables */
  lineInfo = safe_malloc (sizeof (struct line_t) * (maxLine = LINES));
  lineInfo[0].offset = 0;
  for (i = 0 ; i < maxLine ; i++)
  {
    lineInfo[i].type = -1;
    lineInfo[i].continuation = 0;
  }

  while (ch != -1)
  {
    if (redraw || topline != oldtopline)
    {
      clear ();
      move (option (OPTSTATUSONTOP) ? 1 : 0, 0);
      curline = oldtopline = topline;
      lines = 0;
      redraw = 0;
    }

    while (lines < PAGELEN && lineInfo[curline].offset < sb.st_size - 1)
    {
      if (!hideQuoted || lineInfo[curline].type != MT_COLOR_QUOTED)
      {
	display_line (fp, &lineInfo, curline, &lastLine, &maxLine, do_color,
		      hideQuoted);
	lines++;
      }
      curline++;
    }

    if (option (OPTTILDE))
    {
      SETCOLOR (MT_COLOR_TILDE);
      while (lines < PAGELEN)
      {
	addstr ("~\n");
	lines++;
      }
    }

    if (ftell (fp) < sb.st_size - 1)
      sprintf (buffer, "%d%%", (int) (100 * ftell (fp) / sb.st_size));
    else if (topline == 0)
      strcpy (buffer, "all");
    else
      strcpy (buffer, "end");

    move (option (OPTSTATUSONTOP) ? 0 : LINES-2, 0);
    clrtoeol ();
    SETCOLOR (MT_COLOR_STATUS);
    printw ("%-*.*s -- (%s)", COLS-10, COLS-10, banner, buffer);

    SETCOLOR (MT_COLOR_NORMAL);

    refresh ();
    ch = dokey (MENU_PAGER);

    ci_clear_error ();

    if (ch == -1)
    {
#ifdef USE_SLANG_CURSES
      if (Sigwinch)
      {
	mutt_resize_screen ();
	redraw = 1;
	Sigwinch = 0;
      }
#endif
      ch = 0;
      continue;
    }

    rc = ch;

    switch (ch)
    {
      case OP_PAGER_EXIT:

	rc = -1;
	ch = -1;
	break;

      case OP_NEXT_PAGE:

	if (lineInfo[curline].offset < sb.st_size-1)
	{
	  topline = upNLines (PagerContext, lineInfo, curline, hideQuoted);
	}
	else if (option (OPTPAGERSTOP))
	{
	  /* emulate "less -q" and don't go on to the next message. */
	  beep ();
	}
	else
	{
	  /* end of the current message, so display the next message. */
	  rc = OP_MAIN_NEXT_UNDELETED;
	  ch = -1;
	}
	break;

      case OP_PREVIOUS_PAGE:

	if (topline != 0)
	{
	  topline = upNLines (PAGELEN-PagerContext, lineInfo, topline, hideQuoted);
	}
	else
	{
	  beep ();
	  mutt_error ("Top of message is shown.");
	}
	break;

      case OP_PAGER_NEXT_LINE:

	if (lineInfo[curline].offset < sb.st_size-1)
	{
	  if (hideQuoted)
	  {
	    while (lineInfo[topline].type == MT_COLOR_QUOTED &&
		   topline < lastLine)
	      topline++;
	  }
	  else
	    topline++;
	}
	else
	  beep();
	break;

      case OP_PAGER_PREVIOUS_LINE:

	if (topline)
	  topline = upNLines (1, lineInfo, topline, hideQuoted);
	else
	{
	  beep ();
	  mutt_error ("Top of message is shown.");
	}
	break;

      case OP_PAGER_TOP:

	topline = 0;
	break;

      case OP_PAGER_HALF_UP:

	topline = upNLines (PAGELEN/2, lineInfo, topline, hideQuoted);
	break;

      case OP_PAGER_HALF_DOWN:

	topline = upNLines (PAGELEN/2, lineInfo, curline, hideQuoted);
	break;

      case OP_PAGER_SEARCH:

	if (ci_get_field ("Search: ", searchbuf, sizeof (searchbuf), 0) != 0 ||
	    !searchbuf[0])
	  break;

	if ((err = REGCOMP (re, searchbuf, REG_ICASE)) != 0)
	{
#ifdef HAVE_REGCOMP
	  char error[STRING];

	  regerror (err, &re, error, sizeof (error));
	  mutt_error (error);
#else
	  mutt_error ("Error in regular expression.");
#endif
	  REGFREE (re);
	}
	else
	{
	  int i;

	  if (lineInfo[lastLine].offset < sb.st_size - 1)
	  {
	    /* read the rest of the file */
	    i = lastLine;
	    while (display_line (fp, &lineInfo, i, &lastLine, &maxLine, 0, 0) == 0)
	      i++;
	  }

	  buffer[sizeof (buffer) - 1] = 0;
	  for (i = topline + 1; i < lastLine; i++)
	  {
	    if ((!hideQuoted || lineInfo[i].type != MT_COLOR_QUOTED) &&
		!lineInfo[i].continuation)
	    {
	      fseek (fp, lineInfo[i].offset, 0);
	      fgets (buffer, sizeof (buffer) - 1, fp);

	      if (REGEXEC (re, buffer) == 0) break;
	    }
	  }

	  if (i < lastLine)
	    topline = i;
	  else
	  {
	    beep ();
	    mutt_error ("Not found.");
	  }

	  REGFREE (re);
	}
	break;

      case OP_HELP:

	/* don't let the user enter the help-menu from the help screen! */
	if (! InHelp)
	{
	  InHelp = 1;
	  ci_help (MENU_PAGER);
	  redraw = 1;
	  InHelp = 0;
	}
	else
	{
	  beep ();
	  mutt_error ("Help is currently being shown.");
	}
	break;

      case OP_PAGER_HIDE_QUOTED:

	hideQuoted = !hideQuoted;
	if (hideQuoted && lineInfo[topline].type == MT_COLOR_QUOTED)
	  topline = upNLines (1, lineInfo, topline, hideQuoted);
	else
	  redraw = 1;
	break;

      case OP_PAGER_BOTTOM: /* move to the end of the file */

	if (lineInfo[curline].offset < sb.st_size - 1)
	{
	  i = lastLine;
	  while (display_line (fp, &lineInfo, i, &lastLine, &maxLine, 0, 0) == 0)
	    i++;
	  topline = upNLines (PAGELEN, lineInfo, lastLine-1, hideQuoted);
	}
	else
	  beep ();
	break;

      case OP_REDRAW:

	clearok (stdscr, TRUE);
	redraw = 1;
	break;

      case OP_NULL:

	beep ();
	mutt_error ("Key is not bound (type ? for help).");
	break;

	/* --------------------------------------------------------------------
	 * The following are operations on the current message rather than
	 * adjusting the view of the message.
	 */

      case OP_BOUNCE_MESSAGE:

	ci_bounce_message (hdr);
	break;

      case OP_BROWSE_URL:

	ci_browse_url (hdr);
	redraw = 1;
	break;

      case OP_CREATE_ALIAS:

	ci_mkalias (hdr->env);
	break;

      case OP_DISPLAY_ADDRESS:

	mutt_display_address (hdr->env->from);
	break;

      case OP_ENTER_COMMAND:
	
	mutt_enter_command ();
	redraw = 1;
	break;

      case OP_FLAG_MESSAGE:
	
	hdr->flagged = !hdr->flagged;
	hdr->changed = 1;
	break;

#ifdef _PGPPATH
      case OP_FORGET_PASSPHRASE:

	mutt_forget_passphrase ();
	break;
#endif /* _PGPPATH */

      case OP_PIPE_MESSAGE:

	ci_pipe_message (hdr);
	break;

      case OP_PRINT:
	
	mutt_print_message (hdr);
	break;

      case OP_MAIL:

	ci_send_message (0, NULL, NULL, NULL, hdr);
	redraw = 1;
	break;

      case OP_REPLY:

	ci_send_message (SENDREPLY, NULL, NULL, NULL, hdr);
	redraw = 1;
	break;

      case OP_RECALL_MESSAGE:

	ci_send_message (SENDPOSTPONED, NULL, NULL, NULL, hdr);
	redraw = 1;
	break;

      case OP_GROUP_REPLY:

	ci_send_message (SENDREPLY | SENDGROUPREPLY, NULL, NULL, NULL, hdr);
	redraw = 1;
	break;

      case OP_LIST_REPLY:

	ci_send_message (SENDREPLY | SENDLISTREPLY, NULL, NULL, NULL, hdr);
	redraw = 1;
	break;

      case OP_FORWARD_MESSAGE:

	ci_send_message (SENDFORWARD, NULL, NULL, NULL, hdr);
	redraw = 1;
	break;

      case OP_SUBSHELL:
	
	mutt_subshell ();
	break;

      case OP_VIEW_ATTACHMENTS:

	if (mutt_view_attachments (hdr) == 0) redraw = 1;
	break;

      default:
	ch = -1;
    }
  }

  fclose (fp);
  safe_free ((void **)&lineInfo);
  return (rc != -1 ? rc : 0);
}
