/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"
#include "rfc822.h"
#include "rfc1522.h"

#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <stdlib.h>

/*
 * Reads an arbitrarily long header field, and looks ahead for continuation
 * lines.  ``line'' is dynamically increased if more space is required to fit
 * the whole line.
 */
static char *mutt_read_rfc822_line (FILE *f, char *line, size_t *linelen)
{
  size_t offset = 0, l;
  char buf[LONG_STRING];
  char ch, *pbuf, *q;
  int is_cont = 0; /* continuation line? */

  buf[sizeof (buf) - 1] = 0;

  if (line)
    *line = 0;

  FOREVER
  {
    buf[sizeof (buf) - 2] = 0;
    if (fgets (buf, sizeof (buf) - 1, f) == NULL)
      return (line);

    pbuf = buf;
    if (is_cont)
    {
      /* unfold the continuation line, as described by MIME */
      pbuf = skip_whitespace (pbuf + 1);
      *--pbuf = ' '; /* convert to a single space */
    }
    is_cont = 0;

    l = strlen (pbuf);

    /* if we got a full line, remove trailing space */
    if (!buf[sizeof (buf) - 2])
    {
      q = pbuf + l - 1;
      while (q >= pbuf && (*q == ' ' || *q == '\n' || *q == '\t' || *q == '\r'))
      {
        *q-- = 0;
	l--;
      }
    }

    if (offset + l > *linelen)
    {
      /* not enough space to hold the (rest of the) line, grow the buffer */
      if (line)
      {
	*linelen = offset + l;
	safe_realloc ((void **)&line, *linelen + 1);
      }
      else
      {
	*linelen = l;
	line = safe_malloc (*linelen + 1);
      }
    }
    else if (!l && !line)
    {
      /* Avoid a segfault if the first line we read is blank. */
      return (NULL);
    }

    memcpy (line + offset, pbuf, l + 1);
    offset += l;

    /* check to see if we got a full line */
    if (buf[sizeof (buf) - 2])
      continue; /* no, read the next part */

    /* a blank line indicates the end of the message header */
    if (! *line)
      return (line);

    /* check to see if the next line is a continuation line */
    ch = fgetc (f);
    ungetc (ch, f);
    if (ch != ' ' && ch != '\t')
      return (line); /* next line is a separate header field or EOH */
    is_cont = 1;
  }
  /* not reached */
}

static LIST *mutt_parse_references (char *s)
{
  LIST *t, *lst = NULL;

  while ((s = strtok (s, " \t")) != NULL)
  {
    /*
     * some mail clients add other garbage besides message-ids, so do a quick
     * check to make sure this looks like a valid message-id
     */
    if (*s == '<')
    {
      t = (LIST *)safe_malloc (sizeof (LIST));
      t->data = safe_strdup (s);
      t->next = lst;
      lst = t;
    }
    s = NULL;
  }

  return (lst);
}

int mutt_check_encoding (const char *c)
{
  if (strncasecmp ("7bit", c, sizeof ("7bit")-1) == 0)
    return (ENC7BIT);
  else if (strncasecmp ("8bit", c, sizeof ("8bit")-1) == 0)
    return (ENC8BIT);
  else if (strncasecmp ("binary", c, sizeof ("binary")-1) == 0)
    return (ENCBINARY);
  else if (strncasecmp ("quoted-printable", c, sizeof ("quoted-printable")-1) == 0)
    return(ENCQUOTEDPRINTABLE);
  else if (strncasecmp("base64", c, sizeof("base64")-1) == 0)
    return(ENCBASE64);
  else
    return(ENCOTHER);
}

static PARAMETER *parse_parameters (char *s)
{
  PARAMETER *head = 0, *cur = 0, *new;
  char buffer[LONG_STRING], *p, savechar;
  short i;

  while (*s)
  {
    if ((p = strchr(s, '=')) == NULL)
    {
      dprint(1, (debugfile, "parse_parameters: malformed parameter: %s\n", s));
      return (head); /* just bail out now */
    }
    savechar = *p;
    *p = 0;
    new = mutt_new_parameter ();
    new->attribute = safe_strdup (s);
    *p = savechar;
    p = skip_whitespace(p+1);
    if (*p == '"')
    {
      p++;
      for (i=0; *p && *p != '"' && i < sizeof (buffer) - 1; i++, p++)
      {
        if (*p == '\\')
	{
          /* Quote the next character */
          buffer[i] = *(p+1);
          p++;
	  if (!*p)
	    break;
        }
        else
          buffer[i] = *p;
      }
      buffer[i] = 0;
    }
    else
    {
      for (i=0; *p && *p != ' ' && *p != ';' && i < sizeof (buffer) - 1; i++, p++)
        buffer[i] = *p;
      buffer[i] = 0;
    }
    new->value = safe_strdup (buffer);

    /* Add this parameter to the list */
    if (head)
    {
      cur->next = new;
      cur = cur->next;
    }
    else
      head = cur = new;

    /* Find the next parameter */
    if (*p == ';')
      p++;
    else
    {
      while (*p && *p != ';') p++;
      if (!*p) break; /* no more parameters! */
      p++;
    }

    /* Move past any leading whitespace */
    s = skip_whitespace (p);
  }    
  return (head);
}

int mutt_check_mime_type (const char *s)
{
  if (strcasecmp("text", s) == 0)
    return TYPETEXT;
  else if (strcasecmp("image", s) == 0)
    return TYPEIMAGE;
  else if (strcasecmp("audio", s) == 0)
    return TYPEAUDIO;
  else if (strcasecmp("video", s) == 0)
    return TYPEVIDEO;
  else if (strcasecmp("multipart", s) == 0)
    return TYPEMULTIPART;
  else if (strcasecmp("application", s) == 0)
    return TYPEAPPLICATION;
  else if (strcasecmp("message", s) == 0)
    return TYPEMESSAGE;
  else
    return TYPEOTHER;
}

static void parse_content_type (char *s, BODY *ct)
{
  char *pc;
  char buffer[SHORT_STRING];
  short i = 0;

  safe_free((void **)&ct->subtype);
  mutt_free_parameter(&ct->parameter);

  /* First extract any existing parameters */
  if ((pc = strchr(s, ';')) != NULL)
  {
    *pc++ = 0;
    while (*pc && whitespace(*pc)) pc++;
    ct->parameter = parse_parameters(pc);

    /* Some pre-RFC1521 gateways still use the "name=filename" convention */
    if ((pc = mutt_get_parameter("name", ct->parameter)) != 0)
      ct->filename = safe_strdup(pc);
  }
  
  /* Now get the subtype */
  if ((pc = strchr(s, '/')))
  {
    *pc++ = 0;
    while (*pc && !whitespace (*pc) && *pc != ';')
    {
      buffer[i++] = *pc;
      pc++;
    }
    buffer[i] = 0;
    ct->subtype = safe_strdup (buffer);
  }

  /* Finally, get the major type */
  ct->type = mutt_check_mime_type (s);

  if (ct->subtype == NULL)
  {
    /* Some older non-MIME mailers (i.e., mailtool, elm) have a content-type
     * field, so we can attempt to convert the type to BODY here.
     */
    if (ct->type == TYPETEXT)
      ct->subtype = safe_strdup("plain");
    else if (ct->type == TYPEAUDIO)
      ct->subtype = safe_strdup("basic");
    else if (ct->type == TYPEMESSAGE)
      ct->subtype = safe_strdup("rfc822");
    else if (ct->type == TYPEOTHER) {
      ct->type = TYPEAPPLICATION;
      sprintf(buffer, "x-%s", s);
      ct->subtype = safe_strdup(buffer);
    }
  }
}

BODY *mutt_read_mime_header (FILE *fp)
{
  BODY *p = mutt_new_body();
  PARAMETER *parms;
  char *c, *line = NULL;
  size_t linelen = 0;
  
  p->hdr_offset = ftell(fp);

  p->encoding = ENC7BIT; /* default from RFC1521 */
  p->type = TYPETEXT;

  while ((line = mutt_read_rfc822_line (fp, line, &linelen)) && *line != 0)
  {
    /* Find the value of the current header */
    if ((c = strchr(line, ':')))
      c = skip_whitespace (c + 1);
    else
    {
      dprint(1, (debugfile, "read_mime_header: bogus BODY header: %s\n", line));
      break;
    }

    switch (tolower(line[0]))
    {
      case 'c':
	if (strncasecmp("ontent-type", line+1, sizeof("ontent-type")-1) == 0)
	  parse_content_type(c, p);
	else if (strncasecmp("ontent-transfer-encoding", line+1, sizeof("ontent-transfer-encoding")-1) == 0)
	  p->encoding = mutt_check_encoding(c);
	else if (strncasecmp("ontent-disposition", line+1, sizeof("ontent-disposition")-1) == 0)
	{
	  if (strncasecmp("attachment", c, sizeof("attachment")-1) == 0)
	    p->disposition = DISPATTACH;
	  else
	    p->disposition = DISPINLINE;

	  /* Check to see if a default filename was given */
	  if ((c = strchr(c, ';')) != NULL)
	  {
	    c = skip_whitespace(c+1);
	    if ((c = mutt_get_parameter("filename", (parms = parse_parameters(c)))) != 0)
	    {
	      /*
	       * free() here because the content-type parsing routine might
	       * have allocated space if a "name=filename" parameter was
	       * specified.
	       */
	      safe_free ((void **)&p->filename);
	      p->filename = safe_strdup (c); 
	    }
	    mutt_free_parameter(&parms);
	  }
	}
	else if (!strncasecmp ("ontent-description", line+1, sizeof("ontent-description")-1))
	{
	  if (!p->description) p->description = safe_strdup(c);
	}
	break;
    }
  }
  p->offset = ftell(fp); /* Mark the start of the real data */
  if (p->type == TYPETEXT && !p->subtype) p->subtype = safe_strdup("plain");

  safe_free ((void **)&line);

  return (p);
}

/* Routine to parse a MESSAGE/RFC822 body */
BODY *parse_messageRFC822 (FILE *fp, BODY *parent)
{
  BODY *msg;

  msg = mutt_read_mime_header(fp);
  if (msg->type == TYPEMULTIPART)
    msg->parts = parse_multipart(fp, mutt_get_parameter("boundary", msg->parameter));
  else if (msg->type == TYPEMESSAGE)
    msg->parts = parse_messageRFC822(fp, msg);
  else
    msg->length = parent->length - (msg->offset - parent->offset);
  return(msg);
}

BODY *parse_multipart (FILE *fp, char *boundary)
{
  int blen, len, crlf = 0;
  char buffer[LONG_STRING];
  BODY *head = 0, *last = 0, *new = 0;
  short i;

  if (!boundary)
  {
    mutt_error ("Error parsing multipart message (no boundary was found).");
    return (NULL);
  }
  blen = strlen(boundary);
  while (fgets(buffer, LONG_STRING, fp) != NULL)
  {
    len = strlen(buffer);

    /* take note of the line ending.  I'm assuming that either all endings
     * will use <CR><LF> or none will.
     */
    if (buffer[len-2] == '\r') crlf = 1;

    if (buffer[0] == '-' && buffer[1] == '-' && strncmp(buffer+2, boundary, blen) == 0)
    {
      if (last)
      {
        last->length = ftell(fp) - last->offset - len - 1 - crlf;
        if (last->parts && last->parts->length == 0)
          last->parts->length = ftell(fp) - last->parts->offset - len - 1 - crlf;
      }
      /* Remove any trailing whitespace, up to the length of the boundary */
      for(i=len-1; whitespace(buffer[i]) && i>blen+2; i--)
        buffer[i] = 0;
      /* Check for the end boundary */
      if (buffer[2+blen] == '-' && buffer[3+blen] == '-' && buffer[4+blen] == 0)
        break;

      new = mutt_read_mime_header(fp);
      if (head) {
        last->next = new;
        last = new;
      }
      else
        last = head = new;
    }
  }
  
  /* parse recursive MIME parts */
  last = head;
  while (last) {
    fseek(fp, last->offset, 0);
    switch (last->type)
    {
      case TYPEMULTIPART:
	last->parts = parse_multipart(fp, mutt_get_parameter("boundary", last->parameter));
	break;
      case TYPEMESSAGE:
	last->parts = parse_messageRFC822(fp, last);
	break;
    }
    last = last->next;
  }

  return(head);
}

static struct tz_t
{
  char *tzname;
  unsigned char zhours;
  unsigned char zminutes;
  unsigned char zoccident; /* west of UTC? */
  unsigned char xxx;       /* unused */
}
TimeZones[] =
{
  { "pdt",  8, 0, 1, 0 },
  { "pst",  7, 0, 1, 0 },
  { "mdt",  7, 0, 1, 0 },
  { "mst",  6, 0, 1, 0 },
  { "cdt",  6, 0, 1, 0 },
  { "cst",  5, 0, 1, 0 },
  { "edt",  5, 0, 1, 0 },
  { "est",  4, 0, 1, 0 },
  { "met",  1, 0, 0, 0 }, /* this is now officially CET */
  { "cet",  1, 0, 0, 0 },
  { "cest", 2, 0, 0, 0 },
  { "jst",  9, 0, 0, 0 }, /* japan */
  { "gmt",  0, 0, 0, 0 },
  { "utc",  0, 0, 0, 0 },
  { NULL,   0, 0, 0, 0 }
};

/*
 * parses a date string in RFC822 format:
 *
 * Date: [ weekday , ] day-of-month month year hour:minute:second timezone
 *
 */

static int parse_date (char *s, HEADER *h)
{
  int count = 0;
  char *t;
  int year, hour, min, sec;
  int i;

  h->zoccident = 0;

  /* kill the day of the week, if it exists. */
  if ((t = strchr(s, ',')) != 0)
    t = skip_whitespace(t+1);
  else
    t = s;

  while ((t = strtok(t, " \t")) != NULL)
  {
    switch (count)
    {
      case 0: /* day of the month */
	if (!isdigit (*t))
	  return (-1);
	h->day = atoi (t);
	if (h->day > 31)
	  return (-1);
	break;

      case 1: /* month of the year */
	if ((i = mutt_check_month (t)) < 0)
	  return (-1);
	h->month = i;
	break;

      case 2: /* year */
	year = atoi (t);
	if (year < 1900)
	  year += 1900;
	h->year = year - 1971;
	break;

      case 3: /* time of day */
	if (sscanf(t, "%d:%d:%d", &hour, &min, &sec) == 3);
	else if (sscanf(t, "%d:%d", &hour, &min) == 2)
	  sec = 0;
	else
	{
	  dprint(1, (debugfile, "parse_date: could not process time format: %s\n", t));
	  return(-1);
	}
	h->hours = hour;
	h->minutes = min;
	h->seconds = sec;
	break;

      case 4: /* timezone */
	if (*t == '+' || *t == '-')
	{
	  if (*t == '-') h->zoccident = 1;
	  t++;
	  h->zhours = (t[0] - '0') * 10 + (t[1] - '0');
	  h->zminutes = (t[2] - '0') * 10 + (t[3] - '0');
	}
	else
	{
	  struct tz_t *p = TimeZones;
	  int i;

	  for (i = 0; p[i].tzname; i++)
	  {
	    if (!strcasecmp (p[i].tzname, t))
	    {
	      h->zhours = p[i].zhours;
	      h->zminutes = p[i].zminutes;
	      h->zoccident = p[i].zoccident;
	    }
	  }

	  /* ad hoc support for the European MET (now officially CET) TZ */
	  if (!strcasecmp (t, "MET"))
	  {
	    if ((t = strtok (NULL, " \t")) == NULL)
	      return (-1);
	    if (!strcasecmp (t, "DST"))
	      h->zhours++;
	  }
	}
    }
    count++;
    t = 0;
  }
  if (count < 5)
  {
    dprint(1,(debugfile, "parse_date(): error parsing date format, using received time\n"));
    return (-1);
  }
  return (0);
}

/* extract the first substring that looks like a message-id */
static void extract_message_id (char *s, char *id, size_t idlen)
{
  *id = 0;
  idlen--;
  while ((s = strtok(s, " \t")) != 0)
  {
    if (*s == '<')
    {
      while (*s && *s != '>' && idlen > 0)
      {
	*id++ = *s++;
	idlen--;
      }
      *id++ = '>';
      *id = 0;
      return;
    }
    s = NULL;
  }
}

/*
 * Note that this function is also called when new mail is appended to the
 * currently open folder, and NOT just when the mailbox is initially read.
 *
 * NOTE: it is assumed that the mailbox being read has been locked before
 * this routine gets called.  Strange things could happen if it's not!
 */
int mutt_parse_mailbox (CONTEXT *ctx)
{
  struct stat sb;
  char buf[HUGE_STRING];
  char return_path[STRING];
  HEADER *curhdr;
  time_t t;
  int i, count = 0, lines = 0;
  /* if the display is not limited, make new messages visible */
  int make_visible = (ctx->msgcount == ctx->vcount);

  if (!ReadMsginc)
    mutt_error ("Reading messages...");

  /* Save information about the folder at the time we opened it. */
  stat (ctx->folder->name, &sb);
  ctx->folder->size = sb.st_size;
  ctx->folder->mtime = sb.st_mtime;

  dprint(1,(debugfile, "mutt_parse_mailbox(): size=%d, mtime=%d\n",
	    (int)sb.st_size, (int)sb.st_mtime));

  while (fgets (buf, LONG_STRING, ctx->folder->fp) != NULL)
  {
    if ((t = is_from (buf, return_path, sizeof (return_path))))
    {
      long loc = ftell (ctx->folder->fp) - strlen (buf);

      if (ReadMsginc && ((count % ReadMsginc) == 0))
	mutt_error ("Reading message %d...", count);

      /* Save the Content-Length of the previous message */
      if (count > 0)
      {
#define PREVIOUS ctx->hdrs[ctx->msgcount-1]

	PREVIOUS->content->length = loc - PREVIOUS->content->offset - 1;
	if (!PREVIOUS->lines) PREVIOUS->lines = lines ? lines - 1 : 0;

#undef PREVIOUS
      }

      if (ctx->msgcount == ctx->hdrmax)
      {
        /* Need to allocate more memory for structures */
        if (ctx->hdrs)
	{
          safe_realloc((void **)&ctx->hdrs, sizeof(HEADER *) * ((ctx->hdrmax += 25)));
	  safe_realloc((void **)&ctx->v2r, sizeof(short) * ctx->hdrmax);
	}
        else
	{
          ctx->hdrs = (HEADER **)safe_malloc(sizeof(HEADER *) * ((ctx->hdrmax = 25)));
	  ctx->v2r = (short *)safe_malloc(sizeof(short) * ctx->hdrmax);
        }

	for (i = ctx->msgcount; i < ctx->hdrmax; i++)
	{
	  ctx->hdrs[i] = NULL;
	  ctx->v2r[i] = -1;
	}
      }
      
      curhdr = ctx->hdrs[ctx->msgcount] = mutt_new_header();

      if (make_visible)
      {
	curhdr->virtual = ctx->vcount;
	ctx->v2r[ctx->vcount] = ctx->msgcount;
	ctx->vcount++;
      }
      else
	curhdr->virtual = -1;

      curhdr->received = t;
      dprint(3, (debugfile, "parse_mailbox(): message %d received at %ld.\n",
                 ctx->msgcount, (long)curhdr->received));

      curhdr->offset = loc;
      curhdr->index = ctx->msgcount++;
      curhdr->msgno = ctx->msgcount;
	
      curhdr->env = read_rfc822_header (ctx->folder->fp, curhdr);

      if (!curhdr->read && !curhdr->old) ctx->new++;

      /*
       * check to see if there was a FROM address.  If not, use the
       * return-path we obtained from is_from()
       */
      if (!curhdr->env->from)
	rfc822_parse_adrlist(&curhdr->env->from, return_path, "@");
#ifdef _PGPPATH
      /* NOTE: this _must_ be done before the check for metamail! */
      curhdr->pgp = pgp_query (curhdr->content);
      if (!curhdr->pgp)
#endif /* _PGPPATH */
        if (mutt_needs_mailcap (curhdr->content)) curhdr->mailcap = 1;

      count++;

      lines = 0;
    }
    else
      lines++;
  }
  
  /* Only set the content-length of the previous message if we have read more
   * than one message during _this_ invocation.  If this routine is called
   * when new mail is received, we need to make sure not to clobber what
   * previously was the last message since the headers may be sorted.
   */
  if (count > 0)
  {
    ctx->hdrs[ctx->msgcount-1]->content->length = ftell (ctx->folder->fp) - ctx->hdrs[ctx->msgcount-1]->content->offset - 1;

    if (!ctx->hdrs[ctx->msgcount-1]->lines)
      ctx->hdrs[ctx->msgcount-1]->lines = lines ? lines - 1 : 0;
  }

  mutt_error ("");

  return (0);
}

void mutt_parse_mime_message (HEADER *cur)
{
  if (cur->content->type != TYPEMESSAGE && cur->content->type != TYPEMULTIPART)
    return; /* nothing to do */

  fseek (Context->folder->fp, cur->content->offset, 0);

  if (cur->content->type == TYPEMULTIPART)
  {
    if (!cur->content->parts)
      cur->content->parts = parse_multipart (Context->folder->fp, mutt_get_parameter("boundary", cur->content->parameter));
  }
  else
  {
    if (!cur->content->parts)
      cur->content->parts = parse_messageRFC822 (Context->folder->fp, cur->content);
  }
}

/*
 * read_rfc822_header() -- parses a RFC822 header
 *
 * Args:
 *
 * f		stream to read from
 *
 * hdr		header structure of current message (optional).  If hdr is
 *		NULL, then we are reading a postponed message, or called
 *		from ci_edit_headers() so we should keep a list of the
 *		user-defined headers.
 *
 */
ENVELOPE *read_rfc822_header (FILE *f, HEADER *hdr)
{
  ENVELOPE *e = mutt_new_envelope();
  LIST *last = NULL;
  char *line = NULL, *p;
  char buffer[LONG_STRING], in_reply_to[SHORT_STRING] = "";
  long loc, content_length = -1;
  int matched;
  int date_found = 0;
  size_t linelen = 0;

  if (hdr)
  {
    if (!hdr->content)
      hdr->content = mutt_new_body();

    /* set the defaults from RFC1521 */
    hdr->content->type = TYPETEXT;
    hdr->content->subtype = safe_strdup("plain");
    hdr->content->encoding = ENC7BIT;
  }

  loc = ftell (f);
  while ((line = mutt_read_rfc822_line (f, line, &linelen)) != NULL &&
	 *line != 0)
  {
    matched = 0;

    /* find the value of this header */
    if ((p = strchr (line, ':')) == NULL)
    {
      dprint (1, (debugfile, "read_rfc822_header(): missing blank line between header and body in message %d?\n", hdr->index));
      fseek (f, loc, 0); /* return to the beginning of this line */
      break;
    }

    p = skip_whitespace (p+1);

    switch (tolower (line[0]))
    {
      case 'a':
	if (!strncasecmp (line+1, "pparently-to:", 13))
	{
	  rfc822_parse_adrlist(&e->to, p, "@");
	  matched = 1;
	}
	else if (strncasecmp(line+1, "pparently-from:", 15) == 0)
	{
	  rfc822_parse_adrlist(&e->from, p, "@");
	  matched = 1;
	}
	break;
      case 'b':
	if (strncasecmp(line+1, "cc:", 3) == 0)
	{
	  rfc822_parse_adrlist(&e->bcc, p, "@");
	  matched = 1;
	}
	break;
      case 'c':
	if (strncasecmp (line+1, "c:", 2) == 0)
	{
	  rfc822_parse_adrlist(&e->cc, p, "@");
	  matched = 1;
	}
	else if (strncasecmp(line + 1, "ontent-", 7) == 0)
	{
	  if (strncasecmp(line+8, "type:", 5) == 0)
	  {
	    if (hdr) parse_content_type(p, hdr->content);
	    matched = 1;
	  }
	  else if (strncasecmp(line+8, "transfer-encoding:", 18) == 0)
	  {
	    if (hdr) hdr->content->encoding = mutt_check_encoding(p);
	    matched = 1;
	  }
	  else if (strncasecmp(line+8, "length:", 7) == 0)
	  {
	    content_length = atoi(p);
	    matched = 1;
	  }
	}
	break;
      case 'd':
	if (strncasecmp("ate:", line+1, 4) == 0)
	{
	  if (hdr && parse_date(p, hdr) == 0)
	    date_found = 1;
	}
	break;
      case 'f':
	if (strncasecmp(line+1, "rom:", 4) == 0)
	{
	  rfc822_parse_adrlist(&e->from, p, "@");
	  matched = 1;
	}
	else if (strncmp(line+1, "rom ", 4) == 0)
	  matched = 1; /* ignore the message separator */
	break;
      case 'i':
	if (strncasecmp(line+1, "n-reply-to:", 11) == 0)
	{
	  if (hdr)
	    extract_message_id (p, in_reply_to, sizeof (in_reply_to));
	}
	break;
      case 'l':
	if (strncasecmp (line+1, "ines:", 5) == 0)
	{
	  if (hdr) hdr->lines = atoi(p);
	  matched = 1;
	}
	break;
      case 'm':
	if (strncasecmp (line+1, "ime-version:", 12) == 0)
	{
	  if (hdr) hdr->mime = 1;
	  matched = 1;
	}
	else if (strncasecmp (line+1, "essage-id:", 10) == 0)
	{
	  safe_free((void **)&e->message_id);
	  e->message_id = safe_strdup(p);
	}
	break;
      case 'r':
	if (strncasecmp (line+1, "eferences:", 10) == 0)
	{
	  mutt_free_list (&e->references);
	  e->references = mutt_parse_references (p);
	  matched = 1;
	}
	else if (strncasecmp (line+1, "eply-to:", 8) == 0)
	{
	  if (hdr)
	    rfc822_parse_adrlist (&e->reply_to, p, "@");
	}
	break;
      case 's':
	if (strncasecmp (line+1, "ubject:", 7) == 0)
	{
	  if (!e->subject)
	    e->subject = safe_strdup (p);
	  matched = 1;
	}
	else if (strncasecmp (line+1, "ender:", 6) == 0)
	{
	  rfc822_parse_adrlist (&e->sender, p, "@");
	  matched = 1;
	}
	else if (strncasecmp (line+1, "tatus:", 6) == 0)
	{
	  if (hdr)
	  {
	    while (*p)
	    {
	      switch(*p) {
		case 'r':
		  hdr->replied = 1;
		  break;
		case 'O':
		  if (option (OPTMARKOLD))
		    hdr->old = 1;
		  break;
		case 'R':
		  hdr->read = 1;
		  break;
	      }
	      p++;
	    }
	  }
	  matched = 1;
	}
	break;
      case 't':
	if (strncasecmp (line+1, "o:", 2) == 0)
	{
	  rfc822_parse_adrlist (&e->to, p, "@");
	  matched = 1;
	}
	break;
      case 'x':
	if (strncasecmp (line+1, "-status:", 8) == 0)
	{
	  if (hdr)
	  {
	    while (*p)
	    {
	      switch (*p)
	      {
		case 'A':
		  hdr->replied = 1;
		  break;
		case 'D':
		  hdr->deleted = 1;
		  break;
		case 'F':
		  hdr->flagged = 1;
		  break;
		default:
		  break;
	      }
	      p++;
	    }
	  }
	  matched = 1;
	}
	break;
      default:
	break;
    }

    /*
     * if hdr==NULL, then we are using this to parse either a postponed
     * message, or a outgoing message (edit_hdrs), so we want to keep
     * track of the user-defined headers
     */
    if (!matched && !hdr)
    {
      if (last)
      {
	last->next = mutt_new_list ();
	last = last->next;
      }
      else
	last = e->userhdrs = mutt_new_list ();
      last->data = safe_strdup (line);
    }

    loc = ftell (f);
  }

  safe_free ((void **)&line);

  if (hdr)
  {
    hdr->content->hdr_offset = hdr->offset;
    hdr->content->offset = ftell (f);

    /*
     * if we know how long this message is, either just skip over the body,
     * or if we don't know how many lines there are, count them now (this will
     * save time by not having to search for the next message marker).
     */
    if (content_length > 0)
    {
      long loc = ftell (f); /* save the current position */
      long tmploc = loc + content_length + 1;
      struct stat st;
      int valid = 0; /* do we have a valid content-length? */

      fstat (fileno (f), &st);
      if (tmploc < st.st_size) /* make sure we aren't past EOF */
      {
	fseek (f, tmploc, 0);
	if (fgets (buffer, sizeof (buffer), f) != NULL)
	{
	  /*
	   * check to see if the content-length looks valid.  we expect to
	   * to see a valid message separator at this point in the stream
	   */
	  if (strncmp ("From ", buffer, 5))
	  {
	    dprint (1, (debugfile, "read_rfc822_header(): bad content-length in message %d (cl=%ld)\n", hdr->index, content_length));
	    dprint (1, (debugfile, "\tLINE: %s", buffer));
	    fseek (f, loc, 0); /* nope, return the previous position */
	  }
	  else
	    valid = 1;
	}
      }
      else if (tmploc == st.st_size)
	valid = 1; /* last message in the mailbox */

      if (valid)
      {
	/*
	 * good content-length.  check to see if we know how many lines
	 * are in this message.
	 */
	if (hdr->lines == 0)
	{
	  /* count the number of lines in this message */
	  fseek (f, loc, 0);
	  while (content_length-- > 0)
	  {
	    if (fgetc (f) == '\n')
	      hdr->lines++;
	  }
	}

	/* return to the offset of the next message separator */
	fseek (f, tmploc, 0);
      }
      else
      {
	dprint (1, (debugfile, "read_rfc822_header(): bad content-length in message %d\n", hdr->index));
      }
    }

    /*
     * Check for missing or invalid date.  If this is the case, convert the
     * date the message was received to UTC and use that value.
     */
    if (!date_found)
    {
      struct tm *when = gmtime (&hdr->received);

      hdr->year = when->tm_year - 71;
      hdr->month = when->tm_mon;
      hdr->day = when->tm_mday;
      hdr->hours = when->tm_hour;
      hdr->minutes = when->tm_min;
      hdr->seconds = when->tm_sec;

      dprint(1,(debugfile,"read_rfc822_header(): no date found, using received time from msg separator\n"));
    }

    hdr->date_sent = mutt_gmtime (hdr);

    /* if no references were given, try using the In-Reply-To id (if found) */
    if (!e->references && in_reply_to[0])
    {
      e->references = mutt_parse_references (in_reply_to);
      dprint(3,(debugfile,"read_rfc822_header(): no References found, using In-Reply-To\n"));
    }

    /* do RFC1522 decoding */
    rfc1522_decode_adrlist (e->from);
    rfc1522_decode_adrlist (e->to);
    rfc1522_decode_adrlist (e->cc);

    if (e->subject)
      rfc1522_decode (e->subject, strlen (e->subject), 0);
  }

  return (e);
}
