/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"

#include <stdlib.h>
#include <string.h>

#define SORTCODE(x) (Sort & SORTREVERSE) ? -(x) : x

int compare_size (const void *a, const void *b)
{
  HEADER **pa = (HEADER **)a;
  HEADER **pb = (HEADER **)b;

  return (SORTCODE ((*pa)->content->length - (*pb)->content->length));
}

int compare_date_sent (const void *a, const void *b)
{
  HEADER **pa = (HEADER **)a;
  HEADER **pb = (HEADER **)b;

  return (SORTCODE ((*pa)->date_sent - (*pb)->date_sent));
}

int compare_subject (const void *a, const void *b)
{
  HEADER **pa = (HEADER **)a;
  HEADER **pb = (HEADER **)b;
  int rc;

  if (!(*pa)->env->subject)
  {
    if (!(*pb)->env->subject)
      rc = compare_date_sent (pa, pb);
    else
      rc = -1;
  }
  else if (!(*pb)->env->subject)
    rc = 1;
  else
  {
    char *sa = (*pa)->env->subject;
    char *sb = (*pb)->env->subject;

    if (strncasecmp ("re:", sa, 3) == 0) a = skip_whitespace (sa+3);
    if (strncasecmp ("re:", sb, 3) == 0) b = skip_whitespace (sb+3);
    rc = strcmp (sa, sb);
  }
  return (SORTCODE (rc));
}

int compare_from (const void *a, const void *b)
{
  HEADER **ppa = (HEADER **)a;
  HEADER **ppb = (HEADER **)b;
  HEADER *pa = *ppa, *pb = *ppb;
  char buf1[STRING] = { 0 };
  char buf2[STRING] = { 0 };

  mutt_write_address (buf1, sizeof (buf1), pa->env->from);
  mutt_write_address (buf2, sizeof (buf2), pb->env->from);
  return(SORTCODE(strcmp (buf1, buf2)));
}

int compare_date_received (const void *a, const void *b)
{
  HEADER **pa = (HEADER **)a;
  HEADER **pb = (HEADER **)b;
  
  return (SORTCODE ((*pa)->received - (*pb)->received));
}

int compare_order (const void *a, const void *b)
{
  HEADER **ha = (HEADER **)a;
  HEADER **hb = (HEADER **)b;

  return ((*ha)->index - (*hb)->index);
}

sort_t *mutt_get_sort_func (int method)
{
  switch (method & SORTMASK)
  {
    case SORTRECEIVED:
      return (compare_date_received);
    case SORTORDER:
      return (compare_order);
    case SORTDATE:
      return (compare_date_sent);
    case SORTSUBJECT:
      return (compare_subject);
    case SORTFROM:
      return (compare_from);
    case SORTSIZE:
      return (compare_size);
    default:
      return (NULL);
  }
  /* not reached */
}

void mutt_sort_headers (CONTEXT *ctx)
{
  int i;
  sort_t *sortfunc;
  
  if (!ctx->msgcount)
  {
    /*
     * this function gets called by mutt_sync_mailbox(), which may have just
     * deleted all the messages.  the virtual message numbers are not updated
     * in that routine, so we must make sure to zero the vcount member.
     */
    ctx->vcount = 0;
    return; /* nothing to do! */
  }

  mutt_error ("Sorting mailbox...");

  if (Sort == SORTTHREADS)
    mutt_sort_threads (ctx);
  else if ((sortfunc = mutt_get_sort_func (Sort)) == NULL)
  {
    mutt_error ("Could not find sorting function! [report this bug]");
    return;
  }
  else 
    qsort ((void *)ctx->hdrs, ctx->msgcount, sizeof (HEADER *), sortfunc);

  /* adjust the virtual message numbers */
  ctx->vcount = 0;
  for (i=0; i<ctx->msgcount; i++)
  {
    if (ctx->hdrs[i]->virtual != -1)
    {
      ctx->hdrs[i]->virtual = ctx->vcount;
      ctx->v2r[ctx->vcount] = i;
      ctx->vcount++;
    }
    ctx->hdrs[i]->msgno = i+1;
  }

  mutt_error ("");
}
