/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"
#include "mutt_curses.h"

#include <string.h>
#include <ctype.h>
#include <unistd.h>

static void collapse_pathname (char *s)
{
  char *p=s, *q=s;

  while (*p)
  {
    if (*p == '/' && *(p+1) == '/')
    {
      *q++ = '/';
      p += 2;
    }
    else if (*p == '/' && *(p+1) && *(p+1) == '.' && *(p+2) && *(p+2) == '/')
      p += 3;
    else
      *q++ = *p++;
  }
  *q = 0;
}

static void pretty_mailbox (const char *path, char *buf, size_t buflen)
{
  char tmp[_POSIX_PATH_MAX];
  int len;

  if (*path != '/')
  {
    /* get the full pathname */
    getcwd (tmp, sizeof(tmp));
    snprintf (buf, buflen, "%s/%s", tmp, path);
    collapse_pathname (buf);
  }
  else
    strfcpy (buf, path, buflen);

  strfcpy (tmp, Maildir, sizeof (tmp));
  mutt_expand_path (tmp, sizeof (tmp));
  len = strlen (tmp);
  if (strncmp (buf, tmp, len) == 0 && buf[len] == '/')
  {
    strfcpy (tmp, buf+len+1, sizeof (tmp));
    snprintf (buf, buflen, "=%s", tmp);
  }
  else if (len=strlen(Homedir),strncmp(buf, Homedir, len) == 0 && buf[len] == '/')
  {
    strfcpy (tmp, buf+len+1, sizeof (tmp));
    snprintf (buf, buflen, "~/%s", tmp);
  }
}

/*
 * %f = full mailbox path
 * %h = hostname
 * %m = total number of messages
 * %M = number of messages shown (virutal message count when limiting) [option]
 * %n = number of new messages [option]
 * %l = length of mailbox (in bytes)
 * %p = number of postponed messages [option]
 * %s = current sorting method
 * %t = # of tagged messages [option]
 * %v = Mutt version
 *
 * %>X = right justify and pad with "X"
 * %|X = pad with "X" to end of line
 *
 * optional fields:
 *    %?%M/?
 */

void mutt_status_line (char *buf, size_t buflen, char *p)
{
  char *cp;
  char *wptr = buf;
  char tmp[STRING];
  char prefix[LONG_STRING];
  char fmt[SHORT_STRING];
  char optstring[SHORT_STRING];
  char ch;
  char padchar;
  int wlen = 0;
  int optional;
  int num_post;
  int len;

  buflen--; /* save room for the trailing \0 */
  while (*p && wlen < buflen) {
    if (*p == '%') {
      p++;

      if (*p == '?') {
	p++;
	optional = 1;
      }
      else
	optional = 0;

      cp = prefix;
      while (*p && (isdigit(*p) || *p == '.' || *p == '-')) *cp++ = *p++;
      *cp = 0;

      ch = *p;

      cp = optstring;
      if (optional) {
	if (*p) p++;
	while (*p && *p != '?') *cp++ = *p++;
      }
      *cp = 0;

      tmp[0] = 0;
      switch(ch) {
      case '>': /* right justify rest of line */
	p++;
	padchar = *p++;
	mutt_status_line(tmp, sizeof(tmp), p);
	len = COLS - strlen(tmp);
	while (len > 0) {
	  *wptr++ = padchar;
	  wlen++;
	  len--;
	}
	break;
      case '|': /* pad to end of line */
	p++;
	padchar = *p;
	if (buflen > COLS) buflen = COLS;
	while (wlen < buflen) {
	  *wptr++ = padchar;
	  wlen++;
	}
	break;
      case 'h':
	snprintf(fmt, sizeof(fmt), "%%%ss", prefix);
	snprintf(tmp, sizeof(tmp), fmt, Hostname);
	break;
      case 'f':
	snprintf (fmt, sizeof(fmt), "%%%ss", prefix);
	if (Context->folder)
	  pretty_mailbox (Context->folder->name, prefix, sizeof (prefix));
	else
	  strcpy (prefix, "(no folder)");
	snprintf (tmp, sizeof (tmp), fmt, prefix);
	break;
      case 'l':
	snprintf(fmt, sizeof(fmt), "%%%sld", prefix);
	snprintf(tmp, sizeof(tmp), fmt, Context->folder ? Context->folder->size : 0);
	break;
      case 'm':
	snprintf(fmt, sizeof(fmt), "%%%sd", prefix);
	snprintf(tmp, sizeof(tmp), fmt, Context->msgcount);
	break;
      case 'M':
	if (!optional || Context->vcount != Context->msgcount)
	{
	  snprintf (fmt, sizeof(fmt), "%%%sd%%s", prefix);
	  snprintf (tmp, sizeof(tmp), fmt, Context->vcount, optstring);
	}
	break;
      case 'n':
	if (!optional || Context->new)
	{
	  snprintf (fmt, sizeof(fmt), "%%%sd%%s", prefix);
	  snprintf (tmp, sizeof(tmp), fmt, Context->new, optstring);
	}
	break;
      case 'p':
	num_post = mutt_num_postponed ();
	if (!optional || num_post)
	{
	  snprintf (fmt, sizeof (fmt), "%%%sd%%s", prefix);
	  snprintf (tmp, sizeof (tmp), fmt, num_post, optstring);
	}
	break;
      case 's':
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	snprintf (prefix, sizeof (prefix), "%s%s",
		  Sort & SORTREVERSE ? "reverse-" : "",
		  Sort_methods[(Sort & SORTMASK)-1]);
	snprintf(tmp, sizeof(tmp), fmt, prefix);
	break;
      case 't':
	if (!optional || Context->tagged)
	{
	  snprintf (fmt, sizeof (fmt), "%%%sd%%s", prefix);
	  snprintf (tmp, sizeof (tmp), fmt, Context->tagged, optstring);
	}
	break;
      case 'v':
	snprintf (fmt, sizeof (fmt), "%%%ss", prefix);
	snprintf (tmp, sizeof (tmp), fmt, MuttVersion);
	break;
      case 0:
	*buf = 0;
	return;
      default:
	snprintf (tmp, sizeof (tmp), "%%%s%c", prefix, ch);
	break;
      }
      len = strlen(tmp);
      if (len + wlen > buflen)
      {
	if ((len = buflen - wlen) < 0) len = 0;
      }
      memcpy(wptr, tmp, len);
      wptr += len;
      wlen += len;
    }
    else
    {
      *wptr++ = *p;
      wlen++;
    }
    p++;
  }
  *wptr = 0;
}
