/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "muttlib.h"

#include <string.h>

/*
 * This function makes use of the fact that Mutt stores message references in
 * reverse order (i.e., last to first).  This is optiminal since we would like
 * to find the most recent message to which "cur" refers itself.
 */
static HEADER *find_reference (HEADER *cur, CONTEXT *ctx)
{
  LIST *refs = cur->env->references;
  int i;

  while (refs)
  {
    /*
     * If the mailbox is already mostly sorted (such as when new mail arrives),
     * the reference is likely to be just before the current message, so
     * searching backward from this message first could save some time.
     *
     * Note: cur == ctx->hdrs[ cur->msgno - 1 ]
     */
    for (i = cur->msgno - 2 ; i >= 0 ; i--)
    {
      if (ctx->hdrs[i]->env->message_id &&
          !strcmp (refs->data, ctx->hdrs[i]->env->message_id))
        return (ctx->hdrs[i]);
    }

    for (i = cur->msgno ; i < ctx->msgcount ; i++)
    {
      if (ctx->hdrs[i]->env->message_id &&
          !strcmp (refs->data, ctx->hdrs[i]->env->message_id))
        return (ctx->hdrs[i]);
    }

    refs = refs->next;
  } 
  return NULL;
}

static HEADER *find_subject (HEADER *cur, HEADER *tree)
{
  char *a;
  char *b;

  if (!cur->env->subject) return 0;

  if (strncasecmp("re:", cur->env->subject, 3) == 0)
    a = skip_whitespace(cur->env->subject+3);
  else
    a = cur->env->subject;

  /* see if there is a thread with a similar subject */
  while (tree)
  {
    if (tree->env->subject)
    {
      if (strncasecmp ("re:", tree->env->subject, 3) == 0)
	b = skip_whitespace (tree->env->subject+3);
      else
	b = tree->env->subject;
      
      if (strcmp (a, b) == 0) return (tree);
    }
    tree = tree->next;
  }
  return 0;
}

static HEADER *sort_thread (HEADER *list, sort_t *sortfunc)
{
  HEADER *top = NULL;
  HEADER *cur;
  HEADER *tmp;

  while (list)
  {
    cur = list;
    list = list->next;
    cur->next = NULL;
    cur->prev = NULL;

    /* sort the descendants of this message */
    cur->child = sort_thread (cur->child, sortfunc);

    if (top)
    {
      if ((*sortfunc)((void *)&cur, (void *)&top) < 0)
      {
	cur->next = top;
        top->prev = cur;
	top = cur;
      }
      else
      {
	tmp = top;
	while (tmp->next)
	{
	  if ((*sortfunc)((void *)&cur, (void *)&tmp->next) < 0)
	  {
	    cur->next = tmp->next;
            tmp->next->prev = cur;
	    tmp->next = cur;
	    break;
	  }
	  tmp = tmp->next;
	}
	if (!tmp->next)
	{
	  tmp->next = cur;
          cur->prev = tmp;
        }
      }
    }
    else
      top = cur;
  }
  return (top);
}

static void pseudo_threads (HEADER *list)
{
  HEADER *top = NULL;
  HEADER *last = NULL;
  HEADER *cur;
  HEADER *tmp;

  while (list)
  {
    cur = list;
    list = list->next;
    cur->next = 0;
    cur->prev = 0;

    if ((tmp = find_subject (cur, top)) != 0)
    {
      cur->parent = tmp;
      if (tmp->child)
      {
	tmp = tmp->child;
	while (tmp->next) tmp = tmp->next;
	tmp->next = cur;
        cur->prev = tmp;
      }
      else
	tmp->child = cur;
    }
    else
    {
      if (top)
      {
	last->next = cur;
        cur->prev = last;
	last = last->next;
      }
      else
	top = last = cur;
    }
  }
}

/*
 * Since the graphics characters have a value >255, I have to resort to
 * using escape sequences to pass the information to print_enriched_string().
 *
 *   '\001': corner        (ACS_LLCORNER)
 *   '\002': right tee     (ACS_RTEE)
 *   '\003': horiz. line   (ACS_HLINE)
 *   '\004': vertical line (ACS_VLINE)
 *   '\005': space
 *   '\006': greater than  (>)
 *
 * ncurses should automatically use the default ASCII characters instead of
 * graphics chars on terminals which don't support them (see the man page
 * for curs_addch).
 */
static int linearize_tree (HEADER *tree,
			   HEADER **array,
			   int offset,
			   int depth,
			   char *pfx)
{
  char childpfx[SHORT_STRING];
  char mypfx[SHORT_STRING] = { 0 };
  char *a;
  char *b;
    
  while (tree)
  {

    /* determine if the subject has changed from a previous message */
    tree->subject_changed = 0;

    if (tree->env->subject)
    {
      /* get the subject of this message */
      if (strncasecmp ("re:", tree->env->subject, 3) == 0)
        b = skip_whitespace (tree->env->subject+3);
      else
        b = tree->env->subject;

      /* see if it differs from the previous message */
      if (tree->prev && tree->prev->env->subject)
      {
        if (strncasecmp ("re:",tree->prev->env->subject,3) == 0)
          a = skip_whitespace (tree->prev->env->subject+3);
        else
          a = tree->prev->env->subject;
        if (strcmp (a,b) != 0) tree->subject_changed = 1;
      }

      if (!tree->subject_changed && tree->parent)
      {
        if (tree->parent->env->subject)
	{
          if (strncasecmp ("re:", tree->parent->env->subject, 3) == 0)
            a = skip_whitespace (tree->parent->env->subject+3);
          else
            a = tree->parent->env->subject;
          if (strcmp (a,b) != 0) tree->subject_changed = 1;
        }
        else
          tree->subject_changed = 1; /* parent is missing subject */
      }
    }
    
    array[offset] = tree;
    offset++;

    if (depth)
    {
      if (tree->next)
	snprintf (mypfx, sizeof (mypfx), "%s\002\003\006", pfx);
      else
	snprintf (mypfx, sizeof (mypfx), "%s\001\003\006", pfx);
    }
    safe_free ((void **)&tree->tree);
    tree->tree = safe_strdup (mypfx);

    if (tree->child)
    {
      if (depth)
      {
	if (tree->next)
	  snprintf (childpfx, sizeof(childpfx), "%s\004\005", pfx);
	else
	  snprintf (childpfx, sizeof(childpfx), "%s\005\005", pfx);
      }
      else 
	childpfx[0] = 0;
      offset = linearize_tree (tree->child, array, offset, depth+1, childpfx);
    }

    tree = tree->next;
  }
  return (offset);
}

void mutt_sort_threads (CONTEXT *ctx)
{
  HEADER *top = 0;
  HEADER *tmp;
  HEADER *cur;
  int i;

  /* first clear all links */
  for (i = 0; i < ctx->msgcount; i++)
  {
    ctx->hdrs[i]->next = 0;
    ctx->hdrs[i]->child = 0;
    ctx->hdrs[i]->parent = 0;
    ctx->hdrs[i]->prev = 0;
  }

  /* now build the thread tree */
  for (i = 0; i < ctx->msgcount; i++)
  {
    cur = ctx->hdrs[i];

    if ((tmp = find_reference (cur, ctx)) != 0)
    {
      /* found a reference */
      cur->parent = tmp;

      if (tmp->child)
      {
	tmp = tmp->child;
	while (tmp->next) tmp = tmp->next;
	tmp->next = cur;
        cur->prev = tmp;
      }
      else
	tmp->child = cur;
    }
    else
    {
      /* add to the toplevel list of threads */
      if (top)
      {
	tmp = top;
	while (tmp->next) tmp = tmp->next;
	tmp->next = cur;
        cur->prev = tmp;
      }
      else
	top = cur;
    }
  }

  /* sort threads by secondary sorting method */
  top = sort_thread (top, mutt_get_sort_func (SORTDATE));

  if (!option (OPTSTRICTTHREADS))
  {
    /*
     * Create pseudo-threads for messages with the same subject.  This is
     * not always desirable, so it's an option.
     */
    pseudo_threads (top);
  }

  /* put the list into an array */
  linearize_tree (top, ctx->hdrs, 0, 0, "");
}

static void mutt_do_thread_r (HEADER *hdr, int action)
{
  while (hdr)
  {
    mutt_set_flag (hdr, action, 1);
    mutt_do_thread_r (hdr->child, action);
    if (hdr->parent)
      hdr = hdr->next;
    else
      hdr = NULL; /* first message in the thread, so we're done */
  }
}

void mutt_do_thread (HEADER *hdr, int action)
{
  if (Sort != SORTTHREADS)
  {
    mutt_error("Threading is not enabled.");
    return;
  }
  /* find the first message in the thread */
  while (hdr->parent) hdr = hdr->parent;
  mutt_do_thread_r (hdr, action);
}

int mutt_next_thread (HEADER *hdr)
{
  if (Sort != SORTTHREADS)
  {
    mutt_error("Threading is not enabled.");
    return(hdr->virtual);
  }
  while (hdr->parent) hdr = hdr->parent;
  hdr = hdr->next;
  return(hdr ? hdr->virtual : -1);
}

int mutt_previous_thread (HEADER *hdr)
{
  if (Sort != SORTTHREADS)
  {
    mutt_error("Threading is not enabled.");
    return(hdr->virtual);
  }
  while (hdr->parent) hdr = hdr->parent;
  hdr = hdr->prev;
  return (hdr ? hdr->virtual : -1);
}
