/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "keymap.h"

#include <string.h>
#include <stdlib.h>

static char **url_table; 

int mutt_extract_url (HEADER *hdr, LIST **urls)
{
  STATE *s = mutt_new_state();
  FILE *fp;
  LIST *lptr = NULL;
  char temp[_POSIX_PATH_MAX];
  char buf[LONG_STRING];
  char buf2[LONG_STRING];
  char *url;
  int urllen;
  char *p;
  int count = 0;

  mutt_mktemp (temp);
  s->fpin = Context->folder->fp;
  if ((s->fpout = fopen (temp, "w")) == NULL)
  {
    mutt_perror (temp);
    free (s);
    return (-1);
  }
#ifdef _PGPPATH
  if (hdr->pgp)
    endwin ();
#endif
  mutt_copy_header (Context->folder->fp,hdr,s->fpout,0);
  mutt_body_handler (hdr->content, s);
  fclose (s->fpout);
  free (s);

  if ((fp = fopen (temp, "r")) == NULL)
  {
    mutt_perror (temp);
    return (-1);
  }

  while (fgets (buf, sizeof (buf), fp) != 0)
  {
    if ((p = strstr(buf, "://")) != 0)
    {
      /* find the beginning of the method */
      while (p > buf)
      {
        if (!isalpha(*(p-1))) break;
        p--;
      }

      urllen = sizeof (buf2) - 1; /* save room for the trailing \0 */
      url = buf2;
      while (urllen > 0 && *p && *p!=' ' && *p!='\t' && *p !='\n')
      {
        *url++ = *p++;
        urllen--;
      }
      *url = 0;

      /* attempt to remove trailing punctuation */
      if (url > buf2 && !isalpha(*(url-1)) && !isdigit(*(url-1)))
	*(url-1) = 0;

      dprint(2,(debugfile,"mutt_extract_url(): url=%s\n",buf2));
      if (*urls == NULL) { /* first url found */
        *urls = (LIST *)safe_malloc(sizeof(LIST));
        lptr = *urls;
      } else if (lptr == NULL) { /* passed partial LIST */
        lptr = (LIST *)safe_malloc(sizeof(LIST));
        (*urls)->next = lptr;
      } else { /* second url and more */
         lptr->next = (LIST *)safe_malloc(sizeof(LIST));
         lptr = lptr->next;
      }
      lptr->data = safe_strdup(buf2);
      lptr->next = NULL;
      count++;
    }
  }
  fclose (fp);
  mutt_unlink (temp);
  return (count);
}

void url_makeEntry (char *s, int num, size_t size)
{  
  snprintf (s, size, "%2d   %-50s", num+1, url_table[num]);
}

void ci_browse_url (HEADER *hdr)
{
  LIST *urls = NULL;
  LIST *url_ptr;
  MUTTMENU *menu;
  int i, urlmax, done = 0;
  char buf[STRING] = "";
  char cmd[LONG_STRING];
  char *browser;

  if (getenv("DISPLAY") && Web_Xbrowser[0])
    browser = Web_Xbrowser;
  else if (Web_browser[0])
    browser = Web_browser;
  else
  {
    mutt_error("Web_browser is unset.");
    return;
  }

  if (mutt_extract_url (hdr, &urls) == 0)
  {
    mutt_error("No URLs found.");
    return;
  }

  for (urlmax = 0, url_ptr = urls; url_ptr; url_ptr = url_ptr->next, urlmax++);
  url_table = (char **)safe_malloc (sizeof (char *) * (urlmax+1));
  url_table[urlmax] = 0;
  for (i=0, url_ptr = urls; url_ptr; url_ptr = url_ptr->next, i++)
    url_table[i] = url_ptr->data;

  menu = mutt_newMenu ("---Mutt: URL Mode");
  menu->max = urlmax;
  menu->makeEntry = url_makeEntry;
  menu->menu = MENU_URL;

  while (!done)
  {
    switch (mutt_menuLoop (menu))
    {
      case OP_GENERIC_SELECT_ENTRY:

	strfcpy (buf, url_table[menu->current], sizeof (buf));
	if (ci_get_field("Browse URL: ", buf, sizeof (buf), 0) == 0 && buf[0])
	{
	  endwin();
	  if (strstr (browser, "%s"))
	    snprintf (cmd, sizeof (cmd), browser, buf);
	  else
	    snprintf (cmd, sizeof (cmd), "%s '%s'", browser, buf);
	  mutt_system (cmd);
	}
	break;

      case OP_GENERIC_EXIT:

	done = 1;
	break;
    }
  }
  mutt_menuDestroy (&menu);
  safe_free ((void **)&url_table);
  mutt_free_list (&urls);
  return;
}
