/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "keymap.h"

#include <string.h>

typedef struct entry {
  short tagged; /* has the user already added this alias to the list? */
  ALIAS *alias;
} ENTRY;

static ENTRY *AliasTable = NULL;

int alias_search (const char *pat, MUTTMENU *menu)
{
  int i;
  RE_TYPE (re);
  
  if (REGCOMP (re, pat, 0) != 0)
  {
    beep ();
    mutt_error ("Error in regular expression.");
    return (-1);
  }
  
  for (i = menu->current + 1; i < menu->max; i++)
  {
    if (REGEXEC (re, AliasTable[i].alias->name) == 0)
    {
      REGFREE (re);
      return (i);
    }
  }
  
  REGFREE (re);
  
  beep ();
  mutt_error ("Not found.");

  return (-1);
}

/*
 * This is the callback routine from mutt_menuLoop() which is used to generate
 * a menu entry for the requested item number.
 */

void alias_makeEntry (char *s, int num, size_t slen)
{
  char buf[SHORT_STRING] = { 0 };

  mutt_write_address(buf, sizeof(buf), AliasTable[num].alias->addr);
  snprintf(s, slen, "%2d %c %-10s   =>   %s", num+1,
	   AliasTable[num].tagged ? '*' : ' ',
	   AliasTable[num].alias->name,
	   buf);
}

void alias_menu (char *buf, size_t buflen)
{
  ALIAS *aliasp;
  MUTTMENU *menu;
  int i, done = 0;

  if (!Aliases)
  {
    mutt_error ("You have no aliases!");
    return;
  }

  /* tell whoever called me to redraw the screen when I return */
  set_option (OPTNEEDREDRAW);

  menu = mutt_newMenu("---Mutt: Alias Mode");
  menu->makeEntry = alias_makeEntry;
  menu->search = alias_search;
  menu->menu = MENU_ALIAS;

  /* count the number of aliases */
  aliasp = Aliases;
  while (aliasp)
  {
    (menu->max)++;
    aliasp = aliasp->next;
  }

  i = menu->max * sizeof(ENTRY);
  AliasTable = (ENTRY *)safe_malloc(i);
  memset(AliasTable, 0, i);

  for (i = 0, aliasp = Aliases; aliasp; aliasp = aliasp->next, i++)
    AliasTable[i].alias = aliasp;

  while (!done)
  {
    switch (mutt_menuLoop(menu))
    {
      case OP_GENERIC_SELECT_ENTRY:

	AliasTable[menu->current].tagged = !AliasTable[menu->current].tagged;
	menu->redraw = REDRAW_CURRENT;
	break;

      case OP_GENERIC_EXIT:

	done = 1;
	break;
    }
  }

  for (i = 0; i < menu->max; i++)
    if (AliasTable[i].tagged)
      mutt_write_address (buf, buflen, AliasTable[i].alias->addr);

  mutt_menuDestroy (&menu);
  safe_free ((void **)&AliasTable);
}
