/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "rfc822.h"
#include "rfc1522.h"

#include <pwd.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/utsname.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>

enum {
  DT_BOOL,
  DT_NUM,
  DT_STR,
  DT_TRI
};

struct option_t {
  char *option;
  short type;
  short bit; /* DT_BOOL and DT_TRI only */
  void *data;
};

struct option_t MuttVars[] = {
  { "alias_file",      DT_STR,  0, AliasFile },
  { "allow_8bit",      DT_BOOL, OPTALLOW8BIT, NULL },
  { "arrow_cursor",    DT_BOOL, OPTARROWCURSOR, NULL },
  { "askbcc",          DT_BOOL, OPTASKBCC, NULL },
  { "askcc",           DT_BOOL, OPTASKCC, NULL },
  { "askcopy",         DT_BOOL, OPTASKCOPY, NULL },
  { "askdelete",       DT_BOOL, OPTASKDEL, NULL },
  { "askmove",         DT_BOOL, OPTASKMOVE, NULL },
  { "askpostpone",     DT_BOOL, OPTASKPOSTPONE, NULL },
  { "askprint",        DT_BOOL, OPTASKPRINT, NULL },
  { "askrecall",       DT_BOOL, OPTASKRECALL, NULL },
  { "askreplyto",      DT_BOOL, OPTASKREPLYTO, NULL },
  { "attribution",     DT_STR,  0, Attribution },
  { "autoedit",        DT_BOOL, OPTAUTOEDIT,    NULL },
  { "autoinclude",     DT_BOOL, OPTAUTOINCLUDE, NULL },
  { "charset",         DT_STR,  0, Charset },
  { "confirmappend",   DT_BOOL, OPTCONFIRMAPPEND, NULL },
  { "confirmcreate",   DT_BOOL, OPTCONFIRMCREATE, NULL },
  { "confirmfiles",    DT_BOOL,  OPTCONFIRMFILES,  NULL },
  { "confirmfolders",  DT_BOOL, OPTCONFIRMFOLDERS, NULL },
  { "domain",          DT_STR,  0, Fqdn },
#ifdef USE_DSN
  { "dsn_notify",      DT_STR, 0, DsnNotify },
  { "dsn_return",      DT_STR, 0, DsnReturn },
#endif
  { "edit_hdrs",       DT_BOOL, OPTEDITHDRS, NULL },
  { "editor",          DT_STR,  0, Editor },
  { "fcc_attach",      DT_BOOL, OPTFCCATTACH, NULL },
  { "folder",          DT_STR,  0, Maildir },
  { "force_name",      DT_BOOL, OPTFORCENAME, NULL },
  { "forw_format",     DT_STR,  0, ForwFmt },
  { "hdr_format",      DT_STR, 0, Hdr_format },
  { "header",          DT_BOOL, OPTHEADER, NULL },
  { "history",         DT_NUM,    0, &HistSize },
  { "hold",            DT_BOOL,   OPTHOLD, NULL },
  { "in_reply_to",     DT_STR,    0, In_reply_to },
  { "indent_prefix",   DT_STR,  0, Prefix },
  { "keep_empty",      DT_BOOL, OPTKEEPEMPTY, NULL },
  { "localsignature",  DT_STR,  0, Local_sig },
  { "mark_old",        DT_BOOL, OPTMARKOLD, NULL },
  { "mbox",            DT_STR,  0, Inbox },
  { "metoo",           DT_BOOL, OPTMETOO, NULL },
  { "mime_fwd",        DT_BOOL, OPTMIMEFWD, NULL },
  { "pager",           DT_STR,  0, Pager },
  { "pager_context",   DT_NUM,  0, &PagerContext },
  { "pager_format",    DT_STR,  0, PagerFmt },
  { "pager_stop",      DT_BOOL, OPTPAGERSTOP, NULL },
#ifdef _PGPPATH
  { "pgp",             DT_STR,  0, Pgp },
  { "pgp_autosign",    DT_BOOL, OPTPGPAUTOSIGN, NULL },
#endif /* _PGPPATH */
  { "pointnew",        DT_BOOL, OPTPOINTNEW, NULL },
  { "popdelete",       DT_BOOL, OPTPOPDELETE, NULL },
  { "pophost",         DT_STR,  0, PopHost },
  { "popport",         DT_NUM,  0, &PopPort },
  { "poppass",         DT_STR,  0, &PopPass },
  { "popuser",         DT_STR,  0, PopUser },
  { "post_indent_str", DT_STR,  0, PostIndentString },
  { "postponed",       DT_STR,  0, Postponed },
  { "print",           DT_STR,  0, Print },
  { "prompt_after",    DT_BOOL, OPTPROMPTAFTER, NULL },
  { "quote_regexp",    DT_STR,  0, QuoteString },
  { "readmsginc",      DT_NUM,  0, &ReadMsginc },
  { "realname",        DT_STR,  0, Realname },
  { "record",          DT_STR,  0, Outbox },
  { "remotesignature", DT_STR,  0, Remote_sig },
  { "resolve",         DT_BOOL, OPTRESOLVE, NULL },
  { "reverse_alias",   DT_BOOL, OPTREVALIAS, NULL },
  { "save_name",       DT_BOOL, OPTSAVENAME, NULL },
  { "sendmail",        DT_STR,  0, Sendmail },
  { "shell",           DT_STR,  0, Shell },
  { "sigdashes",       DT_BOOL, OPTSIGDASHES, NULL },
  { "signature",       DT_STR,  0, Signature },
  { "spoolfile",       DT_STR,  0, Spoolfile },
  { "status_format",   DT_STR,  0, Status },
  { "status_on_top",   DT_BOOL, OPTSTATUSONTOP, NULL },
  { "strict_threads",  DT_BOOL, OPTSTRICTTHREADS, NULL },
  { "tilde",           DT_BOOL, OPTTILDE, NULL },
  { "timeout",         DT_NUM,  0, &Timeout },
  { "tmpdir",          DT_STR,  0, Tempdir },
  { "tochars",         DT_STR,  0, Tochars },
  { "use_mailcap",     DT_TRI,  M_USEMAILCAP, NULL },
  { "verify_sig",      DT_TRI,  M_VERIFYSIG, NULL },
  { "web_browser",     DT_STR,  0, Web_browser },
  { "web_xbrowser",    DT_STR,  0, Web_Xbrowser },
  { "writemsginc",     DT_NUM,  0, &WriteMsginc },
  { NULL, 0, 0, NULL }
};

static void set_trioption (int opt, int flag)
{
  TriOptions &= ~(0x3 << (2 * opt)); /* first clear the bits */
  TriOptions |= (flag & 0x3) << (2 * opt); /* now set them */
}

int trioption (int opt, int flag)
{
  return(((TriOptions >> opt) & 0x3) == flag);
}

/*
 * given the variable ``s'', return the index into the rc_vars array which
 * matches, or -1 if the variable is not found.
 */
int mutt_option_index (char *s)
{
  int i;

  for (i=0; MuttVars[i].option; i++)
    if (!strcmp (s, MuttVars[i].option)) return (i);
  return (-1);
}

void remove_from_list (LIST **l, const char *s)
{
  char tmp[LONG_STRING], *q = tmp;
  LIST *p;
  LIST *last;
  
  strfcpy (tmp, s, sizeof (tmp));
  while ((q = strtok (q, " \t")))
  {
    /* ``unCMD *'' means delete all current entries */
    if (!strcmp ("*", q))
      mutt_free_list (l);
    else
    {
      p = *l;
      last = NULL;
      while (p)
      {
	if (!strcasecmp (q, p->data))
	{
	  safe_free((void **)&p->data);
	  if (last)
	    last->next = p->next;
	  else
	    (*l) = p->next;
	  safe_free ((void **)&p);
	}
	else
	{
	  last = p;
	  p = p->next;
	}
      }
    }
    q = NULL;
  }
}

static int parse_alternates (char *s, char *err, size_t errlen)
{
  ADDRESS *end = Alternates;
  ADDRESS *newaddr = NULL;
  char *p, *q;

  if (end)
    while (end->next)
      end = end->next;

  p = q = s;
  while (*q)
  {
    *p++ = *q++;
    if (*q == ' ' || *q == '\t' || *q == ',')
    {
      q++;
      while (*q == ' ' || *q == '\t' || *q == ',')
	q++;
      *p++ = ',';
    }
  }
  *p = 0;

  if (mutt_parse_adrlist (&newaddr, s, "@") == -1)
  {
    strfcpy (err, ParseError, errlen);
    return (-1);
  }

  if (end)
    end->next = newaddr;
  else
    Alternates = newaddr;

  return 0;
}

static int add_userheader (char *s, char *errmsg, int errlen)
{
  USERHEADER *tmp;
  char *p;

  if ((p = strchr (s, ':')) == NULL)
  {
    strcpy (errmsg, "invalid header line (missing colon)");
    return (-1);
  }
  *p = 0;
  p = skip_whitespace (p+1);
  if (!*p)
  {
    snprintf (errmsg, errlen, "ignoring empty header field: %s", s);
    return (-1);
  }

  if (User_defined_headers)
  {
    tmp = User_defined_headers;

    FOREVER
    {
      if (strcasecmp (s, tmp->key) == 0)
      {
	/* replace the old value */
	safe_free ((void **)&tmp->val);
	tmp->val = safe_strdup (p);
	return (0);
      }
      if (!tmp->next)
	break;
      tmp = tmp->next;
    }
    tmp->next = safe_malloc (sizeof (USERHEADER));
    tmp = tmp->next;
  }
  else
  {
    tmp = safe_malloc (sizeof (USERHEADER));
    User_defined_headers = tmp;
  }

  memset (tmp, 0, sizeof (USERHEADER));
  tmp->key = safe_strdup (s);
  tmp->val = safe_strdup (p);

  return 0;
}

static void free_userheader (USERHEADER **h)
{
  safe_free ((void **)&(*h)->key);
  safe_free ((void **)&(*h)->val);
  *h = NULL;
}

static void remove_userheader (char *s)
{
  USERHEADER *last = NULL;
  USERHEADER *tmp = User_defined_headers;
  USERHEADER *ptr;

  while ((s = strtok(s, " \t")) != NULL)
  {
    if (strcmp ("*", s) == 0)
    {
      last = User_defined_headers;
      while (last)
      {
	tmp = last;
	last = last->next;
	free_userheader(&tmp);
      }
      User_defined_headers = NULL;
    }
    else
    {
      tmp = User_defined_headers;
      last = NULL;

      while (tmp)
      {
	if (strcasecmp(s, tmp->key) == 0)
	{
	  ptr = tmp;
	  if (last)
	    last->next = tmp->next;
	  else
	    User_defined_headers = tmp->next;
	  tmp = tmp->next;
	  free_userheader(&ptr);
	}
	else
	{
	  last = tmp;
	  tmp = tmp->next;
	}
      }
    }
    s = NULL;
  }
}

static int add_alias (char *s, char *errmsg, int errlen)
{
  ALIAS *tmp = Aliases;
  ALIAS *last = NULL;
  char *p;

  if ((p = strpbrk(s, " \t")) == NULL)
  {
    strfcpy(errmsg, "alias has no address", errlen);
    return(-1);
  }

  *p = 0;
  p = skip_whitespace(p+1);

  /* check to see if an alias with this name already exists */
  while (tmp)
  {
    if (!strcasecmp(tmp->name, s)) break;
    last = tmp;
    tmp = tmp->next;
  }

  if (!tmp)
  {
    /* create a new alias */
    tmp = (ALIAS *)safe_malloc(sizeof(ALIAS));
    memset (tmp, 0, sizeof (ALIAS));
    tmp->name = safe_strdup (s);
  }
  else
  {
    /* override the previous value */
    mutt_free_address (&tmp->addr);
  }

  /* C-CLIENT does not remove tabs, which can cause display problems later */
  mutt_tabs_to_spaces (p);

  if (mutt_parse_adrlist (&tmp->addr, p, "@") == -1)
  {
    strfcpy (errmsg, ParseError, errlen);
    mutt_free_alias (&tmp);
    return (-1);
  }

  if (last)
    last->next = tmp;
  else
    Aliases = tmp;

  return 0;
}

static void remove_aliases (char *s)
{
  ALIAS *tmp;
  ALIAS *last = NULL;

  while ((s = strtok(s, " \t")) != NULL)
  {
    tmp = Aliases;
    while (tmp)
    {
      if (strcasecmp (s, tmp->name) == 0)
      {
	if (last)
	  last->next = tmp->next;
	else
	  Aliases = tmp->next;
	tmp->next = NULL;
	mutt_free_alias (&tmp);
	break;
      }
      last = tmp;
      tmp = tmp->next;
    }
    s = NULL;
  }
}

void mutt_add_to_list (LIST **list, const char *s)
{
  char tmp[LONG_STRING], *q = tmp;
  LIST *last = 0;
  LIST *t;

  strfcpy (tmp, s, sizeof (tmp));
  while ((q = strtok (q, " \t,")))
  {
    /* check to make sure the item is not already on this list */
    last = *list;
    while (last)
    {
      if (!strcasecmp (q, last->data))
      {
	/* already on the list, so just ignore it */
	last = NULL;
	break;
      }
      if (!last->next) break;
      last = last->next;
    }
    if (!*list || last)
    {
      t = (LIST *)safe_malloc(sizeof(LIST));
      memset(t, 0, sizeof(LIST));
      t->data = safe_strdup (q);
      if (last)
      {
	last->next = t;
	last = last->next;
      }
      else
	*list = last = t;
    }
    q = NULL;
  }
}

/*
 * reads the specified initialization file.  returns -1 if errors were found
 * so that we can pause to let the user know...
 */
static int
source_rc (const char *rcfile, int sourcing)
{
  FILE *f;
  short line=0, rc=0;
  char buffer[LONG_STRING], errbuf[SHORT_STRING];

  if ((f = fopen (rcfile, "r")) == NULL)
  {
    printf ("%s: unable to open file\n", rcfile);
    return (-1);
  }

  while (fgets (buffer, LONG_STRING, f) != NULL)
  {
    line++;

    if (mutt_parse_rc_line (buffer, sourcing, errbuf, sizeof (errbuf)) == -1)
    {
      if (!option (OPTBATCHMODE))
	printf ("Error in %s, line %d: %s\n", rcfile, line, errbuf);
      rc = -1;
    }
  }
  fclose (f);

  return (rc);
}

/*
 * Takes a string, and executes any `commands` and replaces the 
 * command string with the first line of the command output
 */
static void mutt_string_cmd_sub (char *s, int slen)
{
   char *tmp;
   int incmd = 0;
   char *cmd, *resp; 
   int cmdind = 0;
   int x,y;

   dprint(1,(debugfile,"mutt_string_cmd_sub(): Substring: %s, slen: %d\n",s,slen));
   tmp = (char *) safe_malloc(sizeof(char *) * slen);
   strfcpy (tmp, s, slen);
   cmd = (char *) safe_malloc(sizeof(char *) * slen);
   resp = (char *) safe_malloc(sizeof(char *) * slen);
   for (x = 0, y = 0; x < slen && y < slen && s[x]; x++) {
     if (incmd) {
       /* dprint(1,(debugfile,"mutt_string_cmd_sub(): cmd: %s\n",s)); */
       if (s[x] == '\\') {
	 x++;
	 if (s[x] == 'n') {
	      cmd[cmdind++] = '\n';
         } 
	 else if (s[x] == 't') {
	      cmd[cmdind++] = '\t';
         }
	 else if (s[x])
	   cmd[cmdind++] = s[x];
       }
       else if (s[x] == '`')
	 incmd = 0;
       else
	 cmd[cmdind++] = s[x];

       if (!incmd) {
	 cmd[cmdind] = 0;
	 if (cmd[0]) {
	   FILE *out = NULL;
	   int rind = 0;

           dprint(1,(debugfile,"mutt_string_cmd_sub(): Command: %s\n",cmd));
	   mutt_create_filter(cmd, NULL, &out);
	   fgets(resp,slen,out);
	   fclose(out);
           dprint(1,(debugfile,"mutt_string_cmd_sub(): Response: %s\n",resp));

	   /* copy response into temp string */
	   while (resp[rind] && y < slen) {
	     if (resp[rind] != '\n')
	       tmp[y++] = resp[rind++];
	     else
	       rind++;
	   }
	   if (y < slen)
	     tmp[y] = 0;
	   else
	     tmp[slen-1] = 0;
           /* reset command */
           cmdind = 0;
         }
       }
     } else {
       if ((s[x] == '`') && (!x || (s[x-1] != '\\'))) incmd = 1;
	 else tmp[y++] = s[x];
     }
   }
   if (y < slen)
     tmp[y] = 0;
   else
     tmp[slen-1] = 0;
   strncpy(s,tmp,slen);
   safe_free((void **)&tmp);
   safe_free((void **)&cmd);
   safe_free((void **)&resp);
}

/*
 * if sourcing is TRUE, we are reading from an init file, otherwise assume
 * that this was entered on a command line.
 */
int mutt_parse_rc_line (char *s, int sourcing, char *errmsg, int errlen)
{
  char *p, *q;
  int unset=0;
  int i;

  *errmsg = 0;

  s = skip_whitespace (s);

  /* Strip those comments!  Allow the comment character (#) to be escaped */
  if (*s == '#')
    return 0; /* nothing to do */

  p = s+1;
  while ((p = strchr (p, '#')))
  {
    if (*(p-1) == '\\')
      strcpy (p-1, p);
    else
    {
      *p = 0;
      break;
    }
  }

  /* Remove any trailing whitespace */
  p = s + strlen (s) - 1;
  while (isspace (*p) && (p >= s))
  {
    *p = 0;
    if (p == s)
      break;
    else
      p--;
  }

  /* Is there anything left on this line? */
  if (!*s)
    return 0;

  if ((p = strchr(s,'`')) && (*(p-1) != '\\')) 
    mutt_string_cmd_sub (s,LONG_STRING);

  if (!strncmp (s, "set", 3) || (!strncmp(s, "unset", 5) && (unset=1)))
  {
    int idx;
    int query=0;
    char keyword[SHORT_STRING], *pk;

    /* position on the next word */
    if ((p = strpbrk(s, " \t")) == NULL)
    {
      strcpy (errmsg, "\"set\" requires argument");
      return (-1);
    }
    p = skip_whitespace (p);
    
    /* this is an alternate way to unset a boolean variable */
    if (strncmp ("no", p, 2) == 0)
    {
      unset = 1;
      p += 2;
    }
    else if (*p == '?')
    {
      query=1;
      p++;
    }

    /* Get the keyword */
    pk = keyword;
    while (*p && !isspace(*p) && *p != '=') *pk++ = *p++;
    *pk = 0;
    p = skip_whitespace (p);
    
    if ((idx = mutt_option_index (keyword)) == -1)
    {
      snprintf(errmsg, errlen, "%s: unknown variable", keyword);
      return(-1);
    }

    if (MuttVars[idx].type == DT_BOOL)
    {
      if (query)
      {
	snprintf(errmsg, errlen, "%s is %sset", keyword,
		 option(MuttVars[idx].bit) ? "" : "un");
	return 0;
      }
      else if (unset)
	unset_option(MuttVars[idx].bit);
      else
	set_option(MuttVars[idx].bit);
    }
    else if (MuttVars[idx].type == DT_STR)
    {
      char *ptr = MuttVars[idx].data;
      
      if (*p != '=')
      {
	if (query || (!sourcing && *p == 0))
	{
	  /* user requested the value of this variable */
	  snprintf (errmsg, errlen, "%s=\"%s\"", MuttVars[idx].option, ptr);
	  return 0;
	}
	else if (unset)
	{
	  *ptr = 0;
	  return 0;
	}
	snprintf (errmsg, errlen, "expected \"=\" but got \"%s\"", p);
	return (-1);
      }
      p = skip_whitespace (p+1);
      if (*p == '"' || *p == '\'')
      {
	char quote = *p++;

	while (*p)
	{
	  if (*p == '\\')
	  {
	    p++;
	    if (*p == 'n')
	    {
	      *ptr++ = '\n';
              p++;
            }
	    else if (*p == 't')
	    {
	      *ptr++ = '\t';
              p++;
            }
	    else if (*p)
	      *ptr++ = *p++;
	    else
	      break;
	  }
	  else if (*p == quote)
	    break;
	  else
	    *ptr++ = *p++;
	}
	*ptr = 0;
      }
      else
      {
	while (*p && !isspace (*p))
	  *ptr++ = *p++;
	*ptr = 0;
      }
    }
    else if (MuttVars[idx].type == DT_NUM)
    {
      short *ptr = MuttVars[idx].data;
      
       if (*p != '=')
       {
	if (!sourcing && *p == 0)
	{
	  snprintf(errmsg, errlen, "%s=%d", MuttVars[idx].option, *ptr);
	  return 0;
	}
	snprintf(errmsg, errlen, "expected \"=\" but got \"%s\"", p);
	return(-1);
      }
      p = skip_whitespace(p+1);
      *ptr = (short)atoi(p);
    }
    else if (MuttVars[idx].type == DT_TRI)
    {
      if (*p != '=')
      {
	if (!sourcing)
	{
	  snprintf(errmsg, errlen, "%s=%s", MuttVars[idx].option,
		   trioption(MuttVars[idx].bit, M_ASK) ? "ask" :
		   (trioption(MuttVars[idx].bit, M_YES) ? "yes" : "no"));
	  return 0;
	}
	else
	{
	  snprintf(errmsg, errlen, "missing = in command");
	  return(-1);
	}
      }
      p = skip_whitespace (p+1);
      if (strcasecmp ("yes", p) == 0)
	set_trioption(MuttVars[idx].bit, M_YES);
      else if (strcasecmp ("no", p) == 0)
	set_trioption(MuttVars[idx].bit, M_NO);
      else if (strcasecmp("ask", p) == 0)
	set_trioption(MuttVars[idx].bit, M_ASK);
      else
      {
	snprintf (errmsg, errlen, "%s: invalid value", p);
	return (-1);
      }
    }
  }
  else if (strncmp(s, "alias", 5) == 0)
  {
    p = skip_whitespace(s+5);
    if (add_alias(p, errmsg, errlen) == -1) return(-1);
  }
  else if (strncmp(s, "unalias", 7) == 0)
    remove_aliases(skip_whitespace(s+7));
  else if (strncmp(s, "alternates", 10) == 0)
  {
    p = skip_whitespace (s+10);
    if (parse_alternates (p, errmsg, errlen) == -1)
      return (-1);
  }
  else if (strncmp(s, "macro", 5) == 0)
  {
    if (mutt_parse_macro (skip_whitespace (s+5), errmsg, errlen) == -1)
      return(-1);
  }
  else if (strncmp (s, "bind", 4) == 0)
  {
    p = skip_whitespace (s+4);
    if (ci_parse_bind (p, errmsg) == -1)
      return(-1);
  }
  else if (strncmp(s, "color", 5) == 0)
  {
    /* If we don't have color support, silently ignore color commands. */
#ifdef HAVE_COLOR
    /* if batchmode, don't try to do color, since we aren't using curses */
    if (!option(OPTBATCHMODE))
    {
      if (ci_parse_color(skip_whitespace(s+5), errmsg, errlen) == -1)
	return(-1);
    }
#endif /* HAVE_COLOR */
  }
  else if (!strncmp (s, "ignore", 6))
  {
    p = skip_whitespace (s+6);
    mutt_add_to_list (&Ignore, p);
    remove_from_list (&UnIgnore, p);
  }
  else if (!strncmp (s, "unignore", 8))
  {
    p = skip_whitespace (s+8);
    mutt_add_to_list (&UnIgnore, p);
    remove_from_list (&Ignore, p);
  }
  else if (!strncmp (s, "lists", 5))
    mutt_add_to_list(&Mailing_lists, skip_whitespace(s+5));
  else if (strncmp (s, "unlists", 7) == 0)
    remove_from_list (&Mailing_lists, skip_whitespace(s+7));
  else if (strncmp (s, "localsite", 9) == 0)
    mutt_add_to_list(&Localsites, skip_whitespace(s+9));
  else if (strncmp(s, "unlocalsite", 11) == 0)
    remove_from_list(&Localsites, skip_whitespace(s+11));
  else if (strncmp(s, "mono", 4) == 0)
  {
    if (!option(OPTBATCHMODE))
    {
      if (ci_parse_mono(skip_whitespace(s+4), errmsg, errlen) != 0) return(-1);
    }
  }
  else if (strncmp (s, "my_hdr", 6) == 0)
  {
    p = skip_whitespace (s+6);
    if (add_userheader (p, errmsg, errlen) == -1)
      return (-1);
  }
  else if (strncmp(s, "unmy_hdr", 8) == 0)
    remove_userheader(skip_whitespace(s+8));
  else if (strncmp(s, "sort", 4) == 0)
  {
    p = skip_whitespace(s+4);
    if (!*p)
    {
      snprintf (errmsg, errlen, "Sorting by %s", Sort_methods[(Sort & SORTMASK)-1]);
      return (0);
    }
    else if (!sourcing)
    {
      beep();
      mutt_error ("Use the sort-mailbox function to change the sorting method.");
      return (0);
    }
    Sort = 0;
    if (strncasecmp(p, "reverse-", 8) == 0)
    {
      Sort = SORTREVERSE;
      p += 8;
    }
    for (i=0; Sort_methods[i]; i++)
    {
      if (strcasecmp(p, Sort_methods[i]) == 0)
      {
	Sort |= i+1;
	break;
      }
    }
    if (i >= SORTMAX)
    {
      snprintf(errmsg, errlen, "%s: unknown sorting method", p);
      Sort = SORTDATE;
      return(-1);
    }
  }
  else if (strncmp (s, "source", 6) == 0)
  {
    char tmp[_POSIX_PATH_MAX];

    p = skip_whitespace (s+6);
    strfcpy (tmp, p, sizeof (tmp));
    mutt_expand_path (tmp, sizeof (tmp));
    if (source_rc (tmp, sourcing) == -1)
      return (-1);
  }
  else if (strncmp (s, "folder-hook", 11) == 0)
  {
    p = skip_whitespace(s+11);
    if ((q = strpbrk(p, " \t")))
    {
      *q = 0;
      q = skip_whitespace (q+1);
      mutt_add_hook (M_FOLDERHOOK, p, q);
    }
  }
  else if (!strncmp (s, "save-hook", 9))
  {
    p = skip_whitespace (s+9);
    if ((q = strpbrk(p, " \t")))
    {
      *q = 0;
      q = skip_whitespace(q+1);
      mutt_add_hook (M_SAVEHOOK, p, q);
    }
  }
  else if (!strncmp (s, "mbox-hook", 9))
  {
    p = skip_whitespace (s + 9);
    if ((q = strpbrk (p, " \t")))
    {
      *q = 0;
      q = skip_whitespace (q + 1);
      mutt_add_hook (M_MBOXHOOK, p, q);
    }
  }
  else
  {
    if ((p = strpbrk(s, " \t")) != 0) *p = 0;
    snprintf (errmsg, errlen, "%s: unknown command", s);
    return (-1);
  }

  return 0;
}

#ifdef DEBUG
static void start_debug (void)
{
  time_t t;
  int i;
  char buf[_POSIX_PATH_MAX];
  char buf2[_POSIX_PATH_MAX];

  /* rotate the old debug logs */
  for (i=3; i>=0; i--)
  {
    snprintf (buf, sizeof(buf), "%s/.muttdebug%d", Homedir, i);
    snprintf (buf2, sizeof(buf2), "%s/.muttdebug%d", Homedir, i+1);
    rename (buf, buf2);
  }
  if ((debugfile = fopen(buf, "w")) != NULL)
  {
    t = time (0);
    fprintf (debugfile, "%s started at %s.\nDebugging at level %d.\n\n",
	     MuttVersion, asctime(localtime(&t)), debuglevel);
  }
}
#endif

void mutt_init (int skip_sys_rc)
{
  struct passwd *pw;
  char buffer[STRING];
  char *p;
  struct utsname utsname;

  /* Get some information about the user */
  pw = getpwuid (getuid ());
  strfcpy (Username, pw->pw_name, sizeof (Username));

  /*
   * on one of the systems I use, getcwd() does not return the same prefix
   * as is listed in the passwd file
   */
  if ((p = getenv ("HOME")) != NULL)
    strfcpy (Homedir, p, sizeof (Homedir));
  else
    strfcpy (Homedir, pw->pw_dir, sizeof (Homedir));

  strfcpy (Realname, pw->pw_gecos, sizeof (Realname));
  if ((p = strchr (Realname, ','))) *p = 0;
  strfcpy (Shell, pw->pw_shell, sizeof (Shell));

#ifdef DEBUG
  /* Start up debugging mode if requested */
  if (debuglevel > 0)
    start_debug();
#endif

  /* And about the host... */
  uname (&utsname);
  strfcpy (Hostname, utsname.nodename, sizeof (Hostname));
  /* some systems report the FQDN instead of just the hostname */
  if ((p = strchr (Hostname, '.')))
    *p = 0;

#ifdef DOMAIN
  strfcpy (Domain, DOMAIN, sizeof (Domain));
#else
#undef USE_DOMAIN
#endif

  /* set the default FQDN (can be overridden by the $domain variable */
#ifdef USE_DOMAIN
# ifdef HIDDEN_HOST
  strfcpy (Fqdn, Domain, sizeof (Fqdn));
# else
  snprintf (Fqdn, sizeof (Fqdn), "%s.%s", Hostname, Domain);
# endif /* HIDDEN_HOST */
#else
  /*
   * "@" has special meaning to rfc822_write_address(); it refers to the
   * local host.  Setting the Fqdn to this will cause rfc822_write_address()
   * to NOT append any domain to the mailbox.
   */
  strcpy (Fqdn, "@");
#endif /* USE_DOMAIN */

  /* Set some defaults */

  set_option (OPTASKRECALL);
  set_option (OPTHOLD);
  set_option (OPTPROMPTAFTER);
  set_option (OPTALLOW8BIT);
  set_option (OPTPOINTNEW);
  set_option (OPTASKDEL);
  set_option (OPTASKMOVE);
  set_option (OPTASKPOSTPONE);
  set_option (OPTASKPRINT);
  set_option (OPTFCCATTACH);
  set_option (OPTMARKOLD);
  set_option (OPTCONFIRMCREATE);
  set_option (OPTCONFIRMFILES);
  set_option (OPTSIGDASHES);
  set_option (OPTRESOLVE);

#ifdef _PGPPATH
  set_trioption (M_VERIFYSIG, M_YES);
#endif
  set_trioption (M_USEMAILCAP, M_ASK);

  if ((p = getenv ("MAIL")))
    strfcpy (Spoolfile, p, sizeof (Spoolfile));
  else
  {
#ifdef HOMESPOOL
    snprintf (Spoolfile, sizeof (Spoolfile), "%s/%s", Homedir, MAILPATH);
#else
    snprintf (Spoolfile, sizeof (Spoolfile), "%s/%s", MAILPATH, Username);
#endif
  }

  strcpy (AliasFile, "~/.muttrc");
  strcpy (Attribution, "%n writes:");
  strcpy (Charset, "iso-8859-1");
#ifdef USE_DSN
  strcpy (DsnNotify, "failure,delay");
  strcpy (DsnReturn, "hdrs");
#endif
  strcpy (ForwFmt, "[%a: %s]");
  strcpy (Hdr_format, "%M %2N %-15.15L (%4l) %s");
  strcpy (In_reply_to, "%i; from %n on %d");
  strcpy (Inbox, "~/mbox");
  strcpy (Maildir, "~/Mail");
  strcpy (Pager, "builtin");  
  strcpy (PagerFmt, "-%S- %C/%T: %-20.20n   %s");
  strcpy (Postponed, "~/postponed");
  strcpy (Prefix, "> ");
  strcpy (Print, "lpr");
  strcpy (QuoteString, "^ *[|>:}#]");
  strfcpy (Sendmail, SENDMAIL, sizeof (Sendmail));
  strcpy (Signature, "~/.signature");
  Sort = SORTDATE;
  strcpy (Status, "---Mutt: %f [%?M/?%m msgs, %?n new, ?%?t tagged, ?%?p post, ?%l bytes]---(%s)%|-");

  if ((p = getenv ("TMPDIR")) != NULL)
    strfcpy (Tempdir, p, sizeof (Tempdir));
  else
    strcpy (Tempdir, "/tmp");

  strcpy (Tochars, " +TCF");
  Metamail[0] = 0;
#ifdef _PGPPATH
  strfcpy (Pgp, _PGPPATH, sizeof (Pgp));
#endif /* _PGPPATH */
  strfcpy (PopUser, Username, sizeof (PopUser));

  if ((p = getenv ("VISUAL")) != NULL)
    strfcpy (Editor, p, sizeof(Editor));
  else if ((p = getenv ("EDITOR")) != NULL)
    strfcpy (Editor, p, sizeof (Editor));
  else
    strcpy (Editor, "vi");

  if ((p = getenv ("REPLYTO")) != NULL)
  {
    char error[SHORT_STRING];

    snprintf (buffer, sizeof (buffer), "Reply-To: %s", p);
    add_userheader (buffer, error, sizeof (error));
  }

  /*
   * Process the global rc file if it exists and the user hasn't explicity
   * requested not to via "-n".
   */
  if (!skip_sys_rc && access (SYSMUTTRC, F_OK) != -1) source_rc (SYSMUTTRC, 1);

  /*
   * Read the user's initialization file.
   */
  mutt_expand_path (Muttrc, sizeof (Muttrc));
  if (access (Muttrc, F_OK) != -1)
  {
    if (!option (OPTBATCHMODE))
      endwin();
    if (source_rc (Muttrc, 1) != 0 && !option(OPTBATCHMODE))
      ci_any_key_to_continue (NULL);
  }

  mutt_init_history ();
}
