/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "config.h"
#include "muttlib.h"
#include "rfc822.h"
#include "rfc1522.h"

#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/wait.h>
#include <errno.h>
#include <signal.h>
#include <pwd.h>

static char *compatible_charsets[] = {
  "US-ASCII",
  "ISO-8859-1",
  "ISO-8859-2",
  "ISO-8859-3",
  "ISO-8859-4",
  "ISO-8859-5",
  "ISO-8859-6",
  "ISO-8859-7",
  "ISO-8859-8",
  "ISO-8859-9",
  "KOI8-R",
  NULL,
};

int mutt_compat_charset (const char *s)
{
  int i;

  for (i=0; compatible_charsets[i]; i++)
    if (strcasecmp(s, compatible_charsets[i]) == 0)
      return(1);
  return(0);
}

BODY *mutt_new_body (void)
{
  BODY *p = (BODY *)safe_malloc(sizeof(BODY));
    
  memset(p, 0, sizeof(BODY));
  p->disposition = DISPATTACH;
  p->use_disp = 1;
  return(p);
}

void mutt_free_body (BODY **p)
{
  BODY *a = *p, *b;

  while (a)
  {
    b = a;
    a = a->next; 

    if (b->parameter) mutt_free_parameter(&b->parameter);
    if (b->unlink) unlink(b->filename);
    safe_free((void **)&b->filename);
    safe_free((void **)&b->content);
    safe_free((void **)&b->subtype);
    safe_free((void **)&b->description);

    if (b->parts) mutt_free_body(&b->parts);

    safe_free((void **)&b);
  }
  *p = 0;
}

PARAMETER *mutt_new_parameter (void)
{
  PARAMETER *p = (PARAMETER *)safe_malloc(sizeof(PARAMETER));
  
  memset(p, 0, sizeof(PARAMETER));
  return(p);
}

void mutt_free_parameter (PARAMETER **p)
{
  PARAMETER *t = *p;
  PARAMETER *o;

  while (t) {
    safe_free((void **)&t->attribute);
    safe_free((void **)&t->value);
    o = t;
    t = t->next;
    safe_free((void **)&o);
  }
  *p = 0;
}

STATE *mutt_new_state (void)
{
  STATE *p = (STATE *)safe_malloc(sizeof(STATE));

  memset(p, 0, sizeof(STATE));
  return p;
}

void mutt_free_state (STATE **p)
{
  safe_free((void **)p);
}

HEADER *mutt_new_header (void)
{
  HEADER *p = (HEADER *)safe_malloc(sizeof(HEADER));

  memset(p, 0, sizeof(HEADER));
  return(p);
}

void mutt_free_list (LIST **list)
{
  LIST *p;
  
  if (!list) return;
  while (*list) {
    p = *list;
    *list = (*list)->next;
    safe_free((void **)&p->data);
    safe_free((void **)&p);
  }
}

void mutt_free_header (HEADER **h)
{
  mutt_free_envelope(&(*h)->env);
  mutt_free_body(&(*h)->content);
  safe_free((void **)&(*h)->tree);
  safe_free((void **)h);
}

void remove_leading_space (char *s)
{
  char *p = s;

  while (*p && isspace(*p)) p++;
  if (p != s && *p) {
    while (*p) *s++ = *p++;
  }
}

/* returns true if the header contained in "s" is in list "t" */
int mutt_matches_ignore (const char *s, LIST *t)
{
  while (t)
  {
    if (!strncasecmp (s, t->data, strlen (t->data)) || *t->data == '*' ||
	(strcasecmp ("From_", t->data) == 0 && strncasecmp ("From ", s, 5) == 0))
      return (1);
    t = t->next;
  }
  return 0;
}

char *mutt_expand_path (char *s, size_t slen)
{
  char p[_POSIX_PATH_MAX] = "";

  if (*s == '~')
  {
    if (*(s+1) == '/' || *(s+1) == 0)
      snprintf (p, sizeof (p), "%s%s", Homedir, s+1);
    else
    {
      char *q = strchr (s + 1, '/');
      struct passwd *pw;

      if (q)
	*q++ = 0;
      pw = getpwnam (s + 1);
      if (pw)
	snprintf (p, sizeof (p), "%s/%s", pw->pw_dir, q ? q : "");
    }
  }
  else
  {
    if (*s == '=' || *s == '+')
      snprintf (p, sizeof (p), "%s/%s", Maildir, s+1);
    else
    {
      if (*s == '>')
	strfcpy (p, Inbox, sizeof (p));
      else if (*s == '<')
	strfcpy (p, Outbox, sizeof (p));
      else if (*s == '!')
	strfcpy (p, Spoolfile, sizeof (p));
      else
	return (s);
    }
    mutt_expand_path (p, sizeof (p));
  }
  strfcpy (s, p, slen); /* replace the string with the expanded version. */
  return (s);
}

/* returns TRUE if the given address belongs to the user. */
int mutt_addr_is_user (ADDRESS *addr)
{
  ADDRESS *a = Alternates;
  int l;

  /* NULL address is assumed to be the user. */
  if (!addr)
    return 1;

  if (!addr->mailbox)
    return 0;

  if (!addr->host || *addr->host == '@')
  {
    if (strcasecmp (addr->mailbox, Username) == 0)
      return 1;
    return 0;
  }

  if (strcasecmp (addr->mailbox, Username) == 0 &&
      (strcasecmp (addr->host, Hostname) == 0 ||
       strcasecmp (addr->host, Fqdn) == 0))
    return 1;

  l = strlen (addr->host);

  while (a)
  {
    if (strcasecmp (addr->mailbox, a->mailbox) == 0)
    {
      if (a->host[0] == '*')
      {
	int r = strlen (&a->host[1]);
	
	if ((l >= r) && (strcasecmp (&addr->host[l-r], &a->host[1]) == 0))
	  return 1;
      }
      else if (strcasecmp (addr->host, a->host) == 0)
	return 1;
    }
    a = a->next;
  }
  return (0);
}

void *safe_malloc (unsigned int siz)
{
  void *p;

  if (siz == 0) return 0;
  if ((p = (void *)malloc(siz)) == 0)
  {
    mutt_error("Out of memory!");
    sleep(1);
    mutt_exit(1);
  }
  return(p);
}

void safe_realloc (void **p, unsigned int siz)
{
  void *r;

  if (siz == 0)
  {
    safe_free((void **)&*p);
    return;
  }
  if ((r = (void *)realloc(*p, siz)) == 0)
  {
    mutt_error("Out of memory!");
    sleep(1);
    mutt_exit(1);
  }
  *p = r;
}

void safe_free (void **p)
{
  if (!*p) return;
  free(*p);
  *p = 0;
}

char *safe_strdup (const char *s)
{
  char *p;

  if (!s || !*s) return 0;
  p = (char *)safe_malloc(strlen(s)+1);
  strcpy(p, s);
  return(p);
}

char *skip_whitespace (char *p)
{
  while (*p && (*p == ' ' || *p == '\t')) p++;
  return(p);
}

int mutt_copy_bytes (FILE *in, FILE *out, long size)
{
  char buffer[LONG_STRING];
  long chunk;

  while (size > 0)
  {
    chunk = (size > sizeof (buffer)) ? sizeof (buffer) : size;
    if ((chunk = fread (buffer, 1, chunk, in)) < 1) break;
    if (fwrite (buffer, 1, chunk, out) != chunk)
    {
      dprint (1, (debugfile, "mutt_copy_bytes(): fwrite() returned short byte count\n"));
      return (-1);
    }
    size -= chunk;
  }

  return 0;
}

ADDRESS *mutt_new_address (void)
{
  ADDRESS *p = (ADDRESS *)safe_malloc(sizeof(ADDRESS));
  
  memset (p, 0, sizeof(ADDRESS));
  return (p);
}

void mutt_free_address (ADDRESS **p)
{
  ADDRESS *t;
  ADDRESS *o;

  if (!*p) return;
  t = *p;
  while (t)
  {
    safe_free((void **)&t->personal);
    safe_free((void **)&t->mailbox);
    safe_free((void **)&t->adl);
    safe_free((void **)&t->host);
    safe_free((void **)&t->error);
    o = t;
    t = t->next;
    safe_free((void **)&o);
  }
  *p = 0;
}

char *mutt_get_parameter (const char *s, PARAMETER *p)
{
  for (; p; p=p->next)
    if (strcasecmp(s, p->attribute) == 0)
      return(p->value);
  return(0);
}

/* returns 1 if Mutt can't display this type of data, 0 otherwise */
int mutt_needs_mailcap (BODY *m)
{
  switch (m->type)
  {
    case TYPETEXT:
      if (strcasecmp ("plain", m->subtype) == 0)
	return 0;
      break;

    case TYPEMULTIPART:
    case TYPEMESSAGE:
      return 0;
  }

  return 1;
}

int mutt_is_text_type (int t, char *s)
{
  if (t == TYPETEXT)
    return 1;
  if (t == TYPEMESSAGE)
  {
    if (strcasecmp("delivery-status", s) == 0)
      return 1;
  }
  return 0;
}

ENVELOPE *mutt_new_envelope (void)
{
  ENVELOPE *p = (ENVELOPE *)safe_malloc(sizeof(ENVELOPE));

  memset(p, 0, sizeof(ENVELOPE));
  return(p);
}

void mutt_free_envelope (ENVELOPE **p)
{
  if (!*p) return;
  mutt_free_address(&(*p)->to);
  mutt_free_address(&(*p)->cc);
  mutt_free_address(&(*p)->bcc);
  mutt_free_address(&(*p)->sender);
  mutt_free_address(&(*p)->from);
  mutt_free_address(&(*p)->reply_to);
  safe_free((void **)&(*p)->subject);
  safe_free((void **)&(*p)->message_id);
  mutt_free_list(&(*p)->references);
  mutt_free_list(&(*p)->userhdrs);
  safe_free((void **)p);
}

void mutt_tabs_to_spaces (char *s)
{
  while (*s)
  {
    if (*s == '\t')
      *s=' ';
    s++;
  }
}

void mutt_mktemp (char *s)
{
  char buf[_POSIX_PATH_MAX];

  strfcpy (buf, Tempdir, sizeof (buf));
  mutt_expand_path (buf, sizeof (buf));
  sprintf (s, "%s/muttXXXXXX", buf);
  mktemp (s);
}

/* convert all characters in the string to lowercase */
char *mutt_strlower (char *s)
{
  char *p = s;

  while (*p)
  {
    *p = tolower (*p);
    p++;
  }
  return (s);
}

/* a wrapper around rfc822_write_address() to prevent buffer overrun */
void mutt_write_address (char *buf, size_t buflen, ADDRESS *addr)
{
  char tmp[LONG_STRING]; /* should be long enough to hold a single address */
  ADDRESS *tmpaddr;
  int len;

  buflen--; /* save room for the terminal \0 */

  /* move past what is already in the buf */
  if (buf[0])
  {
    len = strlen (buf);
    buflen -= len;
    buf += len;

    if (buflen < 2) return; /* not enough room! */
    *buf++ = ',';
    *buf++ = ' ';
    buflen -= 2;
  }

  while (addr)
  {
    tmpaddr = addr->next;
    addr->next = NULL;
    tmp[0] = 0;
    rfc822_write_address(tmp, addr);
    /* add a comma to separate the addresses if this is not a group name */
    if (tmpaddr && tmpaddr->host && addr->host)
    {
      strcat(tmp, ", ");
    }
    addr->next = tmpaddr;
    addr = addr->next;
    len = strlen (tmp);
    if (len < buflen)
    {
      strcpy(buf, tmp);
      buf += len;
      buflen -= len;
    }
    else
      return; /* no more room */
  }
}

/* a wrapper around rfc822_parse_adrlist to check for errors */
int mutt_parse_adrlist (ADDRESS **a, char *s, char *host)
{
  ADDRESS *p, *t = 0;
  char tmp[HUGE_STRING], *cp;

  ParseError[0] = 0;
  strfcpy (tmp, s, sizeof (tmp)); /* make a copy of the list */

  /* remove trailing whitespace and commas */
  cp = tmp + strlen (tmp) - 1;
  while (cp >= tmp && (isspace (*cp) || *cp == ','))
    *cp-- = 0;

  rfc822_parse_adrlist (&t, tmp, host);
  if (ParseError[0])
  {
    mutt_free_address (&t);
    return (-1);
  }
  else
  {
    /* append the new addresses to the list */
    if (*a)
    {
      p = *a;
      while (p->next) p = p->next;
      p->next = t;
    }
    else
      *a = t;
    return 0;
  }
  /* not reached */
}

int mutt_is_valid_mailbox (const char *path)
{
  FILE *fp;
  char buf[SHORT_STRING];
  int rc = 0;

  if (access (path, R_OK|F_OK) != 0)
  {
    snprintf (buf, sizeof (buf), "access: %s", path);
    mutt_perror (buf);
    return (0);
  }
  if ((fp = fopen (path, "r")) == NULL)
    return (0);
  if (fgets (buf, sizeof (buf), fp) == NULL)
    /* empty mailbox, trust the user that this is ok */
    rc = 1;
  else if (strncmp ("From ", buf, 5) == 0) /* MBOX */
    rc = 1;
  else
    mutt_error ("%s does not appear to be a mailbox.", path);
  fclose (fp);
  return (rc);
}

/* write an address without the personal name */
void mutt_simple_address (char *buf, size_t buflen, ADDRESS *addr)
{
  int len;
  
  *buf = 0;
  if (!addr || !addr->mailbox)
    return;
  strfcpy (buf, addr->mailbox, buflen);
  len = strlen (buf);
  buflen -= len;
  buf += len;
  if (addr->host && *addr->host != '@')
    snprintf (buf, buflen, "@%s", addr->host);
}

LIST *mutt_new_list (void)
{
  LIST *p = safe_malloc (sizeof(LIST));

  memset (p, 0, sizeof(LIST));
  return p;
}

int is_local_site (const char *s)
{
  LIST *tmp = Localsites;
  const char *a;
  char *b;
  int len;
  
  if (!s || *s == '@' || strcasecmp (s, Fqdn) == 0)
    return 1;
  while (tmp)
  {
    if (*tmp->data == '*')
    {
      if ((len = strlen (s) - strlen (tmp->data+1)) < 0)
	len = 0;
      a = s + len;
      b = tmp->data + 1;
    }
    else
    {
      a = s;
      b = tmp->data;
    }
    if (a && b && strcasecmp(a, b) == 0) return 1;
    tmp = tmp->next;
  }
  return 0;
}

int mutt_system (const char *s)
{
  int rc;
#ifndef POSIX_SYSTEM
  struct sigaction act;
  struct sigaction oldint;
  struct sigaction oldchld;
#endif

#ifndef POSIX_SYSTEM
  act.sa_handler = SIG_IGN;
  /* must ignore SIGINT and SIGCHLD (POSIX.1 compliance) */
  sigaction (SIGINT, &act, &oldint);
  sigaction (SIGCHLD, &act, &oldchld);
#endif

  rc = system (s);

#ifndef POSIX_SYSTEM
  /* reset SIGINT and SIGCHLD */
  sigaction (SIGINT, &oldint, NULL);
  sigaction (SIGCHLD, &oldchld, NULL);
#endif

  return (rc);
}

void mutt_free_alias (ALIAS **p)
{
  ALIAS *t;

  while (*p)
  {
    t = *p;
    *p = (*p)->next;
    safe_free ((void **)&t->name);
    mutt_free_address (&t->addr);
    free (t);
  }
}
