/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "keymap.h"

#include <unistd.h>
#include <string.h>
#include <sys/stat.h>

static struct {
  CONTEXT *context;
  int msgcount;
  time_t last_modify;
} LOCAL = { 0, 0, 0 };

int mutt_num_postponed (void)
{
  struct stat st;
  char path[_POSIX_PATH_MAX];

  strfcpy (path, Postponed, sizeof (path));
  mutt_expand_path (path, sizeof (path));
  if (stat (path, &st) == -1)
  {
     LOCAL.msgcount = 0;
     LOCAL.last_modify = 0;
     return (0);
  } 

  if (LOCAL.last_modify < st.st_mtime)
  {
    LOCAL.last_modify = st.st_mtime;

    if (access (path, R_OK|F_OK) != 0)
    {
      LOCAL.msgcount = 0;
      return (LOCAL.msgcount);
    }
    if ((LOCAL.context = mutt_open_mailbox (path, NULL, M_NOSORT)) == 0)
      LOCAL.msgcount = 0;
    else
      LOCAL.msgcount = LOCAL.context->msgcount;
  }

  return LOCAL.msgcount;
}

void mutt_postpone_message (const char *tempfile,
			    ENVELOPE *env,
			    const char *msgid,
			    BODY *attach)
{
  time_t now;
  FOLDER *f;
  int len;
  char buf[LONG_STRING];
  FILE *t;

  strfcpy (buf, Postponed, sizeof (buf));
  mutt_expand_path (buf, sizeof (buf));
  if (!(f = folder_open (buf, "a")))
    return;
  if (lock_file (f, TRUE) != 0)
  {
    folder_close (&f);
    mutt_error ("Could not lock %s!", buf);
    return;
  }
  fseek (f->fp, 0, 2);
  
  now = time (0);
  fprintf (f->fp, "From %s %s", Username, ctime(&now));
  mutt_write_rfc822_header (f->fp, env, NULL, -1);

  /*
   * if this was a reply of some sort, <msgid> contians the
   * Message-ID: of message replied to.  Save it using a special X-Mutt-
   * header so it can be picked up if the message is recalled at a later
   * point in time.  This will allow the message to be marked as replied if
   * the same mailbox is still open.
   */
  if (msgid) fprintf(f->fp, "X-Mutt-References: %s\n", msgid);

  /*
   * Save the attachments to this message using
   * X-Mutt-Attachment: <filename> [ <description> ]
   */
  while (attach)
  {
    fprintf(f->fp, "X-Mutt-Attachment: %s", attach->filename);
    if (attach->description)
    {
      fputc(' ', f->fp);
      fputs(attach->description, f->fp);
    }
    fputc('\n', f->fp);
    attach = attach->next;
  }

  fputc('\n', f->fp); /* finish off the header. */
  if ((t = fopen(tempfile, "r")) != 0)
  {
    while ((len = fread(buf, 1, sizeof(buf), t)) > 0)
      fwrite(buf, 1, len, f->fp);
    fclose(t);  
    fputc('\n', f->fp);
  }
  unlink(tempfile);
  unlock_file(f);
  folder_close(&f);
}

static void post_make_entry (char *s, int entry, size_t slen)
{
  char buf[SHORT_STRING] = { 0 };

  mutt_write_address(buf, sizeof(buf), LOCAL.context->hdrs[entry]->env->to);
  snprintf(s, slen, "%2d  %-20.20s     %s", entry+1, buf,
	   LOCAL.context->hdrs[entry]->env->subject);
}

static HEADER *select_msg (void)
{
  MUTTMENU *menu;
  int i, done=0, r=-1;

  menu = mutt_newMenu ("---Postponed Messages");
  menu->makeEntry = post_make_entry;
  menu->menu = MENU_POST;
  menu->max = LOCAL.context->msgcount;

  while (!done)
  {
    switch (i = mutt_menuLoop (menu))
    {
      case OP_DELETE:

	LOCAL.context->hdrs[menu->current]->deleted = 1;
	LOCAL.context->hdrs[menu->current]->changed = 1;
	break;

      case OP_UNDELETE:

	LOCAL.context->hdrs[menu->current]->deleted = 0;
	LOCAL.context->hdrs[menu->current]->changed = 0;
	break;

      case OP_GENERIC_SELECT_ENTRY:

	r = menu->current;
	done = 1;
	break;
      case OP_GENERIC_EXIT:

	done = 1;
	break;
    }
  }

  mutt_menuDestroy (&menu);
  return (r > -1 ? LOCAL.context->hdrs[r] : NULL);
}

int ci_get_postponed_message (FILE *temp, ENVELOPE **env, HEADER **cur,
			      BODY **attach)
{
  HEADER *h;
  BODY *ptr = NULL;
  int code = 0;
  char *p;
  LIST *tmp;
  LIST *last = NULL;
  LIST *next;
  char buf[_POSIX_PATH_MAX];

  *cur = NULL;
  *attach = NULL;

  strfcpy (buf, Postponed, sizeof (buf));
  mutt_expand_path (buf, sizeof (buf));
  if ((LOCAL.context = mutt_open_mailbox (buf, NULL, M_NOSORT)) == NULL)
  {
    mutt_error ("No postponed messages.");
    return (-1);
  }
  
  if (LOCAL.context->msgcount == 1)
  {
    /* only one message, so just use that one. */
    h = LOCAL.context->hdrs[0];
  }
  else if ((h = select_msg ()) == NULL)
  {
    mutt_close_mailbox (LOCAL.context);
    safe_free ((void **)&LOCAL.context);
    return (-1);
  }

  fseek (LOCAL.context->folder->fp, h->offset, 0);
  *env = read_rfc822_header (LOCAL.context->folder->fp, NULL);
  mutt_copy_bytes (LOCAL.context->folder->fp, temp, h->content->length);

  /* finished with this message, so delete it. */
  h->deleted = 1;
  h->changed = 1;

  /* update the count for the status display */
  LOCAL.msgcount = LOCAL.context->msgcount - 1;

  mutt_close_mailbox (LOCAL.context);
  safe_free ((void **)&LOCAL.context);

  tmp = (*env)->userhdrs;
  while (tmp)
  {
    if (Context && strncmp ("X-Mutt-References:", tmp->data, 18) == 0)
    {
      int j;

      p = skip_whitespace(tmp->data + 18);
      for (j=0; j < Context->msgcount; j++)
      {
	if (Context->hdrs[j]->env->message_id &&
	    strcmp (Context->hdrs[j]->env->message_id, p) == 0)
	{
	  *cur = Context->hdrs[j];
	  break;
	}
      }

      /* Remove the X-Mutt-References: header field. */
      next = tmp->next;
      if (last)
	last->next = tmp->next;
      else
	(*env)->userhdrs = tmp->next;
      tmp->next = NULL;
      mutt_free_list (&tmp);
      tmp = next;
      if (*cur)
	code = SENDREPLY;
    }
    else if (strncmp ("X-Mutt-Attachment:", tmp->data, 18) == 0)
    {
      char *q;

      p = skip_whitespace(tmp->data+18);
      if ((q = strchr(p, ' ')) != NULL) *q++ = 0;
      if (*attach)
      {
	ptr->next = mutt_make_attach(p);
	if (ptr->next) ptr = ptr->next;
      }
      else
	*attach = ptr = mutt_make_attach(p);
      if (q)
	ptr->description = safe_strdup (q);

      /* remove the x-mutt-attachment field */
      next = tmp->next;
      if (last)
	last->next = tmp->next;
      else
	(*env)->userhdrs = tmp->next;
      tmp->next = NULL;
      mutt_free_list(&tmp);
      tmp = next;
    }
    else
    {
      last = tmp;
      tmp = tmp->next;
    }
  }
  return code;
}
