/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "keymap.h"

#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>

struct attachptr {
  BODY *content;
  struct attachptr *next;
};

struct attachptr **idx = NULL;

static struct attachptr *gen_attach_list (BODY *m)
{
  struct attachptr *top = NULL, *end = NULL, *new;

  while (m)
  {
    if (m->type == TYPEMULTIPART)
    {
      if (top)
        end->next = gen_attach_list(m->parts);
      else
        end = top = gen_attach_list(m->parts);
      while (end->next) end = end->next;
    }
    else
    {
      new = (struct attachptr *)safe_malloc(sizeof(struct attachptr));
      new->next = NULL;
      new->content = m;

      if (top)
      {
        end->next = new;
        end = new;
      }
      else
        top = end = new;

      if (m->type == TYPEMESSAGE && !strcasecmp(m->subtype, "rfc822") &&
	  is_multipart(m->parts))
      {
	end->next = gen_attach_list(m->parts);
	while (end->next) end = end->next;
      }
    }
    m = m->next;
  }
  return (top);
}

static int can_print (BODY *m)
{
  if (m->type == TYPETEXT)
  {
    if (strcasecmp("plain", m->subtype) == 0) return 1;
  }
  else if (m->type == TYPEAPPLICATION)
  {
    if (strcasecmp("postscript", m->subtype) == 0) return 1;
  }
  return 0;
}

static int da_parse_mailcap (char *type, char *command, int clen) 
{
  char buf[LONG_STRING];
  FILE *fp = NULL;
  int found = FALSE;
  int wrap = FALSE;
  int x = 0;
  int y = 0;
  char *ch;

  /* First try MAILCAP environment variable, then ~/.mailcap, then
   * regular mailcap location (/usr/local/etc/mailcap?)
   */
  if ((fp = fopen (getenv ("MAILCAP"), "r")) == NULL)
  {
    strcpy (buf,"~/.mailcap");
    mutt_expand_path (buf, sizeof (buf));
    if ((fp = fopen (buf,"r")) == NULL)
    {
      /* This should use $(sharedir) from Makefile, but no easy way to 
       * get it here . . .
       */
      strcpy(buf,"/usr/local/etc/mailcap");
      if ((fp = fopen(buf,"r")) == NULL)
      {
	mutt_error("No mailcap file found");
	return 0;
      }
    }
  }
  /* mailcap file (as near as I can tell) is of the format:
   * base/type; command; extradefs
   * where type can be * for matching all
   * where command contains a %s for the filename to pass, default to append
   * where extradefs are of the form:
   *  def1="definition"; def2="define \;";
   * hmm, looks like you can use a \ line continuation as well
   * oh, and # for comments
   */
  while (!found && fgets(buf,sizeof(buf),fp))
  {
    /* Strip comments (ie, everything after #, but not \# */
    if ((ch = strchr(buf,'#')))
      if (*(ch-1) != '\\') *ch = '\0';

    /* Strip trailing white space */
    y = strlen(buf);
    y = (y ? y - 1 : 0);
    while (buf[y] == ' ' || buf[y] == '\t' || buf[y] == '\n') buf[y--] = '\0';

    if (buf[y] == '\\') wrap = TRUE;
    else wrap = FALSE;
    while (wrap && fgets(buf + strlen(buf) - 1, sizeof(buf) - strlen(buf),fp)) 
    {
      if ((ch = strchr(buf,'#')))
	if (*(ch-1) != '\\') *ch = '\0';            
      y = strlen(buf);
      y = (y ? y - 1 : 0);
      while (buf[y] == ' ' || buf[y] == '\t' || buf[y] == '\n') buf[y--] = '\0';
      if (buf[y] == '\\') wrap = TRUE; 
      else wrap = FALSE; 
    }
    dprint(2,(debugfile,"mailcap entry: %s\n",buf));

    if (!strncmp(buf,type,strlen(type)) && (buf[strlen(type)] == ';'))
      found = TRUE;
    else
    {
      /* Check * case */
      char basetype[STRING];

      x = 0;
      while (type[x] && basetype[x] != '/') basetype[x] = type[x++];
      basetype[x] = '\0';
      if (!strncmp(buf,basetype,x-1)) 
	if (buf[x] && buf[x+1] && buf[x+1] == '*') found = TRUE;
    }
  } 
  fclose(fp);

  if (found)
  {
    char *tok = NULL;

    /* hmm, I'll try this strtokq thing */
    tok = strtokq (buf, ";");
    tok = strtokq (NULL, ";");
    if (strlen (tok) < clen)
      strcpy (command,tok);
    else
    {
      strncpy(command,tok,clen);
      command[clen-1] = '\0';
    }
  }
  else
    mutt_error ("mailcap entry for type %s not found", type);
  return found;
}

/* Modified by blong to accept a "suggestion" for file name.  If
 * that file exists, then construct one with unique name but 
 * keep any extension.  This might fail, I guess.
 * Renamed to mutt_adv_mktemp so I only have to change where its
 * called, and not all possible cases.
 */
void mutt_adv_mktemp (char *s)
{
  char buf[_POSIX_PATH_MAX];
  char tmp[_POSIX_PATH_MAX];
  char *period;

  strfcpy (buf, Tempdir, sizeof (buf));
  mutt_expand_path (buf, sizeof (buf));
  if (s[0] == '\0')
  {
    sprintf (s, "%s/muttXXXXXX", buf);
    mktemp (s);
  }
  else
  {
    strfcpy(tmp, s, sizeof (tmp));
    sprintf(s, "%s/%s", buf, tmp);
    mktemp(s); 
    if (s[0] != '\0') return;
    period = strrchr(tmp,'.');
    if (period != NULL) *period = '\0';
    sprintf(s, "%s/%s.XXXXXX",buf,tmp);
    mktemp(s);
    if (period != NULL)
    {
      *period = '.';
      strcat(s,period);
    }
  }
}

void mutt_display_attachment (BODY *a, int type)
{
  STATE *s;
  char tempfile[_POSIX_PATH_MAX] = "";
  char descrip[SHORT_STRING];

  if (a->filename) strfcpy (tempfile, a->filename, sizeof (tempfile));
  mutt_adv_mktemp (tempfile);

  if (type == M_MAILCAP || (type == M_REGULAR && mutt_needs_mailcap (a)))
  {
    char buffer[STRING];
    char type[STRING];
    char command[STRING];

    snprintf (type, sizeof (type), "%s/%s", TYPE (a->type), a->subtype);
    if (da_parse_mailcap (type, command, sizeof (command)))
    {
      mutt_save_attachment (a, tempfile);
      if (!strstr (command,"%s"))
	snprintf (buffer, sizeof (buffer), "%s %s", command, tempfile);
      else
	snprintf (buffer, sizeof (buffer), command, tempfile);
      endwin();
      mutt_system (buffer);
    }
    else
    {
      /* no mailcap type defined */
    }
    mutt_unlink (tempfile);
    return;
  }
  else
  {
    s = mutt_new_state ();
    s->fpout = fopen (tempfile, "w");
    s->fpin = Context->folder->fp;
    s->displaying = 1;
    mutt_body_handler (a, s);
    fclose (s->fpout);
    mutt_free_state (&s);

     if (a->description)
       strfcpy (descrip, a->description, sizeof (descrip));
     else
       snprintf (descrip, sizeof (descrip),
		 "---Attachment: %s/%s", TYPE (a->type), a->subtype);
     mutt_do_pager (descrip, tempfile);
   }
}

static void pipe_attachment (BODY *b, const char *path)
{
  STATE s;

  memset (&s, 0, sizeof (STATE));
  s.fpin = Context->folder->fp;
  endwin ();
  mutt_create_filter (path, &s.fpout, NULL);
  mutt_body_handler (b, &s);
  fclose (s.fpout);
  wait (NULL);
  ci_any_key_to_continue (NULL);
}

void attach_makeEntry (char *b, int num, size_t blen)
{
  char t[SHORT_STRING];
  BODY *m = idx[num]->content;

  snprintf (t, sizeof(t), "%s/%s", TYPE(m->type), m->subtype);
  snprintf (b, blen, " %2d %-20.20s  %-6.6s (%6ld) %s", num+1, t,
	    ENCODING(m->encoding), m->length,
	    m->description ? m->description : (m->filename ? m->filename : ""));
}

void ci_attach (BODY *cur)
{
  struct attachptr *list, *plist;
  int idxlen = 0, count = 0;
  char buffer[STRING];
  MUTTMENU *menu;

  plist = list = gen_attach_list(cur);
  idx = safe_malloc (sizeof (struct attachptr *) * ((idxlen = 5)));
  idxlen = 5;

  for (count=0; plist; count++, plist = plist->next)
  {
    if (count == idxlen)
      safe_realloc ((void **)&idx, ((idxlen += 5)) * sizeof(struct attachptr *));
    idx[count] = plist;
  }

  snprintf (buffer, sizeof (buffer), "---Mutt: Attachment Mode [%d attachment%s]", count, count != 1 ? "s" : "");
  menu = mutt_newMenu (buffer);
  menu->max = count;
  menu->makeEntry = attach_makeEntry;
  menu->menu = MENU_ATTACH;

  FOREVER
  {
    switch (mutt_menuLoop (menu))
    {
      case OP_GENERIC_SELECT_ENTRY:

	mutt_display_attachment (idx[menu->current]->content, M_REGULAR);
	menu->redraw = REDRAW_FULL;
	break;

      case OP_ATTACH_VIEW_MAILCAP:

	mutt_display_attachment (idx[menu->current]->content,  M_MAILCAP);
	menu->redraw = REDRAW_FULL;
	break;

      case OP_ATTACH_VIEW_TEXT:

	mutt_display_attachment (idx[menu->current]->content,  M_AS_TEXT);
	menu->redraw = REDRAW_FULL;
	break;

      case OP_PRINT:

	if (can_print (idx[menu->current]->content))
	  pipe_attachment (idx[menu->current]->content, Print);
	else
	{
	  beep ();
	  mutt_error ("I don't know how to print that!");
	}
	break;

      case OP_ATTACH_PIPE_ENTRY:

	if (ci_get_field ("Pipe to: ", buffer, sizeof (buffer), 0) == 0 &&
	    buffer[0])
	{
	  mutt_expand_path (buffer, sizeof (buffer));
	  pipe_attachment (idx[menu->current]->content, buffer);
	  ci_any_key_to_continue (NULL);
	}
	break;

      case OP_ATTACH_SAVE_ENTRY:

	if (idx[menu->current]->content->filename)
	  strfcpy(buffer, idx[menu->current]->content->filename, sizeof(buffer));
	else
	  buffer[0] = 0;
	if (ci_get_field ("Save to file: ", buffer, sizeof (buffer), M_FILE) != 0 ||
	    !buffer[0])
	  break;
	mutt_expand_path (buffer, sizeof (buffer));
	if (mutt_save_attachment (idx[menu->current]->content, buffer) == 0)
	  mutt_error ("Attachment saved.");
	else
	{
	  beep ();
	  mutt_error ("Error while saving attachment!");
	}
	break;

      case OP_GENERIC_EXIT:

	while (count-- > 0) safe_free((void **)&idx[count]);
	safe_free ((void **)&idx);
	mutt_menuDestroy (&menu);
	return;
    }
  }
  /* not reached */
}
