/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)skip_machdep.C	1.25 96/10/02"

/*
 * System includes
 */
#include <skip_os.h>

/*
 * SKIP includes
 */
#include <skip_proto.h>
#include "skip_keymgr.h"
#include <skip_ioctl.h>
#include <skip_lib.h>
#include "skip_machdep.h"
#include "skip_log.h"
#include "cdp_queue.h"


void skip_machdep_init()
{
	struct rlimit r;

	/* Don't dump core since core files may contain secret data */

	r.rlim_cur = 0;
	r.rlim_max = 0;
	(void) setrlimit(RLIMIT_CORE, &r);
	(void) chdir(SKIP_DIR);
}

	
int open_kernel()
{
	int fd;

	fd = skip_open_device(KEYMGR);

	if (fd < 0) {
		skip_log(SKIP_ERROR, "open of %s failed: %s",
					KEYMGR, strerror(errno));
		exit(1);
	}

	return fd;
}

int open_socket(int port, int quiet)
{
	int sock;
	struct sockaddr_in name;
	int rbufsize = 65535;

	sock = socket(AF_INET, SOCK_DGRAM, 0);

	if (sock < 0 ) {
		if (!quiet)
			skip_log(SKIP_ERROR, "socket() failed: %s",
							strerror(errno));
		return -1;
	}

	(void) setsockopt(sock, SOL_SOCKET, SO_RCVBUF, 
			 (char *)&rbufsize, sizeof(rbufsize));

	name.sin_family = AF_INET;
	name.sin_addr.s_addr = INADDR_ANY;
	name.sin_port = htons(port);

	if (bind(sock, (struct sockaddr *)&name, sizeof (name)))  {
		if (!quiet)
			skip_log(SKIP_ERROR, "bind failed: %s",
							strerror(errno));
		close(sock);
		return -1;
	}

	return sock;
}

int put_kernel_message(int fd, void *buf, int buflen)
{
	int ret;

        ret = skip_putmsg(fd,  buf, buflen);

	if (ret < 0) {
		skip_log(SKIP_ERROR, "put_kernel_message error, ret=%d", ret);
		return -1;
	}

	return ret;
}

int get_kernel_message(int fd, void *buf, int buflen )
{
	int ret;

        ret = skip_getmsg(fd,  buf, buflen);

	if (ret < 0) {
		skip_log(SKIP_ERROR, "get_kernel_message error, ret=%d", ret);
		return -1;
	}

	return ret;
}


//  Wrapper for the select() system call
//
//  Arguments:
//	keyfd		The file descriptor we receive kernel requests on
//	certd_fd	The file descriptor we as a certificate daemon
//			receive CDP requests on
//	cdp_q		Set of ports we are waiting for answers to our own
//			CDP queries on
//
//  Returns:
//	-1	select timed out but no fd's need to be read
//	fd	a file descriptor ready to be read
//
//  Timeouts:
//
//	select_wrapper should not block indefinitely, since on timeouts
//	we manage the cdp_queue, and do housekeeping functions such as
//	kij cache and database cleanup.
//
//	If the cdp_queue is not empty, select_wrapper should timeout
//	after 1 second so that we may service requests in the queue.
//
//	Otherwise, select_wrapper should timeout after 60 seconds.
//
//  Priority:
//
//	File descriptors indicated by select() to be ready are considered
//	in the following order:
//
//		1.  Requests from the kernel.
//		2.  Answers to our own CDP queries
//		3.  Requests to our CDP server port

int select_wrapper(int keyfd, int certd_fd, cdp_queue& cdp_q)
{
	int n;
	fd_set select_fds;
	struct timeval timeout;
	int max_fd = keyfd;
	cdp_queue_entry *p;
	int cdp_queue_empty;

retry:
	FD_ZERO(&select_fds);

	if (keyfd >= 0)
		FD_SET(keyfd, &select_fds);
	if (certd_fd >= 0)
		FD_SET(certd_fd, &select_fds);

	if (certd_fd > max_fd)
		max_fd = certd_fd;

	cdp_queue_empty = 1;

	for (p = cdp_q.first(); p; p = cdp_q.next()) {
		FD_SET(p->request_fd, &select_fds);
		if (p->request_fd > max_fd)
			max_fd = p->request_fd;

		cdp_queue_empty = 0;
	}

	if (cdp_queue_empty) {
		timeout.tv_sec = 60;
		timeout.tv_usec = 0;
	} else {
		timeout.tv_sec = 1;
		timeout.tv_usec = 0;
	}

	if (max_fd < 0) {
		skip_log(SKIP_ERROR, "select_wrapper: nothing to select on");
		exit(1);
	}

	n = select(max_fd + 1, &select_fds, 0, 0, &timeout);

//  XXX
//  what does alarm() do to select?
//  perhaps this should be a return -1 so the reinit check can be done

	if (n < 0) {			// select error
		skip_log(SKIP_ERROR, "select() failed: %s", strerror(errno));
		goto retry;
	}

	if (n == 0)			// timeout
		return -1;

	if (keyfd >= 0 && FD_ISSET(keyfd, &select_fds))
		return keyfd;

	for (p = cdp_q.first(); p; p = cdp_q.next()) {
		if (FD_ISSET(p->request_fd, &select_fds))
			return p->request_fd;
	}

	if (certd_fd >= 0 && FD_ISSET(certd_fd, &select_fds))
		return certd_fd;

	skip_log(SKIP_ERROR, "select() returned an unknown file descriptor");

	goto retry;
}

int parse_ip_address(ip_address *p, char *arg)
{
	if (isdigit(*arg)) 
		p->s_addr = inet_addr(arg);
	else {
		struct hostent *h;
		char **q;

		h = gethostbyname(arg);

		if (h == NULL) {
			skip_log(SKIP_ERROR, "Host lookup of %s failed", arg);
			return -1;
		}

		q = h->h_addr_list;
		bcopy(*q, p, h->h_length);
		p->s_addr = p->s_addr;
	}

	return 0;
}

char sock_info_str[256];
struct sockaddr_in sock_partner;

//  set_sock_partner()
//
//  p == NULL		any addr
//  port == 0		any port

void set_sock_partner(ip_address *p, int port)
{

	if (p == NULL)
		sock_partner.sin_addr.s_addr = INADDR_ANY;
	else
		sock_partner.sin_addr.s_addr = p->s_addr;

	sock_partner.sin_family = AF_INET;
	sock_partner.sin_port = htons(port);
}

Bstream read_sock(int sock)
{
	char buf[65535];
	int len = sizeof(sock_partner);
	int n;

	n = recvfrom(sock, buf, sizeof(buf), 0,
			(struct sockaddr *)&sock_partner, &len);

	if (n < 0) {
		skip_log(SKIP_ERROR, "read_sock: %s", strerror(errno));
		sock_info_str[0] = '\0';
		return Bstream();
	}

	sprintf(sock_info_str, "IP %s:%d",
			inet_ntoa(sock_partner.sin_addr),
			ntohs(sock_partner.sin_port));

	return Bstream(n, buf);
}

void write_sock(int sock, const Bstream& data)
{
	int len = sizeof(sock_partner);
	int n;

	n = sendto(sock, (const char *)data.getdatap(),
		       data.getlength(), 0, 
		       (struct sockaddr *)&sock_partner, len);

	if (n < 0)
		skip_log(SKIP_ERROR, "sendto: %s", strerror(errno));
}

char *get_ip_str(ip_address *p)
{
	static char buf[256];

	sprintf(buf, "%s", inet_ntoa(*p));

	return buf;
}
