/*

    ``gpgp'' Gnome/GTK Front for PGP
    Copyright (C) 1998  Max Valianskiy

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: main.c,v 1.3 1998/09/16 19:18:14 maxcom Stab $ 

*/

#include "../config.h"
#include "../version.h"

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include "gpgp.h"

void about_cb (GtkWidget *widget, void *data);
void quit_cb (GtkWidget *widget, void *data);
void refresh_cb (GtkWidget *widget, void *data);
void unempl_cb (GtkWidget *widget, void *data);
void sign_cb (GtkWidget *widget, void *data);
void sign2_cb (GtkWidget *widget, void* data);
void sign_cancel_cb (GtkWidget *widget, void *data);
void close_cb (GtkWidget *widget, void *data);
void open_cb (GtkWidget *widget, void *data);
void new_cb (GtkWidget *widget, void *data);
void prepare_app();
void lib_properties_cb (GtkWidget *widget, void *data);
void report_error(GtkWidget* widget, void* data);
void addring(gpointer data, gpointer user_data);
void sign_destroy_cb (GnomePgpSign* widget, char* data);
void destroy_cb (GtkWidget *widget, gpointer data);
void unsign_cb (GtkWidget *widget, void *data);
void unsign2_cb (GtkWidget *widget, void* data);
void unsign_cancel_cb (GtkWidget *widget, void *data);
void unsign_destroy_cb (GnomePgpUnSign* widget, char* data);

static int save_state      (GnomeClient        *client,
                            gint                phase,
                            GnomeRestartStyle   save_style,
                            gint                shutdown,
                            GnomeInteractStyle  interact_style,
                            gint                fast,
                            gpointer            client_data);
static void connect_client (GnomeClient *client,
                            gint         was_restarted,
                            gpointer     client_data);

void discard_session (gchar *id);

struct options
   {
    GList* rings; /* rings to open */
    int x,y,h,w; /* window geometry */
   } options;

GnomeApp *app;

int restarted = 0;

/* True if parsing determined that all the work is already done.  */
int just_exit = 0;

/* These are the arguments that our application supports. */
static struct argp_option arguments[] =
{
#define DISCARD_KEY -1
  { "discard-session", DISCARD_KEY, N_("ID"), 0, N_("Discard session"), 1 },
  { NULL, 0, NULL, 0, NULL, 0 }
};

static error_t parse_an_arg (int key, char *arg, struct argp_state *state); 

static struct argp parser =
{
  arguments,                    /* Options.  */
  parse_an_arg,                 /* The parser function.  */
  NULL,                         /* Some docs.  */
  NULL,                         /* Some more docs.  */
  NULL,                         /* Child arguments -- gnome_init fills
                                   this in for us.  */
  NULL,                         /* Help filter.  */
  NULL                          /* Translation domain; for the app it
                                   can always be NULL.  */
};

void parse_args (int argc, char *argv[]);

GnomeUIInfo helpmenu[] = {
            {GNOME_APP_UI_ITEM, 
             N_("About..."), N_("Info about this program"),
             about_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_ABOUT,
             0, 0, NULL},
             GNOMEUIINFO_END
        };

GnomeUIInfo filemenu[] = {
            {GNOME_APP_UI_ITEM, 
             N_("New keyring..."), N_("Create a new keyring"),
             new_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW,
             0, 0, NULL},
	    {GNOME_APP_UI_ITEM, 
             N_("Open keyring..."), N_("Open keyring"),
             open_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN,
             0, 0, NULL},
	    {GNOME_APP_UI_ITEM, 
	  /* WARNING! This menu item should be #2 (CLOSE) */
             N_("Close keyring"), N_("Close keyring"),
             close_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE,
             0, 0, NULL},
	    GNOMEUIINFO_SEPARATOR,
	    {GNOME_APP_UI_ITEM, 
             N_("Refresh All"), N_("Refresh All"),
             refresh_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH,
             0, 0, NULL},
            GNOMEUIINFO_SEPARATOR,
	    {GNOME_APP_UI_ITEM, 
             N_("GnomePgpLib Properties..."), N_("GnomePgpLib Properties"),
             lib_properties_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
             0, 0, NULL},
	    GNOMEUIINFO_SEPARATOR,
	    {GNOME_APP_UI_ITEM, 
             N_("Quit"), N_("Exit from this program"),
             quit_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_QUIT,
             0, 0, NULL},
            GNOMEUIINFO_END
        };

GnomeUIInfo pgpmenu[] = {
            {GNOME_APP_UI_ITEM, 
             N_("Sign and encrypt..."), N_("Sign and encrypt"),
             sign_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD,
             0, 0, NULL},
            {GNOME_APP_UI_ITEM, 
             N_("Decrypt and check signature..."), N_("Decrypt and check signature"),
             unsign_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BACK,
             0, 0, NULL},
	    GNOMEUIINFO_END
        };

GnomeUIInfo mainmenu[] = {
            GNOMEUIINFO_SUBTREE(N_("File"), filemenu),
            GNOMEUIINFO_SUBTREE(N_("Crypt"), pgpmenu),
	    GNOMEUIINFO_SUBTREE(N_("Help"), helpmenu),
            GNOMEUIINFO_END
        };


GnomeUIInfo toolbar[] = {
            {GNOME_APP_UI_ITEM, 
             N_("New"), N_("Create a new keyring"),
             new_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_NEW,
             0, 0, NULL},
	    {GNOME_APP_UI_ITEM, 
             N_("Open"), N_("Open keyring"),
             open_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_OPEN,
             0, 0, NULL},
	    {GNOME_APP_UI_ITEM, 
             N_("Close"), N_("Close key ring"), /* #2 CLOSE2 */
             close_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CLOSE,
             0, 0, NULL},
	    {GNOME_APP_UI_ITEM, 
             N_("Refresh"), N_("Refresh all keyrings"),
             refresh_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
             0, 0, NULL},
            {GNOME_APP_UI_ITEM, 
             N_("Sign/Encrypt"), N_("Sign and encrypt"),
             sign_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_FORWARD,
             0, 0, NULL},
            {GNOME_APP_UI_ITEM, 
             N_("Decrypt/Check"), N_("Decrypt and check signature"),
             unsign_cb, NULL, NULL, 
             GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BACK,
             0, 0, NULL},
	    GNOMEUIINFO_END
        };

struct mainwin
  {
        GtkNotebook* notebook;
  } mainwin;

int error=0;

int
main(int argc, char *argv[])
{
  GnomeClient *client;   

  argp_program_version = VERSION;

  options.rings = g_list_alloc();

  /* i18n */
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  client = gnome_client_new_default();

  gtk_signal_connect (GTK_OBJECT (client), "save_yourself",
                GTK_SIGNAL_FUNC (save_state), (gpointer) argv[0]);
  gtk_signal_connect (GTK_OBJECT (client), "connect",
                GTK_SIGNAL_FUNC (connect_client), NULL);

  /* init gnome and parse params */
  gnome_init ("gpgp", &parser, argc, argv,
	      0, NULL);

  /* prepare_app() makes all the gtk calls necessary to set up a
     minimal Gnome application */
  
  if (!just_exit)
    {
     prepare_app ();
     gtk_main ();
    }

  return 0;
}

/************************/
/*   Prepare applicaton */
/************************/

void
prepare_app()
{
  GtkWidget* mainbox;

  app = GNOME_APP(gnome_app_new ("gpgp", "Gnome PGP"));

  /* GnomePgp init */
  gpgp_init(app, 1);
  
  gtk_widget_set_events(GTK_WIDGET(app), GDK_KEY_PRESS_MASK | GDK_BUTTON_PRESS_MASK);
  gtk_widget_realize (GTK_WIDGET(app));
  gtk_signal_connect (GTK_OBJECT (app), "delete_event",
                      GTK_SIGNAL_FUNC (quit_cb),
                      NULL);

  if (restarted) 
    {
     gtk_widget_set_uposition (GTK_WIDGET(app), options.x, options.y);
     gtk_widget_set_usize     (GTK_WIDGET(app), options.w, options.h);
    }
    else
    {
     void* s;
     char* key;
     char* value;
  
     /* create rings section */
     if (!gnome_config_has_section("gpgp-app/rings"))
       {
        gnome_config_set_string("gpgp-app/rings/0", gpgp_options.defpub);
        gnome_config_set_string("gpgp-app/rings/1", gpgp_options.defsec);
       }

     s = gnome_config_init_iterator("gpgp-app/rings");
     while (gnome_config_iterator_next(s, &key, &value))
     options.rings = g_list_append(options.rings,value);
    }

/* create menus */
  gnome_app_create_menus(app, mainmenu);
  gtk_widget_set_sensitive( filemenu[2].widget, FALSE ); /* CANCEL */

/* create statusbar */
  gnome_app_set_statusbar(app, gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_USER));

/* set toolbar at left side by default */
  gnome_config_set_string("gpgp/Placement/Toolbar",gnome_config_get_string("gpgp/Placement/Toolbar=left"));

/* create toolbar */
  gnome_app_create_toolbar(app, toolbar);
	
/* create main box */  
  mainbox = gtk_vbox_new (FALSE, 0);
  gnome_app_set_contents (GNOME_APP(app), mainbox);

/* create notebook */
  mainwin.notebook = GTK_NOTEBOOK(gtk_notebook_new());
  gtk_notebook_set_tab_pos(mainwin.notebook, GTK_POS_TOP);
  gtk_widget_set_usize(GTK_WIDGET(mainwin.notebook), 600, 200);
  gtk_box_pack_start(GTK_BOX(mainbox), GTK_WIDGET(mainwin.notebook), TRUE, TRUE, 0); 

/* create keyrings page */
  g_list_foreach(options.rings, addring, NULL);

  gtk_notebook_set_page(mainwin.notebook,0);

/* show app */
  gtk_widget_show_all (GTK_WIDGET(app));
}

void addring(gpointer data, gpointer user_data)
{
  GtkWidget* keyringpage;

  if (!data)
     return;

  keyringpage = gnome_pgp_keyring_new();
  gtk_signal_connect (GTK_OBJECT (keyringpage), "error",GTK_SIGNAL_FUNC (report_error),NULL);
  error=0;
  gnome_pgp_keyring_select(GNOME_PGP_KEYRING(keyringpage), data);
  if (!error)
    {
     gtk_notebook_prepend_page(GTK_NOTEBOOK(mainwin.notebook), keyringpage, gtk_label_new(g_basename(data)));
     gtk_widget_show (keyringpage);
     gtk_widget_set_sensitive( filemenu[2].widget, TRUE); /* CLOSE */
     gtk_widget_set_sensitive( toolbar[2].widget, TRUE); /* CLOSE2 */
    }
    else
     gtk_widget_destroy(GTK_WIDGET(keyringpage));
}

/************************/
/*  Error handling      */
/************************/

void err_ok_cb(GtkWidget* widget, GtkWidget* data)
{
 gtk_widget_destroy(data);
}

void real_report_error(char* info)
{
 if (!info) info=_("Internal error, <null> reported");
 gnome_app_error(app, info); 
}

void report_error(GtkWidget* widget, void* data)
{
 char buf[1000];

 error=1;
 
 if (IS_GNOME_PGP_KEYRING(widget))
   {
    strncpy(buf,GNOME_PGP_KEYRING(widget)->error_info,sizeof(buf));
    GNOME_PGP_KEYRING(widget)->error_info=NULL;
   }
 
 if (IS_GNOME_PGP_SIGN(widget))
   {
    strncpy(buf,GNOME_PGP_SIGN(widget)->error_info,sizeof(buf));
    GNOME_PGP_SIGN(widget)->error_info=NULL;
   }
 
  if (IS_GNOME_PGP_UNSIGN(widget))
   {
    strncpy(buf,GNOME_PGP_UNSIGN(widget)->error_info,sizeof(buf));
    GNOME_PGP_UNSIGN(widget)->error_info=NULL;
   }
 
 real_report_error(buf);
}

/* Callbacks functions */

/******************/
/*  Quit          */
/******************/

void
quit_cb (GtkWidget *widget, void *data)
{
  gtk_main_quit ();
  return;
}

/******************/
/*  Refresh       */
/******************/

void for_each_cb (GtkWidget *widget, void* data)
{
 gnome_pgp_keyring_refresh(GNOME_PGP_KEYRING(widget));
}

void
refresh_cb (GtkWidget *widget, void *data)
{
 gtk_container_foreach( GTK_CONTAINER(mainwin.notebook), (GtkCallback) for_each_cb, NULL);
}

/*******************/
/*  Unemplemented  */
/*******************/

void
unempl_cb (GtkWidget *widget, void *data)
{
  real_report_error(_("This feature was not emplemented"));
}

/********************/
/* Sign             */
/********************/

void
sign_cb (GtkWidget *widget, void *data)
{
 GtkWidget* filesel;

 filesel = gtk_file_selection_new(_("Choose file"));
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",GTK_SIGNAL_FUNC(destroy_cb),filesel);
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",GTK_SIGNAL_FUNC(sign2_cb),filesel); 
 gtk_widget_show(filesel);
}
 
void
sign2_cb (GtkWidget *widget, void* data)
{
 GtkWidget* signbox;
 char* filename = strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(data)));
 int fd = open(filename,  O_RDONLY);
 char* buf;
 struct stat stats;

 destroy_cb(widget, data); /* delete file selector */

 if (fd == -1)
   {
    real_report_error(g_strerror(errno));
    return;
   }

 if ( fstat(fd, &stats)==-1)
   {
    real_report_error(g_strerror(errno));
    return;
   }
 
 buf = mmap(NULL, stats.st_size, PROT_READ, MAP_SHARED, fd, 0);

 signbox = gnome_pgp_sign_new(buf,stats.st_size, g_basename(filename));
 gtk_signal_connect (GTK_OBJECT (signbox), "error",GTK_SIGNAL_FUNC (report_error),NULL);
 gtk_signal_connect (GTK_OBJECT (signbox), "cancel",GTK_SIGNAL_FUNC (sign_cancel_cb),NULL);
 gtk_signal_connect (GTK_OBJECT (signbox), "destroy",GTK_SIGNAL_FUNC (sign_destroy_cb),filename);
 
 gtk_widget_show(signbox);
}

void sign_cancel_cb (GtkWidget *widget, void *data)
{
 real_report_error(_("Canceled by user"));
}

void sign_destroy_cb (GnomePgpSign* widget, char* data)
{
  int fd;
  char buf[1000];

  if (!widget->buf) /* cancel or error */
     return;
  
  g_snprintf(buf, sizeof(buf), "%s.pgp", data);
  
  munmap ( widget->buf, widget->bufsize);

  fd = open(buf, O_CREAT | O_TRUNC | O_WRONLY, 0666);
  if (fd==-1)
    {
     real_report_error(g_strerror(errno));
     free(widget->obuf);
     return;
    }
 
 /* TODO: ask for Overwrite and etc here */ 
  if ( write(fd, widget->obuf, widget->obufsize)!=widget->obufsize)
    {
     real_report_error(g_strerror(errno));
     if (widget->obuf) free(widget->obuf);
     close(fd);
     return;
    }
  free(widget->obuf);
  close(fd);
} 

/*******************/
/* UnSign          */
/*******************/

void
unsign_cb (GtkWidget *widget, void *data)
{
 GtkWidget* filesel;

 filesel = gtk_file_selection_new(_("Choose file"));
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",GTK_SIGNAL_FUNC(destroy_cb),filesel);
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",GTK_SIGNAL_FUNC(unsign2_cb),filesel); 
 gtk_widget_show(filesel);
}
 
void
unsign2_cb (GtkWidget *widget, void* data)
{
 GtkWidget* unsignbox;
 char* filename = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(data)));
 int fd = open(filename,  O_RDONLY);
 char* buf;
 struct stat stats;

 destroy_cb(widget, data); /* delete file selector */

 if (fd == -1)
   {
    real_report_error(g_strerror(errno));
    return;
   }

 if ( fstat(fd, &stats)==-1)
   {
    real_report_error(g_strerror(errno));
    return;
   }
 
 buf = mmap(NULL, stats.st_size, PROT_READ, MAP_SHARED, fd, 0);

 unsignbox = gnome_pgp_unsign_new(buf,stats.st_size);
 gtk_signal_connect (GTK_OBJECT (unsignbox), "error",GTK_SIGNAL_FUNC (report_error),NULL);
 gtk_signal_connect (GTK_OBJECT (unsignbox), "cancel",GTK_SIGNAL_FUNC (unsign_cancel_cb),NULL);
 gtk_signal_connect (GTK_OBJECT (unsignbox), "destroy",GTK_SIGNAL_FUNC (unsign_destroy_cb),filename);
 
/* gtk_widget_show(unsignbox); */

 gnome_pgp_unsign_action(GNOME_PGP_UNSIGN(unsignbox));
}

void unsign_cancel_cb (GtkWidget *widget, void *data)
{
 real_report_error(_("Canceled by user"));
}

struct unsign_destroy_pack
  {
   char* buf;
   int   bufsize;
   GtkFileSelection* filesel;
  };
void unsign_destroy_2_cb(GtkWidget* widget, struct unsign_destroy_pack* pack);
void unsign_destroy_cancel_cb(GtkWidget* widget, struct unsign_destroy_pack* pack);

void unsign_destroy_cb (GnomePgpUnSign* widget, char* data)
{
  struct unsign_destroy_pack* pack = g_malloc(sizeof(struct unsign_destroy_pack));
  
  munmap ( widget->buf, widget->bufsize);

  pack->buf=widget->obuf;
  pack->bufsize=widget->obufsize;

  if (!pack->buf)
    {   /* destroy on cancel or error */
     g_free(pack);
     return;
    }

  pack->filesel=GTK_FILE_SELECTION(gtk_file_selection_new(_("Save to...")));
  gtk_file_selection_set_filename(pack->filesel, widget->filename);
  gtk_signal_connect(GTK_OBJECT(pack->filesel->ok_button), "clicked",GTK_SIGNAL_FUNC(unsign_destroy_2_cb),pack);
  gtk_signal_connect(GTK_OBJECT(pack->filesel->cancel_button), "clicked",GTK_SIGNAL_FUNC(unsign_destroy_cancel_cb),pack);
  gtk_widget_show(GTK_WIDGET(pack->filesel));
}

void unsign_destroy_2_cb(GtkWidget* widget, struct unsign_destroy_pack* pack)
{
  int fd;

  fd = open(gtk_file_selection_get_filename(pack->filesel), O_CREAT | O_TRUNC | O_WRONLY, 0666);
  if (fd==-1)
    {
     real_report_error(g_strerror(errno));
     /* we do not free/destroy to let user retry */
     return;
    }
 
 /* TODO: ask for Overwrite and etc here */ 
  if ( write(fd, pack->buf, pack->bufsize)!=pack->bufsize)
    {
     real_report_error(g_strerror(errno));
     close(fd);
     return;
    }
    
  g_free(pack->buf);
  close(fd);
  gtk_widget_destroy(GTK_WIDGET(pack->filesel));
  g_free(pack);
} 

void unsign_destroy_cancel_cb(GtkWidget* widget, struct unsign_destroy_pack* pack)
{
 if (pack->buf) g_free(pack->buf);
 gtk_widget_destroy(GTK_WIDGET(pack->filesel));
 g_free(pack);
}

/******************/
/* About          */
/******************/

void
about_cb (GtkWidget *widget, void *data)
{
  GtkWidget *about;
  gchar *authors[] = {
	  "Max Valianskiy",
          NULL
          };

  about = gnome_about_new ( "Gnome PGP", VERSION,
        		/* copyrigth notice */
                        "(C) 1998 Max Valianskiy and others",
                        (const gchar**) authors,
                        /* another comments */
                        "http://maxcom.ml.org/gpgp",
                        NULL); /* pixmap logo */
  gtk_widget_show (about);

  return;
}

/***********************/
/*  Close              */
/***********************/

void
close_cb (GtkWidget *widget, void *data)
{
 GList *children;
 int num=0;

/* authors of GTK was lazy to save current page as a number */
 children = mainwin.notebook->children;
 while (children)
      {
       if (mainwin.notebook->cur_page == children->data)
	  break;
       children = children->next;
       num++;
      }

 gtk_notebook_remove_page(mainwin.notebook, num);

 if (!mainwin.notebook->cur_page)  /* CLOSE CLOSE2 */
   {
    gtk_widget_set_sensitive( filemenu[2].widget, FALSE );
    gtk_widget_set_sensitive( toolbar[2].widget, FALSE );
   }
}

/**************************/
/* New                    */
/**************************/

struct new_pack
   {
    GnomePgpKeyring* page;
    GtkFileSelection* w;
    GnomeDialog* di;
   };

static void
new2_cancel_cb (GtkWidget *widget, struct new_pack* data)
{
    if (data->di) gtk_widget_destroy(GTK_WIDGET(data->di));
    gtk_widget_destroy(GTK_WIDGET(data->w));
    g_free(data);
    return;
}

void
new2_append_cb(GtkWidget *widget, struct new_pack* data)
{
 GnomePgpKeyring* keyringpage;
 
 FILE* fp = fopen(gtk_file_selection_get_filename(data->w),"w");
 if (!fp)
   {
    real_report_error(g_strerror(errno));
    return;
   }
 fclose(fp);

 keyringpage = GNOME_PGP_KEYRING(gnome_pgp_keyring_new());
 gtk_signal_connect (GTK_OBJECT (keyringpage), "error",GTK_SIGNAL_FUNC (report_error),NULL);
 gnome_pgp_keyring_select(GNOME_PGP_KEYRING(keyringpage), gtk_file_selection_get_filename(data->w));
 gtk_notebook_append_page(GTK_NOTEBOOK(mainwin.notebook), GTK_WIDGET(keyringpage), gtk_label_new(gtk_file_selection_get_filename(data->w)));
 gtk_widget_show(GTK_WIDGET(keyringpage));

 refresh_cb(NULL, NULL);

 new2_cancel_cb(widget, data);
}

void
new2_over_cb (GtkWidget *widget, struct new_pack* data)
{
 if (unlink(gtk_file_selection_get_filename(data->w))==-1)
   {
    real_report_error(g_strerror(errno));
    return;
   }
 new2_append_cb(widget,data);
}
 
void
new2_cb (GtkWidget *widget, struct new_pack* data)
{
 if (g_file_exists(gtk_file_selection_get_filename(data->w)))
  {
   GtkWidget* msgbox;
   GtkWidget* bcancel;
   GtkWidget* bover;
 
   /* create exist dialog */
   msgbox = gnome_message_box_new (_("File already exist"),
	             GNOME_MESSAGE_BOX_QUESTION,
		     NULL);

   bover = gtk_button_new_with_label(_("Overwrite"));
   bcancel = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);

   gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(msgbox)->action_area),bover,
                              FALSE, TRUE, GNOME_PAD_SMALL);
   gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(msgbox)->action_area),bcancel,
                              FALSE, TRUE, GNOME_PAD_SMALL);

   data->di=GNOME_DIALOG(msgbox);

   gtk_signal_connect (GTK_OBJECT (bover), "clicked",GTK_SIGNAL_FUNC (new2_over_cb), data);
   gtk_signal_connect (GTK_OBJECT (bcancel), "clicked",GTK_SIGNAL_FUNC (new2_cancel_cb), data);
   
   gtk_widget_show(bover);
   gtk_widget_show(bcancel);
  
   gtk_widget_hide(GTK_WIDGET(data->w));
   
   gtk_widget_show(msgbox); 
   return;
  } 

  new2_append_cb(widget,data);
}

void
destroy_cb (GtkWidget *widget, gpointer data)
{
 gtk_widget_destroy(GTK_WIDGET(data));
}

void
new_cb (GtkWidget *widget, void *data)
{
 GtkWidget* filesel;
 struct new_pack* ep=g_malloc(sizeof(struct new_pack));

 filesel=gtk_file_selection_new(_("Choose file..."));
 gtk_file_selection_set_filename( GTK_FILE_SELECTION(filesel), gpgp_options.pgppath );

 ep->w=GTK_FILE_SELECTION(filesel);
 ep->page=(GnomePgpKeyring*) data;
 ep->di=NULL;
 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",GTK_SIGNAL_FUNC(destroy_cb),filesel); 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",GTK_SIGNAL_FUNC(new2_cb),ep);
 gtk_widget_show(filesel);
}

/**********************/
/* open               */
/**********************/
void open2_cb (GtkWidget *widget, struct new_pack *data);

void
open_cb (GtkWidget *widget, void *data)
{
 GtkWidget* filesel;
 struct new_pack* ep=g_malloc(sizeof(struct new_pack));

 filesel=gtk_file_selection_new(_("Choose file..."));
 gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel),gpgp_options.pgppath);

 ep->w=GTK_FILE_SELECTION(filesel);
 ep->page=(GnomePgpKeyring*) data;
 ep->di=NULL;
 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button), "clicked",GTK_SIGNAL_FUNC(destroy_cb),filesel); 
 gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button), "clicked",GTK_SIGNAL_FUNC(open2_cb),ep);
 gtk_widget_show(filesel);
}

void
open2_cb (GtkWidget *widget, struct new_pack *data)
{
 GnomePgpKeyring* keyringpage;
 
 keyringpage = GNOME_PGP_KEYRING(gnome_pgp_keyring_new());
 gtk_signal_connect (GTK_OBJECT (keyringpage), "error",GTK_SIGNAL_FUNC (report_error),NULL);
 error=0;
 gnome_pgp_keyring_select(GNOME_PGP_KEYRING(keyringpage), gtk_file_selection_get_filename(data->w));
 if (!error)
   {
    gtk_notebook_prepend_page(GTK_NOTEBOOK(mainwin.notebook),GTK_WIDGET(keyringpage), gtk_label_new(gtk_file_selection_get_filename(data->w)));
    gtk_notebook_set_page(mainwin.notebook, 0);
    gtk_widget_set_sensitive(filemenu[4].widget, TRUE);
    gtk_widget_show(GTK_WIDGET(keyringpage));
   }
   else
    gtk_widget_destroy(GTK_WIDGET(keyringpage));

 new2_cancel_cb(widget, data);
}

/****************************/
/* Session management       */
/****************************/

int savering_n=0;

void savering(GtkNotebookPage* page, char* name)
{
 char buf[1000];

 g_snprintf(buf,sizeof(buf),"%s/%d", name, savering_n++);

 printf("%s=%s\n", buf, GNOME_PGP_KEYRING(page->child)->name);
 
 gnome_config_set_string(buf, GNOME_PGP_KEYRING(page->child)->name);
}

static int
save_state (GnomeClient        *client,
            gint                phase,
            GnomeRestartStyle   save_style,
            gint                shutdown,
            GnomeInteractStyle  interact_style,
            gint                fast,
            gpointer            client_data)
{
  gchar *session_id;
  gchar *sess;
  gchar *buf;
  gchar *argv[3];
  gint x, y, w, h;

  session_id=gnome_client_get_id (client);

  /* The only state that gnome-hello has is the window geometry.
     Get it. */
  gdk_window_get_geometry (GTK_WIDGET(app)->window, &x, &y, &w, &h, NULL);

  /* Save the state using gnome-config stuff. */
  sess = g_copy_strings ("/gpgp-app/Saved-Session-",
                         session_id,
                         NULL);

  buf = g_copy_strings ( sess, "/x", NULL);
  gnome_config_set_int (buf, x);
  g_free(buf);
  buf = g_copy_strings ( sess, "/y", NULL);

  gnome_config_set_int (buf, y);
  g_free(buf);
  buf = g_copy_strings ( sess, "/w", NULL);
  gnome_config_set_int (buf, w);
  g_free(buf);
  buf = g_copy_strings ( sess, "/h", NULL);
  gnome_config_set_int (buf, h);
  g_free(buf);

  g_free(sess);

/* rings */
  savering_n=0;

  sess = g_copy_strings ("/gpgp-app/rings-Saved-Session-",
                         session_id,
                         NULL);

  g_list_foreach(mainwin.notebook->children, (GFunc) savering, sess);

  g_free(sess);

  gnome_config_sync();
 
  /* Here is the real SM code. We set the argv to the parameters need
ed
     to restart/discard the session that we've just saved and call
     the gnome_session_set_*_command to tell the session manager it.
*/
  argv[0] = (char*) client_data;
  argv[1] = "--discard-session";
  argv[2] = session_id;
  gnome_client_set_discard_command (client, 3, argv);

  /* Set commands to clone and restart this application.  Note that w
e
     use the same values for both -- the session management code will
     automatically add whatever magic option is required to set the
     session id on startup.  */
  gnome_client_set_clone_command (client, 1, argv);
  gnome_client_set_restart_command (client, 1, argv);

  return TRUE;
}

/* Connected to session manager. If restarted from a former session:
   reads the state of the previous session. Sets os_* (prepare_app
   uses them) */
void
connect_client (GnomeClient *client, gint was_restarted, gpointer client_data)
{
  gchar *session_id;

  /* Note that information is stored according to our *old*
     session id.  The id can change across sessions.  */
  session_id = gnome_client_get_previous_id (client);

  if (was_restarted && session_id != NULL)
    {
      gchar *sess;
      gchar *buf;
      void  *s;
      char* key;
      char* value;
 
      restarted = 1;

      sess = g_copy_strings ("/gpgp-app/Saved-Session-", session_id, NULL);

      buf = g_copy_strings ( sess, "/x", NULL);
      options.x = gnome_config_get_int (buf);
      g_free(buf);
      buf = g_copy_strings ( sess, "/y", NULL);
      options.y = gnome_config_get_int (buf);
      g_free(buf);
      buf = g_copy_strings ( sess, "/w", NULL);
      options.w = gnome_config_get_int (buf);
      g_free(buf);
      buf = g_copy_strings ( sess, "/h", NULL);
      options.h = gnome_config_get_int (buf);
      g_free(buf);

      g_free(sess);

    /* rings */
      sess = g_copy_strings("/gpgp-app/rings-Saved-Session-", session_id, NULL);
      
      s = gnome_config_init_iterator(sess);
      while (gnome_config_iterator_next(s, &key, &value))
        options.rings = g_list_append(options.rings,value);
      g_free(sess);
    }

  /* If we had an old session, we clean up after ourselves.  */
  if (session_id != NULL)
    discard_session (session_id);

  return;
}

void
discard_session (gchar *id)
{
  gchar *sess;

  sess = g_copy_strings ("/gpgp-app/Saved-Session-", id, NULL);

  /* we use the gnome_config_get_* to work around a bug in gnome-conf
ig
     (it's going under a redesign/rewrite, so i didn't correct it) */
  gnome_config_get_int ("/gpgp-app/Bug/work-around=0");
  gnome_config_clean_section (sess);
  g_free (sess);

  sess = g_copy_strings ("/gpgp-app/rings-Saved-Session-", id, NULL);
  gnome_config_get_int ("/gpgp-app/Bug/work-around=0");
  gnome_config_clean_section (sess);
  g_free (sess);
  
  gnome_config_sync ();

  return;
}

/***************************/
/* Options parseing        */
/***************************/

static error_t
parse_an_arg (int key, char *arg, struct argp_state *state)
{
  if (key == DISCARD_KEY)
    {
      discard_session (arg);
      just_exit = 1;
      return 0;
    }

  /* We didn't recognize it.  */
  return ARGP_ERR_UNKNOWN;
}

void lib_properties_cb (GtkWidget *widget, void *data)
{
 GtkWidget *properties;

 gnome_app_warning(app, _("This feature is under construction")); 

 properties = gnome_pgp_lib_properties_new();

 gtk_widget_show(properties);
}
