//package hal.math;

import java.util.Random;

public class Bignum {
    
// "Compile-time" parameters

// If true, shift divisor up and then down in div routine.
// If false, allocate a temp array and shift up to it, don't shift down.
static final boolean SHIFTDIVUP = false;

// Avoid using the sign bit to simplify mult/div routines
static final int NBITS = 31;
static final int BITMASK = 0x7fffffff;

int d[];        // data of Bignum
int len;        // nonzero words in d, but 1 if Bignum val==0
int words;      // actual words allocated to d
boolean neg;    // true if negative, else positive

// Constructors

public Bignum ()        // initialize to 0
{
    words = 2;
    d = new int[words];
    len = 1;
}

/* Initialize to specified val */
public Bignum (int val)
{
    words = 2;
    d = new int[words];
    len = 1;
    neg = (val < 0);
    if (neg)
        d[0] = -val;
    else
        d[0] = val;
}

/* Initialize to specified array of unsigned vals, with ls one first */
public Bignum (int arr[])
{
    len = arr.length;
    words = len+1;
    d = new int[words];
    for (int i=0; i<len; ++i)
        d[i] = arr[i] & BITMASK;
}

/* Initialize to specified val, with specified internal padding.
 * For internal use. */
Bignum (int val, int pad)
{
    words = pad;
    d = new int[words];
    len = 1;
    neg = (val < 0);
    if (neg)
        d[0] = -val;
    else
        d[0] = val;
}

/* Initialize to a byte array */
public Bignum (byte b[], int off, int blen)
{
    len = ((blen*8)+NBITS-1) / NBITS;
    words = len+1;
    d = new int[words];
    int j = 0;          // index into d
    int dib = 0;        // lsb of byte within d[j]
    for (int i=blen-1; i>=0; --i) {
        int v = b[off+i] & 0xff;
        d[j] |= (v<<dib) & BITMASK;
        dib += 8;
        if (dib >= NBITS) {
            dib -= NBITS;
            ++j;
            if (dib > 0)
                d[j] = v >> (8-dib);
        }
    }
}

public Bignum (byte b[])
{
    this(b, 0, b.length);
}

public Bignum (Bignum a)
{
    copy(a);
}

// Utility routines

/* Change number of words in Bignum.  Not checked for
 * legality. */
protected void resize (int newwords)
{
    int newd[] = new int[newwords];   // zeros array
        int top = Math.min(words,newwords);
    for (int i=0; i<top; ++i)
        newd[i] = d[i];
    d = newd;
    words = newwords;
}

/* Make sure we are at least this big */
protected void minsize (int newlen)
{
    if (newlen > words)
        resize (newlen);
}


// Miscellaneous public entry points

public Bignum copy (Bignum a)
{
    words = a.len + 1;
    d = new int[words];
    System.arraycopy (a.d, 0, d, 0, a.len);
    len = a.len;
    neg = a.neg;
    return this;
}

public static Bignum random(int bits, Random rnd)
{
    int ni = (bits-1)/NBITS + 1;
    int i;
    bits = (bits-1) % NBITS + 1;
    Bignum rslt = new Bignum(0, ni);
    for (i=0; i<ni; ++i) {
        int r = rnd.nextInt() & BITMASK;
        rslt.d[i] = r;
    }
    rslt.d[ni-1] &= (1<<bits)-1;
    // rslt's MSW may be 0, correct it down
    for (i=ni-1; i>0; --i) {
        if (rslt.d[i]!=0)
            break;
    }
    rslt.len = i+1;
    return rslt;
}


/* Returns the number of significant bits.  0 if the value is 0, 1 if the
 * value is 1, 2 if the value is 2 or 3, etc. (floor of log2 of value, plus 1).
 */
public int countbits()
{
    int di = d[len-1];
    int n = _cbits(di);
    return n + NBITS*(len-1);
}

static String tohex(int n, boolean stripzeros)
{
    StringBuffer sb = new StringBuffer(8);
    for (int i=(NBITS-1)/8; i>=0; --i) {
        int b = (n>>(8*i));
        int c = (b>>4) & 0xf;
        if (c!=0 || !stripzeros) {
            sb.append("0123456789abcdef".charAt(c));
            stripzeros = false;
        }
        c = b & 0xf;
        if (c!=0 || !stripzeros || i==0) {
            sb.append("0123456789abcdef".charAt(c));
            stripzeros = false;
        }
    }
    return sb.toString();
}

/* For debugging purposes */
public void dump(java.lang.String s)
{
    System.out.println(s+";len="+len+",words="+words+"(="+d.length+")"+
                       " bits="+countbits());
    if (neg)
        System.out.print("  -0x");
    else
        System.out.print("  0x");
    for (int i=len-1; i>=0; --i) {
        System.out.print(tohex(d[i],i==len-1)+((i==0)?"":";"));
    }
    System.out.println();
}

public int intValue()
{
    if (neg)
        return -d[0];
    else
        return d[0];
}

public int lsw()
{
    return d[0];
}

public boolean even()
{
    return (d[0]&1)==0;
}

public boolean odd()
{
    return (d[0]&1)!=0;
}

public boolean zero()
{
    return len==1 && d[0]==0;
}

public String toString(int radix)
{
    StringBuffer sb = new StringBuffer();
    Bignum tmp = new Bignum(this);
    Bignum r = new Bignum(radix);
    for ( ; ; ) {
        Bignum rem = new Bignum();
        tmp = div2(tmp, r, rem);
        int c = rem.lsw();
        char ch = (char)((c>9)?'a'+c-10:'0'+c);
        sb.insert(0, ch);
        if (tmp.zero())
            break;
    }
    if (radix==16)
        sb.insert(0, "0x");
    return sb.toString();
}

/* Default radix is 10 */
public String toString()
{
    return this.toString(10);
}

/* Return a byte stream, bigendian */
public void getBytes(byte b[], int off)
{
    int blen = (countbits()+7) / 8;
    int j = 0;          // index into d
    int dib = 0;        // lsb of byte within d[j]
    for (int i=blen-1; i>=0; --i) {
        int v = (d[j]>>>dib) & 0xff;
        dib += 8;
        if (dib >= NBITS) {
            dib -= NBITS;
            ++j;
            if (dib > 0)
                v |= (d[j]<<(8-dib)) & 0xff;
        }
        b[i] = (byte)v;
    }
}

public void getBytes(byte b[])
{
    getBytes(b, 0);
}



// "eq" forms of arithmetic.  "this" is first operand, and
// output.

public Bignum addeq (Bignum a)
{
    if (neg ^ a.neg) {
        neg = !neg;
        subeq(a);
        neg = !neg;
        return this;
    }
    int newlen = Math.max(len, a.len);
    minsize(newlen+1);
    len = _add (d, d, newlen, a.d, a.len);
    return this;
}

public Bignum subeq (Bignum a)
{
    if (neg ^ a.neg) {
        neg = !neg;
        addeq(a);
        neg = !neg;
        return this;
    }
    if (acmp2(this,a)<0) {
        len = _sub (d, a.d, a.len, d, len);
        neg = !neg;
    } else {
        len = _sub (d, d, len, a.d, a.len);
    }
    return this;
}

public Bignum muleq (Bignum a)
{
    if (this==a || cmp2(this,a)==0)
        return mulsqeq();
    int newlen = len + a.len;
    Bignum b = new Bignum(this);        // copy of this
        words = newlen+1;
    int newd[] = new int[words];
    len = _mul (newd, b.d, b.len, a.d, a.len);
    d = newd;
    neg = neg ^ a.neg;
    return this;
}

public Bignum mulsqeq ()
{
    int newlen = 2*len;
    Bignum b = new Bignum(this);        // copy of this
    words = newlen+1;
    int newd[] = new int[words];
    len = _mulsq (newd, b.d, b.len);
    d = newd;
    neg = false;
    return this;
}

public Bignum diveq (Bignum a)
{
    if (cmp2(this,a)<0)
        return this.copy(new Bignum(0));
    Bignum quot = new Bignum(0, len - a.len + 1);
    if (a.len == 1) {
        quot.len = _sdiv(quot.d, d, len, a.d[0], d);
        copy(quot);
        return this;
    }
    int remlen[] = new int[1];
    remlen[0] = len;
    minsize(len+1);
    quot.len = _div (quot.d, d, remlen, a.d, a.len);
    copy (quot);
    neg = neg ^ a.neg;
    return this;
}

public Bignum modeq (Bignum a)
{
    if (cmp2(this,a)<0)
        return this;
    Bignum quot = new Bignum(0, len - a.len + 1);
    if (a.len == 1) {
        _sdiv(quot.d, d, len, a.d[0], d);
        len = 1;
        return this;
    }
    int remlen[] = new int[1];
    remlen[0] = len;
    minsize(len+1);
    _div (quot.d, d, remlen, a.d, a.len);
    len = remlen[0];
    return this;
}


public Bignum rshifteq(int sh)
{
    int nw = sh / NBITS;
    sh %= NBITS;
    System.arraycopy(d, nw, d, 0, len-nw);
    len = len-nw;
    _rshift2(d, d, len, sh);
    if (d[len-1]==0)
        --len;
    return this;
}

public Bignum lshifteq(int sh)
{
    int nw = sh / NBITS;
    sh %= NBITS;
    minsize (len+nw+1);
    len = len + nw;
    int dt[];
    if (nw == 0) {
        dt = d;
    } else {
        dt = new int[len];
        System.arraycopy(d, 0, dt, nw, len);
    }
    _lshift2(d, dt, len, sh);
    if (d[len]!=0)
        ++len;
    return this;
}



// two operand entries

public static Bignum add2 (Bignum a1, Bignum a2)
{
    if (a1.neg ^ a2.neg) {
        a1.neg = !a1.neg;
        Bignum diff = sub2 (a1, a2);
        a1.neg = !a1.neg;
        diff.neg = !diff.neg;
        return diff;
    }
    Bignum sum;
    if (a1.len < a2.len) {
        sum = new Bignum (0, a2.len+1);
        sum.len = _add (sum.d, a2.d, a2.len, a1.d, a1.len);
    } else {
        sum = new Bignum (0, a1.len+1);
        sum.len = _add (sum.d, a1.d, a1.len, a2.d, a2.len);
    }
    sum.neg = a1.neg;
    return sum;
}

public static Bignum sub2 (Bignum a1, Bignum a2)
{
    if (a1.neg ^ a2.neg) {
        a1.neg = !a1.neg;
        Bignum sum = add2 (a1, a2);
        a1.neg = !a1.neg;
        sum.neg = !sum.neg;
        return sum;
    }
    Bignum diff;
    if (acmp2(a1,a2)<0) {
        diff = new Bignum (0, a2.len);
        diff.len = _sub(diff.d, a2.d, a2.len, a1.d, a1.len);
        diff.neg = !a1.neg;
    } else {
        diff = new Bignum (0, a1.len);
        diff.len = _sub(diff.d, a1.d, a1.len, a2.d, a2.len);
        diff.neg = a1.neg;
    }
    return diff;
}

public static Bignum mul2 (Bignum a1, Bignum a2)
{
    if (a1==a2 || cmp2(a1,a2)==0)
        return mulsq2(a1);              // faster on squares
    Bignum prod = new Bignum(0, a1.len+a2.len);
    prod.len = _mul(prod.d, a1.d, a1.len, a2.d, a2.len);
    prod.neg = a1.neg ^ a2.neg;
    return prod;
}

public static Bignum mulsq2 (Bignum a1)
{
    Bignum prod = new Bignum(0, a1.len*2);
    prod.len = _mulsq(prod.d, a1.d, a1.len);
    return prod;
}

/* Return quotient, and set rem to the remainder */
/* rem must be different from the other args.  It may also be null
 * to indicate that there is no interest in the remainder */
public static Bignum div2 (Bignum a1, Bignum a2, Bignum rem)
{
    if (rem==null)
        rem = new Bignum(a1);
    else
        rem.copy(a1);
    if (acmp2(a1,a2) <0)
        return new Bignum(0);
    Bignum quot = new Bignum(0, a1.len-a2.len+1);
    if (a2.len == 1) {
        quot.len = _sdiv(quot.d, a1.d, a1.len, a2.d[0], rem.d);
        rem.len = 1;
        quot.neg = a1.neg ^ a2.neg;
        return quot;
    }
    rem.minsize(rem.len+1);
    int remlen[] = new int[1];
    remlen[0] = rem.len;
    rem.minsize(rem.len+1);
    quot.len = _div(quot.d, rem.d, remlen, a2.d, a2.len);
    quot.neg = a1.neg ^ a2.neg;
    rem.len = remlen[0];
    return quot;
}

/* Compare absolute values, neglecting sign */
public static int acmp2 (Bignum a1, Bignum a2)
{
    if (a1.len > a2.len)
        return 1;
    if (a1.len < a2.len)
        return -1;
    for (int i=a1.len-1; i>=0; --i) {
        if (a1.d[i] > a2.d[i])
            return 1;
        if (a1.d[i] < a2.d[i])
            return -1;
    }
    return 0;
}

public static int cmp2 (Bignum a1, Bignum a2)
{
    if (a1.neg) {
        if (a2.neg) {
            return acmp2 (a2, a1);
        } else {
            return -1;
        }
    } else {
        if (a2.neg) {
            return 1;
        } else {
            return acmp2 (a1, a2);
        }
    }
}

public static Bignum rshift2(Bignum a1, int sh)
{
    int nw = sh / NBITS;
    sh %= NBITS;
    Bignum bsh = new Bignum (0, a1.len-nw);
    System.arraycopy(a1.d, nw, bsh.d, 0, a1.len-nw);
    bsh.len = a1.len-nw;
    _rshift2(bsh.d, bsh.d, bsh.len, sh);
    if (bsh.d[bsh.len-1]==0)
        --bsh.len;
    return bsh;
}

public static Bignum lshift2(Bignum a1, int sh)
{
    int nw = sh / NBITS;
    sh %= NBITS;
    Bignum bsh = new Bignum (0, a1.len+nw+1);
    System.arraycopy(a1.d, 0, bsh.d, nw, a1.len);
    bsh.len = a1.len+nw;
    _lshift2(bsh.d, bsh.d, bsh.len, sh);
    if (bsh.d[bsh.len]!=0)
        ++bsh.len;
    return bsh;
}


// Return a power of two
public static Bignum pow2(int n)
{
    int nlen = n/NBITS + 1;
    Bignum p = new Bignum(0, nlen+1);
    p.len = nlen;
    p.d[nlen-1] = 1 << (n % NBITS);
    return p;
}



/* Entry points from here on */
/* Should only be called with positive numbers */

public static Bignum modexp2 (Bignum a1, Bignum a2, Bignum mod)
{
    Bignum rslt = new Bignum(1);
    int ipr = a2.countbits() - 1;
    int ibit = 1 << (ipr % NBITS);
    ipr = ipr / NBITS;
    while (ipr >= 0) {
//        rslt.muleq(rslt);
//        rslt.modeq(mod);
        rslt = mul2(rslt,rslt);
//rslt.dump("pre div after square");
        Bignum quot = new Bignum(rslt);
        div2 (quot,mod,rslt);
//rslt.dump("after square");
        if ((a2.d[ipr] & ibit) != 0) {
//            rslt.muleq(a1);
//            rslt.modeq(mod);
            rslt = mul2(rslt,a1);
//rslt.dump("pre div after mul");
            quot = new Bignum(rslt);
            div2 (quot,mod,rslt);
//rslt.dump("after mul");
        }
        ibit >>>= 1;
        if (ibit==0) {
            --ipr;
            ibit = 1<<(NBITS-1);
        }
    }
    return rslt;
}

// This version tries hard to avoid memory allocation steps within
// the loop.
public static Bignum modexp2a (Bignum a1, Bignum a2, Bignum mod)
{
    Bignum dbl = new Bignum(1, 2*mod.len+1);    // holds double length values
    Bignum sngl = new Bignum(0, mod.len+1);     // quotient of divide
    Bignum rslt = new Bignum(1, mod.len+1);
    int remlen[] = new int[1];
    int ipr = a2.countbits() - 1;
    int ibit = 1 << (ipr % NBITS);
    ipr = ipr / NBITS;
    while (ipr >= 0) {
        // dbl must be all zeros
        for (int i=0; i<mod.len*2; ++i)
            dbl.d[i] = 0;
        dbl.len = _mulsq (dbl.d, rslt.d, rslt.len);
//dbl.dump("pre div after square");
        remlen[0] = dbl.len;
        _div (sngl.d, dbl.d, remlen, mod.d, mod.len);
        System.arraycopy(dbl.d, 0, rslt.d, 0, remlen[0]);
        rslt.len = remlen[0];
//rslt.dump("after square");
        if ((a2.d[ipr] & ibit) != 0) {
            // dbl must be all zeros
            for (int i=0; i<mod.len*2; ++i)
                dbl.d[i] = 0;
            dbl.len = _mul(dbl.d, rslt.d, rslt.len, a1.d, a1.len);
//dbl.dump("pre div after mul");
            remlen[0] = dbl.len;
            _div (sngl.d, dbl.d, remlen, mod.d, mod.len);
            System.arraycopy(dbl.d, 0, rslt.d, 0, remlen[0]);
            rslt.len = remlen[0];
//rslt.dump("after mul");
        }
        ibit >>>= 1;
        if (ibit==0) {
            --ipr;
            ibit = 1<<(NBITS-1);
        }
    }
    return rslt;
}

/* Exponentiation, non-modular */
public static Bignum exp2 (Bignum a1, Bignum a2)
{
    Bignum rslt = new Bignum(1);
    int ipr = a2.countbits() - 1;
    int ibit = 1 << (ipr % NBITS);
    ipr = ipr / NBITS;
    while (ipr >= 0) {
        rslt.muleq(rslt);
        if ((a2.d[ipr] & ibit) != 0) {
            rslt.muleq(a1);
        }
        ibit >>>= 1;
        if (ibit==0) {
            --ipr;
            ibit = 1<<(NBITS-1);
        }
    }
    return rslt;
}

/* Compute the greatest common divisor of the two numbers */
public static Bignum gcd2(Bignum a1, Bignum a2)
{
    Bignum u, v;
    if (cmp2(a1,a2)>0) {
        u = a1;
        v = a2;
    } else {
        u = a2;
        v = a1;
    }
    while (!(v.len==1 && v.d[0]==0)) {
        Bignum r = new Bignum();
        div2(u, v, r);
        u = v;
        v = r;
    }
    return u;
}

/* Compute the multiplicative inverse of a1 mod a2. */
public static Bignum inv2(Bignum a1, Bignum a2)
{
    Bignum u1 = new Bignum(1);
    Bignum u2 = new Bignum(0);
    Bignum u3 = a1;
    Bignum v1 = new Bignum(0);
    Bignum v2 = new Bignum(1);
    Bignum v3 = a2;
    while (!(v3.len==1 && v3.d[0]==0)) {
        Bignum t1;
        Bignum t2;
        Bignum t3 = new Bignum();
        Bignum q = div2(u3, v3, t3);    // t3 = u3 - q*v3
            t2 = sub2(u2,mul2(q,v2));
        t1 = sub2(u1,mul2(q,v1));
        u1 = v1; u2 = v2; u3 = v3;
        v1 = t1; v2 = t2; v3 = t3;
    }
    // At this point u3 = gcd(a1,a2) which should be 1.
        if (u1.neg)
            u1.addeq(a2);
    return u1;
}

// Internal arithmetic routines, mostly from Knuth

/* Do an add operation.  sum is guaranteed big enough, which
 * should be max of a1len, a2len, plus 1.
 * Return new sumlen.
 * sum may be same as a1 and/or a2.
 * a1len must be >= a2len.
 */
static int _add (int sum[], int a1[], int a1len,
                        int a2[], int a2len)
{
    int i;
    boolean bcarry = false;
    int sumlen = a1len + 1;
    for (i=0; i<a2len; ++i) {
        int di = a1[i];
        int s = (a2[i] + di + (bcarry?1:0)) & BITMASK;
        bcarry = ((s==di) && bcarry) || (s < di);
        sum[i] = s;
    }
    for ( ; i<a1len; ++i) {
        int s = (a1[i] + (bcarry?1:0)) & BITMASK;
        bcarry = bcarry && (s==0);
        sum[i] = s;
    }
    if (bcarry) {
        sum[i] = 1;
        return sumlen;
    } else
        return sumlen-1;
}

/* Do a sub operation.  diff is guaranteed big enough,
 * which should be same size as a1.  a1 is guaranteed to be
 * >= a2.
 * Return new diff len.
 * diff may be same as a1 and/or a2.
 */
static int _sub (int diff[], int a1[], int a1len,
                        int a2[], int a2len)
{
    boolean borrow = false;
    int i;
    for (i=0; i<a2len; ++i) {
        int di = a1[i];
        int df = (di - a2[i] - (borrow?1:0)) & BITMASK;
        borrow = ((di==a2[i])&&borrow) || (di < a2[i]);
        diff[i] = df;
    }
    for ( ; i<a1len; ++i) {
        int di = a1[i];
        diff[i] = (di - (borrow?1:0)) & BITMASK;
        borrow = borrow && (di==0);
    }
    // find length of difference.  i==a1len.
        for ( ; diff[i-1]==0 && i>1; --i)
            ;
    return i;
}

/* Do a multiply operation.  prod must be different from a1 and a2,
 * although a1 and a2 can be the same.
 * Prod must be at least as big as a1len*a2len.
 * Return length of prod.
 * prod should be initialized to zeros.
 */
static int _mul (int prod[], int a1[], int a1len, int a2[], int a2len)
{
    int j, i;
/*
System.out.println("_mul called with a1len="+a1len+", a2len="+a2len);
System.out.println("Arrays:");
for (i=0;i<a1len+a2len;++i)
System.out.print("prod["+i+"]="+prod[i]+(i+1==a1len+a2len?"\n":", "));
for (i=0;i<a1len;++i)
System.out.print("a1["+i+"]="+a1[i]+(i+1==a1len?"\n":", "));
for (i=0;i<a2len;++i)
System.out.print("a2["+i+"]="+a2[i]+(i+1==a2len?"\n":", "));
*/
    if ((a1len==1 && a1[0]==0) || (a2len==1 && a2[0]==0)) {
        /* Times zero */
        return 1;
    }
    int newlen = a1len + a2len;
    for (j=0; j<a2len; ++j) {
        long carry = 0;
        long a2j = (long)a2[j];
        int ij = j;                     // i+j
        for (i=0; i<a1len; ++i) {
            long pl = (long)prod[ij];
            pl += (long)a1[i] * a2j + carry;
            carry = pl >>> NBITS;
            prod[ij++] = (int)pl & BITMASK;
        }
        if (carry != 0) {
            prod[ij] = (int)carry;
        }
    }
    if (prod[newlen-1]==0)
        --newlen;
    return newlen;
}

/* Do a squaring operation.  prod must be different from a1,
 * the number to be squared.
 * Prod must be at least as big as a1len squared.
 * Return length of prod.
 * prod should be initialized to zeros.
 * This is almost twice as fast as a regular multiply.
 */
static int _mulsq (int prod[], int a1[], int a1len)
{
    int j, i;
    int newlen = a1len *2;
    for (j=0; j<a1len; ++j) {
        /* Do square term first, then double terms */
        int a2i = a1[j];
        long pl = (long)prod[j*2];
        pl += (long)a2i * a2i;
        long carry = pl >>> NBITS;
        prod [j*2] = (int)pl & BITMASK;
        for (i=j+1; i<a1len; ++i) {
            int a1i = a1[i];
            pl = (long)prod[i+j];
            long m = a1i * (long)a2i * 2L;
            pl += m + carry;            // cannot overflow if NBITS<32!
            carry = pl >>> NBITS;       // carry may be greater than NBITS though
            prod[i+j] = (int)pl & BITMASK;
        }
        if (carry != 0) {
            pl = (long)prod[i+j];  // Could have a 1 in it
            pl += carry;
            prod[i+j] = (int)pl & BITMASK;
            carry = pl >>> NBITS;
            if (carry != 0) {
                prod[i+j+1] = (int)carry;
            }
        }
    }
    if (prod[newlen-1]==0)
        --newlen;
    return newlen;
}


/* Number of bits to shift must be less than NBITS */
/* Output array sh1 can be same as input sh2 */
/* sh1 must be able to hold shlen+1 bytes */
static void _lshift2 (int sh1[], int sh2[], int shlen, int n)
{
    int i;
    int carry = 0;
    for (i=0; i<shlen; ++i) {
        int di = sh2[i];
        sh1[i] = ((di<<n) | carry) & BITMASK;
        carry = di>>>(NBITS-n);
    }
    if (carry!=0) {
        sh1[shlen] = carry;
    }
}
/* Number of bits to shift must be less than NBITS */
/* Output array sh1 can be same as input sh2 */
static void _rshift2 (int sh1[], int sh2[], int shlen, int n)
{
    int i;
    int carry = 0;
    for (i=shlen-1; i>=0; --i) {
        int di = sh2[i];
        sh1[i] = (di>>>n) | carry;
        carry = (di<<(NBITS-n)) & BITMASK;
    }
}

/* Return the floor of the log2 of val, or 0 if val is 0 */
static int _cbits(int val)
{
    int i=0;
    while (val!=0) {
        val>>>=1;
        ++i;
    }
    return i;
}

/* Divide rem by div, with quotient in quot, remainder in rem.
 * Return length of quotient.
 * divlen should be at least 2.
 * remlen is an in-out parameter with length 1.
 * rem must be of size at least remlen[0] + 1.
 * quot must be of size at least divlen - remlen[0] + 1.
 * From Knuth's Algorithm D, annoted with his steps.
 * rem, div, remlen, and quot should all be distinct.
 */
static int _div (int quot[], int rem[], int remlen[], int div[], int divlen)
{
    int i, j;
    int idij;
    int qhat;
    int divlen1 = divlen-1;
    int rlen = remlen[0];
    // D1
        int shift = NBITS - _cbits(div[divlen1]);
    if (shift != 0) {
        if (SHIFTDIVUP) {
            _lshift2(div, div, divlen, shift);
        } else {
            int newdiv[] = new int[divlen];
            _lshift2(newdiv, div, divlen, shift);
            div = newdiv;
        }
        _lshift2(rem, rem, rlen, shift);
        if (rem[rlen] != 0)
            ++rlen;
    }
    // Make sure dividend MSW is less than divisor's MSW
    if (rem[rlen-1] >= div[divlen-1])
        ++rlen;
    int quotlen = rlen - divlen;
    int v1 = div[divlen1];
    int v2 = div[divlen1-1];
    // D2
    // j is low word of part of rem we are working on
    for (j = quotlen-1; j>= 0; --j) {
        int dhi = rem[j+divlen];
        // D3
        if (dhi==v1) {
            qhat = BITMASK;
        } else {
            long dub = ((long)dhi<<NBITS) | rem[j+divlen1];
            qhat = (int)(dub / v1);
            long test = ((dub - (long)qhat*v1)<<NBITS) |
                                                (long)rem[j+divlen1-1];
            while ((v2*(long)qhat) > test) {
                --qhat;
                test += (long)v1<<NBITS;
            }
        }
        // D4
/*
        int carry = 0;
        for (i=0; i<divlen; ++i) {
            int adi = div[i];
            idij = rem[j+i];
            long prod = (long)adi * qhat + carry;
            carry = (int)(prod >>> NBITS);
            int iprod = (int)prod & BITMASK;
            if (iprod > idij)
                ++carry;
            rem[j+i] = (idij - iprod) & BITMASK;
        }
        idij = rem[j+i];
        rem[j+i] = (idij - carry) & BITMASK;
*/
        long borrow = 0;
        for (i=0; i<divlen; ++i) {
            long prod = (long)div[i] * qhat;
            long ldig = (long)rem[j+i] - (prod&BITMASK) - borrow;
            borrow = (prod>>NBITS) - (ldig>>NBITS);
            rem[j+i] = (int)(ldig & BITMASK);
        }
        idij = rem[j+i];
        int carry = (int)borrow;
        rem[j+i] = (idij - carry) & BITMASK;
/**/
        // D5
        quot[j] = qhat;
        if (carry > idij) {
            // D6
            quot[j] = (qhat-1) & BITMASK;
            boolean bcarry = false;
            for (i=0; i<divlen; ++i) {
                idij = rem[i+j];
                int sum = (div[i] + idij + (bcarry?1:0)) & BITMASK;
                bcarry = ((sum==idij) && bcarry) || (sum < idij);
                rem[i+j] = sum;
            }
            // bcarry should always be true here
            if (bcarry)
                rem[i+j] = (rem[i+j] + 1) & BITMASK;
        }
        // D7
    }
    // D8
    if (shift != 0) {
        if (SHIFTDIVUP)
            _rshift2 (div, div, divlen, shift);
        _rshift2 (rem, rem, rlen, shift);
    }
    while (rlen > 1 && rem[rlen-1] == 0)
        --rlen;
    remlen[0] = rlen;
    if (quotlen > 1 && quot[quotlen-1]==0)
        --quotlen;
    return quotlen;
} // _div


/* Divide num by sdiv, with quotient in squot, remainder in rem[0].
 * rem can be the same as num.
 * Return length of quotient.
 * quot should be at least as big as numlen.
 * From Knuth's Exercise 16.
     */
    static int _sdiv (int quot[], int num[], int numlen, int sdiv, int rem[])
    {
        int r = 0;
        for (int j=numlen-1; j>=0; --j) {
            long dub = ((long)r<<NBITS) | num[j];
            long frac = dub / sdiv;
            quot[j] = (int)frac;
            r = (int)(dub % sdiv);
        }
        rem[0] = r;
        if (quot[numlen-1]==0)
            --numlen;
        return numlen;
    } // _sdiv

    } // class Bignum
