/*
 * This code implements the MD5 message-digest algorithm.
 * The algorithm is due to Ron Rivest.  This code was
 * written by Colin Plumb in 1993, no copyright is claimed.
 * This code is in the public domain; do with it what you wish.
 *
 * Equivalent code is available from RSA Data Security, Inc.
 * This code has been tested against that, and is equivalent,
 * except that you don't need to include two pages of legalese
 * with every copy.
 *
 * To compute the message digest of a chunk of bytes, declare an
 * MD5Context structure, pass it to MD5Init, call MD5Update as
 * needed on buffers full of bytes, and then call MD5Final, which
 * will fill a supplied 16-byte array with the digest.
 */

/* The modifications for 386- and Borland- specific inline assembly
 * usage written by Wojciech Pilorz in 1996;
 * This code is also in public domain
 * Description:
 *           This is the replacement module for md5.c, for use
 *           with some Borland C compilers (see 'Required:');
 *           It works as follows:
 *           MD5Init calls checkCPU() routine, (defined in testcpu.c),
 *           which attempts to detect whether the CPU is 386 or better.
 *           If so, and if the check is not disabled (see testcpu.c
 *           for details), checkCPU sets tcpu_CPU386 to a non-zero value;
 *           tcpu_CPUchecked is set in any case to 1, so that
 *           further calls to checkCPU can be avoided.
 *
 *           If tcpu_CPU386 is not 0, MD5Transf386 is called
 *           instead of MD5Transform by MD5Update and MD5Final.
 *           MD5Transf386 contains inline 386 instructions, which
 *           perform almost all operations within 32-bit registers.
 *           It is much faster than C code when compiled by BC++3.0
 *           or BC++ 4.52 (even with 386 code generation and -O2).
 *           (in fact 2...8 times faster, depending on CPU, cache,
 *           compiler version etc.)
 *
 *           Please note that a good optimizing compiler
 *           (like GNU gcc 2.7.2 with -O2) is able to generate
 *           the machine code as good as this one (or better),
 *           so there is no sense in porting this code to gcc.
 *
 * Required: Borland C++ 3.0 or newer or TurboC++ 3.0 or newer
 *           Turbo Assembler 3.0 or newer
 *           testcpu.c compiled and linked in
 * Limitations:
 *           Only real-mode code is supported so far (all memory models);
 *           16-bit protected mode should also work, but have not
 *           been tested much; test before use
 *           32-bit flat memory model does not work in this release
 * Bugs:
 *         - Should be documented better
 *         - Will not work if ASM_MD5 is defined
 *         - Other bugs may be still hidden
 * Comments: Please send to: <wpilorz@priam.umcs.lublin.pl>
 * Release date: 1996/10/22
 */

#define MD5Transform Transform   /*2.63ui*/

#include <string.h>             /* for memcpy() */
#include "md5.h"

#ifndef HIGHFIRST
#define byteReverse(buf, len)   /* Nothing */
#else
void byteReverse(unsigned char *buf, unsigned longs);

#ifndef ASM_MD5
/*
 * Note: this code is harmless on little-endian machines.
 */
void byteReverse(unsigned char *buf, unsigned longs)
{
    uint32 t;
    do {
        t = (uint32) ((unsigned) buf[3] << 8 | buf[2]) << 16 |
            ((unsigned) buf[1] << 8 | buf[0]);
        *(uint32 *) buf = t;
        buf += 4;
    } while (--longs);
}
#endif
#endif

extern int tcpu_CPUchecked; /* initially 0 */
extern int tcpu_CPU386;         /* initially 0  */
void checkCPU(void);       /* this routine sets tcpu_CPUchecked (always)
                              and sets or clears tcpu_CPU386 */
void MD5Transf386(uint32 buf[4], uint32 const in[16]);
static
void MD5Transform_C(uint32 buf[4], uint32 const in[16]);

/* now the time for some compilation-time checks */
#ifndef __TURBOC__
#error Only Borland C compilers are supported
/* The test should be more specific (i.e. test version numbers).
   However, such a specific check would be difficult to test for me
*/
#endif

#ifdef __FLAT__
#error 32-bit flat model is not supported
#endif

/*
 * Start MD5 accumulation.  Set bit count to 0 and buffer to mysterious
 * initialization constants.
 */
void MD5Init(struct MD5Context *ctx)
{
    ctx->buf[0] = 0x67452301;
    ctx->buf[1] = 0xefcdab89;
    ctx->buf[2] = 0x98badcfe;
    ctx->buf[3] = 0x10325476;

    ctx->bits[0] = 0;
    ctx->bits[1] = 0;
    if (! tcpu_CPUchecked)
       checkCPU();
}

/*
 * Update context to reflect the concatenation of another buffer full
 * of bytes.
 */
void MD5Update(struct MD5Context *ctx, unsigned char const *buf, unsigned len)
{
    uint32 t;

    /* Update bitcount */

    t = ctx->bits[0];
    if ((ctx->bits[0] = t + ((uint32) len << 3)) < t)
        ctx->bits[1]++;         /* Carry from low to high */
    ctx->bits[1] += len >> 29;

    t = (t >> 3) & 0x3f;        /* Bytes already in shsInfo->data */

    /* Handle any leading odd-sized chunks */

    if (t) {
        unsigned char *p = (unsigned char *) ctx->in + t;

        t = 64 - t;
        if (len < t) {
            memcpy(p, buf, len);
            return;
        }
        memcpy(p, buf, t);
        byteReverse(ctx->in, 16);
        MD5Transform(ctx->buf, (uint32 *) ctx->in);
        buf += t;
        len -= t;
    }
    /* Process data in 64-byte chunks */
    if (tcpu_CPU386)
      while (len >= 64) {
        memcpy(ctx->in, buf, 64);
        byteReverse(ctx->in, 16);
        MD5Transf386(ctx->buf, (uint32 *) ctx->in);
        buf += 64;
        len -= 64;
      }
    else
      while (len >= 64) {
        memcpy(ctx->in, buf, 64);
        byteReverse(ctx->in, 16);
        MD5Transform_C(ctx->buf, (uint32 *) ctx->in);
        buf += 64;
        len -= 64;
      }

    /* Handle any remaining bytes of data. */

    memcpy(ctx->in, buf, len);
}

/*
 * Final wrapup - pad to 64-byte boundary with the bit pattern
 * 1 0* (64-bit count of bits processed, MSB-first)
 */
void MD5Final(unsigned char digest[16], struct MD5Context *ctx)
{
    unsigned count;
    unsigned char *p;

    /* Compute number of bytes mod 64 */
    count = (ctx->bits[0] >> 3) & 0x3F;

    /* Set the first char of padding to 0x80.  This is safe since there is
       always at least one byte free */
    p = ctx->in + count;
    *p++ = 0x80;

    /* Bytes of padding needed to make 64 bytes */
    count = 64 - 1 - count;

    /* Pad out to 56 mod 64 */
    if (count < 8) {
        /* Two lots of padding:  Pad the first block to 64 bytes */
        memset(p, 0, count);
        byteReverse(ctx->in, 16);
        MD5Transform(ctx->buf, (uint32 *) ctx->in);

        /* Now fill the next block with 56 bytes */
        memset(ctx->in, 0, 56);
    } else {
        /* Pad block to 56 bytes */
        memset(p, 0, count - 8);
    }
    byteReverse(ctx->in, 14);

    /* Append length in bits and transform */
    ((uint32 *) ctx->in)[14] = ctx->bits[0];
    ((uint32 *) ctx->in)[15] = ctx->bits[1];

    MD5Transform(ctx->buf, (uint32 *) ctx->in);
    byteReverse((unsigned char *) ctx->buf, 4);
    memcpy(digest, ctx->buf, 16);
    memset(ctx, 0, sizeof(ctx));        /* In case it's sensitive */
}

void MD5Transform(uint32 buf[4], uint32 const in[16])
{
   if (! tcpu_CPUchecked)
       checkCPU();
        /* the check above is necessary for the code which uses MD5Transform
         * directly (rather than via MD5Init/MD5Update/MD5Final)
         */
        if (tcpu_CPU386)
                MD5Transf386(buf, in);
        else
                MD5Transform_C(buf, in);
}

#ifndef ASM_MD5

/* The four core functions - F1 is optimized somewhat */

/* #define F1(x, y, z) (x & y | ~x & z) */
#define F1(x, y, z) (z ^ (x & (y ^ z)))
#define F2(x, y, z) F1(z, x, y)
#define F3(x, y, z) (x ^ y ^ z)
#define F4(x, y, z) (y ^ (x | ~z))

/* This is the central step in the MD5 algorithm. */
#define MD5STEP(f, w, x, y, z, data, s) \
        ( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x )

/*
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 */
static
void MD5Transform_C(uint32 buf[4], uint32 const in[16])
{
    register uint32 a, b, c, d;

    a = buf[0];
    b = buf[1];
    c = buf[2];
    d = buf[3];

    MD5STEP(F1, a, b, c, d, in[0] + 0xd76aa478, 7);
    MD5STEP(F1, d, a, b, c, in[1] + 0xe8c7b756, 12);
    MD5STEP(F1, c, d, a, b, in[2] + 0x242070db, 17);
    MD5STEP(F1, b, c, d, a, in[3] + 0xc1bdceee, 22);
    MD5STEP(F1, a, b, c, d, in[4] + 0xf57c0faf, 7);
    MD5STEP(F1, d, a, b, c, in[5] + 0x4787c62a, 12);
    MD5STEP(F1, c, d, a, b, in[6] + 0xa8304613, 17);
    MD5STEP(F1, b, c, d, a, in[7] + 0xfd469501, 22);
    MD5STEP(F1, a, b, c, d, in[8] + 0x698098d8, 7);
    MD5STEP(F1, d, a, b, c, in[9] + 0x8b44f7af, 12);
    MD5STEP(F1, c, d, a, b, in[10] + 0xffff5bb1, 17);
    MD5STEP(F1, b, c, d, a, in[11] + 0x895cd7be, 22);
    MD5STEP(F1, a, b, c, d, in[12] + 0x6b901122, 7);
    MD5STEP(F1, d, a, b, c, in[13] + 0xfd987193, 12);
    MD5STEP(F1, c, d, a, b, in[14] + 0xa679438e, 17);
    MD5STEP(F1, b, c, d, a, in[15] + 0x49b40821, 22);

    MD5STEP(F2, a, b, c, d, in[1] + 0xf61e2562, 5);
    MD5STEP(F2, d, a, b, c, in[6] + 0xc040b340, 9);
    MD5STEP(F2, c, d, a, b, in[11] + 0x265e5a51, 14);
    MD5STEP(F2, b, c, d, a, in[0] + 0xe9b6c7aa, 20);
    MD5STEP(F2, a, b, c, d, in[5] + 0xd62f105d, 5);
    MD5STEP(F2, d, a, b, c, in[10] + 0x02441453, 9);
    MD5STEP(F2, c, d, a, b, in[15] + 0xd8a1e681, 14);
    MD5STEP(F2, b, c, d, a, in[4] + 0xe7d3fbc8, 20);
    MD5STEP(F2, a, b, c, d, in[9] + 0x21e1cde6, 5);
    MD5STEP(F2, d, a, b, c, in[14] + 0xc33707d6, 9);
    MD5STEP(F2, c, d, a, b, in[3] + 0xf4d50d87, 14);
    MD5STEP(F2, b, c, d, a, in[8] + 0x455a14ed, 20);
    MD5STEP(F2, a, b, c, d, in[13] + 0xa9e3e905, 5);
    MD5STEP(F2, d, a, b, c, in[2] + 0xfcefa3f8, 9);
    MD5STEP(F2, c, d, a, b, in[7] + 0x676f02d9, 14);
    MD5STEP(F2, b, c, d, a, in[12] + 0x8d2a4c8a, 20);

    MD5STEP(F3, a, b, c, d, in[5] + 0xfffa3942, 4);
    MD5STEP(F3, d, a, b, c, in[8] + 0x8771f681, 11);
    MD5STEP(F3, c, d, a, b, in[11] + 0x6d9d6122, 16);
    MD5STEP(F3, b, c, d, a, in[14] + 0xfde5380c, 23);
    MD5STEP(F3, a, b, c, d, in[1] + 0xa4beea44, 4);
    MD5STEP(F3, d, a, b, c, in[4] + 0x4bdecfa9, 11);
    MD5STEP(F3, c, d, a, b, in[7] + 0xf6bb4b60, 16);
    MD5STEP(F3, b, c, d, a, in[10] + 0xbebfbc70, 23);
    MD5STEP(F3, a, b, c, d, in[13] + 0x289b7ec6, 4);
    MD5STEP(F3, d, a, b, c, in[0] + 0xeaa127fa, 11);
    MD5STEP(F3, c, d, a, b, in[3] + 0xd4ef3085, 16);
    MD5STEP(F3, b, c, d, a, in[6] + 0x04881d05, 23);
    MD5STEP(F3, a, b, c, d, in[9] + 0xd9d4d039, 4);
    MD5STEP(F3, d, a, b, c, in[12] + 0xe6db99e5, 11);
    MD5STEP(F3, c, d, a, b, in[15] + 0x1fa27cf8, 16);
    MD5STEP(F3, b, c, d, a, in[2] + 0xc4ac5665, 23);

    MD5STEP(F4, a, b, c, d, in[0] + 0xf4292244, 6);
    MD5STEP(F4, d, a, b, c, in[7] + 0x432aff97, 10);
    MD5STEP(F4, c, d, a, b, in[14] + 0xab9423a7, 15);
    MD5STEP(F4, b, c, d, a, in[5] + 0xfc93a039, 21);
    MD5STEP(F4, a, b, c, d, in[12] + 0x655b59c3, 6);
    MD5STEP(F4, d, a, b, c, in[3] + 0x8f0ccc92, 10);
    MD5STEP(F4, c, d, a, b, in[10] + 0xffeff47d, 15);
    MD5STEP(F4, b, c, d, a, in[1] + 0x85845dd1, 21);
    MD5STEP(F4, a, b, c, d, in[8] + 0x6fa87e4f, 6);
    MD5STEP(F4, d, a, b, c, in[15] + 0xfe2ce6e0, 10);
    MD5STEP(F4, c, d, a, b, in[6] + 0xa3014314, 15);
    MD5STEP(F4, b, c, d, a, in[13] + 0x4e0811a1, 21);
    MD5STEP(F4, a, b, c, d, in[4] + 0xf7537e82, 6);
    MD5STEP(F4, d, a, b, c, in[11] + 0xbd3af235, 10);
    MD5STEP(F4, c, d, a, b, in[2] + 0x2ad7d2bb, 15);
    MD5STEP(F4, b, c, d, a, in[9] + 0xeb86d391, 21);

    buf[0] += a;
    buf[1] += b;
    buf[2] += c;
    buf[3] += d;
}

#undef F1
#undef F2
#undef F3
#undef F4
#undef MD5STEP

#pragma inline

#define F1(x, y, z) asm {mov eax, z; xor eax, y; \
                         and eax, x; xor eax, z}
#define F2(x, y, z) F1(z, x, y)
#define F3(x, y, z) asm {mov eax, x; xor eax, y; xor eax, z}
#define F4(x, y, z) asm {mov eax, z; not eax; or eax, x; xor eax, y}

#undef MD5_FARDATA
#undef MD5_NEARDATA

#if sizeof(char *) == 4
#define MD5_FARDATA
#define MD_DATAREG es
#define LES_ les
#else
#define MD5_NEARDATA
#define MD_DATAREG ds
#define LES_ mov
#endif

/* #define MD5STEP(f, w, x, y, z, data, s) \
        ( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x ) */
#define MD5STEP(f, w, x, y, z, di, mc, s) \
        f(x, y, z); asm { add eax, MD_DATAREG:[bx+di*4]; \
                                         add eax, (mc); add w, eax; \
                                                                  rol w, s; add w, x}


void MD5Transf386(uint32 buf[4], uint32 const in_[16])
{
/*  register uint32 a = buf[0], b = buf[1], c = buf[2], d = buf[3]; */
#define a esi
#define b edi
#define c ecx
#define d edx

   asm .386

        _SI = 0;
        _DI = 0; /* convince BCC to save and (later) restore di and si */

                  asm {
                LES_  bx, buf
                mov     a, MD_DATAREG:[bx+0*4];
                mov     b, MD_DATAREG:[bx+1*4];
                mov     c, MD_DATAREG:[bx+2*4];
                mov     d, MD_DATAREG:[bx+3*4];
                LES_   bx, in_
        }

  MD5STEP(F1, a, b, c, d, 0 /* in[ 0] */, 0xd76aa478,  7); /* 1 */
  MD5STEP(F1, d, a, b, c, 1 /* in[ 1] */, 0xe8c7b756, 12); /* 2 */
  MD5STEP(F1, c, d, a, b, 2 /* in[ 2] */, 0x242070db, 17); /* 3 */
  MD5STEP(F1, b, c, d, a, 3 /* in[ 3] */, 0xc1bdceee, 22); /* 4 */
  MD5STEP(F1, a, b, c, d, 4 /* in[ 4] */, 0xf57c0faf,  7); /* 5 */
  MD5STEP(F1, d, a, b, c, 5 /* in[ 5] */, 0x4787c62a, 12); /* 6 */
  MD5STEP(F1, c, d, a, b, 6 /* in[ 6] */, 0xa8304613, 17); /* 7 */
  MD5STEP(F1, b, c, d, a, 7 /* in[ 7] */, 0xfd469501, 22); /* 8 */
  MD5STEP(F1, a, b, c, d, 8 /* in[ 8] */, 0x698098d8,  7); /* 9 */
  MD5STEP(F1, d, a, b, c, 9 /* in[ 9] */, 0x8b44f7af, 12); /* 10 */
  MD5STEP(F1, c, d, a, b,10 /* in[10] */, 0xffff5bb1, 17); /* 11 */
  MD5STEP(F1, b, c, d, a,11 /* in[11] */, 0x895cd7be, 22); /* 12 */
  MD5STEP(F1, a, b, c, d,12 /* in[12] */, 0x6b901122,  7); /* 13 */
  MD5STEP(F1, d, a, b, c,13 /* in[13] */, 0xfd987193, 12); /* 14 */
  MD5STEP(F1, c, d, a, b,14 /* in[14] */, 0xa679438e, 17); /* 15 */
  MD5STEP(F1, b, c, d, a,15 /* in[15] */, 0x49b40821, 22); /* 16 */

  MD5STEP(F2, a, b, c, d, 1 /* in[ 1] */, 0xf61e2562,  5); /* 17 */
  MD5STEP(F2, d, a, b, c, 6 /* in[ 6] */, 0xc040b340,  9); /* 18 */
  MD5STEP(F2, c, d, a, b,11 /* in[11] */, 0x265e5a51, 14); /* 19 */
  MD5STEP(F2, b, c, d, a, 0 /* in[ 0] */, 0xe9b6c7aa, 20); /* 20 */
  MD5STEP(F2, a, b, c, d, 5 /* in[ 5] */, 0xd62f105d,  5); /* 21 */
  MD5STEP(F2, d, a, b, c,10 /* in[10] */, 0x02441453,  9); /* 22 */
  MD5STEP(F2, c, d, a, b,15 /* in[15] */, 0xd8a1e681, 14); /* 23 */
  MD5STEP(F2, b, c, d, a, 4 /* in[ 4] */, 0xe7d3fbc8, 20); /* 24 */
  MD5STEP(F2, a, b, c, d, 9 /* in[ 9] */, 0x21e1cde6,  5); /* 25 */
  MD5STEP(F2, d, a, b, c,14 /* in[14] */, 0xc33707d6,  9); /* 26 */
  MD5STEP(F2, c, d, a, b, 3 /* in[ 3] */, 0xf4d50d87, 14); /* 27 */
  MD5STEP(F2, b, c, d, a, 8 /* in[ 8] */, 0x455a14ed, 20); /* 28 */
  MD5STEP(F2, a, b, c, d,13 /* in[13] */, 0xa9e3e905,  5); /* 29 */
  MD5STEP(F2, d, a, b, c, 2 /* in[ 2] */, 0xfcefa3f8,  9); /* 30 */
  MD5STEP(F2, c, d, a, b, 7 /* in[ 7] */, 0x676f02d9, 14); /* 31 */
  MD5STEP(F2, b, c, d, a,12 /* in[12] */, 0x8d2a4c8a, 20); /* 32 */

  MD5STEP(F3, a, b, c, d, 5 /* in[ 5] */, 0xfffa3942,  4); /* 33 */
  MD5STEP(F3, d, a, b, c, 8 /* in[ 8] */, 0x8771f681, 11); /* 34 */
  MD5STEP(F3, c, d, a, b,11 /* in[11] */, 0x6d9d6122, 16); /* 35 */
  MD5STEP(F3, b, c, d, a,14 /* in[14] */, 0xfde5380c, 23); /* 36 */
  MD5STEP(F3, a, b, c, d, 1 /* in[ 1] */, 0xa4beea44,  4); /* 37 */
  MD5STEP(F3, d, a, b, c, 4 /* in[ 4] */, 0x4bdecfa9, 11); /* 38 */
  MD5STEP(F3, c, d, a, b, 7 /* in[ 7] */, 0xf6bb4b60, 16); /* 39 */
  MD5STEP(F3, b, c, d, a,10 /* in[10] */, 0xbebfbc70, 23); /* 40 */
  MD5STEP(F3, a, b, c, d,13 /* in[13] */, 0x289b7ec6,  4); /* 41 */
  MD5STEP(F3, d, a, b, c, 0 /* in[ 0] */, 0xeaa127fa, 11); /* 42 */
  MD5STEP(F3, c, d, a, b, 3 /* in[ 3] */, 0xd4ef3085, 16); /* 43 */
  MD5STEP(F3, b, c, d, a, 6 /* in[ 6] */, 0x04881d05, 23); /* 44 */
  MD5STEP(F3, a, b, c, d, 9 /* in[ 9] */, 0xd9d4d039,  4); /* 45 */
  MD5STEP(F3, d, a, b, c,12 /* in[12] */, 0xe6db99e5, 11); /* 46 */
  MD5STEP(F3, c, d, a, b,15 /* in[15] */, 0x1fa27cf8, 16); /* 47 */
  MD5STEP(F3, b, c, d, a, 2 /* in[ 2] */, 0xc4ac5665, 23); /* 48 */

  MD5STEP(F4, a, b, c, d, 0 /* in[ 0] */, 0xf4292244,  6); /* 49 */
  MD5STEP(F4, d, a, b, c, 7 /* in[ 7] */, 0x432aff97, 10); /* 50 */
  MD5STEP(F4, c, d, a, b,14 /* in[14] */, 0xab9423a7, 15); /* 51 */
  MD5STEP(F4, b, c, d, a, 5 /* in[ 5] */, 0xfc93a039, 21); /* 52 */
  MD5STEP(F4, a, b, c, d,12 /* in[12] */, 0x655b59c3,  6); /* 53 */
  MD5STEP(F4, d, a, b, c, 3 /* in[ 3] */, 0x8f0ccc92, 10); /* 54 */
  MD5STEP(F4, c, d, a, b,10 /* in[10] */, 0xffeff47d, 15); /* 55 */
  MD5STEP(F4, b, c, d, a, 1 /* in[ 1] */, 0x85845dd1, 21); /* 56 */
  MD5STEP(F4, a, b, c, d, 8 /* in[ 8] */, 0x6fa87e4f,  6); /* 57 */
  MD5STEP(F4, d, a, b, c,15 /* in[15] */, 0xfe2ce6e0, 10); /* 58 */
  MD5STEP(F4, c, d, a, b, 6 /* in[ 6] */, 0xa3014314, 15); /* 59 */
  MD5STEP(F4, b, c, d, a,13 /* in[13] */, 0x4e0811a1, 21); /* 60 */
  MD5STEP(F4, a, b, c, d, 4 /* in[ 4] */, 0xf7537e82,  6); /* 61 */
  MD5STEP(F4, d, a, b, c,11 /* in[11] */, 0xbd3af235, 10); /* 62 */
  MD5STEP(F4, c, d, a, b, 2 /* in[ 2] */, 0x2ad7d2bb, 15); /* 63 */
  MD5STEP(F4, b, c, d, a, 9 /* in[ 9] */, 0xeb86d391, 21); /* 64 */

  asm {
       LES_   bx, buf
       add    MD_DATAREG:[bx+0*4], a        /* buf[0] += a; */
       add    MD_DATAREG:[bx+1*4], b        /* buf[1] += b; */
       add    MD_DATAREG:[bx+2*4], c        /* buf[2] += c; */
       add    MD_DATAREG:[bx+3*4], d        /* buf[3] += d; */
        }

  asm .8086
#undef a
#undef b
#undef c
#undef d

}


#endif

