/*  Hey Emacs, please switch to -*-C-*-
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.

 */


/*
 * In order to keep track of encryption keys we store them in a set 
 * of databases.  The "real" index is the KeyID (64 bit long) and
 * you can use this to find a key based on the KeyID.
 * 
 * Then, the string(s) associated with each key ("User ID Packet" in
 * PGP terms) are stored in a separate database for faster lookup.
 *
 */

/* A name must be provided.  From it, we will create two file-names.
 * If these files exists, we will use them. if not, we will create them.
 */


#include "pgplib.h"
#include "pgp_m.h"
#include <errno.h>
#include <string.h>
#include <gdbm.h>
#include <sys/param.h>
#include <stdlib.h>
#include <ctype.h>

#define KEYEXT		".key"
#define STRINGEXT	".string"
#define	REVSTRINGEXT	".revstring"
#define KEYIDEXT	".keyid"

static GDBM_FILE	keydb;
static GDBM_FILE	stringdb;
static GDBM_FILE	revstringdb;
static GDBM_FILE	keyiddb;

int	dbase_open = FALSE;


void
close_databases()
{
    fprintf(stderr, "Fatal error detected by GDBM -- exit\n");
    exit(-1);
}

int
keydb_init(char 	*caller_name,
	   openmode_t	caller_mode)
{
    char 	filename[MAXPATHLEN];
    char 	*p;
    int		mode;

    if ( caller_name == NULL ) {
	fprintf(stderr, "keydb_init called with NULL");
	return(-1);
    }
    switch(caller_mode) {
    case READER: mode = GDBM_READER; break;
    case WRITER: mode = GDBM_WRITER; break;
    case NEWDB:  mode = GDBM_NEWDB ; break;
    default: return(-1);
    }
    filename[0] = NULL;
    strcat(filename, caller_name);
    strcat(filename, KEYEXT);
    keydb = gdbm_open (filename, 0, caller_mode, 0600, close_databases);
    if ( keydb == NULL ) {
	fprintf(stderr, "Error from gdbm_open %s (%s)\n", 
		filename, gdbm_strerror(gdbm_errno));
	return(-1);
    }

    p = rindex(filename, '.');
    *p = NULL;
    strcat(filename, STRINGEXT);
    stringdb = gdbm_open (filename, 0, caller_mode, 0600, close_databases);
    if ( stringdb == NULL ) {
	fprintf(stderr, "Error from gdbm_open %s (%s)\n",
		filename, gdbm_strerror(gdbm_errno));
	return(-1);
    }

    p = rindex(filename, '.');
    *p = NULL;
    strcat(filename, REVSTRINGEXT);
    revstringdb = gdbm_open (filename, 0, caller_mode, 0600, close_databases);
    if (revstringdb == NULL ) {
	fprintf(stderr, "Error from gdbm_open %s (%s)\n",
		filename, gdbm_strerror(gdbm_errno));
	return(-1);
    }

    p = rindex(filename, '.');
    *p = NULL;
    strcat(filename, KEYIDEXT);
    keyiddb = gdbm_open (filename, 0, caller_mode, 0600, close_databases);
    if ( keyiddb == NULL ) {
	fprintf(stderr, "Error from gdbm_open %s (%s)\n",
		filename, gdbm_strerror(gdbm_errno));
	return(-1);
    }
    dbase_open = TRUE;

    return(0);
}


void
keydb_end(void)
{
    if ( keydb != NULL )
	gdbm_close(keydb);
    if ( stringdb !=  NULL )
	gdbm_close(stringdb);
    if ( revstringdb != NULL )
	gdbm_close(revstringdb);
    if ( keyiddb != NULL )
	gdbm_close(keyiddb);
}

/* **************************************************************
 * Below is all the routines to store, fetch, alter and delete
 * PGP compatible keys and certificates.
 * All these routines take and return nicely formatted keys and certificates
 * as pointers to struct's.  The internal format (in the databases)
 * are less nice since it has to be marshalled (and, urk, compatible).
 *
 * In the Key database, the keys (together with UserID and signatures)
 * are stored in PGP format.  This way we can print the
 * key fast by simply "armour" the buffer directly.  It takes some
 * effort to convert the buffer to the correct type of C-struct's,
 * but that is the price we have to pay.
 *
 * The strings (UserID) are stored in a separate database.  This
 * way we can locate one _very_ fast
 *
 * The KeyID that pgp presents (only 32 bit) are stored in a third
 * database.
 * ************************************************************** */



/* 
 * Find a public key in the database.
 * The record also contains signatures and so on but here we only
 * return the actual key and 0.
 * No such key, return 1;
 * Some error: return -1;
 */

int
fetch_only_pubkey(u_quad_t 	key_id,
		  PGPpubkey_t 	*key)
{
    int 	i;
    datum	dbkey, contents;

    if ( dbase_open == FALSE )
	return(-1);

    dbkey.dptr  = (u_char*)&key_id;
    dbkey.dsize = sizeof(key_id);

    contents = gdbm_fetch(keydb, dbkey);
    if (contents.dptr == NULL ) {
	fprintf(stderr, "Key ID=%#.16llx not found\n", key_id);
	return(1);
    }
    i = buf2pubkey(contents.dptr, contents.dsize, &key);
    free(contents.dptr);
    
    if ( i != 0 )
	return(-1);
    
    return(0);
}
b
/* **************************************************************
 * Take a string ("0x12345678") and convert it into the corresponding
 * number, and find the key.  No key: return 1 and set keyid to 0.
 * An error returns -1;
 * ************************************************************** */

int
find_shortid(char 	*short_string,
	     u_quad_t	*keyid)
{

    datum	key, contents;
    u_long	short_id;
    char	*p;

    *keyid = 0LL;

    if ( short_string == NULL || *short_string == 0 )
	return(-1);
    
    short_id = strtoul(short_string, &p, 0);
    if ( p == NULL || *p != 0 )
	return(-1);

    key.dptr  = (char *)&short_id;
    key.dsize = sizeof(short_id);

    contents = gdbm_fetch(keyiddb, key);
    if ( contents.dptr != NULL )
	memcpy(keyid, contents.dptr, sizeof(u_quad_t));
    return(0);
}

int
store_pub_key(PGPpubkey_t *pubkey, 
	      repl_t 	replace)
{
    datum 	key, contents;
    int 	i,j;
    u_quad_t	q;
    u_long	short_id;

    assert ( replace == INSERT || replace == REPLACE);

    if ( dbase_open == FALSE )
	return(-1);

    q = BN_get_keyid(pubkey->N);
    key.dptr = (u_char *)&q;
    key.dsize= sizeof(q);
    i = pubkey2buf(pubkey,  (u_char **)&contents.dptr);
    if ( i < 0 ) {
	return(-1);
    }
    contents.dsize = i;
    j = 0;

    i = gdbm_store(keydb, key, contents, 
		   replace == REPLACE ? GDBM_REPLACE : GDBM_INSERT);
    switch (i) {
    case -1:
	fprintf(stderr, "Reader called STORE\n");
	break;
    case 1:
	break;
    case 0:
	break;
    default:
	fprintf(stderr, "Unknown return from gdbm_store\n");
	break;
    }
    free(contents.dptr);
    if ( i != 0 ) 
	return(i);

    /* The store went well.  We then proceed by adding the shorter
     * id presented by "pgp -kv", a string representing a 32 bit
     * number.  However, we store the string as the number it is
     * supposed to represent (occupies less than half the space)
     */
    
    short_id = (u_long) q;

    key.dptr  = (char *)&short_id;
    key.dsize = sizeof(short_id);
    
    contents.dptr = (char *) &q;
    contents.dsize= sizeof(q);

    i = gdbm_store(keyiddb, key, contents, INSERT);
    if ( i == 1 )
	/* We just hit 1/12.000 probability.  Warn someone. */
	fprintf(stderr, "1/12.000 just stroke.\n");

    return(i);
    
}

int
get_pub_key_record(u_quad_t key_id, u_char **buf, int *len)
{
    datum key, contents;

    if ( dbase_open == FALSE )
	return(-1);

    key.dptr = (u_char *)&key_id;
    key.dsize = sizeof(key_id);
    
    contents = gdbm_fetch (keydb, key);
    if ( contents.dptr == NULL ) {
	return(1);
    }
    *buf = contents.dptr;
    *len = contents.dsize;

    return(0);
}

int
store_pub_key_record(u_quad_t	key_id, 
		     repl_t 	user_replace, 
		     u_char 	*buf, 
		     int 	size)
{
    datum 	key, contents;
    int 	i;

    assert ( user_replace == REPLACE || user_replace == INSERT);

    if ( dbase_open == FALSE )
	return(-1);

    key.dptr = (u_char *)&key_id;
    key.dsize = sizeof(key_id);

    contents.dptr  = buf;
    contents.dsize = size;

    i = gdbm_store(keydb, key, contents, 
		   user_replace == REPLACE ? GDBM_REPLACE : GDBM_INSERT);
    
    return(i);
}
 
/* **************************************************************
 * If the key exists, return 1.  If not, return 0.  Error: -1
 * ************************************************************** */

int
key_exists(u_quad_t	key_id)
{
    datum	key;
    int 	i;

    if ( dbase_open == FALSE )
	return(-1);

    key.dptr  = (u_char*)&key_id;
    key.dsize = sizeof(key_id);

    i = gdbm_exists(keydb, key);
    if ( i == 0)
	return(0);
    else
	return (1);
}

/* **************************************************************
 * Storing a UID consists of two actions.  First, store the KeyID
 * indexed by the UID.  Then, store the UID indexed by the KeyID.
 * ************************************************************** */

int
store_user_id(PGPuserid_t	*user_id, 
	      u_quad_t 		key_id)
{
    datum	key, contents;	
    res_t	what;
    int		used, how_far, found, newlen;
    PGPuserid_t	*uid;
    u_char	*buf;
    /* scratch */
    int 	i;
    u_char	*p;


    if ( dbase_open == FALSE )
	return(-1);

    key.dptr  = user_id->name;
    key.dsize = strlen(user_id->name)+1;
    contents.dptr  = (u_char*) &key_id;
    contents.dsize = sizeof(key_id);
    i = gdbm_store(stringdb, key, contents, GDBM_REPLACE);
    switch (i) {
    case 0:
	i = 0; break;
    case 1:
	fprintf(stderr, "Reader called gdbm_store\n");
	i = -1; 
	break;
    case -1:
	fprintf(stderr, "gdbm_store returned -1\n");
	i = -1; 
	break;
    }
    key.dptr = (char *) &key_id;
    key.dsize = sizeof(key_id);
    contents = gdbm_fetch(revstringdb, key);
    if ( contents.dptr == NULL ) {
	/* This is the first UID for this key */
	contents.dsize = uid2buf(user_id, (u_char **) &contents.dptr);
	i = gdbm_store(revstringdb, key, contents, GDBM_REPLACE);
	free(contents.dptr);
	if ( i != 0 )
	    return(-1);
	else
	    return(0);
    }
    else {
	/* There was already (at least) one UID associated with
	 * this key.  We have to walk through them to check if
	 * the one we have already is here.
	 * Note that we have a PGP compatibale buffer at hand.
	 */
	how_far = 0;
	found = FALSE;
	while( how_far < contents.dsize  && found == FALSE) {

	    i = get_keyentry_from_buffer( contents.dptr  + how_far,
					  contents.dsize - how_far,
					  &what, NULL, NULL, &uid,
					  &used);
	    if ( i != 0 ) {
		free(contents.dptr);
		return(-1);
	    }
	    assert ( what == UID );

	    how_far += used;
	    
	    if ( strcmp(user_id->name, uid->name) == 0 ) {
		/* It's already here */
		found = TRUE;
		break;
	    }
	    else{
		/* Not this one */
		free_uid(uid);
		continue;
	    }		
	}
	if ( found == TRUE )
	    return(1);
	/* Add it (we have the packet at hand. don't we ?)*/
	newlen = uid2buf(user_id, &buf);
	if ( newlen == 0 )
	    return(-1);
	p = realloc(contents.dptr, contents.dsize + newlen);
	contents.dptr   = p;
	memcpy(contents.dptr+contents.dsize, buf, newlen);
	contents.dsize += newlen;

	i = gdbm_store(revstringdb, key, contents, GDBM_REPLACE);
	if ( i != 0 )
	    return(-1);
    }
    return(i);
}


/* **************************************************************
 * Find a string in the database.  We start by checking if we can
 * locate the exact string.  If not, we scan the database looking
 * for matches.
 * If save_case is TRUE we preserve case, if not, everything 
 * is converted to UPPER.
 * Return an the number of KeyID's we found with a match, and an 
 * array (at least) that big of the numbers.
 * ************************************************************** */
 
int
find_keys_from_string(char	*string,
		      int	save_case,
		      u_quad_t	**keys)
{
    datum	key, nextkey, contents;
    u_quad_t	*keyp, tmpkey;
    int		num_left, num_alloc, offset;
    char	upstring[256];
    int		already;

    char	*tmp;
    int		i;

    if ( dbase_open == FALSE )
	return(-1);

    if ( string == NULL || *string == 0 )
	return(0);
    key.dptr  = string;
    key.dsize = strlen(string) + 1;
    contents = gdbm_fetch(stringdb, key);
    if ( contents.dptr != 0 ) {
	/* Perfect match */
	keyp = malloc(sizeof(*keyp));
	if ( keyp == NULL ) {
	    free(contents.dptr);
	    fprintf(stderr, "Out of memory");
	    return(0);
	}
	*keys = (u_quad_t *)contents.dptr;
	return(1);
    }

    /* Hmmmm, we will have to scan the database */

#define	ALLOCATE	100

    offset = num_alloc = num_left = 0;
    keyp = NULL;

    key = gdbm_firstkey(stringdb);
    while ( key.dptr != NULL ) {
	nextkey = gdbm_nextkey ( stringdb, key );

	if (save_case == 0) {
	    memset(upstring, 0, 256);
	    for(i=0; i < key.dsize; i++)
		upstring[i] = toupper(key.dptr[i]);
	    for( tmp = string ; *tmp != NULL; tmp += 1)
		*tmp = toupper(*tmp);
	    tmp = strstr(upstring, string);
	}
	else
	    tmp = strstr(key.dptr, string);
	
	if ( tmp != NULL ) {
	    /* The string matches.  Get the KeyID */
	    if (num_left == 0 ) {
		tmp = realloc(keyp, (num_alloc+1)*(ALLOCATE*sizeof(u_quad_t)));
		if ( tmp == NULL ) {
		    free(keyp);
		    fprintf(stderr, "Out of memory\n");
		    return(0);
		}
		else
		    keyp = (u_quad_t *)tmp;
		num_alloc += 1;
		num_left = ALLOCATE;
	    }
	    contents = gdbm_fetch(stringdb, key);
	    /* just checking */
	    assert(contents.dsize == sizeof(u_quad_t));

	    memcpy(&tmpkey, contents.dptr, sizeof(u_quad_t));
	    assert(tmpkey != 0);
	    /* Check that it isn't already there */
	    already = FALSE;
	    for(i=0; i<offset; i++) {
		if ( keyp[i] == tmpkey) {
		    already = TRUE;
		    break;
		}
	    }
	    if ( already != TRUE ) {
		/* Add this key */
		keyp[offset] = tmpkey;
		offset += 1;
		num_left -= 1;
	    }
	}
	free(key.dptr);
	key = nextkey;
    }
    
    *keys = keyp;
    return(offset);
}

/* **************************************************************
 * Find the UIDs attached to this key.  Return of 1 means no UIDs
 * (or possibly unknown key ?), -1 is error, 0 is OK.
 * The array of pointers to UID's is NULL terminated.
 * ************************************************************** */


int
find_uids_with_key(u_quad_t	key_id,
		   PGPuserid_t	***uidpp)
{
    datum	key, contents;
    int		offset;
    res_t	what;
    int		ent;
    int		used;
    PGPuserid_t	**arr, **tmp;
    /* scratch */
    int		i,j;
    

    if ( dbase_open == FALSE )
	return(-1);

    key.dptr = (char *)&key_id;
    key.dsize = sizeof(key_id);
    
    contents = gdbm_fetch(revstringdb, key);
    if (contents.dptr == NULL )
	return(1);

    /* Ensure our array is NULL terminated */
    arr = calloc(1, sizeof(PGPuserid_t *));
    if ( arr == NULL )
	return(-1);

    ent = 1;
    offset = 0;
    do {
	tmp = realloc(arr, (ent+1) * sizeof(PGPuserid_t *));
	if ( tmp == NULL ) {	
	    free(arr);
	    return(-1);
	}
	arr = tmp;
	arr[ent] = NULL;
	i = get_keyentry_from_buffer(contents.dptr + offset,
				     contents.dsize - offset,
				     &what,
				     NULL, NULL, &arr[ent-1],
				     &used);
	switch ( i ) {
	case -1:
	    for(j=0; j<ent-2; j++)
		free(arr[j]);
	    free(arr);
	    return(-1);
	case 1:
	    *uidpp = arr;
	    free(contents.dptr);
	    return(0);
	case 0:
	    assert (what == UID);
	    offset += used;
	    ent += 1;
	    break;
	default:
	    assert(0);
	}
    } while (offset < contents.dsize);

    free(contents.dptr);
    *uidpp = arr;
    return(0);
}
