/* ./src/secure/readconf.c */

static char *rcsid = "$Id: readconf.c,v 1.9 1995/02/23 15:19:37 viebeg Exp $";

/* 
 *
 * $Id: readconf.c,v 1.9 1995/02/23 15:19:37 viebeg Exp $
 *
 * $Log: readconf.c,v $
 *
 */
 
/*
 *  , STARCOS Version 1.1
 */


/********************************************************************
 * Copyright (C) 1990-1994, GMD Darmstadt. All rights reserved.     *
 *                                                                  *
 *                                                                  *
 *                         NOTICE                                   *
 *                                                                  *
 *    Acquisition, use, and distribution of this module             *
 *    and related materials are subject to restrictions             *
 *    mentioned in each volume of the documentation.                *
 *                                                                  *
 ********************************************************************/

/*----------------Read SC Configuration File------------------------*/
/*------------------------------------------------------------------*/
/* GMD Darmstadt Institute for System Technic (I2)       	    */
/* Rheinstr. 75 / Dolivostr. 15                                     */
/* 6100 Darmstadt                                                   */
/* Project ``SECUDE'' 1993	                                    */
/*------------------------------------------------------------------*/
/*                                                                  */
/* PACKAGE   readconfig                                             */
/*                                      DATE   7.9.93	            */
/*                                        BY   U. Viebeg            */
/* DESCRIPTION                                                      */
/*                                                                  */
/* EXPORT                                                           */
/*   check_SCPSE_configuration() Check consistency of configuration */
/*				 data for an SC-PSE.	 	    */
/*   get_default_configuration() Set "sc_pse_list[]" to default	    */
/*  				 configuration.			    */
/*   display_SC_configuration() Display the SC configuration for    */
/*				the given PSE.  		    */
/*   read_SC_configuration()    Read SC configuration file into     */
/*				global structure "sc_pse_list[]".   */
/*                                                                  */
/* STATIC                                                           */
/*   add_predef_values		Add predefined objects (e.g. SC_PIN)*/
/*				and if app_id is missing the default*/
/*				application id is added.	    */
/*   char2int()			Transform ASCII string to integer   */
/*				value.				    */
/*   check_pse_name()		Check whether the actual read 	    */
/*				PSE name is unique in               */
/*				"sc_pse_list[]".		    */
/*   check_config_list()	Check whether the object list of    */
/*				an SC-PSE contains all mandatory    */
/*				objects.		    	    */
/*   check_obj_info()		Check whether the actual read       */
/* 				object is unique in object list.    */
/*   free_sc_pse_list()		Release all allocated memory in     */
/*   				in "sc_pse_list[]".		    */
/*   get_pse_info()		Get information (e.g. object list)  */
/*                              for current PSE.		    */
/*   get_default_index()        Get index in default object list.   */
/*   get_first_pse_record()     Get first PSE record in     	    */
/*			        file (incl. SC_encrypt, SC_verify). */
/*   get_next_correct_record()  Get next correct record in file and */
/* 				return the parameters.		    */
/*   get_next_word_from_record() Return next word in record.	    */
/*   get_obj_par()		Get object parameters from read     */
/*				record.				    */
/*   handle_mandatory_obj()     Handle mandatory objects in         */
/*				object list.			    */
/*   is_char_relevant()		Check whether character is relevant.*/
/*   is_obj_in_list()	        Check whether given object is part  */
/*			        of given list.			    */ 
/*   is_record_correct()	Check whether parameter in record   */
/*				are correct and return the par.     */
/*   is_record_relevant()	Check whether read record is a 	    */
/*				comment record. 		    */
/*   read_record()	        Read one record from file.	    */
/*   								    */
/*                                                                  */
/* IMPORT              		              		            */
/*   aux_PseObjData2SCObj()	Get information about an SC object  */
/*			        belonging to an SC-PSE.  	    */
/*   aux_PseObjName2SCObj()	Get information about an SC object  */
/*			        belonging to an SC-PSE.  	    */
/*                                                                  */
/*  Auxiliary Functions of SECUDE		                    */
/*   aux_add_error()		Add error to error stack.	    */
/*   aux_dechex()	        Decode printable hex 		    */
/*			        representation into hex.	    */
/*								    */
/*                                                                  */
/*  Global Variables                                                */
/*								    */
/*   sc_pse_list[]		List of the PSEs available  	    */
/*				on the SC, including the list of    */
/*				all objects (PSE specific),         */
/*				-which shall be stored on the SC or */
/*				-which are stored on the SC.	    */
/*   default_sc_pse_list[]	Default SC PSE list.		    */
/*   default_sc_obj_list[]	Default SC object list.		    */
/*   man_sc_obj_list[] 		List of the mandatory objects 	    */
/*				belonging to one SC PSE.   	    */
/*   onekeypair_sc_obj_list[] 	List of the mandatory objects 	    */
/*				for an SC-PSE with one RSA keypair. */
/*   twokeypairs_sc_obj_list[] 	List of the mandatory objects 	    */
/*				for an SC-PSE with two RSA keypairs.*/
/*------------------------------------------------------------------*/

#include  "secure.h"

#ifdef SCA

#include "secsc_config.h"	/* contains default SC configuration lists	 */



#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>



/*
 *    Local declarations
 */



typedef enum {
	NO_KEYWORD, PSE_NAME, APP_ID, NO_DES_KEYS, ONEKEYPAIR, SC_ENCRYPT, SC_VERIFY, OBJ_NAME}               ParType;


typedef struct Par_In_Record {
	ParType         par_type;
	union {
		char           *pse_name;
		OctetString    *app_id;
		Boolean         boolean_flag;
		int		int_parameter;
		SCObjEntry      obj_par;
	}               conf_par;
}               ParInRecord;





#ifdef __STDC__
static int	add_predef_values	(SCPseEntry *sc_pse_entry);
static RC	free_sc_pse_list	();
static int	get_pse_info		(FILE *fp_conf, SCPseEntry *sc_pse_entry, char **next_pse_name);
static int	get_first_pse_record	(FILE *fp_conf, ParInRecord *par_in_record);
static int	get_next_correct_record	(FILE *fp_conf, ParInRecord *par_in_record);
static int	read_record		(FILE *fp, char record[], int rmax);
static Boolean	is_record_correct	(char record[], ParInRecord *par_in_record);
static int	get_obj_par		(char *record, int *rindex, SCObjEntry *obj_par);
static int	is_record_relevant	(char *record);
static char	*get_next_word_from_record	(char *record, int *rindex);
static Boolean	is_char_relevant	(char one_char);
static int	check_pse_name		(char *read_pse_name);
static int	handle_mandatory_obj	(SCPseEntry *sc_pse_entry);
static int	get_default_index	(char *object_name);
static int	check_config_list	(SCPseEntry *sc_pse_entry, char *man_obj_list[]);
static int	is_obj_in_list		(char *obj_name, char *obj_list[]);
static int	check_obj_info		(SCPseEntry *sc_pse_entry, SCObjEntry *new_obj_par);
static int	char2int		(char *s);

#else

static int	add_predef_values();
static int      char2int();
static int      check_pse_name();
static int 	check_config_list();
static int      check_obj_info();
static int      get_pse_info();
static int      get_default_index();
static int      get_first_pse_record();
static int      get_next_correct_record();
static char    *get_next_word_from_record();
static int      handle_mandatory_obj();
static int      get_obj_par();
static Boolean  is_char_relevant();
static int	is_obj_in_list();
static Boolean  is_record_correct();
static int      is_record_relevant();
static RC       free_sc_pse_list();
static int      read_record();

#endif /* not __STDC__ */



/*
 *    Local variables, but global within readconf.c
 */

static	char           *config_file_name = CNULL;



/*--------------------------------------------------------------*/
/*						                */
/* PROC  read_SC_configuration				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Read the SC configuration file and stores the information   */
/*  in the global list "sc_pse_list[]".			        */
/*								*/
/*  Case 1:							*/
/*  If configuration file can not be opened 			*/
/*      => - "sc_pse_list[]" remains unchanged.			*/
/*         - return(0).						*/
/*								*/
/*  Case 2:							*/
/*  If configuration file can be opened, the file is read until */
/*  EOF is found or more than MAX_SC_APP PSEs are read.	        */
/*								*/
/*  In case of error(s): -  all errors are stacked with 	*/
/*  			    "aux_add_error()",			*/
/*                       -  sc_pse_list is set to 0,            */
/*                       -  return (-1).			*/
/*								*/
/****************************************************************/
/*  Handling of SC_encrypt, SC_verify flag:			*/
/*  One record for each flag may be in the configuration file	*/
/*  before the record with the first PSE name.			*/
/*								*/
/****************************************************************/
/*  Handling of mandatory objects:				*/
/*  For each PSE it is checked whether the belonging object list*/
/*  contains all mandatory objects.  If a  			*/
/*  mandatory object is missing, the default values for this    */
/*  object is added to the current object list.			*/
/*  The global list "man_sc_obj_list[]" contains the list of the*/
/*  mandatory objects.						*/
/*								*/
/****************************************************************/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   which_SCconfig	       = USER_CONF			*/
/*				 Search for the SC configuration*/
/*			         in the user directory.		*/
/*			       = SYSTEM_CONF			*/
/*			         Take name of SC configuration  */
/*				 file from define variable      */
/*				 SCCONF.			*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*  -1			       Error			       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*                                                              */
/*   add_predef_values	       Add predefined objects (e.g.     */
/*			       SC_PIN) and if app_id is missing */
/*			       the default application id is 	*/
/*			       added.	    			*/
/*   free_sc_pse_list()	       Release all allocated memory in  */
/*   			       in "sc_pse_list[]".		*/
/*   check_pse_name()	       Check whether the actual read 	*/
/*			       PSE name is unique in            */
/*			       "sc_pse_list[]".		        */
/*   get_pse_info()	       Get information (e.g. object list)*/
/*                             about current PSE. 	        */
/*   get_first_pse_record()    Get first PSE record in file.    */
/*   handle_mandatory_obj()    Handle mandatory objects in      */
/*			       object list.			*/
/*   aux_add_error()	       Add error to error stack.	*/
/*--------------------------------------------------------------*/


/***************************************************************
 *
 * Procedure read_SC_configuration
 *
 ***************************************************************/
#ifdef __STDC__

RC read_SC_configuration(
	WhichSCConfig	  which_SCconfig
)

#else

RC read_SC_configuration(
	which_SCconfig
)
WhichSCConfig	  which_SCconfig;

#endif

{
	int             rc = 0;

	char           *homedir = "";
	int             fd_conf;
	FILE           *fp_conf;
	int             function_rc = 0;
	int             pse_no;
	char           *next_pse_name;

	ParInRecord     par_in_record;


	char           *proc = "read_SC_configuration";

/********************************************************************************/
/*
 *	Compose name of configuration file: 
 */

	switch (which_SCconfig) {

	case USER_CONF: 

		/* 
		 *   Name = "home directory || SC_CONFIG_name"
		 */

		if (config_file_name)
			free (config_file_name);

		if(!(config_file_name = aux_cat_paths(getenv("HOME"), SC_CONFIG_name))) {
			AUX_ADD_ERROR;
			return(-1);
		}

		break;

	case SYSTEM_CONF: 

		/* 
		 *   configuration file name is taken from define variable "SCCONF".
		 */

#ifdef SCCONF
		if (config_file_name)
			free (config_file_name);

		if(!(config_file_name = aux_cpy_String(SCCONF))) {
			AUX_ADD_ERROR;
			return(-1);
		}

#else
		free(config_file_name);
		config_file_name = CNULL;
		return (0);

#endif

		break;
	default:
		aux_add_error(EINVALID, "Invalid value for parameter (which_SCconfig)", CNULL, 0, proc);
		return -1;

	}  /* end switch */
		


#ifdef CONFIGTEST
	if (config_file_name)
		fprintf(stderr, "Name of SC configuration file: %s\n", config_file_name);
#endif



/********************************************************************************/
/*
 *	Open configuration file
 */

	if ((fd_conf = open(config_file_name, O_RDONLY)) < 0) {
#ifdef SECSCTEST
		if (config_file_name)
			fprintf(stderr, "Configuration file %s missing!\n", config_file_name);
#endif
		if (config_file_name) {
			free (config_file_name);
			config_file_name = CNULL;
		}
		return (0);
	}
	if (!(fp_conf = (FILE *)fdopen(fd_conf, "r"))) {
		aux_add_error(ESYSTEM, "Configuration file cannot be opened", config_file_name, char_n, proc);
	        close(fd_conf);
		if (config_file_name) {
			free (config_file_name);
			config_file_name = CNULL;
		}
		return(-1);
	}
/********************************************************************************/
/*
 *	Read configuration file and store information into PSE list
 */

	/* init */
	next_pse_name = CNULL;

	/*
	 * Get first PSE name in configuration file (incl.
	 * SC_encrypt- , SC_verify-flag)
	 */

	rc = get_first_pse_record(fp_conf, &par_in_record);
	if (rc == ERR_flag) {
		aux_add_error(ECONFIG, "PSE: Error in get first PSE name ", CNULL, 0, proc);
		function_rc = ERR_flag;
	}
	if ((rc == EOF_flag) || (rc == EOF_with_ERR) || (par_in_record.par_type != PSE_NAME)) {
		aux_add_error(ECONFIG, "APPPSE: No PSE name in configuration file", CNULL, 0, proc);
		goto err_case;
	}
	/* name of first PSE */
	next_pse_name = par_in_record.conf_par.pse_name;


	/*
	 * Loop for SC-PSE(s) in configuration file (until EOF or too
	 * many PSEs in file ( max number = MAX_SCPSE))
	 * 
	 * Errors are stacked with "aux_add_error()"
	 * 
	 */

	pse_no = 0;
	sc_pse_list[pse_no].pse_name = CNULL;

	while ((pse_no < MAX_SCPSE) && (next_pse_name != CNULL) &&
	       (rc != EOF_flag) && (rc != EOF_with_ERR)) {

		/* check next PSE name */
		rc = check_pse_name(next_pse_name);
		if (rc < 0) {
			AUX_ADD_ERROR;
			function_rc = ERR_flag;
		}

		/*
		 * new entry in sc_pse_list (next_pse_name becomes current
		 * pse_name)
		 */
		sc_pse_list[pse_no].pse_name = next_pse_name;

		next_pse_name = CNULL;

		/*
		 * get parameters (e.g. object list) for PSE until
		 * EOF or next PSE name found
		 */
		rc = get_pse_info(fp_conf, &sc_pse_list[pse_no], &next_pse_name);
		if ((rc == ERR_flag) || (rc == EOF_with_ERR)) {
			aux_add_error(ECONFIG, "PSE: Error in configuration for PSE ", sc_pse_list[pse_no].pse_name, char_n, proc);
			function_rc = ERR_flag;
		}
		

		/*
		 *  Add predefined values to the parameter list:
		 *
		 *	- add all predefined objects
		 *      - if an app_id is missing, 
	         *		the default application id is added.
		 */

		rc = add_predef_values(&sc_pse_list[pse_no]);
		if (rc == ERR_flag) {
			aux_add_error(ECONFIG, "PSE: Error in configuration for PSE ", sc_pse_list[pse_no].pse_name, char_n, proc);
			function_rc = ERR_flag;
		}


		/*
		 * If mandatory objects are missing in current object list =>
		 * add them.
		 */
		rc = handle_mandatory_obj(&sc_pse_list[pse_no]);
		if (rc == ERR_flag) {
			aux_add_error(ECONFIG, "PSE: Error in configuration for PSE ", sc_pse_list[pse_no].pse_name, char_n, proc);
			function_rc = ERR_flag;
		}
		pse_no++;
		sc_pse_list[pse_no].pse_name = CNULL;

	}			/* end while */


	if (pse_no >= MAX_SCPSE) {
		aux_add_error(ECONFIG, "PSE: Too many PSEs in configuration file", CNULL, 0, proc);
		goto err_case;
	}
	if ((rc == EOF_with_ERR) || (function_rc == ERR_flag)) {
		goto err_case;
	}

/*
 *	release allocated memory (normal end)
 */

	close(fd_conf);
	fclose(fp_conf);
	return (0);




/*
 *	release allocated memory (in case or error)
 */

err_case:

	/* release all allocated memory and set "sc_pse_list[]" to 0 */
	rc = free_sc_pse_list();

	close(fd_conf);
	fclose(fp_conf);
	aux_add_error(ECONFIG, "Syntax error in SC configuration file ", config_file_name, char_n, proc);
	if (config_file_name) {
		free(config_file_name);
		config_file_name = CNULL;
	}
	if (next_pse_name != CNULL)
		free(next_pse_name);
	return (-1);

}				/* end read_SC_configuration */








/*--------------------------------------------------------------*/
/*						                */
/* PROC  display_SC_configuration			       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Display the SC configuration for the given PSE. 	        */
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   psename		  	Name of the PSE for which the   */
/*				configuration data shall be 	*/
/*				displayed.			*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*                                                              */
/*--------------------------------------------------------------*/



/***************************************************************
 *
 * Procedure display_SC_configuration
 *
 ***************************************************************/
#ifdef __STDC__

RC display_SC_configuration(
	char		*psename
)

#else

RC display_SC_configuration(
	psename
)
	char		*psename;

#endif

{
	int		SC_available;
	SCPseEntry     *sc_pse_entry;
	SCObjType	type;
	Boolean		predef_obj;
	Boolean		headline;
	char 	       *sm_mode;
	OctetString    *enc_app_id;

	int             n, i;


	/*
	 *  Perform SC configuration
	 */

	if ((SC_available = SC_configuration()) == -1) {
		AUX_ADD_ERROR;
		return (-1);
	}
	if (SC_available == FALSE) {

		/*
		 *  no SC configuration file found)
		 */

		aux_add_error(ECONFIG, "No SC configuration file found.", CNULL, 0, proc);
		return (-1);
	}


	/*
	 *  Get entry of the SC-PSE
	 */

	sc_pse_entry = aux_PseName2SCPse(psename);
	if (sc_pse_entry == (SCPseEntry * ) 0) 	{
		if ((config_file_name) && (strlen(config_file_name)))
			fprintf(stderr, "\n\nUsed SC configuration file:        %s\n\n", config_file_name);
		aux_add_error(EINVALID, "PSE is not an SC-PSE", psename, char_n, proc);
		return (-1);
	}

	fprintf(stderr, "\n\nCONFIGURATION for PSE:          %s\n", sc_pse_entry->pse_name);

	if ((config_file_name) && (strlen(config_file_name)))
		fprintf(stderr, "\n\nUsed SC configuration file:     %s\n\n", config_file_name);


	fprintf(stderr, "Global flags:\n");
	if (SC_encrypt == TRUE)
		fprintf(stderr, "   SC_encrypt:                  TRUE\n");
	else
		fprintf(stderr, "   SC_encrypt:                  FALSE\n");


	if (SC_verify == TRUE)
		fprintf(stderr, "   SC_verify:                   TRUE\n");
	else
		fprintf(stderr, "   SC_verify:                   FALSE\n");


	/*
	 *  Display configuration of the given PSE
	 */

	if (!(enc_app_id = aux_enchex(sc_pse_entry->app_id))) {
		aux_add_error(ECONFIG, "Cannot encode hex representation of the application id", CNULL, char_n, proc);
		return (-1);
	}

	enc_app_id->octets[enc_app_id->noctets] = '\0';

        fprintf(stderr, "\nApplication-id:                 %s\n",enc_app_id->octets);
	aux_free_OctetString(&enc_app_id);


/*NO_OF_DESKeys*/ 	
	fprintf(stderr, "Number of DES keys:             %-2d\n", sc_pse_entry->no_of_DESKeys);


/*one keypair*/ 	
	if (sc_pse_entry->onekeypair == TRUE)
		fprintf(stderr, "One keypair\n");
	else
		fprintf(stderr, "Two keypairs\n");


	fprintf(stderr, "\n\nList of the SC-objects:\n");

	type = SC_KEY_TYPE;	/* display the key-objects first */
	headline = TRUE;

repeat:

	for (i = 0; sc_pse_entry->sc_obj_list[i].name; i++) {

		if (sc_pse_entry->sc_obj_list[i].type != type)
			continue;

		/*
		 * Predefined objects are not displayed
		 */

		predef_obj = FALSE;

		for (n = 0; predef_sc_obj_list[n]; n++) {
			if (strcmp(sc_pse_entry->sc_obj_list[i].name, predef_sc_obj_list[n]) == 0) {
				predef_obj = TRUE;
				break;
			}
		}
		if (predef_obj == FALSE) {

			/*
			 *  Display Object
	        	 */

/*id=key*/ 		if (sc_pse_entry->sc_obj_list[i].type == SC_KEY_TYPE) {

/*key headline*/ 		if (headline == TRUE) {
					fprintf(stderr, "\n   KEYs:      Object Name       Key-Id\n");
					headline = FALSE;
				}

				fprintf(stderr, "                  %-16s%2d\n", sc_pse_entry->sc_obj_list[i].name, sc_pse_entry->sc_obj_list[i].sc_id);


			} else {

/*file headline*/ 		if (headline == TRUE) {
					fprintf(stderr, "\n   FILEs:     Object Name       File-Id     Max. Size     Sec. Mess. Mode\n");
					headline = FALSE;
				}

				if (sc_pse_entry->sc_obj_list[i].sm_SCT == SCT_SEC_NORMAL)
					sm_mode= " NONE";
				else if (sc_pse_entry->sc_obj_list[i].sm_SCT == SCT_INTEGRITY)
					sm_mode= " INTG";
				else if (sc_pse_entry->sc_obj_list[i].sm_SCT == SCT_CONCEALED)
					sm_mode= " CONC";
				else if (sc_pse_entry->sc_obj_list[i].sm_SCT == SCT_COMBINED)
					sm_mode= " COMB";
				
	 			fprintf(stderr, "                  %-16s%2d	       %4d	       %s\n", 
sc_pse_entry->sc_obj_list[i].name, 
sc_pse_entry->sc_obj_list[i].sc_id,
sc_pse_entry->sc_obj_list[i].size,
sm_mode);


			}
		}

	}		/* end for objects */
	if (type == SC_KEY_TYPE) {
		type = SC_FILE_TYPE;	/* display the key-objects first */
		headline = TRUE;
		goto repeat;
	}



	fprintf(stderr, "\n\n");

}				/* end display_SC_configuration */






/*--------------------------------------------------------------*/
/*						                */
/* PROC  get_default_configuration			       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Store the default configuration list "default_sc_pse_list[]"*/
/*  in the configuration list "sc_pse_list[]". For each         */
/*  PSE the "default_sc_obj_list[]" is used as default          */
/*  object list.						*/
/*								*/
/*??????????????????????????????????????????????????????????????*/
/*	Default value for the application id missing		*/
/*??????????????????????????????????????????????????????????????*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*                                                              */
/*--------------------------------------------------------------*/


/***************************************************************
 *
 * Procedure get_default_configuration
 *
 ***************************************************************/
#ifdef __STDC__

RC get_default_configuration(
)

#else

RC get_default_configuration(
)

#endif

{
	int             n = 0;
	int             i = 0;



	for (n = 0; default_sc_pse_list[n]; n++) {

		sc_pse_list[n].pse_name = default_sc_pse_list[n];
		sc_pse_list[n].app_id = NULLOCTETSTRING;

		/* get SC object list */
		for (i = 0; default_sc_obj_list[i].name; i++)
			sc_pse_list[n].sc_obj_list[i] = default_sc_obj_list[i];
		sc_pse_list[n].sc_obj_list[i].name = CNULL;
	}

	sc_pse_list[n].pse_name = CNULL;

	return (0);

}				/* end get_default_configuration */




/*--------------------------------------------------------------*/
/*						                */
/* PROC  free_sc_pse_list				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Release all allocated memory in "sc_pse_list[]" and set 	*/
/*  "sc_pse_list[]" to 0.					*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/


/***************************************************************
 *
 * Procedure free_sc_pse_list
 *
 ***************************************************************/
#ifdef __STDC__

static RC free_sc_pse_list(
)

#else

static RC free_sc_pse_list(
)

#endif

{

	int             n = 0;
	int             i = 0;

	char 		*proc  = "free_sc_pse_list";




	for (n = 0; sc_pse_list[n].pse_name; n++) {

		if (sc_pse_list[n].pse_name)
			free(sc_pse_list[n].pse_name);
		if (sc_pse_list[n].app_id)
			aux_free_OctetString(&sc_pse_list[n].app_id);

		for (i = 0; sc_pse_list[n].sc_obj_list[i].name; i++) {
			if (sc_pse_list[n].sc_obj_list[i].name)
				free(sc_pse_list[n].sc_obj_list[i].name);
		}		

	}

	sc_pse_list[0].pse_name = CNULL;

	return (0);

}





/*--------------------------------------------------------------*/
/*						                */
/* PROC  get_pse_info					       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Read information (object list) for the current PSE 	        */
/*  and returns the information in paramter "sc_pse_entry" .    */
/*								*/
/*  Read file until next PSE or EOF. Errors are stacked         */
/*  with "aux_add_error()".					*/
/*								*/
/****************************************************************/
/*  Handling of the flags SC_Encrypt and SC_verify		*/
/*  These flags are ignored within the list of the PSE          */
/*  parameters.							*/
/*								*/
/****************************************************************/
/*  APPLICATION_ID, NO_DES_KEYS, ONEKEYPAIR:			*/
/*  These parameters are expected to be in the records between	*/
/*  the PSE name and the belonging object list.	   	        */
/*								*/
/****************************************************************/
/*  Handling of the OBJECTS					*/
/*  If the number of objects read for one PSE is greater        */
/*  than MAX_SCOBJ, all further objects are stored in 		*/
/*  "sc_pse_entry->sc_obj_list[MAX_SCOBJ]".			*/
/*  Within the object-list all other PSE parameters are         */
/*  ignored.							*/
/*  								*/
/****************************************************************/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   fp_conf			Pointer to configuration file 	*/
/*   *sc_pse_entry  		Pointer to an entry in the      */
/*				"sc_pse_list[]".		*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*   sc_pse_entry  		Entry of the "sc_pse_list":	*/
/*				- app_id			*/
/*				- onekeypair flag		*/
/*				- no_of_DESKeys			*/
/*				- object_list			*/
/*   next_pse_name		name of the next PSE		*/
/*				(memory for next_pse_name is    */
/*				 allocated)			*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*  ERR_flag		       Error			       	*/
/*  EOF_flag		       EOF found		       	*/
/*  EOF_with_ERR	       Error and EOF found .       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   check_obj_info()	       Check whether the actual read    */
/* 			       object is unique in object list. */
/*   get_next_correct_record() Get next correct record in file  */
/* 			       and return the parameters.       */
/*   aux_add_error()	       Add error to error stack.	*/
/*--------------------------------------------------------------*/



/***************************************************************
 *
 * Procedure get_pse_info
 *
 ***************************************************************/
#ifdef __STDC__

static int get_pse_info(
	FILE		 *fp_conf,
	SCPseEntry	 *sc_pse_entry,
	char		**next_pse_name
)

#else

static int get_pse_info(
	fp_conf,
	sc_pse_entry,
	next_pse_name
)
FILE		 *fp_conf;
SCPseEntry	 *sc_pse_entry;
char		**next_pse_name;

#endif

{
	int             rc = 0;


	int             obj_no;
	ParInRecord     par_in_record;
	Boolean         found_next_pse_name;
	Boolean         found_object_name;

	int             function_rc = 0;


	char           *proc = "get_pse_info";

	/* init parameters for an PSE*/
	sc_pse_entry->app_id = NULLOCTETSTRING;
	sc_pse_entry->no_of_DESKeys = 1;
	sc_pse_entry->onekeypair = TRUE;
	


	/*
	 * Get next correct record in file and return all parameters
	 */

	rc = get_next_correct_record(fp_conf, &par_in_record);
	if (rc == ERR_flag) {
/*	   	aux_add_error (ECONFIG , "OBJ: Error in get correct record ", CNULL, 0, proc); */
		function_rc = ERR_flag;
	}
	if ((rc == EOF_flag) || (rc == EOF_with_ERR)) {
		return (rc);
	}

	/*
	 * Loop for parameters belonging to one PSE:
	 *    - APPLICATION IDentifier, this parameter is optional 
	 *    - NO_DES_KEYS, this parameter is optional, default value is 1
	 *    - ONEKEYPAIR, this parameter is optional, default value is TRUE
	 *
	 * Loop ends in case of an error, EOF found, first object or next PSE name found.
	 */


	found_object_name = FALSE;
	found_next_pse_name = FALSE;

	while ((rc != EOF_flag) && (rc != EOF_with_ERR) &&
	       (found_next_pse_name == FALSE) && (found_object_name == FALSE)) {

		switch (par_in_record.par_type) {
		case PSE_NAME:
			/* get name of the next PSE in file */
			*next_pse_name = par_in_record.conf_par.pse_name;
			found_next_pse_name = TRUE;
			break;

		case NO_DES_KEYS:
			sc_pse_entry->no_of_DESKeys = par_in_record.conf_par.int_parameter;
			break;

		case ONEKEYPAIR:
			sc_pse_entry->onekeypair = par_in_record.conf_par.boolean_flag;
			break;

		case SC_ENCRYPT:
		case SC_VERIFY:

			/*
			 * within the list of the PSE parameters the flags 
			 * "SC_ENCRYPT", "SC_VERIFY" are ignored.
			 */

			break;

		case APP_ID:
			sc_pse_entry->app_id = par_in_record.conf_par.app_id;
			break;

		case OBJ_NAME:
			found_object_name = TRUE;
			break;

		default:
			aux_add_error(ECONFIG, "Invalid parameter record for PSE ", sc_pse_entry->pse_name, char_n, proc);
			function_rc = ERR_flag;
			break;

		}		/* end switch */


		if ((found_next_pse_name == FALSE) && (found_object_name == FALSE)) {
			rc = get_next_correct_record(fp_conf, &par_in_record);

			if (rc == ERR_flag) {
/*	    			aux_add_error (ECONFIG , "OBJ: Error in get correct record ", CNULL, 0, proc);*/
				function_rc = ERR_flag;
			}
			if ((rc == EOF_flag) || (rc == EOF_with_ERR)) 
				return (rc);

		}
	}			/* end while */




	/*
	 * Loop for objects belonging to the actual PSE (until error or
	 * EOF or next PSE name found)
	 */

	obj_no = 0;
	sc_pse_entry->sc_obj_list[obj_no].name = CNULL;

	while ((rc != EOF_flag) && (rc != EOF_with_ERR) &&
	       (found_next_pse_name == FALSE)) {

		switch (par_in_record.par_type) {
		case PSE_NAME:
			/* get name of the next PSE in file */
			*next_pse_name = par_in_record.conf_par.pse_name;
			found_next_pse_name = TRUE;
			break;

		case NO_DES_KEYS:
		case SC_ENCRYPT:
		case SC_VERIFY:
		case APP_ID:
		case ONEKEYPAIR:

			/*
			 * within the object list PSE parameters as  
			 * "NO_OF_DES_KEYS", "SC_ENCRYPT", "SC_VERIFY", "APPLICATION_ID" 
			 * and "ONEKEYPAIR 
			 * are ignored.
			 */

			break;

		case OBJ_NAME:

			/*
			 * check new object (e.g. compare with already
			 * entered objects)
			 */
			rc = check_obj_info(sc_pse_entry, &par_in_record.conf_par.obj_par);
			if (rc < 0) {
				aux_add_error(ECONFIG, "OBJ: Invalid object for PSE", sc_pse_entry->pse_name, char_n, proc);
				function_rc = ERR_flag;
			}

			/*
			 * enter new object in obj_list[obj_no].
			 */
			sc_pse_entry->sc_obj_list[obj_no] = par_in_record.conf_par.obj_par;

#ifdef CONFIGTEST
			fprintf(stderr, "Name of new entered object : %s\n", sc_pse_entry->sc_obj_list[obj_no].name);
#endif


			obj_no++;

			/*
			 * if obj_no >= MAX_SCOBJ => obj_no will be set to
			 * MAX_SCOBJ
			 */
			if (obj_no >= MAX_SCOBJ) {
				aux_add_error(ECONFIG, "OBJ: Too many objects for PSE ", sc_pse_entry->pse_name, char_n, proc);
				obj_no = MAX_SCOBJ;
				function_rc = ERR_flag;
			}
			sc_pse_entry->sc_obj_list[obj_no].name = CNULL;
			break;

		default:
			aux_add_error(ECONFIG, "Invalid parameter record for PSE ", sc_pse_entry->pse_name, char_n, proc);
			function_rc = ERR_flag;
			break;

		}		/* end switch */

		if (found_next_pse_name == FALSE) {
			rc = get_next_correct_record(fp_conf, &par_in_record);
			if ((rc == ERR_flag) || (rc == EOF_with_ERR)) {
/*	         aux_add_error (ECONFIG , "OBJ: Error in get correct record ", CNULL, 0, proc); */
				function_rc = ERR_flag;
			}
		}
	}			/* end while */


	/*
	 * Return value
	 */

	if (rc == EOF_with_ERR)
		return (EOF_with_ERR);

	if (rc == EOF_flag) {
		if (function_rc == ERR_flag)
			return (EOF_with_ERR);
		else
			return (EOF_flag);
	} else
		return (function_rc);




}				/* end get_pse_info */










/*--------------------------------------------------------------*/
/*						                */
/* PROC  get_first_pse_record				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Read configuration file until first record with PSE name is */
/*  found or EOF.						*/
/*								*/
/****************************************************************/
/*  Handling of SC_encrypt, SC_verify flag:			*/
/*  The first relevant records of the file may contain the 	*/
/*  SC_ENCRYPT and/or SC_VERIFY flags. According to these flags */
/*  the global variables SC_encrypt, SC_verify are set.		*/
/*								*/
/****************************************************************/
/*  Case 1:							*/
/*  If pse_name can be found:					*/
/*     => - pse_name is set in "par_in_record->conf_par.pse_name"*/
/*        - return(0)						*/
/*								*/
/*  Case 2:							*/
/*  If no pse_name can be found (EOF reached):			*/
/*     => return (EOF_with_ERR)					*/
/*								*/
/*  Case 3:							*/
/*  If records with PSE parameters (e.g. OBJECT)  	        */
/*  are found before the record with an PSE name:		*/
/*     => - error(s) are stacked with "aux_add_error()",	*/
/*        - "par_in_record->conf_par.pse_name" is set to the the*/
/*          found PSE name,					*/
/*        - return(ERR_flag).					*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   fp_conf			Pointer to configuration file 	*/
/*   *par_in_record		Pointer to structure.		*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*   par_in_record		If not ERR, this structure	*/
/*			        contains the correct pse_name	*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*  ERR_flag		       Error			       	*/
/*  EOF_with_ERR	       Error and EOF found	       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   get_next_correct_record()  Get next correct record in file */
/* 				and return the parameters.      */
/*   aux_add_error()	       Add error to error stack.	*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure get_first_pse_record
 *
 ***************************************************************/
#ifdef __STDC__

static int get_first_pse_record(
	FILE		 *fp_conf,
	ParInRecord	 *par_in_record
)

#else

static int get_first_pse_record(
	fp_conf,
	par_in_record
)
FILE		 *fp_conf;
ParInRecord	 *par_in_record;

#endif

{


	int             function_rc = 0;
	Boolean         pse_name_found = FALSE;
	int             rc;

	char           *proc = "get_first_pse_record";

#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  get_first_pse_record\n\n");*/
#endif


	do {

		rc = get_next_correct_record(fp_conf, par_in_record);
		if ((rc == EOF_flag) || (rc == EOF_with_ERR)) {
			aux_add_error(ECONFIG, "PSE: Cannot find first pse_name in file. ", CNULL, 0, proc);
			return (EOF_with_ERR);
		}
		if (rc == ERR_flag)
			function_rc = 0;	/* ignore all leading invalid
						 * records */

		/* check type of record */
		switch (par_in_record->par_type) {

		case PSE_NAME:
			pse_name_found = TRUE;
			break;

		case OBJ_NAME:
			aux_add_error(ECONFIG, "PSE: Object record found before pse_name", CNULL, 0, proc);
			/* release memory for object name */
			free(par_in_record->conf_par.obj_par.name);
			par_in_record->conf_par.obj_par.name = CNULL;
			function_rc = ERR_flag;
			break;

		case NO_DES_KEYS:
			aux_add_error(ECONFIG, "PSE: Number of DES keys found before pse_name", CNULL, 0, proc);
			function_rc = ERR_flag;
			break;

		case ONEKEYPAIR:
			aux_add_error(ECONFIG, "PSE: One keypair flag found before pse_name", CNULL, 0, proc);
			function_rc = ERR_flag;
			break;

		case SC_ENCRYPT:
			SC_encrypt = par_in_record->conf_par.boolean_flag;
			break;

		case SC_VERIFY:
			SC_verify = par_in_record->conf_par.boolean_flag;
			break;

		default:
			aux_add_error(ECONFIG, "PSE: Unknown record type found before pse_name", CNULL, 0, proc);
			function_rc = ERR_flag;
			break;

		}		/* end switch */


	} while (pse_name_found == FALSE);


	/*
	 * Return value
	 */

	return (function_rc);

}				/* end get_first_pse_record */








/*--------------------------------------------------------------*/
/*						                */
/* PROC  get_next_correct_record			       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Get next correct record in file and return all parameters	*/
/*  i.e. jumb over all leading comment-, blank- or  empty 	*/
/*  and incorrect records.					*/
/*								*/
/*  This function reads the file until a correct record is found*/
/*  or EOF. 							*/
/*								*/
/*  In case of an error (e.g. invalid keyword, invalid par.)    */
/*  the error together with the actual record is stacked  	*/
/*  in function "is_record_correct" with "aux_add_error()".	*/
/*  								*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   fp_conf			Pointer to configuration file 	*/
/*   *par_in_record		Pointer to structure.		*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*   par_in_record		If not EOF, this structure	*/
/*			        contains the correct par.	*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*  ERR_flag		       Error			       	*/
/*  EOF_flag		       EOF found		       	*/
/*  EOF_with_ERR	       Error and EOF found	       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   is_record_correct()       Check whether parameter in record*/
/*			       are correct and return the par.  */
/*   is_record_relevant()      Check whether read record is a 	*/
/*			       comment record. 		        */
/*   read_record()	       Read one record from file.	*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure get_next_correct_record
 *
 ***************************************************************/
#ifdef __STDC__

static int get_next_correct_record(
	FILE		 *fp_conf,
	ParInRecord	 *par_in_record
)

#else

static int get_next_correct_record(
	fp_conf,
	par_in_record
)
FILE		 *fp_conf;
ParInRecord	 *par_in_record;

#endif

{


	int             function_rc = 0;
	Boolean         found_relevant_record = FALSE;
	Boolean         found_correct_record = FALSE;
	char            record[MAX_RECORD + 1];	/* one character for '\0'             */
	int             rc;


#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  get_next_correct_record\n\n");*/
#endif


	do {

		rc = read_record(fp_conf, record, MAX_RECORD);

		if (rc != EOF_flag) {
			found_relevant_record = is_record_relevant(record);
			if (found_relevant_record == TRUE) {
				found_correct_record = is_record_correct(record, par_in_record);
				if (found_correct_record == FALSE)
					function_rc = ERR_flag;
			}
		}
	} while ((found_correct_record == FALSE) && (rc != EOF_flag));


	/*
	 * Return value
	 */

	if (rc == EOF_flag) {
		if (function_rc == ERR_flag)
			return (EOF_with_ERR);
		else
			return (EOF_flag);
	} else
		return (function_rc);

}				/* end get_next_correct_record */






/*--------------------------------------------------------------*/
/*						                */
/* PROC  read_record					       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Read one record or max n chars from file. The read chars	*/
/*  are concatenated with '\0' and returned in parameter 	*/
/*  "record".						        */
/*  								*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   fp_conf			Pointer to configuration file 	*/
/*   rmax			Max no. of chars to be read.	*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*   record			Read record			*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*  EOF_flag		       EOF found		       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure read_record
 *
 ***************************************************************/
#ifdef __STDC__

static int read_record(
	FILE	 *fp,
	char	  record[],
	int	  rmax
)

#else

static int read_record(
	fp,
	record,
	rmax
)
FILE	 *fp;
char	  record[];
int	  rmax;

#endif

{

	int             ctr;	/* counter for read chars */
	int             c;	/* single read character */
	int             NL_found;


#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  read_record\n\n");*/
#endif

	ctr = 0;
	NL_found = 0;


	do {
		c = getc(fp);
		if (c != EOF) {
			if (c == '\n')
				NL_found = 1;
			record[ctr] = c;
			ctr++;
		}		/* end if */
	} while ((c != EOF) && (ctr < rmax) && (NL_found == 0));

	if (NL_found == 1) {
		record[ctr - 1] = '\0';
		return (ctr);
	}
	if (c == EOF)
		return (EOF_flag);


	record[ctr] = '\0';
	return (ctr);		/* return number of read characters */


}				/* read_record */








/*--------------------------------------------------------------*/
/*						                */
/* PROC  is_record_relevant				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check delivered record and 					*/
/*        return FALSE, if record is empty or			*/
/*        	        if record consists of blanks or		*/
/*                      if record contains a comment.		*/
/*  Otherwise tailing comments are cut off in record and TRUE is*/
/*  returned.							*/
/*  The delivered record is supposed to be a string (null-	*/
/*  terminated). 						*/
/*  								*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   TRUE	         	Record is relevant.       	*/
/*   FALSE	         	Record is not relevant.       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure is_record_relevant
 *
 ***************************************************************/
#ifdef __STDC__

static int is_record_relevant(
	char	 *record
)

#else

static int is_record_relevant(
	record
)
char	 *record;

#endif

{


	int             i;
	Boolean         relevant_sign_found;
	Boolean         comment_sign_found;

#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  is_record_relevant\n\n");*/
#endif


	/*
	 * is record empty ?
	 */

	if ((record == CNULL) || ((int)strlen(record) == 0))
		return (FALSE);


	/* get first character which is != BLANK, COMMENT, TAB */

	i = 0;
	relevant_sign_found = FALSE;
	while ((i < (int)strlen(record)) &&
	       (record[i] != COMMENT) && (relevant_sign_found == FALSE)) {

		if ((record[i] == BLANK_CHAR) ||
		    (record[i] == TAB_CHAR) ||
		    (record[i] == CR_CHAR))
			i++;
		else
			relevant_sign_found = TRUE;
	}			/* end while */


	if (relevant_sign_found == FALSE)
		return (FALSE);	/* record not relevant */


	/* cut off tailing comment in relevant record */
	comment_sign_found = FALSE;
	while ((i < (int)strlen(record)) && (comment_sign_found == FALSE)) {

		if (record[i] == COMMENT) {
			record[i] = '\0';
			comment_sign_found = TRUE;
		} else
			i++;
	}			/* end while */


	return (TRUE);

}				/* is_record_relevant */






/*--------------------------------------------------------------*/
/*						                */
/* PROC  is_record_correct		      		       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check whether record contains correct parameters. If values */
/*  are correct, all parameters are returned in "par_in_record".*/
/*								*/
/*  In case of an error (e.g. invalid keyword, invalid par.)    */
/*  the error together with the actual record is stacked with 	*/
/*  "aux_add_error()" and FALSE is returned.		 	*/
/*  								*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   record			Pointer to the record to be	*/
/*				checked.		 	*/
/*   *par_in_record		Pointer to structure.		*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*   par_in_record		In case of correct parameter    */
/*				values, this structure		*/
/*			        contains the correct par.	*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   TRUE	               Record is correct.	       	*/
/*   FALSE		       Record contains invalid values.	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   aux_dechex()	       Decode printable hex 		*/
/*			       representation into hex.		*/
/*   char2int()		       Transform ASCII string to integer*/
/*			       value.				*/
/*   get_next_word_from_record() Return next word in record.	*/
/*   get_obj_par()	       Get object parameters from read  */
/*			       record.				*/
/*   is_char_relevant()	       Check whether character is       */
/*			       relevant.			*/
/*   aux_add_error()	       Add error to error stack.	*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure is_record_correct
 *
 ***************************************************************/
#ifdef __STDC__

static Boolean is_record_correct(
	char		  record[],
	ParInRecord	 *par_in_record
)

#else

static Boolean is_record_correct(
	record,
	par_in_record
)
char		  record[];
ParInRecord	 *par_in_record;

#endif

{


	int             rindex;	/* index for record 	              */

	/* where to start read word in record */
	int             rc;
	char           *word;	/* one word in record	              */
	OctetString	appid_hex;
	int		no;

	char           *proc = "is_record_correct";

#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  is_record_correct\n\n");*/
#endif


/*******************************************************************************************/
/*
 * Input:		 1. A relevant record.
 *			 2. A tailing comment has been cut off.
 *			 3. record is a null-terminated string.
 *
 * Next to do:		 1. Get first word in record, which is supposed to be one of the
 *		            following keywords: APP_KEY_WORD, PSE_KEY_WORD, OBJ_KEY_WORD, ....
 *			 2. Get the parameters according to the keyword
 */


#ifdef CONFIGTEST
	if (record)
		fprintf(stderr, " relevant record: \n%s\n\n", record);
#endif

	/*
	 * get keyword (first word in record)
	 */

	rindex = 0;
	word = get_next_word_from_record(record, &rindex);
	if ((word == CNULL) || ((int)strlen(word) == 0)) {
		aux_add_error(ECONFIG, "PAR: No keyword found in record", record, char_n, proc);
		return (FALSE);
	}

	/*
	 * keyword  analyse
	 */

	if (!strncmp(word, APP_ID_KEY_WORD, strlen(APP_ID_KEY_WORD)))
		par_in_record->par_type = APP_ID;
	else if (!strncmp(word, OBJ_KEY_WORD, strlen(OBJ_KEY_WORD)))
		par_in_record->par_type = OBJ_NAME;
	else if (!strncmp(word, APP_KEY_WORD, strlen(APP_KEY_WORD)))
		par_in_record->par_type = PSE_NAME;
	else if (!strncmp(word, PSE_KEY_WORD, strlen(PSE_KEY_WORD)))
		par_in_record->par_type = PSE_NAME;
	else if (!strncmp(word, DES_KEYS_WORD, strlen(DES_KEYS_WORD)))
		par_in_record->par_type = NO_DES_KEYS;
	else if (!strncmp(word, SC_ENC_KEY_WORD, strlen(SC_ENC_KEY_WORD)))
		par_in_record->par_type = SC_ENCRYPT;
	else if (!strncmp(word, SC_VER_KEY_WORD, strlen(SC_VER_KEY_WORD)))
		par_in_record->par_type = SC_VERIFY;
	else if (!strncmp(word, ONEKEYPAIR_WORD, strlen(ONEKEYPAIR_WORD)))
		par_in_record->par_type = ONEKEYPAIR;

	else {
		par_in_record->par_type = NO_KEYWORD;
		free(word);
		aux_add_error(ECONFIG, "PAR: Unknown keyword found in record", record, char_n, proc);
		return (FALSE);
	}			/* end else */
	free(word);


	/*
	 * get parameters depending on keyword
	 */

	switch (par_in_record->par_type) {

	case OBJ_NAME:
		/* init object name */
		par_in_record->conf_par.obj_par.name = CNULL;

		/* get object parameters */
		rc = get_obj_par(record, &rindex, &par_in_record->conf_par.obj_par);
		if (rc == ERR_flag) {
/*	                    aux_add_error(ECONFIG, "PAR: Invalid parameter for object in record", record, char_n, proc); */
			if (par_in_record->conf_par.obj_par.name != CNULL)
				free(par_in_record->conf_par.obj_par.name);
			return (FALSE);
		}
		break;

	case PSE_NAME:
		/* get pse_name */
		word = get_next_word_from_record(record, &rindex);
		if ((word == CNULL) || ((int)strlen(word) == 0)) {
			aux_add_error(ECONFIG, "PAR: No PSE name found in record", record, char_n, proc);
			return (FALSE);
		}
		par_in_record->conf_par.pse_name = word;
		break;

	case APP_ID:
		/* get app_id */
		word = get_next_word_from_record(record, &rindex);
		if ((word == CNULL) || ((int)strlen(word) == 0)) {
			aux_add_error(ECONFIG, "PAR: No application id found in record", record, char_n, proc);
			return (FALSE);
		}
		appid_hex.noctets = strlen(word);
		appid_hex.octets = word;
		if (!(par_in_record->conf_par.app_id = aux_dechex(&appid_hex))) {
			aux_add_error(ECONFIG, "Cannot decode hex representation of the application id", record, char_n, proc);
			return (FALSE);
		}
	
		/* check length of application id */
		if (par_in_record->conf_par.app_id->noctets !=  APPID_L) {
			aux_add_error(ECONFIG, "PAR: Invalid length of application id in record", record, char_n, proc);
			return (FALSE);
		}

		break;

	case NO_DES_KEYS:
		/* get max number of DES keys which can be installed on the SC */
		word = get_next_word_from_record(record, &rindex);
		if ((word != CNULL) && ((int)strlen(word) != 0)) {
			no = char2int(word);
			if ((no <= 0) || (no > MAXNO_DES_KEYS)) {
				aux_add_error(ECONFIG, "PAR: Invalid number of DES keys.", record, char_n, proc);
			        free(word);
				return (FALSE);
			}
			par_in_record->conf_par.int_parameter = no;
		}
		else {
			aux_add_error(ECONFIG, "PAR: Number of DES keys in record missing", record, char_n, proc);
			return (FALSE);
		}
		free(word);
		break;


	case SC_ENCRYPT:
	case SC_VERIFY:
	case ONEKEYPAIR:
		/* get_boolean_flag */
		par_in_record->conf_par.boolean_flag = TRUE;

		word = get_next_word_from_record(record, &rindex);
		if ((word != CNULL) && ((int)strlen(word) != 0)) {
			if (!strncmp(word, TRUE_WORD, strlen(TRUE_WORD)))
				par_in_record->conf_par.boolean_flag = TRUE;
			else if (!strncmp(word, FALSE_WORD, strlen(FALSE_WORD)))
				par_in_record->conf_par.boolean_flag = FALSE;
			else {
				aux_add_error(ECONFIG, "PAR: Invalid boolean flag in record", record, char_n, proc);
				free(word);
				return (FALSE);
			}
		}		/* end if */
		free(word);

		break;


	default:
		par_in_record->par_type = NO_KEYWORD;
		aux_add_error(ECONFIG, "PAR: Unknown keyword found in record", record, char_n, proc);
		return (FALSE);

	}			/* end switch */


	return (TRUE);


}				/* is_record_correct */








/*--------------------------------------------------------------*/
/*						                */
/* PROC  get_obj_par					       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Get parameters for object from delivered record.		*/
/*								*/
/*  The delivered record is supposed to be a string (null-	*/
/*  terminated).  This function gets the parameters from record */
/*  from the offset record + *rindex.				*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   record			Pointer to record.		*/
/*   rindex			Pointer to index in record.	*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*   obj_par			Parameters for object.    	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*   ERR_flag		       error				*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   char2int()		       Transform ASCII string to integer*/
/*			       value.				*/
/*   get_next_word_from_record() Return next word in record.	*/
/*   is_char_relevant()	       Check whether character is       */
/*			       relevant.			*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure get_obj_par
 *
 ***************************************************************/
#ifdef __STDC__

static int get_obj_par(
	char		 *record,
	int		 *rindex,
	SCObjEntry	 *obj_par
)

#else

static int get_obj_par(
	record,
	rindex,
	obj_par
)
char		 *record;
int		 *rindex;
SCObjEntry	 *obj_par;

#endif

{


	int             i;
	char           *word;
	int             no;

	char           *proc = "get_obj_par";




	/*
	 * get object_name
	 */

	word = get_next_word_from_record(record, rindex);
	if ((word == CNULL) || ((int)strlen(word) == 0)) {
		aux_add_error(ECONFIG, "PAR: No object name found in record", record, char_n, proc);
		return (ERR_flag);
	}
	obj_par->name = word;


	/*
	 * get object type (KEY, FILE)
	 */

	word = get_next_word_from_record(record, rindex);
	if ((word != CNULL ) && ((int)strlen(word) != 0)) {
		if (!strncmp(word, SC_KEY_WORD, strlen(SC_KEY_WORD)))
			obj_par->type = SC_KEY_TYPE;
		else if (!strncmp(word, SC_FILE_WORD, strlen(SC_FILE_WORD)))
			obj_par->type = SC_FILE_TYPE;
		else {
			aux_add_error(ECONFIG, "PAR: Invalid type of object in record", record, char_n, proc);
			free(word);
			return (ERR_flag);
		}
	}
	 /* end if */ 
	else {
		aux_add_error(ECONFIG, "PAR: Type of object in record missing", record, char_n, proc);
		return (ERR_flag);
	}
	free(word);


	/*
	 * get id depending on type
	 */


	switch (obj_par->type) {

	case SC_KEY_TYPE:

		/*
		 * get key number
		 */

		word = get_next_word_from_record(record, rindex);
		if ((word != CNULL) && (strlen(word) != 0)) {
			no = char2int(word);
			if ((no < MIN_KEY_NUMBER) || (no > MAX_KEY_NUMBER)) {
				aux_add_error(ECONFIG, "PAR: Invalid key number in record ", record, char_n, proc);
			        free(word);
				return (ERR_flag);
			}
			obj_par->sc_id = no;


		} else {
			aux_add_error(ECONFIG, "PAR: Key number in record missing", record, char_n, proc);
			return (ERR_flag);
		}
		free(word);

		/* init parameters which are not used for a key_id */
		obj_par->size = 0;

		break;

	case SC_FILE_TYPE:

		/*
		 * get short index	
		 */

		word = get_next_word_from_record(record, rindex);
		if ((word != CNULL) && (strlen(word) != 0)) {
			no = char2int(word);
			if ((no < MIN_SHORT_INDEX) || (no > MAX_SHORT_INDEX)) {
				aux_add_error(ECONFIG, "PAR: Invalid short index for SC-file in record. ", record, char_n, proc);
			        free(word);
				return (ERR_flag);
			}
			obj_par->sc_id = no;
		} else {
			aux_add_error(ECONFIG, "PAR: Short index for file on SC missing", record, char_n, proc);
			return (ERR_flag);
		}
		free(word);


		/*
		 * get size = number of bytes to be reserved for WEF file on
		 * SC
		 */

		word = get_next_word_from_record(record, rindex);
		if ((word != CNULL) && (strlen(word) != 0)) {
			if ((no = char2int(word)) <= 0) {
				aux_add_error(ECONFIG, "PAR: Invalid no. of bytes for file on SC.", record, char_n, proc);
			        free(word);
				return (ERR_flag);
			}
			obj_par->size = no;
		}
		 /* end if */ 
		else {
			aux_add_error(ECONFIG, "PAR: File size in record missing", record, char_n, proc);
			return (ERR_flag);
		}
		free(word);

		break;
	default:
		break;

	}			/* end switch */



	/*
	 * Init secure messaging parameters with default value,
	 *    (secure messaging parameters are optional)
	 */

	obj_par->sm_SCT = SCT_SEC_NORMAL;
	obj_par->sm_SC_read.command = ICC_SEC_NORMAL;
	obj_par->sm_SC_read.response = ICC_SEC_NORMAL;
	obj_par->sm_SC_write.command = ICC_SEC_NORMAL;
	obj_par->sm_SC_write.response = ICC_SEC_NORMAL;



	/*
	 * get secure messaging parameter
	 */

	word = get_next_word_from_record(record, rindex);

	if ((word != CNULL) && (strlen(word) != 0)) {
		if (!strncmp(word, NORM_WORD, strlen(NORM_WORD)))
			obj_par->sm_SCT = SCT_SEC_NORMAL;
		else if (!strncmp(word, INTG_WORD, strlen(INTG_WORD))) {
			obj_par->sm_SCT = SCT_INTEGRITY;
			obj_par->sm_SC_read.command = ICC_SEC_NORMAL;
			obj_par->sm_SC_read.response = ICC_AUTHENTIC;
			obj_par->sm_SC_write.command = ICC_AUTHENTIC;
			obj_par->sm_SC_write.response = ICC_SEC_NORMAL;
		}
		else if (!strncmp(word, CONC_WORD, strlen(CONC_WORD))) {
			obj_par->sm_SCT = SCT_CONCEALED;
			obj_par->sm_SC_read.command = ICC_SEC_NORMAL;
			obj_par->sm_SC_read.response = ICC_CONCEALED;
			obj_par->sm_SC_write.command = ICC_CONCEALED;
			obj_par->sm_SC_write.response = ICC_SEC_NORMAL;
		}
		else if (!strncmp(word, COMB_WORD, strlen(COMB_WORD))) {
			obj_par->sm_SCT = SCT_COMBINED;
			obj_par->sm_SC_read.command   = ICC_SEC_NORMAL;
			obj_par->sm_SC_read.response  = ICC_COMBINED;
			obj_par->sm_SC_write.command  = ICC_COMBINED;
			obj_par->sm_SC_write.response = ICC_SEC_NORMAL;
		}
		else {
			aux_add_error(ECONFIG, "PAR: Invalid secure messaging parameter for object in record", record, char_n, proc);
		        free(word);
			return (ERR_flag);
		}
		free(word);
	}		/* end if */


	return(0);

}				/* get_obj_par */





/*--------------------------------------------------------------*/
/*						                */
/* PROC  get_next_word_from_record			       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  This function returns the next word found in record from the*/
/*  offset record + *rindex.					*/
/*  The null-pointer is returned, if no word could be found.	*/
/*  The value of *rindex is incremented and points to the next  */
/*  character in record after the returned word.		*/
/*  Memory for the returned word is allocated by this function.	*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   record			Pointer to record.		*/
/*							       	*/
/* INOUT						       	*/
/*   rindex			Pointer to index in record.	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   NULL	         	No word found 		       	*/
/*   ptr. to word					       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   is_char_relevant()	       Check whether character is       */
/*			       relevant.			*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure get_next_word_from_record
 *
 ***************************************************************/
#ifdef __STDC__

static char *get_next_word_from_record(
	char	 *record,
	int	 *rindex
)

#else

static char *get_next_word_from_record(
	record,
	rindex
)
char	 *record;
int	 *rindex;

#endif

{

	char           *word;
	int             i;
	int             start_word;
	int             len_word;

	char           *proc = "get_next_word_from_record";

#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  get_next_word_from_record\n\n");*/
#endif


	/*
	 * input check
	 */

	if ((record == CNULL) || ((int)strlen(record) == 0) || (*rindex >= (int)strlen(record))) {
		return (CNULL);
	}

	/*
	 * jump over all leading not relevant characters
	 */

	while ((*rindex < (int)strlen(record)) && ((is_char_relevant(record[*rindex])) == FALSE)) {

		(*rindex)++;

	}


	if (*rindex >= (int)strlen(record))
		return (CNULL);	/* no relevant character found */


	/*
	 * get relevant characters until first not relevant character
	 */

	start_word = *rindex;
	len_word = 0;

	while ((*rindex < (int)strlen(record)) && ((is_char_relevant(record[*rindex])) == TRUE)) {

		(*rindex)++;

	}


	/*
	 * allocate memory for word, get word from record and return ptr to
	 * word.
	 */

	len_word = *rindex - start_word;


#ifdef CONFIGTEST
/*	fprintf(stderr, "in record: %s\n", record);
	fprintf(stderr, "strlen(record): %d\n", strlen(record));
	fprintf(stderr, "start_word: %d\n", start_word);
	fprintf(stderr, "len_word: %d\n", len_word);
	fprintf(stderr, "*rindex: %d\n", *rindex);*/

#endif



	if (len_word <= 0)
		return (CNULL);

	word = (char *) malloc(len_word + 1);
	if (!word) {
		aux_add_error(EMALLOC, "Word", CNULL, 0, proc);
		return (CNULL);
	}
	for (i = 0; i < len_word;)
		word[i++] = record[start_word++];
	word[i] = '\0';		/* terminate word with null */

#ifdef CONFIGTEST
/*	fprintf(stderr, "\nfound word: %s\n\n", word);*/

#endif


	return (word);

}				/* get_next_word_from_record */








/*--------------------------------------------------------------*/
/*						                */
/* PROC  is_char_relevant				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Return TRUE if delivered character is  relevant else return */
/*  FALSE.							*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   one_char			One character.			*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   TRUE         	       Character is relevant.	       	*/
/*   FALSE         	       Character is not relevant.      	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure is_char_relevant
 *
 ***************************************************************/
#ifdef __STDC__

static Boolean is_char_relevant(
	char	  one_char
)

#else

static Boolean is_char_relevant(
	one_char
)
char	  one_char;

#endif

{


	if (!one_char)
		return (FALSE);

	if ((one_char == BLANK_CHAR) ||
	    (one_char == COMMENT) ||
	    (one_char == COMMA) ||
	    (one_char == TAB_CHAR) ||
	    (one_char == EQUAL) ||
	    (one_char == CR_CHAR) ||
	    (one_char == '\n') ||
	    (one_char == '\0'))
		return (FALSE);


	return (TRUE);

}				/* is_char_relevant */





/*--------------------------------------------------------------*/
/*						                */
/* PROC  check_pse_name					       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check whether the actual read PSE name is unique in 	*/
/*  "sc_pse_list[]".			   	    		*/
/*								*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   read_pse_name		PSE name			*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*   ERR_flag		       Error			       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure check_pse_name
 *
 ***************************************************************/
#ifdef __STDC__

static int check_pse_name(
	char	 *read_pse_name
)

#else

static int check_pse_name(
	read_pse_name
)
char	 *read_pse_name;

#endif

{

	char           *proc = "check_pse_name";

#ifdef CONFIGTEST
	fprintf(stderr, "READ_CONFIG function:  check_pse_name\n\n");
#endif


	/* check the given pse_name */
	if (read_pse_name) {
		if ((int)strlen(read_pse_name) > MAXL_PSENAME) {
			aux_add_error(ECONFIG, "PSE: PSE name too long", read_pse_name, char_n, proc);
			return (ERR_flag);
		}
	} else {
		aux_add_error(EINVALID, "PSE: PSE name empty", CNULL, 0, proc);
		return (ERR_flag);
	}


	if (aux_PseName2SCPse(read_pse_name)) {
		aux_add_error(ECONFIG, "PSE: PSE name not unique.", read_pse_name, char_n, proc);
		return (ERR_flag);
	}
	return (0);

}				/* check_pse_name */





/*--------------------------------------------------------------*/
/*						                */
/* PROC  add_predef_values				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Add predefined values to the parameter list for one 	*/
/*  PSE.							*/
/*								*/
/*      - If the application identifier for the actual PSE is 	*/
/*        missing, the default application id is added.		*/
/*	- All predefined objects are added to the object list.  */
/*        If a predefined object name or the belonging sc_id	*/
/*	  are already defined in the SCCONFFILE, Err_flag is 	*/
/*	  returned.						*/
/*								*/		  
/*  Memory for all added values are allocated explicitly => it 	*/
/*  can be released with free().				*/
/*								*/
/*  The global list "predef_sc_obj_list[]" contains the list of */
/*  the predefined objects.					*/
/*  The global list "default_sc_obj_list[]" contains the list of*/
/*  the  defaults values for objects.				*/
/*  								*/
/*  If the number of objects is greater	than MAX_SCOBJ, ERR_flag*/
/*  is returned.				 		*/
/*  								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   sc_pse_entry		One entry in SC PSE list.	*/
/*   								*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*   ERR_flag		       Error			      	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   get_default_index()       Get index in default object list.*/
/*--------------------------------------------------------------*/
static
/***************************************************************
 *
 * Procedure add_predef_values
 *
 ***************************************************************/
#ifdef __STDC__

int add_predef_values(
	SCPseEntry	 *sc_pse_entry
)

#else

int add_predef_values(
	sc_pse_entry
)
SCPseEntry	 *sc_pse_entry;

#endif

{

	int             n, i;
	int             def_ind;
	Boolean         predef_obj_found;
	Boolean		SC_id_found;
	OctetString	appid_hex;
	int             function_rc = 0;
	char		err_msg[256];


	char           *proc = "add_predef_values";

#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  add_predef_values\n\n");*/
#endif

	if (!sc_pse_entry) {
		aux_add_error(EINVALID, "Invalid parameter value for sc_pse_entry", CNULL, 0, proc);
		return (ERR_flag);
	}




	/*
	 *  If application id is missing, take default application id
	 */

	if ((!sc_pse_entry->app_id) || (!sc_pse_entry->app_id->octets)) {

		appid_hex.noctets = strlen(DEF_SC_APP_ID);
		appid_hex.octets  = DEF_SC_APP_ID;
		if (!(sc_pse_entry->app_id = aux_dechex(&appid_hex))) {
			aux_add_error(ECONFIG, "Cannot decode hex representation of the default application id", CNULL, 0, proc);
			function_rc = ERR_flag;
		}
	}



	/*
	 * Loop for the list of the predefined objects:
	 */

	for (n = 0; predef_sc_obj_list[n]; n++) {

		def_ind = get_default_index(predef_sc_obj_list[n]);
		if ((def_ind < 0) || (!default_sc_obj_list[def_ind].name)) {
			aux_add_error(ECONFIG, "OBJ: Invalid default list for object ", predef_sc_obj_list[n], char_n, proc);
			return (ERR_flag);
		}

		predef_obj_found = FALSE;
		SC_id_found = FALSE;
		i = 0;
		while ((i <= MAX_SCOBJ) &&
		       (sc_pse_entry->sc_obj_list[i].name) && 
			(predef_obj_found == FALSE) && (SC_id_found == FALSE)) {

			/*
			 *  Check uniqueness of object name
			 */

			if (strcmp(sc_pse_entry->sc_obj_list[i].name, predef_sc_obj_list[n]) == 0) {
				aux_add_error(ECONFIG, "PAR: Object is a predefined object", predef_sc_obj_list[n], char_n, proc);
				predef_obj_found = TRUE;
			}
			else {

				/*
				 *  Check uniqueness of identifier on the SC
				 */

				if (( sc_pse_entry->sc_obj_list[i].type == default_sc_obj_list[def_ind].type) &&
				    ( sc_pse_entry->sc_obj_list[i].sc_id == default_sc_obj_list[def_ind].sc_id)) {

					sprintf(err_msg, "PAR: SC_id %d belongs to a predefined object!", sc_pse_entry->sc_obj_list[i].sc_id);
					aux_add_error(ECONFIG, "Identifier not unique", err_msg, char_n, proc);
					SC_id_found = TRUE;
				}
				else
					i++;
				    
			} 

		}		/* end while */

		if ((predef_obj_found == TRUE) || (SC_id_found == TRUE)) {
			function_rc = ERR_flag;
		}
		else  {

			/*
			 * Object name and belonging SC-id (file, key) is unique
			 * = > Add predefined object to object list 
			 */

			if (i >= MAX_SCOBJ) {
				aux_add_error(ECONFIG, "OBJ: Too many objects for PSE ", sc_pse_entry->pse_name, char_n, proc);
				return (ERR_flag);
			}
			sc_pse_entry->sc_obj_list[i] = default_sc_obj_list[def_ind];


			/*
			 *  Allocate memory for object name
			 */

	                sc_pse_entry->sc_obj_list[i].name = (char *) malloc (strlen(default_sc_obj_list[def_ind].name)+1);
			if (!sc_pse_entry->sc_obj_list[i].name) {
				aux_add_error(EMALLOC, "object name", CNULL, 0, proc);
				return (ERR_flag);
			}
			strcpy (sc_pse_entry->sc_obj_list[i].name, default_sc_obj_list[def_ind].name);

			i++;
			sc_pse_entry->sc_obj_list[i].name = CNULL;

		}		/* end else */
	}			/* end for */

	return (function_rc);


}				/* add_predef_values */







/*--------------------------------------------------------------*/
/*						                */
/* PROC  handle_mandatory_obj				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check whether object list contains all mandatory objects    */
/*  ("man_sc_obj_list[]"). If a mandatory object is missing, the*/
/*  default values for this object is added to the current 	*/
/*  object list ( in this case memory for the object name is	*/
/*  allocated explicitly.					*/
/*								*/
/*  The global list "man_sc_obj_list[]" contains the list of the*/
/*  mandatory objects.						*/
/*  The global list "default_sc_obj_list[]" contains the list of*/
/*  the  defaults values for objects.				*/
/*  								*/
/*  If the number of objects is greater	than MAX_SCOBJ, ERR_flag*/
/*  is returned.				 		*/
/*  								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   sc_pse_entry		One entry in SC PSE list.       */
/*   								*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*   ERR_flag		       Error			      	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   get_default_index()       Get index in default object list.*/
/*--------------------------------------------------------------*/
static
/***************************************************************
 *
 * Procedure handle_mandatory_obj
 *
 ***************************************************************/
#ifdef __STDC__

int handle_mandatory_obj(
	SCPseEntry	 *sc_pse_entry
)

#else

int handle_mandatory_obj(
	sc_pse_entry
)
SCPseEntry	 *sc_pse_entry;

#endif

{

	int             n, i;
	int             def_ind;
	Boolean         man_obj_found;

	char           *proc = "handle_mandatory_obj";

#ifdef CONFIGTEST
/*	   fprintf(stderr, "READ_CONFIG function:  handle_mandatory_obj\n\n");*/
#endif

	if (!sc_pse_entry) {
		aux_add_error(ECONFIG, "Invalid parameter value for sc_pse_entry", CNULL, 0, proc);
		return (ERR_flag);
	}



	/*
	 * Loop for the list of the mandatory objects:
	 */

	for (n = 0; man_sc_obj_list[n]; n++) {

		man_obj_found = FALSE;
		i = 0;
		while ((i <= MAX_SCOBJ) &&
		       (sc_pse_entry->sc_obj_list[i].name) && (man_obj_found == FALSE)) {

			if (strcmp(sc_pse_entry->sc_obj_list[i].name, man_sc_obj_list[n]) == 0)
				man_obj_found = TRUE;
			else
				i++;
		}		/* end while */


		if (man_obj_found == FALSE) {

			/*
			 * Mandatory object missing in object list => add
			 * mandatory object.
			 */

			if (i >= MAX_SCOBJ) {
				aux_add_error(ECONFIG, "OBJ: Too many objects for PSE ", sc_pse_entry->pse_name, char_n, proc);
				return (ERR_flag);
			}
			def_ind = get_default_index(man_sc_obj_list[n]);
			if ((def_ind < 0) || (!default_sc_obj_list[def_ind].name)) {
				aux_add_error(ECONFIG, "OBJ: Mandatory object missing", man_sc_obj_list[n], char_n, proc);
				return (ERR_flag);
			}
			sc_pse_entry->sc_obj_list[i] = default_sc_obj_list[def_ind];


			/*
			 *  Allocate memory for object name
			 *  => free(obj_name) causes no error
			 */

	                sc_pse_entry->sc_obj_list[i].name = (char *) malloc (strlen(default_sc_obj_list[def_ind].name));
			if (!sc_pse_entry->sc_obj_list[i].name) {
				aux_add_error(EMALLOC, "object name", CNULL, 0, proc);
				return (ERR_flag);
			}
			strcpy (sc_pse_entry->sc_obj_list[i].name, default_sc_obj_list[def_ind].name);

			i++;
			sc_pse_entry->sc_obj_list[i].name = CNULL;

		}		/* end if */
	}			/* end for */

	return (0);


}				/* handle_mandatory_obj */






/*--------------------------------------------------------------*/
/*						                */
/* PROC  get_default_index				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Get index in default object list ("sc_obj_list[]") for the  */
/*  delivered object name.					*/
/*  								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   object_name		Object name.			*/
/*   								*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   index (>= 0)	       Index in default object list.  	*/
 /*   ERR_flag		       No entry found.			*//* */
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/
static
/***************************************************************
 *
 * Procedure get_default_index
 *
 ***************************************************************/
#ifdef __STDC__

int get_default_index(
	char	 *object_name
)

#else

int get_default_index(
	object_name
)
char	 *object_name;

#endif

{
	int             n;
	char           *proc = "get_default_index";

	if (!object_name) {
		aux_add_error(EINVALID, "OBJ: Invalid value for parameter 'object_name' ", CNULL, 0, proc);
		return (ERR_flag);
	}
	for (n = 0; default_sc_obj_list[n].name; n++) {
		if (strcmp(default_sc_obj_list[n].name, object_name) == 0)
			return (n);
	}

	return (ERR_flag);


}				/* end get_default_index */




/*--------------------------------------------------------------*/
/*						         	*/
/* PROC  check_SCPSE_configuration			       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check consistency of configuration data for an SC-		*/
/*  PSE.	    						*/
/*								*/
/*								*/
/*  1) Perform SC configuration (read SC configuration file 	*/
/*     (".starcosrc")).       					*/
/*								*/
/*  2) Check whether given PSE (pse_name) is an SC-PSE.   	*/
/*     								*/
/*     Case 1:  PSE is an SC-PSE:				*/
/*	        Depending on the value of the configuration flag*/
/*	        "onekeypair" it is checked whether the 		*/
/*		configuration file contains the necessary	*/
/*		objects for this kind of PSE.			*/
/*								*/
/*     Case 2: PSE is not an SC-PSE:				*/
/*             => return error					*/
/*  								*/
/*  3) Depending on the configuration flag "onekeypair" it is   */
/*     checked,							*/
/*     a) whether the size of "SCToc" is sufficient for the     */
/*        relevant objects in the objectlist for the given 	*/
/*        PSE.							*/
/*							        */
/*     b) whether the free space on the SC is sufficient for 	*/
/*        the relevant objects in the objectlist for the given 	*/
/*        PSE.							*/
/*							        */
/*							        */
/* IN			     DESCRIPTION		       	*/
/*   pse_name		       Name of the PSE for which	*/
/*			       the configuration data shall be  */
/*			       checked.				*/
/*   onekeypair	       		Pointer to the flag	  	*/
/* OUT			     DESCRIPTION		       	*/
/*   *onekeypair	       TRUE means that the PSE is  	*/
/*			       configured as an one keypair PSE.*/
/*			       FALSE means that the PSE 	*/
/*			       configured as a two keypairs PSE.*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0		  	       Configuration data for the given */
/*			       PSE are consistent.		*/
/*   -1		               - Necessary objects missing in	*/
/*			         configuration data.		*/
/*   		               - error (e.g. PSE not an SC-PSE)	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*   check_config_list()	Check whether the object list of*/
/*				an SC-PSE contains all 		*/
/*				mandatory objects.		*/
/*   is_obj_in_list()	        Check whether given object is   */
/*			        part of given list.	        */ 
/*   aux_PseName2SCPse()	Get information about an SC PSE.*/
/*   aux_PseObjName2SCObj()	Get information about an SC     */
/*        			object belonging to an SC-PSE.	*/
/*   aux_add_error()		Add error to error stack.	*/
/*			         		       		*/
/*--------------------------------------------------------------*/

/***************************************************************
 *
 * Procedure check_SCPSE_configuration
 *
 ***************************************************************/
#ifdef __STDC__

RC check_SCPSE_configuration(
	char	 *pse_name,
	Boolean	 *onekeypair
)

#else

RC check_SCPSE_configuration(
	pse_name,
	onekeypair
)
char	 *pse_name;
Boolean	 *onekeypair;

#endif

{

#define SC_OBJ_LIST	sc_pse_entry->sc_obj_list
#define ONEKEY_LIST	onekeypair_sc_obj_list
#define TWOKEY_LIST	twokeypairs_sc_obj_list

	int		SC_available;
	SCPseEntry     *sc_pse_entry;
	SCObjEntry     *sc_obj_entry;
	Boolean		data_consistent;
	unsigned int	i;
	Boolean		obj_relevant;
	unsigned int	no_of_toc_entries;
	unsigned int	no_of_WEFs;
	unsigned int	no_of_units;
	unsigned int 	rest;
	char		err_msg[256];
	unsigned int	req_Toc_size;
	unsigned int	free_units;





	char            *proc = "check_SCPSE_configuration";



#ifdef SECSCTEST
	fprintf(stderr, "SECSC-Function: %s\n", proc);
	fprintf(stderr, "                   PSE-Name: %s\n", pse_name);
#endif


	if (pse_name == CNULL) {
		aux_add_error(EINVALID, "No PSE name specified", CNULL, 0, proc);
		return (-1);
	}
	if(! onekeypair){
		aux_add_error(EINVALID, "No parameter provided", CNULL, 0, proc);
		return(- 1);
	}


	/*
	 *  Perform SC configuration
	 */

	if ((SC_available = SC_configuration()) == -1) {
		AUX_ADD_ERROR;
		return (-1);
	}
	if (SC_available == FALSE) {

		/*
		 *  no SC configuration file found)
		 */

		aux_add_error(ECONFIG, "No SC configuration file found.", CNULL, 0, proc);
		return (-1);
	}



	/*
	 *  Check whether PSE is an SC-PSE
	 */

	sc_pse_entry = aux_PseName2SCPse(pse_name);
	if (sc_pse_entry == (SCPseEntry * ) 0) 	{
		aux_add_error(EINVALID, "PSE is not an SC-PSE", pse_name, char_n, proc);
		return (-1);
	}

	*onekeypair = sc_pse_entry->onekeypair;

	/*
	 *  Check consistency of the configuration data for the given PSE.
	 */

	if (*onekeypair == TRUE) {
		data_consistent = check_config_list(sc_pse_entry, onekeypair_sc_obj_list);
		free_units = MAX_SC_SPACE1;
	}
	else {
		data_consistent = check_config_list(sc_pse_entry, twokeypairs_sc_obj_list);
		free_units = MAX_SC_SPACE2;
	}


	if (data_consistent == -1) {
		aux_add_error(ECONFIG, "Inconsistent configuration data for SC-PSE ", pse_name, char_n, proc);	
		return (-1);
	}

	
	/*
	 *  Depending on parameter "onekeypair" the relevant objects are counted
	 *  and the units, these objects will need on the SC, are accumulated.
	 */

	i = 0;
	no_of_toc_entries = 0;
	no_of_WEFs = 0;
	no_of_units = 0;

	while ((i <= MAX_SCOBJ) && (SC_OBJ_LIST[i].name)) {

		/*
		 *  First check whether object is relevant
		 */
		
		obj_relevant = TRUE;
#ifdef SECSCTEST
		fprintf(stderr, "SC_OBJ_LIST[i].name: %s\n", SC_OBJ_LIST[i].name);
#endif

		if (is_obj_in_list(SC_OBJ_LIST[i].name, predef_sc_obj_list) == TRUE) {

			/*
			 *  Predefined objects as SC_PIN, SC_PUK, PSE_PIN, the
			 *    device key set don't count
			 */

			obj_relevant = FALSE;
		}
		else {
			if (*onekeypair == TRUE) {
		        	if ((is_obj_in_list(SC_OBJ_LIST[i].name, ONEKEY_LIST) == FALSE) &&
		                    (is_obj_in_list(SC_OBJ_LIST[i].name, TWOKEY_LIST) == TRUE)) 
					/*
					 *  Object not relevant, 
					 *    - is only needed for an SC with two key pairs
					 */

					obj_relevant = FALSE;
			}
			else {
			    	if ((is_obj_in_list(SC_OBJ_LIST[i].name, TWOKEY_LIST) == FALSE) &&
			            (is_obj_in_list(SC_OBJ_LIST[i].name, ONEKEY_LIST) == TRUE)) 
					/*
					 *  Object not relevant, 
					 *    - is only needed for an SC with one key pair
					 */

					obj_relevant = FALSE;
			}

		}
		
		
		if (obj_relevant == TRUE) {

			/*
			 *  Object is relevant
			 */
#ifdef SECSCTEST
			fprintf(stderr, "Relevant SC_OBJ_LIST[i].name: %s\n", SC_OBJ_LIST[i].name);
#endif
			if (strcmp(SC_OBJ_LIST[i].name, SCToc_name) != 0)
				no_of_toc_entries++;

			if (SC_OBJ_LIST[i].type == SC_FILE_TYPE) {
				no_of_WEFs++;
				rest = SC_OBJ_LIST[i].size % UNIT_SIZE;
				if (rest > 0)
					no_of_units += SC_OBJ_LIST[i].size / UNIT_SIZE + 1;
				else
					no_of_units += SC_OBJ_LIST[i].size / UNIT_SIZE;

#ifdef SECSCTEST
				fprintf(stderr, "SC_OBJ_LIST[i].size: %d\n", SC_OBJ_LIST[i].size);
				fprintf(stderr, "rest:                %d\n", rest);
				fprintf(stderr, "no_of_units:         %d\n", no_of_units);
#endif
			}

		}
		i++;
	}


	/*  
	 *     Compare the number of objects with the size of the object "SCToc". If
	 *     the given size of "SCToc" is too small, an error will be returned.
	 *     Compare the computed units with the free space on the SC. If the
	 *     free space on the SC is too small an error is returned.
	 */


	sc_obj_entry = aux_PseObjName2SCObj(pse_name, SCToc_name);
	if (sc_obj_entry == NULLSCOBJENTRY) {
		aux_add_error(ECONFIG, "SCToc entry for SC_PSE missing ", pse_name, char_n, proc);
		return (-1);
	}

	/* no. of entries plus header of Toc */
	req_Toc_size = (no_of_toc_entries + 1) * MAX_TOC_ENTRY;

#ifdef SECSCTEST

	fprintf(stderr, "                   No of Toc Entries:      %d\n", no_of_toc_entries);
	fprintf(stderr, "                   No of WEFs:             %d\n", no_of_WEFs);
	fprintf(stderr, "                   Required space for Toc: %d\n", req_Toc_size);
	fprintf(stderr, "                   Size of Toc:            %d\n", sc_obj_entry->size);
#endif

	if (req_Toc_size > sc_obj_entry->size) {
		sprintf(err_msg, "(Required size of SCToc for PSE %s is %d)", pse_name, req_Toc_size);
		aux_add_error(ECONFIG, "Size of SCToc is too small.", (char *) err_msg, char_n, proc);
		return (-1);
	}
			

	/*
	 *  Each WEF will take one additional unit
	 */

	no_of_units += no_of_WEFs;  

#ifdef SECSCTEST
	fprintf(stderr, "                   Required number of units:   %d\n", no_of_units);
	fprintf(stderr, "                   Number of free units on SC: %d\n", MAX_SC_SPACE);
#endif

	if (no_of_units > free_units) {
		sprintf(err_msg, "(Available space for PSE %s is %d, required size is %d)", pse_name, free_units * UNIT_SIZE, no_of_units * UNIT_SIZE);
		aux_add_error(ECONFIG, "Not enough space on SC!", (char *) err_msg, char_n, proc);
		return (-1);
	}



	return (0);


}				/* end check_SCPSE_configuration */




/*--------------------------------------------------------------*/
/*						                */
/* PROC  check_config_list				       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check whether a given object list of an SC-PSE contains all */
/*  mandatory objects.						*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   sc_pse_entry		One entry in SC PSE list.       */
/*   man_obj_list		List of the mandatory objects.  */
/*   								*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0		         	All mandatory objects found in  */
/*				given "sc_pse_entry".		*/
/*   -1			      	Objects missing.	      	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/
static
/***************************************************************
 *
 * Procedure check_config_list
 *
 ***************************************************************/
#ifdef __STDC__

int check_config_list(
	SCPseEntry	 *sc_pse_entry,
	char		 *man_obj_list[]
)

#else
int check_config_list(
	sc_pse_entry,
	man_obj_list
)
SCPseEntry	 *sc_pse_entry;
char		 *man_obj_list[];

#endif

{

	int             n, i;
	Boolean         man_obj_found;

	char           *proc = "check_config_list";


	if (!sc_pse_entry) {
		aux_add_error(EINVALID, "Invalid parameter value for sc_pse_entry", CNULL, 0, proc);
		return (-1);
	}

	/*
	 * Loop for the list of the mandatory objects:
	 */

	for (n = 0; man_obj_list[n]; n++) {

		man_obj_found = FALSE;
		i = 0;
		while ((i <= MAX_SCOBJ) &&
		       (sc_pse_entry->sc_obj_list[i].name) && (man_obj_found == FALSE)) {

			if (strcmp(sc_pse_entry->sc_obj_list[i].name, man_obj_list[n]) == 0)
				man_obj_found = TRUE;
			else
				i++;
		}		/* end while */


		if (man_obj_found == FALSE) {

			/*
			 * Mandatory object missing.
			 */

			aux_add_error(ECONFIG, "Mandatory SC-object missing in configuration file", man_obj_list[n], char_n,  proc);
			return (-1);

		}		
	}			

	return (0);


}				/* check_config_list */




/*--------------------------------------------------------------*/
/*						                */
/* PROC  is_obj_in_list				       		*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check whether a given object is part of a given list. 	*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   obj_name			Name of object.			*/
/*   obj_list			List of objects.		*/
/*   								*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0		         	All mandatory objects found in  */
/*				given "sc_pse_entry".		*/
/*   -1			      	Objects missing.	      	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/
static
/***************************************************************
 *
 * Procedure is_obj_in_list
 *
 ***************************************************************/
#ifdef __STDC__
int is_obj_in_list(
	char	*obj_name,
	char	*obj_list[]
)

#else
int is_obj_in_list(
	obj_name,
	obj_list
)
char	 *obj_name;
char	 *obj_list[];

#endif

{

	int             i;
	Boolean         obj_found;

	char           *proc = "is_obj_in_list";


	if (!obj_list) {
		aux_add_error(EINVALID, "Invalid parameter value for obj_list", CNULL, 0, proc);
		return (-1);
	}


	obj_found = FALSE;
	i = 0;
	while ((obj_list[i]) && (obj_found == FALSE)) {

		if (strcmp(obj_name, obj_list[i]) == 0)
			obj_found = TRUE;
		else
			i++;
	}		

	return(obj_found);


}				/* is_obj_in_list */




/*--------------------------------------------------------------*/
/*						                */
/* PROC  check_obj_info					       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Check new object for an SC-PSE.				*/
/*  (e.g. compare with objects already entered for the actual   */
/*   PSE). The object name and the type/sc_id combination has	*/
/*   to be unique.						*/
/*								*/
/*								*/
/* IN			     DESCRIPTION		       	*/
/*   sc_pse_entry		Entry in sc_pse_list for the    */
/*                              actual PSE.			*/
/*   new_obj_par		Parameter for the new object.	*/
/*   								*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/*							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*   0	         	       o.k			       	*/
/*   ERR_flag		       Error			       	*/
/*							       	*/
/* CALLED FUNCTIONS	     DESCRIPTION		       	*/
/*--------------------------------------------------------------*/
static
/***************************************************************
 *
 * Procedure check_obj_info
 *
 ***************************************************************/
#ifdef __STDC__

int check_obj_info(
	SCPseEntry	 *sc_pse_entry,
	SCObjEntry	 *new_obj_par
)

#else

int check_obj_info(
	sc_pse_entry,
	new_obj_par
)
SCPseEntry	 *sc_pse_entry;
SCObjEntry	 *new_obj_par;

#endif

{
	char		err_msg[256];

	char           *proc = "check_obj_info";

#ifdef CONFIGTEST
	fprintf(stderr, "READ_CONFIG function:  check_obj_info\n\n");
#endif


	/* check input parameters */
	if ((!sc_pse_entry) || (!sc_pse_entry->pse_name) ||
	    (!new_obj_par) || (!new_obj_par->name)) {
		aux_add_error(ECONFIG, "PAR: Invalid input values for check_obj_info", CNULL, 0, proc);
		return (ERR_flag);
	}
	/* check whether object name is already in use */
	if (aux_PseObjName2SCObj(sc_pse_entry->pse_name, new_obj_par->name)) {
		aux_add_error(ECONFIG, "PAR: Object name not unique", new_obj_par->name, char_n, proc);
		return (ERR_flag);
	}
	/* check whether type/sc_id combination is already in use */
	if (aux_PseObjData2SCObj(sc_pse_entry->pse_name, new_obj_par->type, new_obj_par->sc_id)) {
		if (new_obj_par->type == SC_KEY_TYPE)
			sprintf(err_msg, "PAR: key id %d (object %s) not unique!", new_obj_par->sc_id, new_obj_par->name);
		else 
			sprintf(err_msg, "PAR: file id %d (object %s) not unique!", new_obj_par->sc_id, new_obj_par->name);

		aux_add_error(ECONFIG, "Identifier not unique", err_msg, char_n, proc);

		return (ERR_flag);
	}
	return (0);

}				/* check_obj_info */




/*--------------------------------------------------------------*/
/*						                */
/* PROC  char2int					       	*/
/*							       	*/
/* DESCRIPTION						       	*/
/*								*/
/*  Transform an character string (ASCII) into an integer value */
/*  and returns this integer value.			 	*/
/*								*/
/*  If a character in string is not a digit, -1 is returned.	*/
/*							        */
/* IN			     DESCRIPTION		       	*/
/*   s 	   		       Pointer to char_string to be	*/
/*			       transformed.			*/
/*							       	*/
/* OUT							       	*/
/*							       	*/
/* RETURN		     DESCRIPTION	      	       	*/
/*    -1		       Error			        */
/*   int_value	 	       Integer value.			*/
/*--------------------------------------------------------------*/



/***************************************************************
 *
 * Procedure char2int
 *
 ***************************************************************/
#ifdef __STDC__

static int char2int(
	char	 *s
)

#else

static int char2int(
	s
)
char	 *s;

#endif

{

	unsigned int    n;

	char           *proc = "char2int";

#ifdef CONFIGTEST
	fprintf(stderr, "\nATOI: string: %s", s);
#endif


	if (!s) {
		aux_add_error(ECONFIG, "PAR: ASCII string = NULL", CNULL, 0, proc);
		return (-1);
	}
	n = 0;

	while (*s) {
		if ((*s < '0') || (*s > '9')) {
			aux_add_error(ECONFIG, "PAR: ASCII character is not a digit", CNULL, 0, proc);
			return (-1);
		}
		n = 10 * n + *s - '0';
		s++;
	}

#ifdef CONFIGTEST
	fprintf(stderr, ", integer: %d\n", n);
#endif
	return (n);
}				/* end char2int */


#else
readconf_dummy() {}
#endif				
