/*
 * S.F.S. v. 0.5alpha-linux
 *
 * Authors:	Giuseppe Cattaneo, <cattaneo@udsab.dia.unisa.it>
 *		Giuseppe Persiano, <giuper@udsab.dia.unisa.it>
 *		Andrea Cozzolino, <andcoz@mikonos.dia.unisa.it>
 *		Angelo Celentano, <angcel@mikonos.dia.unisa.it>
 *		Aniello Del Sorbo, <anidel@mikonos.dia.unisa.it>
 *		Ermelindo Mauriello, <ermmau@mikonos.dia.unisa.it>
 *		Raffaele Pisapia, <rafpis@mikonos.dia.unisa.it>
 *
 * Permission to use, copy, and modify this software without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies of any software which is or includes a copy or
 * modification of this software and in all copies of the supporting
 * documentation for such software.
 *
 * This software maybe be used for any purpose provided
 * the above copyright notice is retained.  It is supplied
 * as is, with no warranty expressed or implied.
 *
 * This code derives from: 
 *		 The LINUX User-Space NFS Server 1.5
 *	    by Mark Shand, Donald J. Becker, Rick Sladkey,
 *		 Orest Zborowski, Fred N. van Kempen *
 */

/*
 * A umount(8) for Linux 0.99.
 * umount.c,v 1.1.1.1 1993/11/18 08:40:51 jrs Exp
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <rpc/rpc.h>
#include "tcfs_prot.h"
#include "sundries.h"
#include "mount.h"

#ifdef notyet
/* Nonzero for force umount (-f).  This needs kernel support we don't have.  */
int force = 0;
#endif

/* Nonzero for chatty (-v).  This is a nonstandard flag (not in BSD).  */
int verbose = 0;

/* True if ruid != euid.  */
int suid = 0;

/* Get the pid of the SFSD daemon from file */
pid_t getsfsdpid (void)
{
	FILE *sfsp;
	pid_t sfspid;
	char pid[10];

	if ( (sfsp = fopen ("/etc/tcfsd.pid", "r")) == NULL )
	{
		fprintf (stderr, "Error: unable to open /etc/tcfsd.pid!\n");
		exit (1);
	}

	fgets (pid, 10, sfsp);
	sfspid = (pid_t) atoi (pid);
	
	fclose (sfsp);

	return sfspid;
}

/* Umount a single device.  Return a status code, so don't exit
   on a non-fatal error.  We lock/unlock around each umount.  */
static int
umount_one (const char *spec, const char *node)
{
  int umnt_err;
  int isroot;
  pid_t sfsdpid;
  enum clnt_stat cs;
  sfs_lookupargs args;
  sfs_lookupres res;

  args.name=(char*)malloc((strlen(node)+1)*sizeof(char));
  res.work=(char*)malloc(NFS_MAXPATHLEN+1*sizeof(char));
  res.remote=(char*)malloc(NFS_MAXPATHLEN+1*sizeof(char));
  res.dummy=(char*)malloc(NFS_MAXPATHLEN+1*sizeof(char));

  /* Special case for root.  As of 0.99pl10 we can (almost) unmount root;
     the kernel will remount it readonly so that we can carry on running
     afterwards.  The readonly remount is illegal if any files are opened
     for writing at the time, so we can't update mtab for an unmount of
     root.  As it is only really a remount, this doesn't matter too
     much.  [sct May 29, 1993] */

  isroot = (streq (node, "/") || streq (node, "root"));

  if (!isroot)
	  lock_mtab ();

  strcpy (args.name,node);

  args.mode=0;
  cs=callrpc ("localhost",ADM_PROGRAM,ADM_VERSION,ADMPROC_LOOKUP, (xdrproc_t)xdr_sfs_lookupargs, (char*)&args, (xdrproc_t)xdr_sfs_lookupres, (char*)&res);

  if (cs!=0)
  {
	fprintf (stderr, "Error: could not make an rpc call to mountd.\n");
	exit (1);
  }

  if (res.work==NULL)
  {
	args.mode=1;
  	cs=callrpc ("localhost",ADM_PROGRAM,ADM_VERSION,ADMPROC_LOOKUP, (xdrproc_t)xdr_sfs_lookupargs, (char*)&args, (xdrproc_t)xdr_sfs_lookupres, (char*)&res);
  }

  if (cs!=0)
  {
	fprintf (stderr, "Error: could not make an rpc call to mountd.\n");
	exit (1);
  }

  sfsdpid = getsfsdpid();
  kill (sfsdpid, SIGHUP);	/* Tell sfsd to close its file handlers */

  if (umount (res.work) >= 0)
    {
      if (verbose)
	printf ("%s umounted\n", res.work);
      
      if (!isroot)
	update_mtab (node, NULL);

      if (umount (res.dummy) >= 0)
        {
	  enum sfsstat statsfs;
	  sfs_unstickargs unstickargs;
	  
          if (verbose)
	    printf ("%s umounted\n", res.dummy);

	  if (!isroot)
	    {
	     update_mtab (res.dummy, NULL);

 	     if (verbose)
		printf ("/etc/mtab updated\n");

	     unlock_mtab ();
	    }

	  unstickargs.name=(char*)malloc((NFS_MAXPATHLEN+1)*sizeof(char));

	  strcpy (unstickargs.name, res.dummy);
  	  callrpc ("localhost",ADM_PROGRAM,ADM_VERSION,ADMPROC_UNSTICK, (xdrproc_t)xdr_sfs_unstickargs, (char*)&unstickargs, (xdrproc_t)xdr_sfsstat, (char*)&statsfs);

	  switch (statsfs)
	  {
		case SFS_OK:
          		return 0;
		default:
			clnt_perrno (statsfs);
			fprintf (stderr, "Error: bof %d\n", statsfs);
			exit (0);
	  }
        }
    }

  /* Umount failed, complain, but don't die.  */
  umnt_err = errno;
  if (!isroot)
	  unlock_mtab ();

  switch (umnt_err)
    {
    case ENXIO:
      error ("umount: %s: invalid block device", spec); break;
    case EINVAL:
      error ("umount: %s: not mounted", spec); break;
    case EIO:
      error ("umount: %s: can't write superblock", spec); break;
    case EBUSY:
      error ("umount: %s: device is busy", spec); break;
    case ENOENT:
      error ("umount: %s: not mounted", spec); break;
    case EPERM:
      error ("umount: %s: must be superuser to umount", spec); break;
    case EACCES:
      error ("umount: %s: block devices not permitted on fs", spec); break;
    default:
      error ("umount: %s: %s", spec, strerror (umnt_err)); break;
    }
  return 1;
}

extern char version[];

static struct option longopts[]=
{
	{ "force", 0, 0, 'f'},
	{ "help", 0, 0, 'h'},
	{ "verbose", 0, 0, 'v'},
	{ "version", 0, 0, 'V'},
	{ NULL, 0, 0, 0}
};

char *usage_string = "\
usage: sfsumount [-hV]\n\
       sfsumount [-v] special | node\n\
";

static void
usage (FILE *fp, int n)
{
  fprintf (fp, "%s", usage_string);
  exit (n);
}

int
main (int argc, char *argv[])
{
  int c;
  string_list options;
  struct mntent *mnt;
  struct mntent mntbuf;
  struct mntent *fs;
  char *file;
  int result = 0;

  while ((c = getopt_long (argc, argv, "fhvV", longopts, NULL)) != EOF)
    switch (c)
      {
      case 'f':			/* force umount (needs kernel support) */
#if 0
	++force;
#else
	die (2, "umount: forced umount not supported yet");
#endif
	break;
      case 'h':			/* help */
	usage (stdout, 0);
	break;
      case 'v':			/* make noise */
	++verbose;
	break;
      case 'V':			/* version */
	printf ("%s\n", version);
	exit (0);
      case 0:
	break;
      case '?':
      default:
	usage (stderr, 1);
      }

  if (getuid () != geteuid ())
    {
      suid = 1;
    }

  argc -= optind;
  argv += optind;

  if (argc != 1)
    usage (stderr, 2);
  else
    {
      file = canonicalize (*argv); /* mtab paths are canonicalized */

      open_mtab ("r");
      mnt = getmntfile (file);
      if (mnt)
	{
	  /* Copy the structure and strings becuase they're in static areas. */
	  mntbuf = *mnt;
	  mnt = &mntbuf;
	  mnt->mnt_fsname = xstrdup (mnt->mnt_fsname);
	  mnt->mnt_dir = xstrdup (mnt->mnt_dir);
	}
      close_mtab ();

      if (suid)
	{
	  if (!mnt)
	    die (2, "umount: %s is not mounted", file);
	  if (!(fs = getfsspec (file)) && !(fs = getfsfile (file)))
	    die (2, "umount: %s is not in the fstab", file);
	  if (!streq (mnt->mnt_fsname, fs->mnt_fsname)
	      || !streq (mnt->mnt_dir, fs->mnt_dir))
	    die (2, "umount: %s mount disagrees with the fstab", file);
	  options = parse_list (fs->mnt_opts);
	  while (options)
	    {	
	      if (streq (car (options), "user"))
		break;
	      options = cdr (options);
	    }
	  if (!options)
	    die (2, "umount: only root can unmount %s from %s",
		 fs->mnt_fsname, fs->mnt_dir);
	}

      if (mnt)
	result = umount_one (xstrdup (mnt->mnt_fsname), xstrdup(mnt->mnt_dir));
      else
	result = umount_one (*argv, *argv);
    }
  exit (result);
}
