/*
 * T.C.F.S. 2.0 Alpha 1 
 *
 *      	   This  program  handles  RPC  "NFS"  data  requests
 *              adopting a secure transfer protocol.
 *                 This   is  an  unsecure   and  unchecked  version,
 *              use at your own risk.
 *
 *              Please, report Bugs to: <tcfs@edu-gw.dia.unisa.it>
 *
 * Authors:	Giuseppe Cattaneo, <cattaneo@udsab.dia.unisa.it>
 *		Giuseppe Persiano, <giuper@udsab.dia.unisa.it>
 *		Andrea Cozzolino, <andcoz@edu-gw.dia.unisa.it>
 *		Angelo Celentano, <angcel@edu-gw.dia.unisa.it>
 *		Aniello Del Sorbo, <anidel@edu-gw.dia.unisa.it>
 *		Ermelindo Mauriello, <ermmau@edu-gw.dia.unisa.it>
 *		Raffaele Pisapia, <rafpis@edu-gw.dia.unisa.it>
 *
 *   Permission to  use, copy, and modify  this software  without fee
 * is hereby granted, provided that this entire notice is included in
 * all copies  of  any  software  which  is  or  includes a  copy  or
 * modification of this software and in all copies  of the supporting
 * documentation for such software.
 *
 *   This  software is  distribuited  under  the  GNU General  Public
 * License  (version  2, June  1991). Check  the  file  'COPING'  for
 * more  infos. Some  parts of  this  software  derive  from the  NFS
 * implementation in the Linux kernel 2.0.x.
 *
 * This software  maybe be used  for any  purpose provided  the above
 * copyright  notice  is retained.  It  is  supplied  as is,  with no
 * warranty expressed or implied.
 *
 */

/* -+-_== */

/*
 * linux/fs/tcfs/tcfsiod.c
 *
 * Async TCFS RPC call support.
 *
 * When a process wants to place an asynchronous RPC call, it reserves
 * an tcfsiod slot, fills in all necessary fields including the callback
 * handler field, and enqueues the request.
 *
 * This will wake up tcfsiod, which calls tcfs_tcrpc_doio to collect the
 * reply. It then dispatches the result to the caller via the callback
 * function, including result value and request pointer. It then re-inserts
 * itself into the free list.
 *
 * Copyright (C) 1996, Olaf Kirch <okir@monad.swb.de>
 */

#include <linux/sched.h>
#include <linux/tcfs_fs.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/tcrpcsock.h>
#include <linux/tcfsiod.h>

static struct tcfsiod_req *	free_list = NULL;
static int			active = 0;

#undef DEBUG_TCFSIOD
#ifdef DEBUG_TCFSIOD
#define dprintk(args...)	printk(## args)
#else
#define dprintk(args...)	/* nothing */
#endif


/*
 * Reserve an tcfsiod slot and initialize the request struct
 */
struct tcfsiod_req *
tcfsiod_reserve(struct tcfs_server *server)
{
	struct tcfsiod_req	*req;

	if (!(req = free_list)) {
		dprintk("BIO: tcfsiod_reserve: no free tcfsiods\n");
		return NULL;
	}
	free_list = req->rq_next;
	memset(&req->rq_rpcreq, 0, sizeof(struct tcrpc_ioreq));

	if (tcrpc_reserve(server->rsock, &req->rq_rpcreq, 1) < 0) {
		dprintk("BIO: tcfsiod_reserve failed to reserve RPC slot\n");
		req->rq_next = free_list;
		free_list = req;
		return NULL;
	}

	req->rq_server = server;
	return req;
}

void
tcfsiod_release(struct tcfsiod_req *req)
{
	dprintk("BIO: tcfsiod_release called\n");
	tcrpc_release(req->rq_server->rsock, &req->rq_rpcreq);
	memset(&req->rq_rpcreq, 0, sizeof(struct tcrpc_ioreq));
	req->rq_next = free_list;
	free_list = req;
}

/*
 * Transmit a request and put it on tcfsiod's list of pending requests.
 */
void
tcfsiod_enqueue(struct tcfsiod_req *req)
{
	dprintk("BIO: enqueuing request %p\n", &req->rq_rpcreq);
	wake_up(&req->rq_wait);
	schedule();
}

/*
 * This is the main tcfsiod loop.
 */
int
tcfsiod(void)
{
	struct tcfsiod_req	request, *req = &request;
	int			result;

	dprintk("BIO: tcfsiod %d starting\n", current->pid);
	while (1) {
		/* Insert request into free list */
		memset(req, 0, sizeof(*req));
		req->rq_next = free_list;
		free_list = req;

		/* Wait until user enqueues request */
		dprintk("BIO: before: now %d tcfsiod's active\n", active);
		dprintk("BIO: tcfsiod %d waiting\n", current->pid);
		interruptible_sleep_on(&req->rq_wait);

		if (current->signal & ~current->blocked)
			break;
		if (!req->rq_rpcreq.rq_slot)
			continue;
		dprintk("BIO: tcfsiod %d woken up; calling tcfs_tcrpc_doio.\n",
				current->pid);
		active++;
		dprintk("BIO: before: now %d tcfsiod's active\n", active);
		do {
			result = tcfs_tcrpc_doio(req->rq_server,
						&req->rq_rpcreq, 1);
		} while (!req->rq_callback(result, req));
		active--;
	}

	return 0;
}
