#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/stat.h>
#include <linux/tcfs_fs.h>
#include <linux/fcntl.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/mm.h>

#include <asm/segment.h>	/* for fs functions */
#include <linux/tcfs_fs.h>
#include <linux/tcfs.h>

/* These are the function needed to handle keys hash table */
#include <linux/tcfs/hash.h>
#include <linux/tcfs/cipherinterface.h>

static int tcfs_dir_open(struct inode * inode, struct file * file);
static int tcfs_dir_read(struct inode *, struct file *, char *, int);
static int tcfs_readdir(struct inode *, struct file *, void *, filldir_t);
int tcfs_lookup(struct inode *, const char *, int, struct inode **);
static int tcfs_create(struct inode *, const char *, int, int, struct inode **);
static int tcfs_mkdir(struct inode *, const char *, int, int);
static int tcfs_rmdir(struct inode *, const char *, int);
static int tcfs_unlink(struct inode *, const char *, int);
static int tcfs_symlink(struct inode *, const char *, int, const char *);
static int tcfs_link(struct inode *, struct inode *, const char *, int);
static int tcfs_mknod(struct inode *, const char *, int, int, int);
static int tcfs_rename(struct inode *, const char *, int,
		      struct inode *, const char *, int, int);

static struct file_operations tcfs_dir_operations = {
	NULL,			/* lseek - default */
	tcfs_dir_read,		/* read - bad */
	NULL,			/* write - bad */
	tcfs_readdir,		/* readdir */
	NULL,			/* select - default */
	tcfs_ioctl,			/* ioctl - default */
	NULL,			/* mmap */
	tcfs_dir_open,		/* open - revalidate */
	NULL,			/* no special release code */
	NULL			/* fsync */
};

struct inode_operations tcfs_dir_inode_operations = {
	&tcfs_dir_operations,	/* default directory file-ops */
	tcfs_create,		/* create */
	tcfs_lookup,		/* lookup */
	tcfs_link,		/* link */
	tcfs_unlink,		/* unlink */
	tcfs_symlink,		/* symlink */
	tcfs_mkdir,		/* mkdir */
	tcfs_rmdir,		/* rmdir */
	tcfs_mknod,		/* mknod */
	tcfs_rename,		/* rename */
	NULL,			/* readlink */
	NULL,			/* follow_link */
	NULL,			/* readpage */
	NULL,			/* writepage */
	NULL,			/* bmap */
	NULL,			/* truncate */
	NULL			/* permission */
};

static void purgepath(char *argv)
{
	char *a,*o,*p,*s;
	int l;
	
	a=argv;
	o=NULL;
	
	while ((p=strchr(a,'/'))!=NULL)	{
		s=strchr(p+1,'/');
		if (s!=NULL) {
			if (strncmp(p+1,"..",s-p-1)==0) {
				if (o!=NULL) {
					l=strlen(s+1);
					strcpy(o+1,s+1);
					*(o+1+l)='\0';
					p=o;
					o=NULL;
				}
			}
			o=p;
			a=p+1;
		} else {
			break;
		}
	}
}

static inline void revalidate_dir(struct tcfs_server * server, struct inode * dir)
{
	struct tcfs_fattr fattr;

	if (jiffies - TCFS_READTIME(dir) < TCFS_ATTRTIMEO(dir)) {
		return;
	}
	TCFS_READTIME(dir) = jiffies;
	if (tcfs_proc_getattr(server, TCFS_FH(dir), &fattr) == 0) {
		tcfs_refresh_inode(dir, &fattr);
		if (fattr.mtime.seconds == TCFS_OLDMTIME(dir)) {
			if ((TCFS_ATTRTIMEO(dir) <<= 1) > server->acdirmax)
				TCFS_ATTRTIMEO(dir) = server->acdirmax;
			return;
		}
		TCFS_OLDMTIME(dir) = fattr.mtime.seconds;
	}
	if (tcfs_proc_geteattr(&(dir->i_sb->u.tcfs_sb.x_server),dir->u.tcfs_i.pathname,&(dir->u.tcfs_i.tcfs_fl.cflag))<0) {
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_proc_geteattr returned error\n");
#endif
		dir->u.tcfs_i.tcfs_fl.bf.mb=0;
		dir->u.tcfs_i.tcfs_fl.bf.spure=0;
		dir->u.tcfs_i.tcfs_fl.bf.unknow=0;
	}
	/* invalidate directory cache here when we _really_ start caching */
}

static int tcfs_dir_open(struct inode * dir, struct file * file)
{
	revalidate_dir(TCFS_SERVER(dir), dir);
	return 0;
}

static int tcfs_dir_read(struct inode *inode, struct file *filp, char *buf,
			int count)
{
	return -EISDIR;
}

static struct tcfs_entry *c_entry = NULL;

/*
 * We need to do caching of directory entries to prevent an
 * incredible amount of RPC traffic.  Only the most recent open
 * directory is cached.  This seems sufficient for most purposes.
 * Technically, we ought to flush the cache on close but this is
 * not a problem in practice.
 */

static int tcfs_readdir(struct inode *inode, struct file *filp,
		       void *dirent, filldir_t filldir)
{
	static kdev_t c_dev = 0;
	static int c_ino;
	static int c_size;

	int result=0;
	int i=0, index = 0,k=0;
	struct tcfs_entry *entry;
	struct hash_entry *htmp=NULL;
	struct gid_hash_entry *ghtmp=NULL;
	void *ks=NULL;
	char doit=0;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_readdir %s\n",inode->u.tcfs_i.pathname);
#endif
	TCFS_SEMR_DOWN(inode);

	if (!inode || !S_ISDIR(inode->i_mode)) {
		printk("tcfs_readdir: inode is NULL or not a directory\n");
		TCFS_SEMR_UP(inode);
		return -EBADF;
	}
	revalidate_dir(TCFS_SERVER(inode), inode);

	/* initialize cache memory if it hasn't been used before */

	if (c_entry == NULL) {
		i = sizeof (struct tcfs_entry)*TCFS_READDIR_CACHE_SIZE;
		c_entry = (struct tcfs_entry *) kmalloc(i, GFP_KERNEL);
		if (c_entry == NULL) {
			printk("tcfs_readdir: no MEMORY for cache\n");
			TCFS_SEMR_UP(inode);
			return -ENOMEM;
		}
		for (i = 0; i < TCFS_READDIR_CACHE_SIZE; i++) {
			c_entry[i].name = (char *) kmalloc(TCFS_MAXNAMLEN + 1,
				GFP_KERNEL);
			if (c_entry[i].name == NULL) {
				printk("tcfs_readdir: no MEMORY for cache\n");
				while (--i>=0)
					kfree(c_entry[i].name);
				kfree(c_entry);
				c_entry = NULL;
			TCFS_SEMR_UP(inode);
			return -ENOMEM;
			}
		}
	}
	entry = NULL;

	/* try to find it in the cache */

	if (inode->i_dev == c_dev && inode->i_ino == c_ino) {
		for (i = 0; i < c_size; i++) {
			if (filp->f_pos == c_entry[i].cookie) {
				if (i == c_size - 1) {
					if (c_entry[i].eof) {
					TCFS_SEMR_UP(inode);
					return 0;
					}
				}
				else
					entry = c_entry + (index = i + 1);
				break;
			}
		}
	}
	/* if we didn't find it in the cache, revert to an tcfs call */

	entry = NULL;
	if (!entry) {
		result = tcfs_proc_readdir(TCFS_SERVER(inode), TCFS_FH(inode),
			filp->f_pos, TCFS_READDIR_CACHE_SIZE, c_entry);
		if (result < 0) {
			c_dev = 0;
			TCFS_SEMR_UP(inode);
			return result;
		}
		if (result > 0) {
			c_dev = inode->i_dev;
			c_ino = inode->i_ino;
			c_size = result;
			entry = c_entry + (index = 0);
		}
	}
	if (result>0 && TCFS_IS_SHARED(inode)) {
		ghtmp=gid_hash_lookup(inode->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			TCFS_SEMR_UP(inode);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (result>0 && TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				TCFS_SEMR_UP(inode);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	if (doit) {
		for (i=0;i<result;i++) {
			if (strcmp(c_entry[i].name,".")!=0 && strcmp(c_entry[i].name,"..")!=0) {
				k=strlen(c_entry[i].name);
				k=tcfsdecode(c_entry[i].name,k+1);
				tcfs_decrypt(c_entry[i].name,(k & 0xfff8)+8,ks);
			}
		}
	}
	/* if we found it in the cache or from an tcfs call, return results */
	if (!entry) {
		return 0;
	}
	while (index < c_size) {
		int nextpos = entry->cookie;
		if (filldir(dirent, entry->name, strlen(entry->name), filp->f_pos, entry->fileid) < 0)
			break;
		filp->f_pos = nextpos;
		/* revalidate the cache if we slept in filldir() */
		if (inode->i_dev != c_dev)
			break;
		if (inode->i_ino != c_ino)
			break;
		if (nextpos != entry->cookie)
			break;
		index++;
		entry++;
	}
	TCFS_SEMR_UP(inode);
	return 0;
}

/*
 * free cache memory
 * called from cleanup_module
 */

void tcfs_kfree_cache(void)
{
	int i;

	if (c_entry == NULL)
		return;
	for (i = 0; i < TCFS_READDIR_CACHE_SIZE; i++)
		kfree(c_entry[i].name);
	kfree(c_entry);
	c_entry = NULL;
}
 

/*
 * Lookup caching is a big win for performance but this is just
 * a trial to see how well it works on a small scale.
 * For example, bash does a lookup on ".." 13 times for each path
 * element when running pwd.  Yes, hard to believe but true.
 * Try pwd in a filesystem mounted with noac.
 *
 * It trades a little cpu time and memory for a lot of network bandwidth.
 * Since the cache is not hashed yet, it is a good idea not to make it too
 * large because every lookup looks through the entire cache even
 * though most of them will fail.
 */

static struct tcfs_lookup_cache_entry {
	kdev_t dev;
	int inode;
	char filename[TCFS_MAXNAMLEN + 1];
	struct tcfs_fh fhandle;
	struct tcfs_fattr fattr;
	int expiration_date;
} tcfs_lookup_cache[TCFS_LOOKUP_CACHE_SIZE];

static struct tcfs_lookup_cache_entry *tcfs_lookup_cache_index(struct inode *dir,
							     const char *filename)
{
	struct tcfs_lookup_cache_entry *entry;
	int i;

	for (i = 0; i < TCFS_LOOKUP_CACHE_SIZE; i++) {
		entry = tcfs_lookup_cache + i;
		if (entry->dev == dir->i_dev
		    && entry->inode == dir->i_ino
		    && !strncmp(filename, entry->filename, TCFS_MAXNAMLEN))
			return entry;
	}
	return NULL;
}

static int tcfs_lookup_cache_lookup(struct inode *dir, const char *filename,
				   struct tcfs_fh *fhandle,
				   struct tcfs_fattr *fattr)
{
	static int tcfs_lookup_cache_in_use = 0;

	struct tcfs_lookup_cache_entry *entry;

	if (!tcfs_lookup_cache_in_use) {
		memset(tcfs_lookup_cache, 0, sizeof(tcfs_lookup_cache));
		tcfs_lookup_cache_in_use = 1;
	}
	if ((entry = tcfs_lookup_cache_index(dir, filename))) {
		if (jiffies > entry->expiration_date) {
			entry->dev = 0;
			return 0;
		}
		*fhandle = entry->fhandle;
		*fattr = entry->fattr;
		return 1;
	}
	return 0;
}

static void tcfs_lookup_cache_add(struct inode *dir, const char *filename,
				 struct tcfs_fh *fhandle,
				 struct tcfs_fattr *fattr)
{
	static int tcfs_lookup_cache_pos = 0;
	struct tcfs_lookup_cache_entry *entry;

	/* compensate for bug in SGI TCFS server */
	if (fattr->size == -1 || fattr->uid == -1 || fattr->gid == -1
	    || fattr->atime.seconds == -1 || fattr->mtime.seconds == -1)
		return;
	if (!(entry = tcfs_lookup_cache_index(dir, filename))) {
		entry = tcfs_lookup_cache + tcfs_lookup_cache_pos++;
		if (tcfs_lookup_cache_pos == TCFS_LOOKUP_CACHE_SIZE)
			tcfs_lookup_cache_pos = 0;
	}
	entry->dev = dir->i_dev;
	entry->inode = dir->i_ino;
	strcpy(entry->filename, filename);
	entry->fhandle = *fhandle;
	entry->fattr = *fattr;
	entry->expiration_date = jiffies + (S_ISDIR(fattr->mode)
		? TCFS_SERVER(dir)->acdirmin : TCFS_SERVER(dir)->acregmin);
}

static void tcfs_lookup_cache_remove(struct inode *dir, struct inode *inode,
				    const char *filename)
{
	struct tcfs_lookup_cache_entry *entry;
	kdev_t dev;
	int fileid;
	int i;

	if (inode) {
		dev = inode->i_dev;
		fileid = inode->i_ino;
	}
	else if ((entry = tcfs_lookup_cache_index(dir, filename))) {
		dev = entry->dev;
		fileid = entry->fattr.fileid;
	}
	else
		return;
	for (i = 0; i < TCFS_LOOKUP_CACHE_SIZE; i++) {
		entry = tcfs_lookup_cache + i;
		if (entry->dev == dev && entry->fattr.fileid == fileid)
			entry->dev = 0;
	}
}

static void tcfs_lookup_cache_refresh(struct inode *file,
				     struct tcfs_fattr *fattr)
{
	struct tcfs_lookup_cache_entry *entry;
	kdev_t dev = file->i_dev;
	int fileid = file->i_ino;
	int i;

	for (i = 0; i < TCFS_LOOKUP_CACHE_SIZE; i++) {
		entry = tcfs_lookup_cache + i;
		if (entry->dev == dev && entry->fattr.fileid == fileid)
			entry->fattr = *fattr;
	}
}

int tcfs_lookup(struct inode *dir, const char *__name, int len,
		      struct inode **result)
{
	struct tcfs_fh fhandle;
	struct tcfs_fattr fattr;
	char *name=NULL; /*[TCFS_MAXPATHLEN+1];*/
	int error,k;
	struct hash_entry * htmp=NULL;
	struct gid_hash_entry *ghtmp=NULL;
	void *ks=NULL;
	char doit=0;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_lookup %s\n",__name);
#endif
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL || current->uid!=dir->i_uid) {
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}	

	TCFS_SEMR_DOWN(dir);

	while (name==NULL) {
		name=kmalloc(TCFS_MAXPATHLEN+1,GFP_KERNEL);
		schedule();
	}
	*result = NULL;
	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_lookup: inode is NULL or not a directory\n");
		kfree(name);
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		kfree(name);
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	memset(name,'\0',TCFS_MAXPATHLEN);
	memcpy(name,__name,len);
	name[len] = '\0';
	/* Handle cryptographed file names */
	if (doit && (strcmp(name,".")!=0 && strcmp(name,"..")!=0)) {
		k=len+1;
		tcfs_encrypt(name,(k & 0xfff8)+8,ks);
		tcfsencode(name,(k & 0xfff8)+8);
		len=strlen(name);
	}
	if (len == 1 && name[0] == '.') { /* cheat for "." */
		*result = dir;
		kfree(name);
		TCFS_SEMR_UP(dir);
		return 0;
	}
	if ((TCFS_SERVER(dir)->flags & TCFS_MOUNT_NOAC)
	    || !tcfs_lookup_cache_lookup(dir, name, &fhandle, &fattr)) {
		if ((error = tcfs_proc_lookup(TCFS_SERVER(dir), TCFS_FH(dir),
		    name, &fhandle, &fattr))) {
			TCFS_SEMR_UP(dir);
			kfree(name);
			iput(dir);
			return error;
		}
		tcfs_lookup_cache_add(dir, name, &fhandle, &fattr);
	}
	if (!(*result = tcfs_fhget(dir->i_sb, &fhandle, &fattr))) {
		TCFS_SEMR_UP(dir);
		kfree(name);
		iput(dir);
		return -EACCES;
	}
	strcpy((*result)->u.tcfs_i.pathname,dir->u.tcfs_i.pathname);
	strcat((*result)->u.tcfs_i.pathname,"/");
	strcat((*result)->u.tcfs_i.pathname,name);
	purgepath((*result)->u.tcfs_i.pathname);
	if (S_ISLNK((*result)->i_mode) 
		|| S_ISCHR((*result)->i_mode) 
		|| S_ISFIFO((*result)->i_mode) 
		|| S_ISSOCK((*result)->i_mode)
		|| S_ISBLK((*result)->i_mode)) {
		(*result)->u.tcfs_i.tcfs_fl.cflag=0;
	} else {
		if ((error=tcfs_proc_geteattr(&(dir->i_sb->u.tcfs_sb.x_server),
			(*result)->u.tcfs_i.pathname,&((*result)->u.tcfs_i.tcfs_fl.cflag)))<0)
		{
#ifdef DEBUG_TCFS
	printk("TCFS: xattrproc_geteattr return %d\n",error);
#endif
			TCFS_SEMR_UP(dir);
			kfree(name);
			iput((*result));
			iput(dir);
			return error;
		}  
	}
	(*result)->u.tcfs_i.readcount=0;
	if (S_ISREG((*result)->i_mode) && 
		(TCFS_IS_SHARED((*result)) || 
		TCFS_IS_SECURE((*result)->u.tcfs_i.tcfs_fl.cflag)) && 
		(*result)->i_size>0 && (*result)->u.tcfs_i.tcfs_fl.bf.spure>0) {
		(*result)->i_size-=8-(*result)->u.tcfs_i.tcfs_fl.bf.spure;
	}
	up(&((*result)->u.tcfs_i.wrt));
	up(&((*result)->u.tcfs_i.mutex));
	kfree(name);
	TCFS_SEMR_UP(dir);
	iput(dir);
	return 0;
}

static int tcfs_create(struct inode *dir, const char *name, int len, int mode,
		      struct inode **result)
{
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;
	char *newname=NULL; /*[TCFS_MAXPATHLEN+1];*/
	struct hash_entry *htmp=NULL;
	int error,k;
	struct gid_hash_entry *ghtmp=NULL;
	void *ks=NULL;
	char doit=0;
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_create %s\n",name);
#endif
	TCFS_SEMR_DOWN(dir);
	
	while (newname==NULL) {
		newname=kmalloc(TCFS_MAXPATHLEN+1,GFP_KERNEL);
		schedule();
	}
	*result = NULL;
	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_create: inode is NULL or not a directory\n");
		kfree(newname);
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		kfree(newname);
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	sattr.mode = mode;
	sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;
	memset(newname,0,TCFS_MAXPATHLEN+1);
	strcpy(newname,name);
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			TCFS_SEMR_UP(dir);
			kfree(newname);
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				TCFS_SEMR_UP(dir);
				kfree(newname);
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	if (doit && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
			k=strlen(newname)+1;
			tcfs_encrypt(newname,(k & 0xfff8)+8,ks);
			tcfsencode(newname,(k & 0xfff8)+8);
	}
	if ((error = tcfs_proc_create(TCFS_SERVER(dir), TCFS_FH(dir),
		newname, &sattr, &fhandle, &fattr))) {
		kfree(newname);
		TCFS_SEMR_UP(dir);
		iput(dir);
		return error;
	}
	if (!(*result = tcfs_fhget(dir->i_sb, &fhandle, &fattr))) {
		kfree(newname);
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -EACCES;
	}
	tcfs_lookup_cache_add(dir, newname, &fhandle, &fattr);

	strcpy((*result)->u.tcfs_i.pathname,dir->u.tcfs_i.pathname);
	strcat((*result)->u.tcfs_i.pathname,"/");
	strcat((*result)->u.tcfs_i.pathname,newname);
	purgepath((*result)->u.tcfs_i.pathname);
	(*result)->u.tcfs_i.tcfs_fl.cflag=0;
	(*result)->u.tcfs_i.tcfs_fl.bf.mb=dir->u.tcfs_i.tcfs_fl.bf.mb; /* Inherited Xflag */
	(*result)->u.tcfs_i.tcfs_fl.bf.gshar=dir->u.tcfs_i.tcfs_fl.bf.gshar; /* Inherited Gflag */
	(*result)->u.tcfs_i.readcount=0;
	error=tcfs_proc_seteattr(NULL,&((*result)->i_sb->u.tcfs_sb.x_server),(*result)->u.tcfs_i.pathname,(*result)->u.tcfs_i.tcfs_fl.cflag,NULL);
	up(&((*result)->u.tcfs_i.wrt));
	up(&((*result)->u.tcfs_i.mutex));
	TCFS_SEMR_UP(dir);
	kfree(newname);
	iput(dir);
	return error;
}

static int tcfs_mknod(struct inode *dir, const char *name, int len,
		     int mode, int rdev)
{
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;
	char newname[TCFS_MAXPATHLEN+1];
	struct hash_entry *htmp=NULL;
	int error,k;
	struct gid_hash_entry *ghtmp;
	void *ks;
	char doit=0;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_mknod %s\n",name);
#endif
	TCFS_SEMR_DOWN(dir);
	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_mknod: inode is NULL or not a directory\n");
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	sattr.mode = mode;
	sattr.uid = sattr.gid = (unsigned) -1;
	if (S_ISCHR(mode) || S_ISBLK(mode))
		sattr.size = rdev; /* get out your barf bag */
	else
		sattr.size = (unsigned) -1;
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;
	memset(newname,'\0',TCFS_MAXPATHLEN+1);
	strcpy(newname,name);
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			TCFS_SEMR_UP(dir);
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				TCFS_SEMR_UP(dir);
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	if (doit && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
		k=strlen(newname)+1;
		tcfs_encrypt(newname,(k & 0xfff8)+8,htmp->ks);
		tcfsencode(newname,(k & 0xfff8)+8);
	}
	error = tcfs_proc_create(TCFS_SERVER(dir), TCFS_FH(dir),
		newname, &sattr, &fhandle, &fattr);
	if (!error)
	{
		tcfs_lookup_cache_add(dir, newname, &fhandle, &fattr);
		/* The parent dir inode count may have changed ! */
		tcfs_lookup_cache_remove( NULL, dir, NULL);
	}
	TCFS_SEMR_UP(dir);
	iput(dir);
	return error;
}

static int tcfs_mkdir(struct inode *dir, const char *name, int len, int mode)
{
	struct tcfs_sattr sattr;
	struct tcfs_fattr fattr;
	struct tcfs_fh fhandle;
	char newname[TCFS_MAXPATHLEN+1],tmp[TCFS_MAXPATHLEN+1];
	struct hash_entry *htmp=NULL;
	int error,k;
	struct gid_hash_entry *ghtmp=NULL;
	void *ks=NULL;
	char doit=0;
	unsigned int newmode=0;
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_mkdir %s\n",name);
#endif
	TCFS_SEMR_DOWN(dir);
	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_mkdir: inode is NULL or not a directory\n");
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	sattr.mode = mode;
	sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;
	memset(newname,'\0',TCFS_MAXPATHLEN+1);
	strcpy(newname,name);
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			TCFS_SEMR_UP(dir);
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
		newmode=TCFS_GSHARED_FL;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				TCFS_SEMR_UP(dir);
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
			newmode=TCFS_SECURE_FL;
		}
	}
	if (doit && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
		k=strlen(newname)+1;
		tcfs_encrypt(newname,(k & 0xfff8)+8,ks);
		tcfsencode(newname,(k & 0xfff8)+8);
	}
	error = tcfs_proc_mkdir(TCFS_SERVER(dir), TCFS_FH(dir),
		newname, &sattr, &fhandle, &fattr);
	if (!error) {
		if (fattr.fileid == dir->i_ino)
			printk("Sony NewsOS 4.1R buggy tcfs server?\n");
		else
			tcfs_lookup_cache_add(dir, newname, &fhandle, &fattr);
	}
	if (!error && 
		(TCFS_IS_SHARED(dir) ||
		TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag))) {
			strcpy(tmp,dir->u.tcfs_i.pathname);
			strcat(tmp,"/");
			strcat(tmp,newname);
			tmp[strlen(tmp)+1]='\0';            
			k=tcfs_proc_seteattr(NULL,&(dir->i_sb->u.tcfs_sb.x_server),tmp,newmode,NULL);
			TCFS_SEMR_UP(dir);
			iput(dir);
			return k;
	}
	TCFS_SEMR_UP(dir);
	iput(dir);
	return error;
}

static int tcfs_rmdir(struct inode *dir, const char *name, int len)
{
	int error,k;
	char newname[TCFS_MAXPATHLEN+1];
	struct hash_entry *htmp;
	struct gid_hash_entry *ghtmp;
	void *ks=NULL;
	char doit=0;
#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_rmdir %s\n",name);
#endif
	TCFS_SEMR_DOWN(dir);
	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_rmdir: inode is NULL or not a directory\n");
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	memset(newname,'\0',TCFS_MAXPATHLEN+1);
	strcpy(newname,name);
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			TCFS_SEMR_UP(dir);
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				TCFS_SEMR_UP(dir);
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	if (doit && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
		k=strlen(newname)+1;
		tcfs_encrypt(newname,(k & 0xfff8)+8,ks);
		tcfsencode(newname,(k & 0xfff8)+8);
	}
	error = tcfs_proc_rmdir(TCFS_SERVER(dir), TCFS_FH(dir), newname);
	tcfs_lookup_cache_remove(dir, NULL, name);
	TCFS_SEMR_UP(dir);
	iput(dir);
	return error;
}

static int tcfs_sillyrename(struct inode *dir, const char *name, char *newname,int len,void *ks)
{
	struct inode	*inode;
	char		silly[256];
	int		slen, ret, k;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_sillyrename %s %s\n",name,newname);
#endif
	dir->i_count++;	
	if (tcfs_lookup(dir, name, len, &inode) < 0) {
		return -EIO;		/* arbitrary */
	}
	if (inode->i_count == 1 || TCFS_RENAMED_DIR(inode)) {
		iput(inode);
		return -EIO;
	}
	slen = sprintf(silly, ".tcfs%ld", inode->i_ino);

	if (len == slen && !strncmp(newname, silly, len)) {
		iput(inode);
		return -EIO;		/* DWIM */
	}
	if (TCFS_IS_SHARED(inode) || TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
		k=slen+1;
		tcfs_encrypt(silly,(k & 0xfff8)+8,ks);
		tcfsencode(silly,(k & 0xfff8)+8);
	}
	
	ret = tcfs_proc_rename(TCFS_SERVER(dir), TCFS_FH(dir), newname,
					       TCFS_FH(dir), silly, 0);
	if (ret >= 0) {
		tcfs_lookup_cache_remove(dir, NULL, name);
		tcfs_lookup_cache_remove(dir, NULL, silly);
		TCFS_RENAMED_DIR(inode) = dir;
		strcpy(inode->u.tcfs_i.pathname,dir->u.tcfs_i.pathname);
		strcat(inode->u.tcfs_i.pathname,"/");
		strcat(inode->u.tcfs_i.pathname,silly);
		purgepath(inode->u.tcfs_i.pathname);
		dir->i_count++;
	}
	iput(inode);
	return ret;
}

void tcfs_sillyrename_cleanup(struct inode *inode)
{
	struct inode	*dir = TCFS_RENAMED_DIR(inode);
	char		silly[256];
	int		error, slen;

	slen = sprintf(silly, ".tcfs%ld", inode->i_ino);
	if ((error = tcfs_unlink(dir, silly, slen)) < 0) {
		printk("TCFS silly_rename cleanup failed (err = %d)\n",
					-error);
	}
	TCFS_RENAMED_DIR(inode) = NULL;
}

static int tcfs_unlink(struct inode *dir, const char *name, int len)
{
	int error,k;
	char newname[TCFS_MAXPATHLEN+1];
	struct hash_entry * htmp=NULL;
	struct gid_hash_entry *ghtmp;
	void *ks=NULL;
	char doit=0;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_unlink %s\n",name);
#endif
	TCFS_SEMR_DOWN(dir);

	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_unlink: inode is NULL or not a directory\n");
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	memset(newname,'\0',TCFS_MAXPATHLEN+1);
	strncpy(newname,name,len);
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			TCFS_SEMR_UP(dir);
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				TCFS_SEMR_UP(dir);
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	if (doit && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
		k=len+1;
		tcfs_encrypt(newname,(k & 0xfff8)+8,ks);
		tcfsencode(newname,(k & 0xfff8)+8);
	}
	if ((error = tcfs_sillyrename(dir, name, newname, len, ks)) < 0) {
		error = tcfs_proc_remove(TCFS_SERVER(dir), TCFS_FH(dir), newname);
		tcfs_lookup_cache_remove(dir, NULL, newname);
	}
	TCFS_SEMR_UP(dir);
	iput(dir);
	return error;
}

static int tcfs_symlink(struct inode *dir, const char *name, int len,
		       const char *symname)
{
	struct tcfs_sattr sattr;
	int error,k;
	char newname[TCFS_MAXPATHLEN+1];
	struct hash_entry *htmp=NULL;
	struct gid_hash_entry *ghtmp=NULL;
	void *ks=NULL;
	char doit=0;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_symlink %s->%s\n",name,symname);
#endif
	TCFS_SEMR_DOWN(dir);
	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_symlink: inode is NULL or not a directory\n");
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	if (strlen(symname) > TCFS_MAXPATHLEN) {
		TCFS_SEMR_UP(dir);
		iput(dir);
		return -ENAMETOOLONG;
	}
	memset(newname,'\0',TCFS_MAXPATHLEN+1);
	strcpy(newname,name);
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			TCFS_SEMR_UP(dir);
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				TCFS_SEMR_UP(dir);
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	if (doit && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
		k=strlen(newname)+1;
		tcfs_encrypt(newname,(k & 0xfff8)+8,ks);
		tcfsencode(newname,(k & 0xfff8)+8);
	}
	sattr.mode = S_IFLNK | S_IRWXUGO; /* SunOS 4.1.2 crashes without this! */
	sattr.uid = sattr.gid = sattr.size = (unsigned) -1;
	sattr.atime.seconds = sattr.mtime.seconds = (unsigned) -1;
	error = tcfs_proc_symlink(TCFS_SERVER(dir), TCFS_FH(dir),
		newname, symname, &sattr);
	TCFS_SEMR_UP(dir);
	iput(dir);
	return error;
}

static int tcfs_link(struct inode *oldinode, struct inode *dir,
		    const char *name, int len)
{
	int error,k;
	char newname[TCFS_MAXPATHLEN+1];
	struct hash_entry * htmp=NULL;
	struct gid_hash_entry *ghtmp=NULL;
	void *ks=NULL;
	char doit=0;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_link %s\n",name);
#endif
	if (!oldinode) {
		printk("tcfs_link: old inode is NULL\n");
		iput(oldinode);
		iput(dir);
		return -ENOENT;
	}
	if (!dir || !S_ISDIR(dir->i_mode)) {
		printk("tcfs_link: dir is NULL or not a directory\n");
		iput(oldinode);
		iput(dir);
		return -ENOENT;
	}
	if (len > TCFS_MAXNAMLEN) {
		iput(oldinode);
		iput(dir);
		return -ENAMETOOLONG;
	}
	memset(newname,'\0',TCFS_MAXPATHLEN+1);
	strcpy(newname,name);
	if (TCFS_IS_SHARED(dir)) {
		ghtmp=gid_hash_lookup(dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			iput(oldinode);
			iput(dir);
			return -EACCES;
		}
		ks=ghtmp->ks;
		doit=1;
	} else {
		if (TCFS_IS_SECURE(dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				iput(oldinode);
				iput(dir);
				return -EACCES;
			}
			ks=htmp->ks;
			doit=1;
		}
	}
	if (doit && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
		k=strlen(newname)+1;
		tcfs_encrypt(newname,(k & 0xfff8)+8,ks);
		tcfsencode(newname,(k & 0xfff8)+8);
	}
	error = tcfs_proc_link(TCFS_SERVER(oldinode), TCFS_FH(oldinode),
		TCFS_FH(dir), newname);

	tcfs_lookup_cache_remove(dir, oldinode, NULL);
	iput(oldinode);
	iput(dir);
	return error;
}

static int tcfs_rename(struct inode *old_dir, const char *old_name, int old_len,
		      struct inode *new_dir, const char *new_name, int new_len,
		      int must_be_dir)
{
	int error,k=0;
	struct hash_entry *htmp=NULL;
	char oldname[TCFS_MAXPATHLEN+1], newname[TCFS_MAXPATHLEN+1];
	struct gid_hash_entry *ghtmp=NULL;
	void *oks=NULL, *nks=NULL;
	char doit1=0,doit2=0;

#ifdef DEBUG_TCFS
	printk("TCFS: tcfs_rename %s -> %s\n",old_name,new_name);
#endif
	if (!old_dir || !S_ISDIR(old_dir->i_mode)) {
		printk("tcfs_rename: old inode is NULL or not a directory\n");
		iput(old_dir);
		iput(new_dir);
		return -ENOENT;
	}
	if (!new_dir || !S_ISDIR(new_dir->i_mode)) {
		printk("tcfs_rename: new inode is NULL or not a directory\n");
		iput(old_dir);
		iput(new_dir);
		return -ENOENT;
	}
	if (old_len > TCFS_MAXNAMLEN || new_len > TCFS_MAXNAMLEN) {
		iput(old_dir);
		iput(new_dir);
		return -ENAMETOOLONG;
	}
	
	memset(oldname,0,TCFS_MAXPATHLEN);
	memset(newname,0,TCFS_MAXPATHLEN);
	strcpy(oldname,old_name);
	if (TCFS_IS_SHARED(old_dir)) {
		ghtmp=gid_hash_lookup(old_dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			iput(old_dir);
			iput(new_dir);
			return -EACCES;
		}
		oks=ghtmp->ks;
		doit1=1;
	} else {
		if (TCFS_IS_SECURE(old_dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				iput(old_dir);
				iput(new_dir);
				return -EACCES;
			}
			oks=htmp->ks;
			doit1=1;
		}
	}
	if (TCFS_IS_SHARED(new_dir)) {
		ghtmp=gid_hash_lookup(new_dir->i_gid);
		if (ghtmp==NULL || ghtmp->ks==NULL) {
			iput(old_dir);
			iput(new_dir);
			return -EACCES;
		}
		nks=ghtmp->ks;
		doit2=1;
	} else {
		if (TCFS_IS_SECURE(new_dir->u.tcfs_i.tcfs_fl.cflag)) {
			htmp=hash_lookup(current->uid);
			if (htmp==NULL) {
				iput(old_dir);
				iput(new_dir);
				return -EACCES;
			}
			nks=htmp->ks;
			doit2=1;
		}
	}
	if (doit1 && (strcmp(oldname,".")!=0 && strcmp(oldname,"..")!=0)) {
		k=old_len+1;
		tcfs_encrypt(oldname,(k & 0xfff8)+8,oks);
		tcfsencode(oldname,(k & 0xfff8)+8);
	}
	strcpy(newname,new_name);
	if (doit2 && (strcmp(newname,".")!=0 && strcmp(newname,"..")!=0)) {
		k=new_len+1;
		tcfs_encrypt(newname,(k & 0xfff8)+8,nks);
		tcfsencode(newname,(k & 0xfff8)+8);
	}
	error = tcfs_proc_rename(TCFS_SERVER(old_dir),
		TCFS_FH(old_dir), oldname,
		TCFS_FH(new_dir), newname,
		must_be_dir);
	tcfs_lookup_cache_remove(old_dir, NULL, oldname);
	tcfs_lookup_cache_remove(new_dir, NULL, newname);
	iput(old_dir);
	iput(new_dir);
	return error;
}

/*
 * Many tcfs protocol calls return the new file attributes after
 * an operation.  Here we update the inode to reflect the state
 * of the server's inode.
 */

void tcfs_refresh_inode(struct inode *inode, struct tcfs_fattr *fattr)
{
	int was_empty,k=0;
	if (!inode || !fattr) {
		printk("tcfs_refresh_inode: inode or fattr is NULL\n");
		return;
	}
	if (inode->i_ino != fattr->fileid) {
		printk("tcfs_refresh_inode: inode number mismatch\n");
		return;
	}
	was_empty = (inode->i_mode == 0);
	inode->i_mode = fattr->mode;
	inode->i_nlink = fattr->nlink;
	inode->i_uid = fattr->uid;
	inode->i_gid = fattr->gid;

	if (was_empty==0) {
		if (tcfs_proc_geteattr(&(inode->i_sb->u.tcfs_sb.x_server),inode->u.tcfs_i.pathname,&(inode->u.tcfs_i.tcfs_fl.cflag))<0) {
			inode->u.tcfs_i.tcfs_fl.cflag=0;
		}
	}
	/* Size changed from outside: invalidate caches on next read */
	if (S_ISREG(inode->i_mode) && 
		(TCFS_IS_SECURE(inode->u.tcfs_i.tcfs_fl.cflag) ||
		TCFS_IS_SHARED(inode)) &&
		inode->i_size>0 && inode->u.tcfs_i.tcfs_fl.bf.spure>0) {
		k=8-inode->u.tcfs_i.tcfs_fl.bf.spure;
	}
	if (inode->i_size != fattr->size-k) 
		TCFS_CACHEINV(inode);
	if (TCFS_OLDMTIME(inode) != fattr->mtime.seconds)
		TCFS_ATTRTIMEO(inode) = TCFS_MINATTRTIMEO(inode);
	inode->i_size = fattr->size;
	if (S_ISCHR(inode->i_mode) || S_ISBLK(inode->i_mode))
		inode->i_rdev = to_kdev_t(fattr->rdev);
	else
		inode->i_rdev = 0;
	inode->i_blocks = fattr->blocks;
	inode->i_atime = fattr->atime.seconds;
	inode->i_mtime = fattr->mtime.seconds;
	inode->i_ctime = fattr->ctime.seconds;
	if (S_ISREG(inode->i_mode))
		inode->i_op = &tcfs_file_inode_operations;
	else if (S_ISDIR(inode->i_mode))
		inode->i_op = &tcfs_dir_inode_operations;
	else if (S_ISLNK(inode->i_mode))
		inode->i_op = &tcfs_symlink_inode_operations;
	else if (S_ISCHR(inode->i_mode))
		inode->i_op = &chrdev_inode_operations;
	else if (S_ISBLK(inode->i_mode))
		inode->i_op = &blkdev_inode_operations;
	else if (S_ISFIFO(inode->i_mode)) {
		if (was_empty)
			init_fifo(inode);
	} else
		inode->i_op = NULL;
	tcfs_lookup_cache_refresh(inode, fattr);
}
