#include <linux/mm.h>
#include <linux/malloc.h>
/* On some kernel versions this is wait.h */
#include <asm/semaphore.h>

#include <linux/tcfs_fs.h>

struct hash_entry *hash_table[HASH_SIZE];
struct gid_hash_entry *gid_hash_table[HASH_SIZE];

/* Since tcfsiod makes async calls to these function we must avoid 
   conflicts during hash functions .... this means hash funcs have to
   be atomic in their exectution and mutually exclusive one-to-one.
   So we use a semaphore to syncronize theme */

struct semaphore tcfs_hash_sem = MUTEX;
struct semaphore tcfs_hash_sem_gid = MUTEX;
   
/* Here we use the division method as hash function */
#define hashpos(uid)	(uid % HASH_SIZE)
void hash_debug(void);

void 
init_hash(void)
{
	int i;
#ifdef HASH_DEBUG
	printk("TCFS: init_hash\n");
#endif
	for (i=0;i<HASH_SIZE;i++) {
		hash_table[i]=NULL;
		gid_hash_table[i]=NULL;
	}
}

void 
cleanup_hash(void)
{
	int i;
	struct hash_entry * htmp, *htmp1=NULL;
	struct gid_hash_entry *ghtmp, *ghtmp1;
#ifdef HASH_DEBUG
	printk("TCFS: cleanup_hash\n");
#endif
	down(&tcfs_hash_sem);
	for (i=0;i<HASH_SIZE;i++) {
		htmp=hash_table[i];
		while (htmp!=NULL) {
			htmp1=htmp->next;
			if (htmp->ks!=NULL)
				kfree(htmp->ks);
			kfree(htmp);
			htmp=htmp1;
		}
	}
	up(&tcfs_hash_sem);
	down(&tcfs_hash_sem_gid);
	for (i=0;i<HASH_SIZE;i++) {
		ghtmp=gid_hash_table[i];
		while (htmp!=NULL) {
			ghtmp1=ghtmp->next;
			if (ghtmp->ks!=NULL)
				kfree(ghtmp->ks);
			htmp=ghtmp->key;
			while (htmp!=NULL) {
				htmp1=htmp->next;
				kfree(htmp);
				htmp=htmp1;
			}
			kfree(htmp);
			htmp=htmp1;
		}
	}
	up(&tcfs_hash_sem_gid);
}

struct hash_entry * 
hash_lookup(int uid)
{
	struct hash_entry * htmp;
#ifdef HASH_DEBUG
	printk("TCFS: hash_lookup %d\n",uid);
#endif
	down(&tcfs_hash_sem);
	htmp=hash_table[hashpos(uid)];
	while (htmp!=NULL && htmp->uid!=uid)
		htmp=htmp->next;
	up(&tcfs_hash_sem);
	return htmp;
}

struct hash_entry * 
hash_add(int uid,char *deskey, void *ks)
{
	int pos;
	struct hash_entry * htmp;
#ifdef HASH_DEBUG
	printk("TCFS: hash_add\n");
#endif
	htmp=hash_lookup(uid);
	down(&tcfs_hash_sem);
	if (htmp!=NULL) {
		if(!memcmp(deskey,htmp->deskey,(size_t)KEYSIZE))
                        htmp->count++;
		up(&tcfs_hash_sem);
		return NULL;
	}
	pos=hashpos(uid);
	htmp=(struct hash_entry *)kmalloc(sizeof(struct hash_entry),GFP_KERNEL);
	if (htmp==NULL) {
		printk("TCFS: Unable to get a free page\n");
		up(&tcfs_hash_sem);
	}
	htmp->uid=uid;
	htmp->count=1;
	htmp->permanent=0;
	memcpy(htmp->deskey,deskey,KEYSIZE);
	htmp->ks=ks;
	htmp->next=hash_table[pos];
	hash_table[pos]=htmp;
#ifdef HASH_DEBUG
	hash_debug();
#endif
	up(&tcfs_hash_sem);
	return htmp;
}

void 
hash_rem_count(int uid)
{
	struct hash_entry * htmp1, * htmp2;

#ifdef HASH_DEBUG
	printk("TCFS: hash_rem %d\n",uid);
#endif

	down(&tcfs_hash_sem);
	htmp2=NULL;
	htmp1=hash_table[hashpos(uid)];

	while(htmp1!=NULL && htmp1->uid!=uid) {
		htmp2=htmp1;
		htmp1=htmp1->next;
	}
	if (htmp1==NULL) { /* Element not present in hash_table */
		up(&tcfs_hash_sem);
		return;
	}
	if (htmp1->count>0) {
		htmp1->count--;
	}

#ifdef HASH_DEBUG
  hash_debug();
#endif
	if (htmp1->permanent==0 && htmp1->count==0) {/* Key must be destroyed */
		if (htmp2==NULL) {/* The element is the first */
			hash_table[hashpos(uid)]=htmp1->next;
			kfree(htmp1->ks);
			kfree(htmp1);
			up(&tcfs_hash_sem);
			return;
		}

		htmp2->next=htmp1->next;
		kfree(htmp1->ks);
		kfree(htmp1);
		up(&tcfs_hash_sem);
		return;

	}

	up(&tcfs_hash_sem);

}


void 
hash_rem_all (int uid)
{
	struct hash_entry *htmp1, *htmp2;

#ifdef HASH_DEBUG
	printk("TCFS: hash_rem_count\n");
#endif

	down(&tcfs_hash_sem);
	htmp2=NULL;
	htmp1=hash_table[hashpos(uid)];

	while(htmp1!=NULL && htmp1->uid!=uid) {
		htmp2=htmp1;
		htmp1=htmp1->next;
	}

	if (htmp1==NULL) { /* Element not present into the hash table */
		up(&tcfs_hash_sem);
		return;
	}

	htmp1->count=0;

	if (htmp1->permanent==0){ /* Key must be destroyed */
		if (htmp2==NULL) { /* This is the first element */
			hash_table[hashpos(uid)]=htmp1->next;
			kfree(htmp1->ks);
			kfree(htmp1);
#ifdef HASH_DEBUG
          hash_debug();
#endif
			up(&tcfs_hash_sem);
			return;
		}
      
		htmp2->next=htmp1->next;
		kfree(htmp1->ks);
		kfree(htmp1);
	}
#ifdef HASH_DEBUG
  hash_debug();
#endif
	up(&tcfs_hash_sem);
}

void 
hash_rem_permanent (int uid)
{
  struct hash_entry * htmp1, * htmp2;

#ifdef HASH_DEBUG
  printk("TCFS: hash_rem_permanent\n");
#endif

  down(&tcfs_hash_sem);
  htmp2=NULL;
  htmp1=hash_table[hashpos(uid)];

  while(htmp1!=NULL && htmp1->uid!=uid) {
      htmp2=htmp1;
      htmp1=htmp1->next;
    }

  if (htmp1==NULL) {/* Element not present into the hash table */
      up(&tcfs_hash_sem);
      return;
    }

  htmp1->permanent=0;

  if (htmp1->count==0){ /* Key must be destroyed */
      if (htmp2==NULL){ /* This is the first element */
	  hash_table[hashpos(uid)]=htmp1->next;
	  kfree(htmp1->ks);
	  kfree(htmp1);

#ifdef HASH_DEBUG
	  hash_debug();
#endif

	  up(&tcfs_hash_sem);
	  return;
	}
      
      htmp2->next=htmp1->next;
      kfree(htmp1->ks);
      kfree(htmp1);
    }

#ifdef HASH_DEBUG
  hash_debug();
#endif

  up(&tcfs_hash_sem);
}

 
/* Here there are the functions to handle GID hash_table used to store
 * data about users key */


struct gid_hash_entry * 
gid_hash_lookup(int gid)
/* search gid entry into hash table */
{
	struct gid_hash_entry * htmp;
#ifdef HASH_DEBUG
	printk("TCFS: hash_lookup %d\n",gid);
#endif
	down(&tcfs_hash_sem_gid);
	htmp=gid_hash_table[hashpos(gid)];
	while (htmp!=NULL && htmp->gid!=gid)
		htmp=htmp->next;
	up(&tcfs_hash_sem_gid);
	return htmp;
}

struct gid_hash_entry * 
gid_hash_add(int gid)
{
	int pos=0;
	struct gid_hash_entry * htmp;
#ifdef HASH_DEBUG
	printk("TCFS: hash_add\n");
#endif
	htmp=gid_hash_lookup(gid);
	down(&tcfs_hash_sem_gid);
	if (htmp!=NULL) {
		up(&tcfs_hash_sem_gid);
		return NULL;
	}
	pos=hashpos(gid);
	htmp=(struct gid_hash_entry *)kmalloc(sizeof(struct gid_hash_entry),GFP_KERNEL);
	if (htmp==NULL) {
		printk("TCFS: Unable to get a free page\n");
		up(&tcfs_hash_sem_gid);
	}
	htmp->gid=gid;
	htmp->ks=NULL;
	htmp->key=NULL;
	htmp->next=gid_hash_table[pos];
	htmp->ins=0;
	gid_hash_table[pos]=htmp;
#ifdef HASH_DEBUG
	hash_debug();
#endif
	up(&tcfs_hash_sem_gid);
	return htmp;
}

struct hash_entry * 
gid_uid_lookup(struct hash_entry*key,int uid)
{
	struct hash_entry *htmp=key;
	
	while (htmp!=NULL && htmp->uid!=uid)
		htmp=htmp->next;
	return htmp;
}

struct gid_hash_entry *
gid_uid_add(int gid,int uid,char *deskey)
{
	struct gid_hash_entry *htmp;
	struct hash_entry *uhtmp;
	
	htmp=gid_hash_lookup(gid);
	if (htmp==NULL)
		return NULL;
		
	uhtmp=gid_uid_lookup(htmp->key,uid);
	if (uhtmp!=NULL){
		if(!memcmp(deskey,uhtmp->deskey,(size_t)KEYSIZE+KEYSIZE/8))
			uhtmp->count++;	
		return NULL;
	}
	down(&tcfs_hash_sem_gid);
	uhtmp=(struct hash_entry *)kmalloc(sizeof(struct hash_entry),GFP_KERNEL);
	if (uhtmp==NULL) {
		printk("TCFS: Unable to get a free page\n");
		up(&tcfs_hash_sem_gid);
	}
	uhtmp->uid=uid;
	uhtmp->count=1;
	uhtmp->permanent=0;
	memcpy(uhtmp->deskey,deskey,KEYSIZE+KEYSIZE/8);
	uhtmp->ks=NULL;
	uhtmp->next=htmp->key;
	htmp->key=uhtmp;
	htmp->ins++;
	up(&tcfs_hash_sem_gid);
	return htmp;
}

void 
gid_hash_rem(int gid)
{
	struct gid_hash_entry * htmp1, * htmp2;
	struct hash_entry *uhtmp, *uhtmp1;
#ifdef HASH_DEBUG
	printk("TCFS: hash_rem\n");
#endif
	down(&tcfs_hash_sem_gid);
	htmp2=NULL;
	htmp1=gid_hash_table[hashpos(gid)];
	while(htmp1!=NULL && htmp1->gid!=gid) {
		htmp2=htmp1;
		htmp1=htmp1->next;
	}
	if (htmp1==NULL) { /* Element not present in hash_table */
		up(&tcfs_hash_sem_gid);
		return;
	}
	if (htmp2==NULL) {/* The element is the first */
		gid_hash_table[hashpos(gid)]=htmp1->next;
		if (htmp1->ins>0) {
			printk("TCFS: Cant' remove group %d, user keys not empty",gid);
			up(&tcfs_hash_sem_gid);
			return;
		}		
/*		uhtmp=htmp1->key;
		while (uhtmp!=NULL) {
			uhtmp1=uhtmp->next;
			kfree(uhtmp);
			uhtmp=uhtmp1;
		}*/
		if (htmp1->ks!=NULL)
			kfree(htmp1->ks);
		kfree(htmp1);
#ifdef HASH_DEBUG
	hash_debug();
#endif
		up(&tcfs_hash_sem_gid);
		return;
	}
	if (htmp1->ins>0) {
		printk("TCFS: Cant' remove group %d, user keys not empty",gid);
		up(&tcfs_hash_sem_gid);
		return;
	}
	htmp2->next=htmp1->next;
	uhtmp=htmp1->key;
	while (uhtmp!=NULL) {
		uhtmp1=uhtmp->next;
		kfree(uhtmp);
		uhtmp=uhtmp1;
	}
	if (htmp1->ks!=NULL)
		kfree(htmp1->ks);
	kfree(htmp1);
#ifdef HASH_DEBUG
	hash_debug();
#endif
	up(&tcfs_hash_sem_gid);
}

void 
gid_uid_rem_count(int gid,int uid)
{
	struct hash_entry *uhtmp,*uhtmp1;
	struct gid_hash_entry *htmp;

	htmp=gid_hash_lookup(gid);
	if (htmp==NULL) 
		return;

	down(&tcfs_hash_sem_gid);

	uhtmp1=NULL;
	uhtmp=htmp->key;
	while(uhtmp!=NULL) {
		if (uhtmp->uid==uid) {
			if (uhtmp1==NULL) {
				if (uhtmp->count>0) 
					uhtmp->count--;
				if (uhtmp->permanent==0 && uhtmp->count==0) {
					htmp->key=uhtmp->next;
					kfree(uhtmp);
					uhtmp=NULL;
					htmp->ins--;
				}
			} else {
				if (uhtmp->count>0) 
					uhtmp->count--;
				if (uhtmp->permanent==0 && uhtmp->count==0) {
					uhtmp1->next=uhtmp->next;
					kfree(uhtmp);
					uhtmp=NULL;
					htmp->ins--;
				}
			}
		} else {
			uhtmp1=uhtmp;
			uhtmp=uhtmp->next;
		}
	}	
	up(&tcfs_hash_sem_gid);
}

void 
gid_uid_rem_all(int gid,int uid)
{
	struct hash_entry *uhtmp,*uhtmp1;
	struct gid_hash_entry *htmp;

	htmp=gid_hash_lookup(gid);
	if (htmp==NULL) 
		return;

	down(&tcfs_hash_sem_gid);

	uhtmp1=NULL;
	uhtmp=htmp->key;
	/* Lookup for UID struct */
	while (uhtmp!=NULL && uhtmp->uid != uid) {
		uhtmp1=uhtmp;
		uhtmp=uhtmp->next;
	}
	
	if (uhtmp==NULL) {
		up(&tcfs_hash_sem_gid);
		return;
	}
	
	uhtmp->count=0;
	
	if (uhtmp->permanent==0) {
		if (uhtmp1==NULL) {
			htmp->key=uhtmp->next;
			kfree(uhtmp);
			htmp->ins --;
		} else {
			uhtmp1->next=uhtmp->next;
			kfree(uhtmp);
			htmp->ins --;
		}
	}
#ifdef HASH_DEBUG
	hash_debug();
#endif
	up(&tcfs_hash_sem_gid);
}

void 
gid_uid_rem_permanent(int gid,int uid)
{
	struct hash_entry *uhtmp,*uhtmp1;
	struct gid_hash_entry *htmp;

	htmp=gid_hash_lookup(gid);
	if (htmp==NULL) 
		return;

	down(&tcfs_hash_sem_gid);

	uhtmp1=NULL;
	uhtmp=htmp->key;
	/* Lookup for UID struct */
	while (uhtmp!=NULL && uhtmp->uid != uid) {
		uhtmp1=uhtmp;
		uhtmp=uhtmp->next;
	}

	if (uhtmp==NULL) {
		up(&tcfs_hash_sem_gid);
		return;
	}
	
	uhtmp->permanent=0;
	
	if (uhtmp->count==0) {
		if (uhtmp1==NULL) {
			htmp->key=uhtmp->next;
			kfree(uhtmp);
			htmp->ins --;
		} else {
			uhtmp1->next=uhtmp->next;
			kfree(uhtmp);
			htmp->ins --;
		}
	}
#ifdef HASH_DEBUG
	hash_debug();
#endif
	up(&tcfs_hash_sem_gid);
}

#ifdef HASH_DEBUG
void 
hash_debug(void)
{
	int i;
	struct hash_entry * htmp;
	for (i=0;i<HASH_SIZE;i++) {
		printk("%d:",i);
		htmp=hash_table[i];
		while (htmp!=NULL) {
			printk("%d(%d)-->",htmp->uid,htmp->count);
			htmp=htmp->next;
		}
		printk("\n");
	}
}
#endif
