/*____________________________________________________________________________
	Copyright (C) 1994-1998 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: cast5.cpp,v 1.10.8.1 1998/11/12 03:05:48 heller Exp $
____________________________________________________________________________*/

#include <assert.h>
#include <string.h>
#include "cast5.h"
#include "cast5box.c"	/* S-box contents are defined in this file */

#if PGP_MACINTOSH
	#if TARGET_CPU_PPC
		#define MAC_68K 			0
	#else
		#define MAC_68K 			1

		#if A4_GLOBALS
		#define MAC68K_ALIGN_HACK	0	// change, depending on alignment in
										// code resource
		#else
		#define MAC68K_ALIGN_HACK	1	// change, depending on alignment in
										// app
		#endif
		
		#if MAC68K_ALIGN_HACK
			static short  ___AlignHack[ 1 ] = {0xABCD};
		#endif
	#endif
#endif

/* CAST is uniformly big-endian - byte access macros */
#if MAC_68K	/* [ */

// casting to uchar will do results in better code
#define B0(x) ((uchar)((x) >> 24))
#define B1(x) ((uchar)((x) >> 16))
#define B2(x) ((uchar)((x) >> 8 ))
#define B3(x) ((uchar)((x)))

#else	/* ][ */

#define B0(x) ((x) >> 24 & 255)
#define B1(x) ((x) >> 16 & 255)
#define B2(x) ((x) >> 8 & 255)
#define B3(x) ((x) & 255)

#endif	/* ] */


/* Some macros used in the key scheduling code */
#define x0 B0(x0123)
#define x1 B1(x0123)
#define x2 B2(x0123)
#define x3 B3(x0123)
#define x4 B0(x4567)
#define x5 B1(x4567)
#define x6 B2(x4567)
#define x7 B3(x4567)
#define x8 B0(x89AB)
#define x9 B1(x89AB)
#define xA B2(x89AB)
#define xB B3(x89AB)
#define xC B0(xCDEF)
#define xD B1(xCDEF)
#define xE B2(xCDEF)
#define xF B3(xCDEF)
#define z0 B0(z0123)
#define z1 B1(z0123)
#define z2 B2(z0123)
#define z3 B3(z0123)
#define z4 B0(z4567)
#define z5 B1(z4567)
#define z6 B2(z4567)
#define z7 B3(z4567)
#define z8 B0(z89AB)
#define z9 B1(z89AB)
#define zA B2(z89AB)
#define zB B3(z89AB)
#define zC B0(zCDEF)
#define zD B1(zCDEF)
#define zE B2(zCDEF)
#define zF B3(zCDEF)

/*
 * This expands a 128-bit key to a 32-word scheduled key, where each round
 * uses two words: a 32-bit XOR mask and a 5-bit rotate amount.  Shorter keys
 * are just padded with zeros, and if they are 80 bits or less, the cipher
 * is reduced to 12 rounds (not implemented here).
 *
 * The feed-forward used with x0123 through yCDEF prevent any weak keys,
 * and the substitution to set up the xkey tables ensure that the subround
 * keys are not easily derivable from each other, so linear cryptanalysis
 * won't do very well against CAST.
 */
void
CAST5schedule( word32 xkey[32], byte const *k)
{
	word32 x0123, x4567, x89AB, xCDEF;
	word32 z0123, z4567, z89AB, zCDEF;
	int i;

	/* Initialize x0123456789ABCDEF with input key */
#if __POWERPC__ || __MC68K__
	x0123 = ((word32 *)k)[0];	/* Big-endian and unaligned access okay */
	x4567 = ((word32 *)k)[1];
	x89AB = ((word32 *)k)[2];
	xCDEF = ((word32 *)k)[3];
#else
	x0123= (word32)k[ 0]<<24 | (word32)k[ 1]<<16 | (word32)k[ 2]<<8 | k[ 3];
	x4567= (word32)k[ 4]<<24 | (word32)k[ 5]<<16 | (word32)k[ 6]<<8 | k[ 7];
	x89AB= (word32)k[ 8]<<24 | (word32)k[ 9]<<16 | (word32)k[10]<<8 | k[11];
	xCDEF= (word32)k[12]<<24 | (word32)k[13]<<16 | (word32)k[14]<<8 | k[15];
#endif
	/* Now set up the key table */

	for (i = 0; i < 4; i++) {
		z0123 = x0123 ^ S5[xD] ^ S6[xF] ^ S7[xC] ^ S8[xE] ^ S7[x8];
		z4567 = x89AB ^ S5[z0] ^ S6[z2] ^ S7[z1] ^ S8[z3] ^ S8[xA];
		z89AB = xCDEF ^ S5[z7] ^ S6[z6] ^ S7[z5] ^ S8[z4] ^ S5[x9];
		zCDEF = x4567 ^ S5[zA] ^ S6[z9] ^ S7[zB] ^ S8[z8] ^ S6[xB];

		x0123 = z89AB ^ S5[z5] ^ S6[z7] ^ S7[z4] ^ S8[z6] ^ S7[z0];
		x4567 = z0123 ^ S5[x0] ^ S6[x2] ^ S7[x1] ^ S8[x3] ^ S8[z2];
		x89AB = z4567 ^ S5[x7] ^ S6[x6] ^ S7[x5] ^ S8[x4] ^ S5[z1];
		xCDEF = zCDEF ^ S5[xA] ^ S6[x9] ^ S7[xB] ^ S8[x8] ^ S6[z3];

		switch (i) {
		  case 0:
			/* Masking keys, rounds 0..7 */
			xkey[ 0] = S5[z8] ^ S6[z9] ^ S7[z7] ^ S8[z6] ^ S5[z2];
			xkey[ 2] = S5[zA] ^ S6[zB] ^ S7[z5] ^ S8[z4] ^ S6[z6];
			xkey[ 4] = S5[zC] ^ S6[zD] ^ S7[z3] ^ S8[z2] ^ S7[z9];
			xkey[ 6] = S5[zE] ^ S6[zF] ^ S7[z1] ^ S8[z0] ^ S8[zC];
			
			xkey[ 8] = S5[x3] ^ S6[x2] ^ S7[xC] ^ S8[xD] ^ S5[x8];
			xkey[10] = S5[x1] ^ S6[x0] ^ S7[xE] ^ S8[xF] ^ S6[xD];
			xkey[12] = S5[x7] ^ S6[x6] ^ S7[x8] ^ S8[x9] ^ S7[x3];
			xkey[14] = S5[x5] ^ S6[x4] ^ S7[xA] ^ S8[xB] ^ S8[x7];
			break;
		  case 1:
			/* Masking keys, rounds 8..15 */
			xkey[16] = S5[z3] ^ S6[z2] ^ S7[zC] ^ S8[zD] ^ S5[z9];
			xkey[18] = S5[z1] ^ S6[z0] ^ S7[zE] ^ S8[zF] ^ S6[zC];
			xkey[20] = S5[z7] ^ S6[z6] ^ S7[z8] ^ S8[z9] ^ S7[z2];
			xkey[22] = S5[z5] ^ S6[z4] ^ S7[zA] ^ S8[zB] ^ S8[z6];
			
			xkey[24] = S5[x8] ^ S6[x9] ^ S7[x7] ^ S8[x6] ^ S5[x3];
			xkey[26] = S5[xA] ^ S6[xB] ^ S7[x5] ^ S8[x4] ^ S6[x7];
			xkey[28] = S5[xC] ^ S6[xD] ^ S7[x3] ^ S8[x2] ^ S7[x8];
			xkey[30] = S5[xE] ^ S6[xF] ^ S7[x1] ^ S8[x0] ^ S8[xD];
			break;
		  case 2:
			/* Rotation keys, rounds 0..7 */
			xkey[ 1] = (S5[z8]^S6[z9]^S7[z7]^S8[z6]^S5[z2]) & 31;
			xkey[ 3] = (S5[zA]^S6[zB]^S7[z5]^S8[z4]^S6[z6]) & 31;
			xkey[ 5] = (S5[zC]^S6[zD]^S7[z3]^S8[z2]^S7[z9]) & 31;
			xkey[ 7] = (S5[zE]^S6[zF]^S7[z1]^S8[z0]^S8[zC]) & 31;
			
			xkey[ 9] = (S5[x3]^S6[x2]^S7[xC]^S8[xD]^S5[x8]) & 31;
			xkey[11] = (S5[x1]^S6[x0]^S7[xE]^S8[xF]^S6[xD]) & 31;
			xkey[13] = (S5[x7]^S6[x6]^S7[x8]^S8[x9]^S7[x3]) & 31;
			xkey[15] = (S5[x5]^S6[x4]^S7[xA]^S8[xB]^S8[x7]) & 31;
			break;
		  case 3:
			/* Rotation keys, rounds 8..15 */
			xkey[17] = (S5[z3]^S6[z2]^S7[zC]^S8[zD]^S5[z9]) & 31;
			xkey[19] = (S5[z1]^S6[z0]^S7[zE]^S8[zF]^S6[zC]) & 31;
			xkey[21] = (S5[z7]^S6[z6]^S7[z8]^S8[z9]^S7[z2]) & 31;
			xkey[23] = (S5[z5]^S6[z4]^S7[zA]^S8[zB]^S8[z6]) & 31;
		
			xkey[25] = (S5[x8]^S6[x9]^S7[x7]^S8[x6]^S5[x3]) & 31;
			xkey[27] = (S5[xA]^S6[xB]^S7[x5]^S8[x4]^S6[x7]) & 31;
			xkey[29] = (S5[xC]^S6[xD]^S7[x3]^S8[x2]^S7[x8]) & 31;
			xkey[31] = (S5[xE]^S6[xF]^S7[x1]^S8[x0]^S8[xD]) & 31;
			break;
		}
	}
	x0123 = x4567 = x89AB = xCDEF = 0;
	z0123 = z4567 = z89AB = zCDEF = 0;
}

#undef x0
#undef x1
#undef x2
#undef x3
#undef x4
#undef x5
#undef x6
#undef x7
#undef x8
#undef x9
#undef xA
#undef xB
#undef xC
#undef xD
#undef xE
#undef xF
#undef z0
#undef z1
#undef z2
#undef z3
#undef z4
#undef z5
#undef z6
#undef z7
#undef z8
#undef z9
#undef zA
#undef zB
#undef zC
#undef zD
#undef zE
#undef zF

/* Some macros used in the encryption/decryption code */
#define ROL(x,r) ((x)<<(r) | (x)>>(32-(r)))


#if MAC_68K	/* [ */

#pragma parameter ROLMac(__D0, __D1)
extern pascal UInt32 ROLMac(ulong x, ulong r) = 0xE3B8;

#undef ROL
#define ROL(x, r)	ROLMac( x, r )
	
#endif	/* ] MAC_68K */

#ifdef __GNUC__
#if __i386__
/* Redefine using GCC inline assembler */
#undef ROL
#define ROL(x,r) ({unsigned _y; \
		__asm__("rol %%cl,%0" : "=g" (_y) : "0" (x), "c" (r)); _y;})
#endif /* __i386__ */
#endif /* __GNUC__ */

#define F1(x,xkey,i) (ROL((xkey)[2*(i)] + (x), (xkey)[2*(i)+1]))
#define F2(x,xkey,i) (ROL((xkey)[2*(i)] ^ (x), (xkey)[2*(i)+1]))
#define F3(x,xkey,i) (ROL((xkey)[2*(i)] - (x), (xkey)[2*(i)+1]))


#if MAC_68K

// small 's' uses local register variable inited to point at global
#define G1(x) (((s1[B0(x)] ^ s2[B1(x)]) - s3[B2(x)]) + s4[B3(x)])
#define G2(x) (((s1[B0(x)] - s2[B1(x)]) + s3[B2(x)]) ^ s4[B3(x)])
#define G3(x) (((s1[B0(x)] + s2[B1(x)]) ^ s3[B2(x)]) - s4[B3(x)])

#else

#define G1(x) (((S1[B0(x)] ^ S2[B1(x)]) - S3[B2(x)]) + S4[B3(x)])
#define G2(x) (((S1[B0(x)] - S2[B1(x)]) + S3[B2(x)]) ^ S4[B3(x)])
#define G3(x) (((S1[B0(x)] + S2[B1(x)]) ^ S3[B2(x)]) - S4[B3(x)])

#endif


#if __POWERPC__ // {
/*
 * PowerPC assembly version
 * This is 604-tuned. For our purposes, the 604 can dispatch
 * and execute two ALU and one load/store instruction per cycle.
 * Loads from cache are available cycle after next.
 *
 * Let's figure out the critical path through a round.
 * - Add/subf/xor in key word
 * - Rotate
 * - Extract fields
 * - Load B0
 * - Load B1
 * - (stall)
 * - Merge B0 & B1
 * - Merge in B2
 * - Merge in B3
 * - XOR with other half
 *
 * Thus, a round should be doable in 10 cycles.  We have 16
 * instructions (6 loads and 10 ALU ops), so that should be
 * easy.  Perhaps it's worth interleaving and doing two at once?
 * That would give us 30 ops, 10 load/store and 20 ALU, which
 * might allow an optimal schedule.
 *
 * Okay... the only tricky part is the load & merge...
 * Merge with key
 * Rotate
 * Extract 1
 * Load 1 / Extract 2
 * Load 2 / Extract 3 / L1
 * Load 3 / Extract 4 / L2
 * Load 4 / L3 / Merge 1 and 2
 * L4 / Merge 3
 * Merge 4
 * XOR with other half
 */
// 4567890123456789012345678901234567890123456789012345678901234567890
// Register conventions:
// r0 - volatile
// r1 - stack pointer, preserved
// r2 - TOC pointer, preserved
// r3 - First argument (and return value)
// r4-r10 - More arguments, volatile
// r11-r12 - volatile
// r13-r31 - preserved
// LR - return address, volatile
// CTR, XER and MQ are all volatile
//
// The calling stack has some slots reserved for us to use, but
// the interesting part is that there are a few hundred bytes
// *below* the current stack pointer (the so-called "red zone")
// that interrupt handlers won't use and we can use to save out
// registers.

#define t1	r3
#define left	r6
#define right	r7
#define t2	r8
#define s1	r9
#define s2	r10
#define s3	r11
#define s4	r12
#define t3	r13	// Needs to be saved somewhere
#define t4	r14	// Needs to be saved somewhere

// The CAST round.  op1/op2/op3 are add/xor/subf in rotating order.
// These do round n from x into y.
#define ROUND(x,y,op1,op2,op3,n) \
	op3		r0,x,r0			;\
	rlwnm	r0,r0,t1,0,31	;\
	rlwinm	t1,r0,10,22,29	;\
	lwzx	t2,s1,t1		;\
	rlwinm	t1,r0,18,22,29	;\
	lwzx	t3,s2,t1		;\
	rlwinm	t1,r0,26,22,29	;\
	lwzx	t4,s3,t1		;\
	rlwinm	t1,r0,2,22,29	;\
	op1		t2,t3,t2		;\
	lwzx	t3,s4,t1		;\
	op2		t2,t4,t2		;\
	lwz		r0,8*n+8(xkey)	;\
	op3		t2,t3,t2		;\
	lwz		t1,8*n+12(xkey)	;\
	xor		y,t2,y
	
asm void
CAST5encrypt(
	register byte const 	*in,
	register byte 			*out,
	register word32 const 	*xkey)
{
	lwz		left,0(in)	// PPC is big-endian, as is CAST,
	lwz		right,4(in) // and unaligned loads are okay
	stw		t3,-8(SP)
	stw		t4,-4(SP)
	lwz		s1,S1(RTOC)
	lwz		s2,S2(RTOC)
	lwz		s3,S3(RTOC)
	lwz		s4,S4(RTOC)

	lwz		r0,0(xkey)
	lwz		t1,4(xkey)

	add		r0,right,r0
	rlwnm	r0,r0,t1,0,31
	rlwinm	t1,r0,10,22,29
	lwzx	t2,s1,t1
	rlwinm	t1,r0,18,22,29
	lwzx	t3,s2,t1
	rlwinm	t1,r0,26,22,29
	lwzx	t4,s3,t1
	rlwinm	t1,r0,2,22,29
	xor		t2,t3,t2
	lwzx	t3,s4,t1
	subf	t2,t4,t2
	lwz		r0,8*0+8(xkey)
	add		t2,t3,t2
	lwz		t1,8*0+12(xkey)
	xor		left,t2,left
	
//	ROUND(right, left, xor, subf, add, 0)
	ROUND(left, right, subf, add, xor, 1)
	ROUND(right, left, add, xor, subf, 2)
	ROUND(left, right, xor, subf, add, 3)
	ROUND(right, left, subf, add, xor, 4)
	ROUND(left, right, add, xor, subf, 5)
	ROUND(right, left, xor, subf, add, 6)
	ROUND(left, right, subf, add, xor, 7)
	ROUND(right, left, add, xor, subf, 8)
	ROUND(left, right, xor, subf, add, 9)
	ROUND(right, left, subf, add, xor, 10)
	ROUND(left, right, add, xor, subf, 11)
	ROUND(right, left, xor, subf, add, 12)
	ROUND(left, right, subf, add, xor, 13)
	ROUND(right, left, add, xor, subf, 14)
//	ROUND(left, right, xor, subf, add, 15)

// Last round: omit loading of keys for next round
	add		r0,left,r0
	stw		left,4(out)
	rlwnm	r0,r0,t1,0,31
	lwz		t4,-4(SP)
	rlwinm	t1,r0,10,22,29
	lwzx	t2,s1,t1
	rlwinm	t1,r0,18,22,29
	lwzx	t3,s2,t1
	rlwinm	t1,r0,26,22,29
	lwzx	left,s3,t1
	rlwinm	t1,r0,2,22,29
	xor		t2,t3,t2
	lwzx	t3,s4,t1
	subf	t2,left,t2
	add		t2,t3,t2
	lwz		t3,-8(SP)
	xor		right,t2,right
	
	stw		right,0(out)
	blr
}


// Get rid of those #defines...
#undef ROUND
#undef left
#undef right
#undef s1
#undef s2
#undef s3
#undef s4
#undef t1
#undef t2
#undef t3
#undef t4

#if !CAST5_INTERLEAVED_ENCRYPT // {
/*********** Version for assembly calling ************/
/*
 * DO NOT CALL THIS FROM C!
 * Its calling convention is incompatible!
 * DO NOT CALL THIS FROM C!
 *
 * The arguments are declared for ease of reference.
 * Input comes in from "lin" and "right".  The encrypted result is
 * returned in "left" and "rout".  The other registers are for
 * necessary pointers and ARE NOT MODIFIED, so the caller doesn't
 * have to set them up again.  src and dest are not used at all.
 *
 * Most rounds operate on "left" and "right".  Input is in r4 and r5, while
 * output is in r8 and r9 (left and right are swapped, as the
 * algorithm requires).  The first and last rounds are slightly
 * different to accomodate this.
 */
// The CAST round.  op1/op2/op3 are add/xor/subf in rotating order.
// These do round n from x into y
#define ROUND(x,y,op1,op2,op3,n) \
	op3		r0,x,t1			;\
	rlwnm	r0,r0,t2,0,31	;\
	rlwinm	t1,r0,10,22,29	;\
	lwzx	t1,s1,t1		;\
	rlwinm	t2,r0,18,22,29	;\
	lwzx	t2,s2,t2		;\
	rlwinm	t3,r0,26,22,29	;\
	lwzx	t3,s3,t3		;\
	rlwinm	r0,r0,2,22,29	;\
	op1		t2,t2,t1		;\
	lwzx	r0,s4,r0		;\
	op2		t3,t3,t2		;\
	lwz		t1,8*n+8(xkey)	;\
	op3		r0,r0,t3		;\
	lwz		t2,8*n+12(xkey)	;\
	xor		y,y,r0

#define s1 r10
#define s2 r11
#define s3 r12
#define s4 r13

asm static void
CAST5asm(
	register word32 const *xkey,					// r3
	register word32 lin, register word32 right,		// r4 and r5
	register word32 const *src,						// r6 (unused)
	register word32 *dest,							// r7 (unused)
	register word32 rout, register word32 left,		// r8 and r9
	         word32 const *s1,						// r10
	         word32 const *s2,						// r11
	         word32 const *s3,						// r12
	         word32 const *s4)						// r13
{
#define t1 r14
#define t2 rout
#define t3 left

// First round: fetch of first key is not overlapped, use "lin".
	lwz		t1,0(xkey)
	lwz		t2,4(xkey)

	add		r0,right,t1
	rlwnm	r0,r0,t2,0,31
	rlwinm	t1,r0,10,22,29
	lwzx	t1,s1,t1
	rlwinm	t2,r0,18,22,29
	lwzx	t2,s2,t2
	rlwinm	t3,r0,26,22,29
	lwzx	t3,s3,t3
	rlwinm	r0,r0,2,22,29
	xor		t2,t2,t1
	lwzx	r0,s4,r0
	subf	t3,t3,t2
	lwz		t1,8(xkey)
	add		r0,r0,t3
	lwz		t2,12(xkey)
	xor		left,r0,lin	// A little dance here...

#undef t3
#define t3 lin

//	ROUND(right, left, xor, subf, add, 0)
	ROUND(left, right, subf, add, xor, 1)
	ROUND(right, left, add, xor, subf, 2)
	ROUND(left, right, xor, subf, add, 3)
	ROUND(right, left, subf, add, xor, 4)
	ROUND(left, right, add, xor, subf, 5)
	ROUND(right, left, xor, subf, add, 6)
	ROUND(left, right, subf, add, xor, 7)
	ROUND(right, left, add, xor, subf, 8)
	ROUND(left, right, xor, subf, add, 9)
	ROUND(right, left, subf, add, xor, 10)
	ROUND(left, right, add, xor, subf, 11)
	ROUND(right, left, xor, subf, add, 12)
	ROUND(left, right, subf, add, xor, 13)
	ROUND(right, left, add, xor, subf, 14)
//	ROUND(left, right, xor, subf, add, 15)

// Last round: do not fetch next round's keys, leave result in "rout"
	add		r0,left,t1
	rlwnm	r0,r0,t2,0,31
	rlwinm	t1,r0,10,22,29
	lwzx	t1,s1,t1
	rlwinm	t2,r0,18,22,29
	lwzx	t2,s2,t2
	rlwinm	t3,r0,26,22,29
	lwzx	t3,s3,t3
	rlwinm	r0,r0,2,22,29
	xor		t2,t2,t1
	lwzx	r0,s4,r0
	subf	t3,t3,t2
	add		r0,r0,t3
	xor		rout,r0,right	
	blr
}
#undef t1
#undef t2
#undef t3
#undef t4
#undef ROUND

#define out0 r8
#define out1 r9

asm void
CAST5encryptCFB(
	register word32 const *xkey,	// r3
	register word32 iv0,			// r4
	register word32 iv1,			// r5
	register word32 const *src,		// r6
	register word32 *dest,			// r7
	register word32 len)			// r8
{
	mflr	r0
	stw		r13,-16(SP)
	mtctr	len
	stw		r0,-12(SP)
	stw		r14,-8(SP)
	
	lwz		s1,S1(RTOC)
	lwz		s2,S2(RTOC)
	lwz		s3,S3(RTOC)
	lwz		s4,S4(RTOC)

@loop:

	bl		CAST5asm

	lwz		iv0,0(src);		addi	dest,dest,8
	lwz		iv1,4(src);		xor		iv0,out0,iv0
	stw		iv0,-8(dest);	xor		iv1,out1,iv1
	stw		iv1,-4(dest);	addi	src,src,8

	bdnz	@loop		
	
	lwz		r13,-16(SP)
	lwz		r0,-12(SP)
	lwz		r14,-8(SP)
	mtlr	r0
	blr	
}

#if !DOUBLE_DECRYPT // {
asm void
CAST5decryptCFB(
	register word32 const *xkey,	// r3
	register word32 iv0,			// r4
	register word32 iv1,			// r5
	register word32 const *src,		// r6
	register word32 *dest,			// r7
	register word32 len)			// r8
{
	mflr	r0
	stw		r13,-16(SP)
	mtctr	len
	stw		r0,-12(SP)
	stw		r14,-8(SP)
	stw		r15,-4(SP)
	
	lwz		s1,S1(RTOC)
	lwz		s2,S2(RTOC)
	lwz		s3,S3(RTOC)
	lwz		s4,S4(RTOC)

@loop:
	bl		CAST5asm

	lwz		iv0,0(src);		addi	dest,dest,8
	lwz		iv1,4(src);		xor		out0,out0,iv0
	stw		out0,-8(dest);	xor		out1,out1,iv1
	stw		out1,-4(dest);	addi	src,src,8

	bdnz	@loop		
	
	lwz		r13,-16(SP)
	lwz		r0,-12(SP)
	lwz		r14,-8(SP)
	mtlr	r0
	lwz		r15,-4(SP)
	blr	
}
#endif // } !DOUBLE_DECRYPT

#undef out0
#undef out1

#undef s1
#undef s2
#undef s3
#undef s4


#if DOUBLE_DECRYPT // {
/*******************************************************************

			The Tricky Part
			
	This interleaves two CAST cipher block operations to make
	optimal use of the PPC 604's pipeline.  (Other PPC models
	are also sped up, to a lesser degree.)

 *******************************************************************/
/*
 * Here is how it's pipelined.  Repeat the middle block as often as necessary.
 * Each line in the overlapping part has 2 simple ALU operations and one load.
 * load resuts are accessed 2 cycles later.  (Key fetches are done earlier.)
 * Code on a PowerPC 604 just doesn't run any faster than this.
 *
 * Key a
 * Rotate a
 * Extract 1a
 * Load 1a / Extract 2a	
 * Load 2a / Extract 3a
 * Load 3a / Extract 4a
 * Merge 1/2a / Load 4a		Key b
 * Merge 2/3a				Rotate b				Load key c/d
 * Merge 3/4a				Extract 1b				Load rotate c/d
 * XOR a					Load 1b / Extract 2b
 * Key c					Load 2b / Extract 3b
 * Rotate c					Load 3b / Extract 4b
 * Extract 1c				Merge 1/2b / Load 4b
 * Load 1c / Extract 2c		Merge 2/3b
 * Load 2c / Extract 3c		Merge 3/4b
 * Load 3c / Extract 4c		XOR b
 * Merge 1/2c / Load 4c		Key d
 * Merge 2/3c				Rotate d				Load key e/f
 * Merge 3/4c				Extract 1d				Load rotate e/f
 * XOR c					Load 1d / Extract 2d
 * 							Load 2d / Extract 3d
 * 							Load 3d / Extract 4d
 * 							Merge 1/2d / Load 4d
 * 							Merge 2/3d
 * 							Merge 3/4d
 * 							XOR d
 */
// The CAST round.  op1/op2/op3 are add/xor/subf in rotating order.
// These do the "a" half round n from x1 into y1 and from x2 into y2,
// and the corresponding parts of the "b" half: part of round n-1 and
// part of round n.
// Note: Current op1 is previous round's op2, and so on.
//  B half				B half		A half			A half
#define ROUND(x,y1,y2,b1,b2,op1,op2,op3,n) \
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;		op3    ta,x,key;		\
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		rlwnm  ta,ta,rot,0,31;	\
	op3    t2,t2,t1;	lwzx   tb,s4,tb;			rlwinm t1,ta,10,22,29;	\
	op1    t3,t3,t2;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;	\
	op2    tb,tb,t3;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;	\
	xor    b2,b1,tb;				lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;	\
	op3    tb,b2,key;				op1  t2,t2,t1;	lwzx   ta,s4,ta;		\
	rlwnm  tb,tb,rot,0,31;			op2  t3,t3,t2;	lwz key,8*n+8(xkey);	\
	rlwinm t1,tb,10,22,29;			op3  ta,ta,t3;	lwz rot,8*n+12(xkey);	\
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		xor	   y2,y1,ta;		\

#define out0 r10
#define out1 r11
#define out2 r12
#define out3 r13
#define s1	r14
#define s2	r15
#define s3	r16
#define s4	r17

#define ta	r18
#define tb	r19
#define	t1	r20
#define	t2	r21
#define	t3	r22

#define	key	r23
#define	rot	r24

asm void
CAST5decryptCFB(
	register word32 const *xkey,	// r3
	register word32 in0,			// r4
	register word32 in1,			// r5
	register word32 const *src,		// r6
	register word32 *dest,			// r7
	register word32 len)			// r8
#define in2	r8
#define in3	r9
{
	rlwinm	len,len,31,1,31		// Divide by 2
	stw		r13,-48(sp)
	stw		r14,-44(sp)
	stw		r15,-40(sp)
	stw		r16,-36(sp)
	stw		r17,-32(sp)
	stw		r18,-28(sp)
	stw		r19,-24(sp)
	stw		r20,-20(sp)
	stw		r21,-16(sp)
	stw		r22,-12(sp)
	stw		r23,-8(sp)
	stw		r24,-4(sp)

	mtctr	len				// Repeat len/2 times....

	lwz		in2,0(src)
	lwz		in3,4(src)
	
	lwz		key,0(xkey)
	lwz		rot,4(xkey)

	lwz		s1,S1(RTOC)
// The core CAST function
	lwz		s2,S2(RTOC);							add    ta,in1,key;
	lwz		s3,S3(RTOC);							rlwnm  ta,ta,rot,0,31;
	lwz		s4,S4(RTOC);							rlwinm t1,ta,10,22,29;
									lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
									lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
	b		@entry
@loop:
	subf   t3,t3,t2;	 xor out3,out3,in1;			lwz key,0(xkey);
	add    tb,tb,t3;		/**/					lwz rot,4(xkey);
	xor    out2,out2,tb;	/**/					 lwz in2,16(src);
	 xor out2,out2,in0;		/**/					 lwz in3,20(src);
	 stw out0,0(dest);		/**/					/**/
	 stw out1,4(dest);		/**/					add    ta,in1,key;
	 stw out2,8(dest);		/**/					rlwnm  ta,ta,rot,0,31;
	 stw out3,12(dest);		/**/					rlwinm t1,ta,10,22,29;
	 addi src,src,16;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
	 addi dest,dest,16;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
@entry:
									lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;
	add    tb,in3,key;				xor  t2,t2,t1;	lwzx   ta,s4,ta;
	rlwnm  tb,tb,rot,0,31;			subf t3,t3,t2;	lwz key,8(xkey);
	rlwinm t1,tb,10,22,29;			add  ta,ta,t3;	lwz rot,12(xkey);
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		xor    out1,in0,ta;

//	ROUND(in1,  in0,  out1, XXX,  in3,  xor, subf, add, 0)
	ROUND(out1, in1,  out0, in2,  out3, subf, add, xor, 1)
	ROUND(out0, out1, out1, in3,  out2, add, xor, subf, 2)
	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 3)
	ROUND(out0, out1, out1, out2, out2, subf, add, xor, 4)
	ROUND(out1, out0, out0, out3, out3, add, xor, subf, 5)
	ROUND(out0, out1, out1, out2, out2, xor, subf, add, 6)
	ROUND(out1, out0, out0, out3, out3, subf, add, xor, 7)
	ROUND(out0, out1, out1, out2, out2, add, xor, subf, 8)
	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 9)
	ROUND(out0, out1, out1, out2, out2, subf, add, xor, 10)
	ROUND(out1, out0, out0, out3, out3, add, xor, subf, 11)
	ROUND(out0, out1, out1, out2, out2, xor, subf, add, 12)
	ROUND(out1, out0, out0, out3, out3, subf, add, xor, 13)
	ROUND(out0, out1, out1, out2, out2, add, xor, subf, 14)
//	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 15)
// Final wrapup.  Steps that are part of the CFB (as opposed to CAST)
// are indented one space.  /**/ marks an instruction slot.
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;		add    ta,out1,key;
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		rlwnm  ta,ta,rot,0,31;
	add    t2,t2,t1;	lwzx   tb,s4,tb;			rlwinm t1,ta,10,22,29;
	xor    t3,t3,t2;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
	subf   tb,tb,t3;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
	xor    out3,out3,tb; 			lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;
	add    tb,out3,key;			 	xor  t2,t2,t1;	lwzx	ta,s4,ta;
	rlwnm  tb,tb,rot,0,31;			subf t3,t3,t2;	 lwz in0,8(src);
	rlwinm t1,tb,10,22,29;			add  ta,ta,t3;	 lwz in1,12(src);
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		 xor	out0,out0,ta;
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;	 	 xor out0,out0,in2;
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		 xor out1,out1,in3;
	xor    t2,t2,t1;	lwzx   tb,s4,tb;		 	/**/

	bdnz	@loop

	subf   t3,t3,t2;	 xor out3,out3,in1;		 lwz in2,16(src)
	add    tb,tb,t3;							 lwz in3,20(src)
	xor    out2,out2,tb;						 stw out0,0(dest)
	 xor out2,out2,in0;							 stw out1,4(dest)
												 stw out2,8(dest)
												 stw out3,12(dest)

// Done.  Now reload all the saved registers.  Since we've been
// memory-bound for a long time, there's no advantage to scheduling this.

	lwz		r13,-48(sp)
	lwz		r14,-44(sp)
	lwz		r15,-40(sp)
	lwz		r16,-36(sp)
	lwz		r17,-32(sp)
	lwz		r18,-28(sp)
	lwz		r19,-24(sp)
	lwz		r20,-20(sp)
	lwz		r21,-16(sp)
	lwz		r22,-12(sp)
	lwz		r23,-8(sp)
	lwz		r24,-4(sp)

	blr
}
#undef in2
#undef in3

#endif // } DOUBLE_DECRYPT

#else // } CAST5_INTERLEAVED_ENCRYPT {

// Same round function as above
#define ROUND(x,y1,y2,b1,b2,op1,op2,op3,n) \
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;		op3    ta,x,key;		\
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		rlwnm  ta,ta,rot,0,31;	\
	op3    t2,t2,t1;	lwzx   tb,s4,tb;			rlwinm t1,ta,10,22,29;	\
	op1    t3,t3,t2;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;	\
	op2    tb,tb,t3;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;	\
	xor    b2,b1,tb;				lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;	\
	op3    tb,b2,key;				op1  t2,t2,t1;	lwzx   ta,s4,ta;		\
	rlwnm  tb,tb,rot,0,31;			op2  t3,t3,t2;	lwz key,8*n+8(xkey);	\
	rlwinm t1,tb,10,22,29;			op3  ta,ta,t3;	lwz rot,8*n+12(xkey);	\
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		xor	   y2,y1,ta;		\

#define out0 r10
#define out1 r11
#define out2 r12
#define out3 r13
#define s1	r14
#define s2	r15
#define s3	r16
#define s4	r17

#define ta	r18
#define tb	r19
#define	t1	r20
#define	t2	r21
#define	t3	r22

#define	key	r23
#define	rot	r24

asm void
CAST5encryptCFBdbl(
	register word32 const *xkey,	// r3
	register word32 in0,			// r4
	register word32 in1,			// r5
	register word32 in2,			// r6
	register word32 in3,			// r7
	register word32 const *src,		// r8
	register word32 *dest,			// r9
	register word32 len)			// r10
{
	mtctr	len					// Repeat len times....
	stw		r13,-48(sp)
	stw		r14,-44(sp)
	stw		r15,-40(sp)
	stw		r16,-36(sp)
	stw		r17,-32(sp)
	stw		r18,-28(sp)
	stw		r19,-24(sp)
	stw		r20,-20(sp)
	stw		r21,-16(sp)
	stw		r22,-12(sp)
	stw		r23,-8(sp)
	stw		r24,-4(sp)

	lwz		key,0(xkey)
	lwz		rot,4(xkey)

	lwz		s1,S1(RTOC)
// The core CAST function starts here
	lwz		s2,S2(RTOC);							add    ta,in1,key;
	lwz		s3,S3(RTOC);							rlwnm  ta,ta,rot,0,31;
	lwz		s4,S4(RTOC);							rlwinm t1,ta,10,22,29;
									lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
									lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
	b		@entry

@loop:
	subf   t3,t3,t2;	 	/**/					 lwz key,0(xkey);
	add    tb,tb,t3;		/**/					 lwz rot,4(xkey);
	xor    out2,out2,tb;	/**/					 lwz in2,8(src);
	 xor in2,out2,in2;		/**/					 lwz in3,12(src);
	 stw in0,0(dest);		/**/					 xor in3,out3,in3;
	 stw in1,4(dest);		/**/					add    ta,in1,key;
	 stw in2,8(dest);		/**/					rlwnm  ta,ta,rot,0,31;
	 stw in3,12(dest);		/**/					rlwinm t1,ta,10,22,29;
	 addi src,src,16;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
	 addi dest,dest,16;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
@entry:
									lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;
	add    tb,in3,key;				xor  t2,t2,t1;	lwzx   ta,s4,ta;
	rlwnm  tb,tb,rot,0,31;			subf t3,t3,t2;	lwz key,8(xkey);
	rlwinm t1,tb,10,22,29;			add  ta,ta,t3;	lwz rot,12(xkey);
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		xor    out1,in0,ta;

//	ROUND(in1,  in0,  out1, XXX,  in3,  xor, subf, add, 0)
	ROUND(out1, in1,  out0, in2,  out3, subf, add, xor, 1)
	ROUND(out0, out1, out1, in3,  out2, add, xor, subf, 2)
	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 3)
	ROUND(out0, out1, out1, out2, out2, subf, add, xor, 4)
	ROUND(out1, out0, out0, out3, out3, add, xor, subf, 5)
	ROUND(out0, out1, out1, out2, out2, xor, subf, add, 6)
	ROUND(out1, out0, out0, out3, out3, subf, add, xor, 7)
	ROUND(out0, out1, out1, out2, out2, add, xor, subf, 8)
	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 9)
	ROUND(out0, out1, out1, out2, out2, subf, add, xor, 10)
	ROUND(out1, out0, out0, out3, out3, add, xor, subf, 11)
	ROUND(out0, out1, out1, out2, out2, xor, subf, add, 12)
	ROUND(out1, out0, out0, out3, out3, subf, add, xor, 13)
	ROUND(out0, out1, out1, out2, out2, add, xor, subf, 14)
//	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 15)
// Final wrapup.  Steps that are part of the CFB (as opposed to CAST)
// are indented one space.  /**/ marks an instruction slot.
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;		add    ta,out1,key;
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		rlwnm  ta,ta,rot,0,31;
	add    t2,t2,t1;	lwzx   tb,s4,tb;			rlwinm t1,ta,10,22,29;
	xor    t3,t3,t2;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
	subf   tb,tb,t3;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
	xor    out3,out3,tb; 			lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;
	add    tb,out3,key;			 	xor  t2,t2,t1;	lwzx	ta,s4,ta;
	rlwnm  tb,tb,rot,0,31;			subf t3,t3,t2;	 lwz in0,0(src);
	rlwinm t1,tb,10,22,29;			add  ta,ta,t3;	 lwz in1,4(src);
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		 xor	out0,out0,ta;
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;		 xor in0,out0,in0;
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		 xor in1,out1,in1;
	xor    t2,t2,t1;	lwzx   tb,s4,tb;			 /**/

	bdnz	@loop

	subf   t3,t3,t2;	 						 lwz in2,8(src)
	add    tb,tb,t3;							 lwz in3,12(src)
	xor    out2,out2,tb;						 stw in0,0(dest)
	 xor in2,out2,in2;							 stw in1,4(dest)
	 xor in3,out3,in3;							 stw in2,8(dest)
												 stw in3,12(dest)

// Done.  Now reload all the saved registers.  Since we've been
// memory-bound for a long time, there's no advantage to scheduling this.

	lwz		r13,-48(sp)
	lwz		r14,-44(sp)
	lwz		r15,-40(sp)
	lwz		r16,-36(sp)
	lwz		r17,-32(sp)
	lwz		r18,-28(sp)
	lwz		r19,-24(sp)
	lwz		r20,-20(sp)
	lwz		r21,-16(sp)
	lwz		r22,-12(sp)
	lwz		r23,-8(sp)
	lwz		r24,-4(sp)

	blr
}

asm void
CAST5decryptCFBdbl(
	register word32 const *xkey,	// r3
	register word32 in0,			// r4
	register word32 in1,			// r5
	register word32 in2,			// r6
	register word32 in3,			// r7
	register word32 const *src,		// r8
	register word32 *dest,			// r9
	register word32 len)			// r10
{
	mtctr	len					// Repeat len times....
	stw		r13,-48(sp)
	stw		r14,-44(sp)
	stw		r15,-40(sp)
	stw		r16,-36(sp)
	stw		r17,-32(sp)
	stw		r18,-28(sp)
	stw		r19,-24(sp)
	stw		r20,-20(sp)
	stw		r21,-16(sp)
	stw		r22,-12(sp)
	stw		r23,-8(sp)
	stw		r24,-4(sp)

	lwz		key,0(xkey)
	lwz		rot,4(xkey)

	lwz		s1,S1(RTOC)
// The core CAST function starts here
	lwz		s2,S2(RTOC);							add    ta,in1,key;
	lwz		s3,S3(RTOC);							rlwnm  ta,ta,rot,0,31;
	lwz		s4,S4(RTOC);							rlwinm t1,ta,10,22,29;
									lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
									lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
	b		@entry
@loop:
	subf   t3,t3,t2;	 	/**/					 lwz key,0(xkey);
	add    tb,tb,t3;		/**/					 lwz rot,4(xkey);
	xor    out2,out2,tb;	/**/					 lwz in2,8(src);
	 xor out2,out2,in2;		/**/					 lwz in3,12(src);
	 stw out0,0(dest);		/**/					 xor out3,out3,in3;
	 stw out1,4(dest);		/**/					add    ta,in1,key;
	 stw out2,8(dest);		/**/					rlwnm  ta,ta,rot,0,31;
	 stw out3,12(dest);		/**/					rlwinm t1,ta,10,22,29;
	 addi src,src,16;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
	 addi dest,dest,16;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
@entry:
									lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;
	add    tb,in3,key;				xor  t2,t2,t1;	lwzx   ta,s4,ta;
	rlwnm  tb,tb,rot,0,31;			subf t3,t3,t2;	lwz key,8(xkey);
	rlwinm t1,tb,10,22,29;			add  ta,ta,t3;	lwz rot,12(xkey);
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		xor    out1,in0,ta;

//	ROUND(in1,  in0,  out1, XXX,  in3,  xor, subf, add, 0)
	ROUND(out1, in1,  out0, in2,  out3, subf, add, xor, 1)
	ROUND(out0, out1, out1, in3,  out2, add, xor, subf, 2)
	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 3)
	ROUND(out0, out1, out1, out2, out2, subf, add, xor, 4)
	ROUND(out1, out0, out0, out3, out3, add, xor, subf, 5)
	ROUND(out0, out1, out1, out2, out2, xor, subf, add, 6)
	ROUND(out1, out0, out0, out3, out3, subf, add, xor, 7)
	ROUND(out0, out1, out1, out2, out2, add, xor, subf, 8)
	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 9)
	ROUND(out0, out1, out1, out2, out2, subf, add, xor, 10)
	ROUND(out1, out0, out0, out3, out3, add, xor, subf, 11)
	ROUND(out0, out1, out1, out2, out2, xor, subf, add, 12)
	ROUND(out1, out0, out0, out3, out3, subf, add, xor, 13)
	ROUND(out0, out1, out1, out2, out2, add, xor, subf, 14)
//	ROUND(out1, out0, out0, out3, out3, xor, subf, add, 15)
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;		add    ta,out1,key;
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		rlwnm  ta,ta,rot,0,31;
	add    t2,t2,t1;	lwzx   tb,s4,tb;			rlwinm t1,ta,10,22,29;
	xor    t3,t3,t2;				lwzx t1,s1,t1;	rlwinm t2,ta,18,22,29;
	subf   tb,tb,t3;				lwzx t2,s2,t2;	rlwinm t3,ta,26,22,29;
	xor    out3,out3,tb; 			lwzx t3,s3,t3;	rlwinm ta,ta,2,22,29;
	add    tb,out3,key;			 	xor  t2,t2,t1;	lwzx	ta,s4,ta;
	rlwnm  tb,tb,rot,0,31;			subf t3,t3,t2;	 lwz in0,0(src);
	rlwinm t1,tb,10,22,29;			add  ta,ta,t3;	 lwz in1,4(src);
	lwzx   t1,s1,t1;	rlwinm t2,tb,18,22,29;		 xor	out0,out0,ta;

// Final wrapup.  Steps that are part of the CFB (as opposed to CAST)
// are indented one space.  /**/ marks an instruction slot.
// Note the "bdz" buried in there, just before the cycle that starts
// loading keys to restart.  Since all PPCs have a separate branch unit,
// having two branches takes no time, and it simplifies it considerably.
	lwzx   t2,s2,t2;	rlwinm t3,tb,26,22,29;		 xor out0,out0,in0;
	lwzx   t3,s3,t3;	rlwinm tb,tb,2,22,29;		 xor out1,out1,in1;
	xor    t2,t2,t1;	lwzx   tb,s4,tb;			 /**/

	bdnz	@loop	// Should take 0 cycles

@exit:
	subf   t3,t3,t2;	 						 lwz in2,8(src)
	add    tb,tb,t3;							 lwz in3,12(src)
	xor    out2,out2,tb;						 stw out0,0(dest)
	 xor out2,out2,in2;							 stw out1,4(dest)
	 xor out3,out3,in3;							 stw out2,8(dest)
												 stw out3,12(dest)

// Done.  Now reload all the saved registers.  Since we've been
// memory-bound for a long time, there's no advantage to scheduling this.

	lwz		r13,-48(sp)
	lwz		r14,-44(sp)
	lwz		r15,-40(sp)
	lwz		r16,-36(sp)
	lwz		r17,-32(sp)
	lwz		r18,-28(sp)
	lwz		r19,-24(sp)
	lwz		r20,-20(sp)
	lwz		r21,-16(sp)
	lwz		r22,-12(sp)
	lwz		r23,-8(sp)
	lwz		r24,-4(sp)

	blr
}

#endif // } CAST5_INTERLEAVED_ENCRYPT

#undef out0
#undef out1
#undef out2
#undef out3
#undef s1
#undef s2
#undef s3
#undef s4

#undef ta
#undef tb
#undef t1
#undef t2
#undef t3

#else // } !__POWERPC__ {   Portable C code version of above

#if __MC68K__ && ! __CFM68K__ // {

// Bloody annoying Metrowerks compiler doesn't support
// '020 addressing modes like (a0,d0.w*4).  So we have to
// define them with opword...  Sigh.
// Note if you care: accessing "xkey" with offsets rather
// than postincrement has been tried.  It's the same speed
// or slower, and takes more space.
#define ROUND1(x,y,n) \
	move.l	(xkey)+,k1;	\
	add.l	x,k1;			\
	move.l	(xkey)+,k2;	\
	rol.l	k2,k1;			\
	swap	k1;				\
	move.l	k1,k2;			\
	lsr.l	%8,k1;			\
	and.l	mask,k1;		\
	and.l	mask,k2;		\
/*	move.l	(s1,k1.w*4),t;	*/	opword	0x2A31, 0x3400;	\
	swap	k1;				\
/*	move.l	(s2,k2.w*4),e;	*/	opword	0x2C32, 0x4400;	\
	eor.l	e,t;			\
	swap	k2;				\
/*	sub.l	(s3,k1.w*4),t;	*/	opword	0x9AB3, 0x3400;	\
/*	add.l	(s4,k2.w),t;	*/	opword	0xDAB4, 0x4400;	\
	eor.l	t,y

#define ROUND2(x,y,n) \
	move.l	(xkey)+,k1;	\
	eor.l	x,k1;			\
	move.l	(xkey)+,k2;	\
	rol.l	k2,k1;			\
	swap	k1;				\
	move.l	k1,k2;			\
	lsr.l	%8,k1;			\
	and.l	mask,k1;		\
	and.l	mask,k2;		\
/*	move.l	(s1,k1.w*4),t;	*/	opword 0x2A31, 0x3400;	\
	swap	k1;				\
/*	sub.l	(s2,k2.w*4),t;	*/	opword 0x9AB2, 0x4400;	\
	swap	k2;				\
/*	add.l	(s3,k1.w*4),t;	*/	opword 0xDAB3, 0x3400;	\
/*	move.l	(s4,k2.w*4),e;	*/	opword 0x2C34, 0x4400;	\
	eor.l	e,t;			\
	eor.l	t,y

#define ROUND3(x,y,n) \
	move.l	(xkey)+,k1;	\
	sub.l	x,k1;			\
	move.l	(xkey)+,k2;	\
	rol.l	k2,k1;			\
	swap	k1;				\
	move.l	k1,k2;			\
	lsr.l	%8,k1;			\
	and.l	mask,k1;		\
	and.l	mask,k2;		\
/*	move.l	(s1,k1.w*4),t;	*/	opword 0x2A31, 0x3400;	\
	swap	k1;				\
/*	add.l	(s2,k2.w*4),t;	*/	opword 0xDAB2, 0x4400;	\
	swap	k2;				\
/*	move.l	(s3,k1.w*4),e;	*/	opword 0x2C33, 0x3400;	\
	eor.l	e,t;			\
/*	sub.l	(s4,k2.w*4),t;	*/	opword 0x9AB4, 0x4400;	\
	eor.l	t,y

// *** DANGER WILL ROBINSON ***
// If you touch these register assignments, you'll have to fix
// all the opwords above!
#define left	d0
#define right	d1
#define mask	d2
#define k1		d3
#define k2		d4
#define t		d5
// The 680x0 can't EOR mem->reg, arrgh!
#define e		d6

#define xkey	a0
#define s1		a1
#define s2		a2
#define s3		a3
#define s4		a4

asm void
CAST5encrypt( byte const *in, byte *out, word32 const *xkey)
{
	move.l	12(sp),xkey
	move.l	4(sp),a1	/* in pointer */
	movem.l	d2-d6/a2-a4,-(sp)
	move.l	(a1)+,left
	move.l	(a1),right
	lea		S1,s1
	lea		S2,s2
	lea		S3,s3
	lea		S4,s4
	move.l	#0xFF00FF,mask

	ROUND1(right, left, 0)
	ROUND2(left, right, 1)
	ROUND3(right, left, 2)
	ROUND1(left, right, 3)
	ROUND2(right, left, 4)
	ROUND3(left, right, 5)
	ROUND1(right, left, 6)
	ROUND2(left, right, 7)
	ROUND3(right, left, 8)
	ROUND1(left, right, 9)
	ROUND2(right, left, 10)
	ROUND3(left, right, 11)
	ROUND1(right, left, 12)
	ROUND2(left, right, 13)
	ROUND3(right, left, 14)
	ROUND1(left, right, 15)

	move.l	8+4*8(sp),a1
	movem.l	(sp)+,d2-d6/a2-a4
	move.l	right,(a1)+
	move.l	left,(a1)
	rts
}
#undef ROUND1
#undef ROUND2
#undef ROUND3
#undef left
#undef right
#undef mask
#undef k1
#undef k2
#undef t
#undef xkey
#undef s1
#undef s2
#undef s3
#undef s4

#else // } !__MC68K__ && ! __CFM68K__ {

void
CAST5encrypt( byte const *in, byte *out, word32 const *xkey)
{
	word32 l, r, t;

#if MAC_68K
	/* substituting these generates significantly better code */
	register const word32 *	s1 = S1;
	register const word32 *	s2 = S2;
	register const word32 *	s3 = S3;
	register const word32 *	s4 = S4;
#endif

#if __POWERPC__ || __MC68K__
	l = ((word32 *)in)[0];	// Big-endian and unaligned access okay.
	r = ((word32 *)in)[1];
#else
	l = (word32)in[0]<<24 | (word32)in[1]<<16 | (word32)in[2]<<8 | in[3];
	r = (word32)in[4]<<24 | (word32)in[5]<<16 | (word32)in[6]<<8 | in[7];
#endif

	t = F1(r, xkey,  0); l ^= G1(t);
	t = F2(l, xkey,  1); r ^= G2(t);
	t = F3(r, xkey,  2); l ^= G3(t);
	t = F1(l, xkey,  3); r ^= G1(t);
	t = F2(r, xkey,  4); l ^= G2(t);
	t = F3(l, xkey,  5); r ^= G3(t);
	t = F1(r, xkey,  6); l ^= G1(t);
	t = F2(l, xkey,  7); r ^= G2(t);
	t = F3(r, xkey,  8); l ^= G3(t);
	t = F1(l, xkey,  9); r ^= G1(t);
	t = F2(r, xkey, 10); l ^= G2(t);
	t = F3(l, xkey, 11); r ^= G3(t);
	/* Stop here if only doing 12 rounds */
	t = F1(r, xkey, 12); l ^= G1(t);
	t = F2(l, xkey, 13); r ^= G2(t);
	t = F3(r, xkey, 14); l ^= G3(t);
	t = F1(l, xkey, 15); r ^= G1(t);

#if __POWERPC__ || __MC68K__
	((word32 *)out)[0] = r;
	((word32 *)out)[1] = l;
#else
	out[0] = B0(r);
	out[1] = B1(r);
	out[2] = B2(r);
	out[3] = B3(r);
	out[4] = B0(l);
	out[5] = B1(l);
	out[6] = B2(l);
	out[7] = B3(l);
#endif
}

#endif // } !__MC68K__

#if !CAST5_INTERLEAVED_ENCRYPT // {
void
CAST5encryptCFB(word32 const *xkey, word32 iv0, word32 iv1,
	word32 const *src, word32 *dest, word32 len)
{
	word32 iv[2] = {iv0, iv1};

	while (len--) {
		CAST5encrypt((byte const *)iv, (byte *)iv, xkey);
		*dest++ = iv[0] ^= *src++;
		*dest++ = iv[1] ^= *src++;
	}
}

void
CAST5decryptCFB(word32 const *xkey, word32 iv0, word32 iv1,
	word32 const *src, word32 *dest, word32 len)
{
	word32 iv[2] = {iv0, iv1};
	word32 out[2];

	while (len--) {
		CAST5encrypt((byte const *)iv, (byte *)out, xkey);
		*dest++ = out[0] ^ (iv[0] = *src++);
		*dest++ = out[1] ^ (iv[1] = *src++);
	}
}

#else // } CAST5_INTERLEAVED_ENCRYPT {

void
CAST5encryptCFBdbl(word32 const *xkey,
	word32 iv0, word32 iv1, word32 iv2, word32 iv3,
	word32 const *src, word32 *dest, word32 len)
{
	word32 iv[4] = {iv0, iv1, iv2, iv3};

	while (len--) {
		CAST5encrypt((byte const *)iv, (byte *)iv, xkey);
		*dest++ = iv[0] ^= *src++;
		*dest++ = iv[1] ^= *src++;
		CAST5encrypt((byte const *)(iv+2), (byte *)(iv+2), xkey);
		*dest++ = iv[2] ^= *src++;
		*dest++ = iv[3] ^= *src++;
	}
}

void
CAST5decryptCFBdbl(word32 const *xkey,
	word32 iv0, word32 iv1, word32 iv2, word32 iv3,
	word32 const *src, word32 *dest, word32 len)
{
	word32 iv[4] = {iv0, iv1, iv2, iv3};
	word32 out[4];

	while (len--) {
		CAST5encrypt((byte const *)iv, (byte *)out, xkey);
		*dest++ = out[0] ^ (iv[0] = *src++);
		*dest++ = out[1] ^ (iv[1] = *src++);
		CAST5encrypt((byte const *)(iv+2), (byte *)(out+2), xkey);
		*dest++ = out[2] ^ (iv[2] = *src++);
		*dest++ = out[3] ^ (iv[3] = *src++);
	}
}

#endif // } CAST5_INTERLEAVED_ENCRYPT

#endif // } !__POWERPC__


#if 1	// [

void
CAST5decrypt(byte const *in, byte *out, word32 const *xkey)
{
	word32 l, r, t;

#if MAC_68K
	// substituting these generates significantly better code
	register const word32 *	s1 = S1;
	register const word32 *	s2 = S2;
	register const word32 *	s3 = S3;
	register const word32 *	s4 = S4;
#endif

#if __POWERPC__ || __M68K__
	r = ((word32 const *)in)[0];	/* Big-endian, unaligned okay */
	l = ((word32 const *)in)[1];
#else
	r = (word32)in[0]<<24 | (word32)in[1]<<16 | (word32)in[2]<<8 | in[3];
	l = (word32)in[4]<<24 | (word32)in[5]<<16 | (word32)in[6]<<8 | in[7];
#endif

	t = F1(l, xkey, 15); r ^= G1(t);
	t = F3(r, xkey, 14); l ^= G3(t);
	t = F2(l, xkey, 13); r ^= G2(t);
	t = F1(r, xkey, 12); l ^= G1(t);
	/* Start here if only doing 12 rounds */
	t = F3(l, xkey, 11); r ^= G3(t);
	t = F2(r, xkey, 10); l ^= G2(t);
	t = F1(l, xkey,  9); r ^= G1(t);
	t = F3(r, xkey,  8); l ^= G3(t);
	t = F2(l, xkey,  7); r ^= G2(t);
	t = F1(r, xkey,  6); l ^= G1(t);
	t = F3(l, xkey,  5); r ^= G3(t);
	t = F2(r, xkey,  4); l ^= G2(t);
	t = F1(l, xkey,  3); r ^= G1(t);
	t = F3(r, xkey,  2); l ^= G3(t);
	t = F2(l, xkey,  1); r ^= G2(t);
	t = F1(r, xkey,  0); l ^= G1(t);

#if __POWERPC__ || __M68K__
	((word32 *)out)[0] = l;		/* Big-endian, unaligned okay */
	((word32 *)out)[1] = r;
#else
	out[0] = B0(l);
	out[1] = B1(l);
	out[2] = B2(l);
	out[3] = B3(l);
	out[4] = B0(r);
	out[5] = B1(r);
	out[6] = B2(r);
	out[7] = B3(r);
#endif
}

#endif // ] USE_CAST_DECRYPT_CODE



	static word32
ContinueCRC32(
	const word32 *	words,
	size_t			numWords,
	word32			startCRC)
{
	word32			k4C11DB7	= 0x04C11DB7;
	word32 			crc			= startCRC;
	word32 const *	curWord 		= words;
	word32 const *	lastWord		= words + (numWords - 1);

	while( curWord <= lastWord )
	{
		int				remaining;
		word32			theWord;
		unsigned char	bytes[ 4 ];
		const unsigned char *	curByte;
		
		/* we want an endian-independent CRC */
		/* place the bytes in big-endian order */
		theWord	= *curWord++;
		bytes[ 0 ]	= (theWord >> 24) & 0xFF;
		bytes[ 1 ]	= (theWord >> 16) & 0xFF;
		bytes[ 2 ]	= (theWord >> 8) & 0xFF;
		bytes[ 3 ]	= theWord & 0xFF;
		
		remaining	= 4;
		curByte		= bytes;
		while ( remaining-- != 0 )
		{
			SInt32			check;

			crc ^= ((SInt32 ) *curByte) << 8;
			++curByte;

			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
			check = crc; crc <<= 1; if( check < 0 ) crc ^= k4C11DB7;
		}
	}

	return( crc );
}




	Boolean
VerifyCastSBoxesValid()
	{
	Boolean			valid	= TRUE;
	unsigned long	crc	= 0;

	// Force reference to our static so linker does not remove it.
	#if MAC68K_ALIGN_HACK
	{
		crc	= ContinueCRC32( (word32 *)___AlignHack, sizeof( ___AlignHack ),
					crc );
		crc = 0;
	}
	#endif

	pgpAssert( ( (long)&S1[ 0 ] & 0x3 ) == 0 );
	pgpAssert( ( (long)&S2[ 0 ] & 0x3 ) == 0 );
	pgpAssert( ( (long)&S3[ 0 ] & 0x3 ) == 0 );
	pgpAssert( ( (long)&S4[ 0 ] & 0x3 ) == 0 );
	pgpAssert( ( (long)&S5[ 0 ] & 0x3 ) == 0 );
	pgpAssert( ( (long)&S6[ 0 ] & 0x3 ) == 0 );
	pgpAssert( ( (long)&S7[ 0 ] & 0x3 ) == 0 );
	pgpAssert( ( (long)&S8[ 0 ] & 0x3 ) == 0 );

	crc	= ContinueCRC32( S1, sizeof( S1 ) / sizeof( S1[ 0 ] ), crc );
	crc	= ContinueCRC32( S2, sizeof( S2 ) / sizeof( S2[ 0 ] ), crc );
	crc	= ContinueCRC32( S3, sizeof( S3 ) / sizeof( S3[ 0 ] ), crc );
	crc	= ContinueCRC32( S4, sizeof( S4 ) / sizeof( S4[ 0 ] ), crc );
	crc	= ContinueCRC32( S5, sizeof( S5 ) / sizeof( S5[ 0 ] ), crc );
	crc	= ContinueCRC32( S6, sizeof( S6 ) / sizeof( S6[ 0 ] ), crc );
	crc	= ContinueCRC32( S7, sizeof( S7 ) / sizeof( S7[ 0 ] ), crc );
	crc	= ContinueCRC32( S8, sizeof( S8 ) / sizeof( S8[ 0 ] ), crc );

	valid	= ( crc == 0xD8D4C8D7 );

	return( valid );
	}
	

#if 0	// KEEP [
/*
 * Appendix C.
 * This appendix provides test vectors for the CAST-128 cipher described
 * in Section 5 and in Appendices A and B.
 * C.1. Single Key-Plaintext-Ciphertext Set
 * 128-bit key       = 01 23 45 67 12 34 56 78 23 45 67 89 34 56 78 9A (hex)
 * 64-bit plaintext  = 01 23 45 67 89 AB CD EF (hex)
 * 64-bit ciphertext = 23 8B 4F E5 84 7E 44 B2 (hex)
 * C.2. Full Maintenance Test
 * A maintenance test for CAST-128 has been defined to verify the correctness
 * of implementations.  It is defined in pseudo-code as follows, where a and
 * b are 128-bit vectors, aL and aR are the leftmost and rightmost halves of
 * a, bL and bR are the leftmost and rightmost halves of b, and encrypt(d,
 * k) is the encryption in ECB mode of block d under key k.
 * 
 * .
 * Initial 
 * a = 01 23 45 67 12 34 56 78 23 45 67 89 34 56 78 9A (hex)
 * Initial 
 * b = 01 23 45 67 12 34 56 78 23 45 67 89 34 56 78 9A (hex)
 * do 1,000,000 times
 * {
 *     aL = encrypt( aL , b)
 *     aR = encrypt( aR , b)
 *     bL = encrypt( bL , a)
 *     bR = encrypt( bR , a)
 * }
 * Verify 
 * a == EE A9 D0 A2 49 FD 3B A6 B3 43 6F B8 9D 6D CA 92 (hex)
 * Verify 
 * b == B2 C9 5E B0 0C 31 AD 71 80 AC 05 B8 E8 3D 69 6E (hex)
 * 	
 * 
 */
	int
main(void)
{
	byte a[16] = 
	{
		0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
		0x23, 0x45, 0x67, 0x89, 0x34, 0x56, 0x78, 0x9A
	};
	byte b[16] = 
	{
		0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
		0x23, 0x45, 0x67, 0x89, 0x34, 0x56, 0x78, 0x9A
	};
	byte test[8] =
	{
		0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF
	};
	static byte const expect1[8] =
	{
		 0x23, 0x8B, 0x4F, 0xE5, 0x84, 0x7E, 0x44, 0xB2
	};
	static byte const expecta[16] =
	{
		0xEE, 0xA9, 0xD0, 0xA2, 0x49, 0xFD, 0x3B, 0xA6,
		0xB3, 0x43, 0x6F, 0xB8, 0x9D, 0x6D, 0xCA, 0x92
	};
	static byte const expectb[16] =
	{
		0xB2, 0xC9, 0x5E, 0xB0, 0x0C, 0x31, 0xAD, 0x71,
		0x80, 0xAC, 0x05, 0xB8, 0xE8, 0x3D, 0x69, 0x6E
	};
	word32	xkey[32];
	long	count;

	printf("Simple sanity check...\n");
	
	printhex("       Key =", a, sizeof(a));
	CAST5schedule(xkey, a);
	printhex(" Plaintext =", test, sizeof(test));
	CAST5encrypt(test, test, xkey);
	printhex("Ciphertext =", test, sizeof(test));
	printhex("    Expect =", expect1, sizeof(expect1));
	if (memcmp(test, expect1, sizeof(test)) != 0)
		printf("		***** ERROR *****\n");
	
	printf("Full maintenance test (4,000,000 encryptions)...\n");

	for (count = 0 ;count < 1000000; count++) {
		CAST5schedule(xkey, b);
		CAST5encrypt(a, a, xkey);
		CAST5encrypt(a+8, a+8, xkey);
		CAST5schedule(xkey, a);
		CAST5encrypt(b, b, xkey);
		CAST5encrypt(b+8, b+8, xkey);
	}
	printhex("     a =", a, sizeof(a));
	printhex("Expect =", expecta, sizeof(expecta));
	if (memcmp(a, expecta, sizeof(a)) != 0)
		printf("		***** ERROR *****\n");
	printhex("     b =", b, sizeof(b));
	printhex("Expect =", expectb, sizeof(expectb));
	if (memcmp(b, expectb, sizeof(b)) != 0)
		printf("		***** ERROR *****\n");

	return 0;
}
#endif	// ] KEEP

