/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: PGPclientLibUtils.cp,v 1.42.2.1.2.1 1998/11/12 03:07:26 heller Exp $
____________________________________________________________________________*/

#include <Folders.h>
#include <LowMem.h>
#include <fp.h>
#include <string.h>

#include <LActiveScroller.h>
#include <LCaption.h>
#include <LGrafPortView.h>
#include <LIconPane.h>
#include <LRadioGroupView.h>
#include <LScrollBar.h>
#include <LTabGroup.h>
#include <PP_Messages.h>
#include <UAttachments.h>
#include <UDrawingState.h>
#include <UGAColorRamp.h>
#include <UEnvironment.h>
#include <UGraphicUtils.h>
#include <URegistrar.h>

// Appearance classes
#include <LBevelButton.h>
#include <LCheckBox.h>
#include <LEditText.h>
#include <LIconControl.h>
#include <LLittleArrows.h>
#include <LMultiPanelView.h>
#include <LPopupButton.h>
#include <LProgressBar.h>
#include <LPushButton.h>
#include <LRadioButton.h>
#include <LStaticText.h>
#include <LTabsControl.h>
#include <LTextGroupBox.h>
#include <LWindowHeader.h>

// Appearance Manager Implementations
#include <LAMControlImp.h>
#include <LAMPopupButtonImp.h>
#include <LAMPushButtonImp.h>
#include <LAMStaticTextImp.h>
#include <LAMTabsControlImp.h>
#include <LAMTrackActionImp.h>

// Grayscale Implementations
#include <LGABevelButtonImp.h>
#include <LGACheckBoxImp.h>
#include <LGAEditTextImp.h>
#include <LGAIconControlImp.h>
#include <LGALittleArrowsImp.h>
#include <LGAPopupButtonImp.h>
#include <LGAProgressBarImp.h>
#include <LGAPushButtonImp.h>
#include <LGARadioButtonImp.h>
#include <LGAStaticTextImp.h>
#include <LGATabsControlImp.h>
#include <LGATextGroupBoxImp.h>
#include <LGAWindowHeaderImp.h>

#include "MacBasics.h"
#include "MacDebug.h"
#include "MacDialogs.h"
#include "MacEnvirons.h"
#include "MacErrors.h"
#include "MacFiles.h"
#include "MacDesktop.h"
#include "MacResources.h"

#include "pflPrefs.h"
#include "pgpBase.h"
#include "pgpDebug.h"
#include "pgpMem.h"
#include "pgpSDKPrefs.h"
#include "pgpFeatures.h"

#include "CPGPActiveScroller.h"
#include "CPGPAMEditTextImp.h"
#include "CPGPAMTabsControlImp.h"
#include "CPGPModalGrafPortView.h"
#include "CServerStatusDialog.h"
#include "CShareholderList.h"
#include "PGPOpenPrefs.h"
#include "pgpAdminPrefs.h"
#include "pgpClientPrefs.h"
#include "pflPrefTypes.h"
#include "PGPclientLibUtils.h"
#include "pgpClientLib.h"
#include "pgpClientLibDialogs.h"
#include "pgpKeys.h"
#include "pgpRandomPool.h"
#include "CComboError.h"
#include "PowerPlantLeaks.h"
#include "StSaveCurResFile.h"
#include "CString.h"
#include "PGPSharedEncryptDecrypt.h"

static const ResType	kAdminConfiguredType	=	'PGPa';
static const SInt16		PGPa_Configured			=	0;

const ResID	kSelectPublicKeySFDialogResID 	= 4770;
const ResID	kSelectPrivateKeySFDialogResID 	= 4771;
const ResID	kSelectRandomSeedSFDialogResID 	= 4772;
const ResID	kSelectKeySharesSFDialogResID	= 4773;

const ResID	kFileNameStringResID 			= 4749;

enum
{
	kPGPPrivateKeyFileNameStrIndex	= 1,
	kPGPPublicKeyFileNameStrIndex,
	kPGPRandomSeedFileNameStrIndex,
	kPGPGroupsFileNameStrIndex
};

enum
{
	kShowAllFilesCheckboxItem	= 10
};

const ResID	STRx_UtilStrings				= 8946;
enum
{
	kSignerKeyImportStringID	= 1
};

static FSSpec		sLibraryFileSpec		= {0, };
static long			sLibraryFileID			= 0;
static short		sLibraryFileRefNum		= -1;
static PGPUInt32	sLibraryFileRefCount	= 0;
static PGPPrefRef	sClientPrefsRef			= kInvalidPGPPrefRef;

#if PGP_BUSINESS_SECURITY
static PGPPrefRef	sAdminPrefsRef			= kInvalidPGPPrefRef;
#endif

	static void
RegisterPowerPlantClasses(void)
{
	RegisterClass_( LActiveScroller );
	RegisterClass_( LCaption );
	RegisterClass_( LGrafPortView );
	RegisterClass_( LIconPane );
	RegisterClass_( LRadioGroupView );
	RegisterClass_( LTabGroup );
	RegisterClass_( LView );
	
	// Appearance classes:
	
	RegisterClass_( LBevelButton );
	RegisterClass_( LCheckBox );
	RegisterClass_( LEditText );
	RegisterClass_( LIconControl );
	RegisterClass_( LLittleArrows );
	RegisterClass_( LMultiPanelView );
	RegisterClass_( LPopupButton );
	RegisterClass_( LProgressBar );
	RegisterClass_( LPushButton );
	RegisterClass_( LRadioButton );
	RegisterClass_( LStaticText );
	RegisterClass_( LTabsControl );
	RegisterClass_( LTextGroupBox );
	RegisterClass_( LWindowHeader );

	// Register Appearance Manager implementations
	// Note: Do NOT call RegisterAppearanceClient here. That is the
	// client library responsibility.
	
	RegisterClassID_( LAMControlImp, 		LBevelButton::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LCheckBox::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LIconControl::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LProgressBar::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LTextGroupBox::imp_class_ID );		
	RegisterClassID_( LAMControlImp, 		LWindowHeader::imp_class_ID );
	RegisterClassID_( CPGPAMEditTextImp,	LEditText::imp_class_ID );
	RegisterClassID_( LAMPopupButtonImp,	LPopupButton::imp_class_ID );
	RegisterClassID_( LAMPushButtonImp,		LPushButton::imp_class_ID );
	RegisterClassID_( LAMControlImp, 		LRadioButton::imp_class_ID );
	RegisterClassID_( LAMStaticTextImp,		LStaticText::imp_class_ID );
	RegisterClassID_( CPGPAMTabsControlImp, LTabsControl::imp_class_ID );
	RegisterClassID_( LAMTrackActionImp, 	LLittleArrows::imp_class_ID );
		
	RegisterClass_( CPGPActiveScroller );
	RegisterClass_( CPGPModalGrafPortView );
	RegisterClass_( CServerStatusDialog );
	RegisterClass_( CShareholderList );
}

	OSStatus
SetLibraryFSSpec(const FSSpec *fileSpec)
{
	OSStatus	status;
	CInfoPBRec	cpb;
	
	// This routine should be called once ONLY from the library INIT routine.
	pgpAssert( sLibraryFileSpec.name[0] == 0 );
	pgpAssert( sLibraryFileID == 0 );
	
	status = FSpGetCatInfo( fileSpec, &cpb );
	if( IsntErr( status ) )
	{
		HParamBlockRec	pb;
		
		sLibraryFileSpec = *fileSpec;
		
		// Create a file ID so we can track this file later.
		
		pgpClearMemory( &pb, sizeof( pb ) );
		
		pb.fidParam.ioVRefNum	= sLibraryFileSpec.vRefNum;
		pb.fidParam.ioNamePtr	= sLibraryFileSpec.name;
		pb.fidParam.ioSrcDirID	= sLibraryFileSpec.parID;
		
		if( IsntErr( PBCreateFileIDRefSync( &pb ) ) )
		{
			sLibraryFileID = pb.fidParam.ioFileID;
		}
	}
	
	return( status );
}

	OSStatus
PGPGetClientLibFSSpec(FSSpec *fileSpec)
{
	OSStatus	status;
	CInfoPBRec	cpb;
	
	pgpAssertAddrValid( fileSpec, FSSpec );
	AssertSpecIsValid( &sLibraryFileSpec, "PGPGetClientLibFSSpec" );
	
	status = FSpGetCatInfo( &sLibraryFileSpec, &cpb );
	if( IsErr( status ) && sLibraryFileID != 0 )
	{
		// Find the library using the file ID.
		
		HParamBlockRec	pb;
		FSSpec			specCopy;
		
		pgpClearMemory( &pb, sizeof( pb ) );
		
		specCopy = sLibraryFileSpec;
		
		pb.fidParam.ioVRefNum	= specCopy.vRefNum;
		pb.fidParam.ioNamePtr	= specCopy.name;
		pb.fidParam.ioFileID	= sLibraryFileID;
		
		status = PBResolveFileIDRefSync( &pb );
		if( IsntErr( status ) )
		{
			specCopy.parID = pb.fidParam.ioSrcDirID;
			
			status = FSpGetCatInfo( &specCopy, &cpb );
			if( IsntErr( status ) )
			{
				sLibraryFileSpec = specCopy;
			}
		}
	}

	AssertNoErr( status, "GetPGPclientLibFileSpec" );
	
	*fileSpec = sLibraryFileSpec;
	
	return( status );
}

	static void
InitializePowerPlant(void)
{
	static Boolean	firstTime = TRUE;
	
	if( firstTime )
	{
#if PGP_DEBUG
		gDebugThrow 	= debugAction_LowLevelDebugger;
		gDebugSignal 	= debugAction_LowLevelDebugger;
#endif

		UEnvironment::InitEnvironment();
		UGraphicUtils::InitGraphicsUtilities();

		UQDGlobals::SetQDGlobals((QDGlobals*)(*((long*)::LMGetCurrentA5()) - 
				(sizeof(QDGlobals) - sizeof(GrafPtr))));
		
		MacLeaks_Suspend();

		#if USE_MAC_DEBUG_LEAKS
		{ CForceInitLPeriodical	temp; }
		#endif

		RegisterPowerPlantClasses();

		MacLeaks_Resume();
		
		LCommander::SetDefaultCommander( NULL );
		LPane::SetDefaultView( NULL );

		firstTime = FALSE;
	}
}

	PGPError
EnterPGPclientLib(
	PGPContextRef 		context,
	PGPclientLibState 	*state)
{
	CComboError		err;
	
	pgpAssertAddrValid( state, PGPclientLibState );
	
	InitializePowerPlant();

	pgpClearMemory( state, sizeof( *state ) );
	
	GetPort( &state->savedPort );
	
	state->savedResLoad 	= LMGetResLoad();
	state->savedZone		= GetZone();
	state->savedResFile		= CurResFile();
	state->savedView 		= LPane::GetDefaultView();
	state->savedCommander 	= LCommander::GetDefaultCommander();
	
	SetResLoad( TRUE );
	LCommander::SetDefaultCommander( NULL );
	LPane::SetDefaultView( NULL );
	
	if( sLibraryFileRefCount == 0 )
	{
		FSSpec	libraryFileSpec;

		pgpAssert( sLibraryFileRefNum == -1 );
		pgpAssert( ! PGPPrefRefIsValid( sClientPrefsRef ) );
	#if PGP_BUSINESS_SECURITY
		pgpAssert( ! PGPPrefRefIsValid( sAdminPrefsRef ) );
	#endif
	
		err.err	= PGPGetClientLibFSSpec( &libraryFileSpec );
		if( err.IsntError() )
		{
			sLibraryFileRefNum = FSpOpenResFile( &libraryFileSpec, fsRdPerm );
			if( sLibraryFileRefNum > 0 )
			{
				PGPMemoryMgrRef	memMgr;
				
				memMgr = PGPGetContextMemoryMgr( context );
				pgpAssert( PGPMemoryMgrRefIsValid( memMgr ) );
				
				err.pgpErr = PGPOpenClientPrefs( memMgr, &sClientPrefsRef );

			#if PGP_BUSINESS_SECURITY	// [
				if( err.IsntError() ) 
				{
					err.pgpErr = PGPOpenAdminPrefs( memMgr, &sAdminPrefsRef );
				}
			#endif	// ] PGP_BUSINESS_SECURITY
			
				if( err.IsntError() ) 
				{
					++sLibraryFileRefCount;
					UseResFile( sLibraryFileRefNum );
				}
				else
				{
					CloseResFile( sLibraryFileRefNum );
					UseResFile( state->savedResFile );
					sLibraryFileRefNum = -1;
					
					if( PGPPrefRefIsValid( sClientPrefsRef ) )
					{
						(void) PGPClosePrefFile( sClientPrefsRef );
						sClientPrefsRef = kInvalidPGPPrefRef;
					}

				#if PGP_BUSINESS_SECURITY	// [
					if( PGPPrefRefIsValid( sAdminPrefsRef ) )
					{
						(void) PGPClosePrefFile( sAdminPrefsRef );
						sAdminPrefsRef = kInvalidPGPPrefRef;
					}
				#endif	// ] PGP_BUSINESS_SECURITY
				}
			}
			else
			{
				err.pgpErr = kPGPError_FileNotFound;
			}
		}
	}
	else
	{
		/* Library already open. Just increment the count */
		
		pgpAssert( sLibraryFileRefNum > 0 );
		
		++sLibraryFileRefCount;
		UseResFile( sLibraryFileRefNum );
	}

	state->clientPrefsRef 	= sClientPrefsRef;
#if PGP_BUSINESS_SECURITY
	state->adminPrefsRef 	= sAdminPrefsRef;
#else
	state->adminPrefsRef 	= kInvalidPGPPrefRef;
#endif

	return( err.ConvertToPGPError() );
}

	void
ExitPGPclientLib(const PGPclientLibState *state)
{
	pgpAssertAddrValid( state, PGPclientLibState );

	--sLibraryFileRefCount;
	if( sLibraryFileRefCount == 0 )
	{
		pgpAssert( sLibraryFileRefNum > 0 );
		pgpAssert( PGPPrefRefIsValid( sClientPrefsRef ) );
	#if PGP_BUSINESS_SECURITY
		pgpAssert( PGPPrefRefIsValid( sAdminPrefsRef ) );
	#endif
	
		CloseResFile( sLibraryFileRefNum );
		sLibraryFileRefNum = -1;

		if( PGPPrefRefIsValid( sClientPrefsRef ) )
		{
			(void) PGPClosePrefFile( sClientPrefsRef );
			sClientPrefsRef = kInvalidPGPPrefRef;
		}
			
	#if PGP_BUSINESS_SECURITY
		if( PGPPrefRefIsValid( sAdminPrefsRef ) )
		{
			(void) PGPClosePrefFile( sAdminPrefsRef );
			sAdminPrefsRef = kInvalidPGPPrefRef;
		}
	#endif
	}

	LCommander::SetDefaultCommander( state->savedCommander );
	LPane::SetDefaultView( state->savedView );

	SetPort( state->savedPort );
	SetResLoad( state->savedResLoad );
	SetZone( state->savedZone );
	UseResFile( state->savedResFile );
}
	
	PGPError
PGPGetPGPFileName(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	StringPtr 		fileName)
{
	PGPError			err;
	PGPclientLibState	state;
	
	PGPValidatePtr( fileName );
	fileName[0] = 0;
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		short		strIndex;
		
		switch( whichFile )
		{
			case kPGPFileSelector_PublicKeys:
				strIndex = kPGPPublicKeyFileNameStrIndex;
				break;
				
			case kPGPFileSelector_PrivateKeys:
				strIndex = kPGPPrivateKeyFileNameStrIndex;
				break;
				
			case kPGPFileSelector_RandomSeed:
				strIndex = kPGPRandomSeedFileNameStrIndex;
				break;
			
			case kPGPFileSelector_Groups:
				strIndex = kPGPGroupsFileNameStrIndex;
				break;
				
			default:
				pgpDebugMsg( "PGPGetPGPFileName(): Invalid selector" );
				err = kPGPError_BadParams;
		}

		if( IsntPGPError( err ) )
		{
			GetIndString( fileName, kFileNameStringResID, strIndex );
		}
	}
	
	ExitPGPclientLib( &state );
	
	return( err );
}


	PGPError
PGPGetPGPFileDefaultFSSpec(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	FSSpec			*fileSpec)
{
	CComboError	err;
	Str255		fileName;
	
	PGPValidateParam( IsntNull( fileSpec ) );
	
	err.pgpErr = PGPGetPGPFileName( context, whichFile, fileName );
	if( err.IsntError() )
	{
		short	vRefNum;
		long	dirID;
		
		err.err = FindPGPPreferencesFolder( kOnSystemDisk, &vRefNum, &dirID );
		if( err.IsntError() )
		{
			err.err = FSMakeFSSpec( vRefNum, dirID, fileName, fileSpec );
		}
	}
	
	return( err.ConvertToPGPError() );
}

	PGPError
PGPSetPGPFileFSSpec(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	const FSSpec	*fileSpec)
{
	PGPError			err = kPGPError_NoErr;
	PGPsdkPrefSelector	prefSelector;
	
	PGPValidateParam( PGPContextRefIsValid( context ) );
	PGPValidateParam( IsntNull( fileSpec ) );

	switch( whichFile )
	{
		case kPGPFileSelector_PublicKeys:
			prefSelector = kPGPsdkPref_PublicKeyring;
			break;
			
		case kPGPFileSelector_PrivateKeys:
			prefSelector = kPGPsdkPref_PrivateKeyring;
			break;
			
		case kPGPFileSelector_RandomSeed:
			prefSelector = kPGPsdkPref_RandomSeedFile;
			break;

		case kPGPFileSelector_Groups:
			prefSelector = kPGPsdkPref_GroupsFile;
			break;
		
		default:
			pgpDebugMsg( "PGPSetPGPFileFSSpec(): Invalid selector" );
			err = kPGPError_BadParams;
	}
	
	if( IsntPGPError( err ) )
	{
		PGPFileSpecRef	fileRef;
		
		err = PGPNewFileSpecFromFSSpec(	context, fileSpec, &fileRef );
		if( IsntPGPError( err ) )
		{
			err = PGPsdkPrefSetFileSpec( context, prefSelector, fileRef );
			if( IsntPGPError( err ) )
			{
				err = PGPsdkSavePrefs( context );
			}
			
			(void) PGPFreeFileSpec( fileRef );
		}
	}
	
	return( err );
}

	static pascal Boolean
VisibleItemFileFilterProc(CInfoPBRec *cpb)
{
	Boolean	shouldFilter = FALSE;
	
	pgpAssertAddrValid( cpb, CInfoPBRec);

	if( cpbIsFile( cpb ) )
	{
		if( ( cpb->hFileInfo.ioFlFndrInfo.fdFlags & kIsInvisible ) != 0 )
		{
			shouldFilter = TRUE;
		}
	}
	else
	{
		if( ( cpb->dirInfo.ioDrUsrWds.frFlags & kIsInvisible ) != 0 )
		{
			shouldFilter = TRUE;
		}
	}
	
	return( shouldFilter );
}

typedef struct SFInfo
{
	StandardFileReply	sfReply;
	Boolean				showAllFiles;
	PGPFileSelector		fileSelector;

} SFInfo;

	static pascal Boolean
KeyFileFilterHook(
	CInfoPBRec 		*cpb,
	const SFInfo 	*sfInfo)
{
	Boolean	shouldFilter = FALSE;
	
	shouldFilter = VisibleItemFileFilterProc( cpb );
	if( ! shouldFilter )
	{
		if( cpbIsFile( cpb ) && ! sfInfo->showAllFiles )
		{
			// Show only the files possibly created by PGP.
			switch( cpbFileType( cpb ) )
			{
				case 'PKey':	// ancient MacPGP 2.6.2 filetype
				case kPGPMacFileType_PubRing:
					if( sfInfo->fileSelector != kPGPFileSelector_PublicKeys )
						shouldFilter = TRUE;
					break;
					
				case 'SKey':	// ancient MacPGP 2.6.2 filetype
				case kPGPMacFileType_PrivRing:
					if( sfInfo->fileSelector != kPGPFileSelector_PrivateKeys )
						shouldFilter = TRUE;
					break;
					
				case kPGPMacFileType_RandomSeed:
					if( sfInfo->fileSelector != kPGPFileSelector_RandomSeed )
						shouldFilter = TRUE;
					break;

				case kPGPMacFileType_KeyShares:
					if( sfInfo->fileSelector != kPGPFileSelector_KeyShares )
						shouldFilter = TRUE;
					break;
					
				default:
					shouldFilter = TRUE;
					break;
			}
		}
	}
		
	return( shouldFilter );
}

	static pascal short
KeyFileDialogHook(short item, DialogPtr dialog, SFInfo *sfInfo)
{
	pgpAssertAddrValid( dialog, DialogRecord );
	pgpAssertAddrValid( sfInfo, SFInfo );

	switch( item )
	{
		case kShowAllFilesCheckboxItem:
		{
			sfInfo->showAllFiles = ToggleDialogCheckbox( dialog,
										kShowAllFilesCheckboxItem );
			item = sfHookRebuildList;
			break;
		}
	}
	return item;
}

	PGPError
PGPSelectPGPFile(
	PGPContextRef	context,
	PGPFileSelector whichFile,
	FSSpec			*fileSpec)
{
	PGPError			err;
	PGPclientLibState 	state;
	
	PGPValidateParam( IsntNull( fileSpec ) );
	
	err = EnterPGPclientLib( context, &state );
	if( IsntPGPError( err ) )
	{
		ResID	dialogResID;
		
		switch( whichFile )
		{
			case kPGPFileSelector_PublicKeys:
				dialogResID = kSelectPublicKeySFDialogResID;
				break;
				
			case kPGPFileSelector_PrivateKeys:
				dialogResID = kSelectPrivateKeySFDialogResID;
				break;
				
			case kPGPFileSelector_RandomSeed:
				dialogResID = kSelectRandomSeedSFDialogResID;
				break;
				
			case kPGPFileSelector_KeyShares:
				dialogResID = kSelectKeySharesSFDialogResID;
				break;

			default:
				pgpDebugMsg( "PGPSelectPGPFileFSSpec(): Invalid selector" );
				err = kPGPError_BadParams;
		}
		
		if( IsntPGPError( err ) )
		{
			SFInfo				sfInfo;
			SFTypeList			typeList;
			DlgHookYDUPP		cbHook;
			FileFilterYDUPP		filterUPP;
			static Point		where={0,0};
			
			pgpClearMemory( &sfInfo, sizeof( sfInfo ) );
			
			cbHook 		= NewDlgHookYDProc( KeyFileDialogHook );
			filterUPP	= NewFileFilterYDProc( KeyFileFilterHook );
			typeList[0] = 0;
			
			sfInfo.showAllFiles = FALSE;
			sfInfo.fileSelector	= whichFile;
			
			CustomGetFile( filterUPP, -1, typeList, &sfInfo.sfReply,
					dialogResID, where, cbHook, NULL, NULL, NULL,
					&sfInfo );
								
			DisposeRoutineDescriptor( cbHook );
			DisposeRoutineDescriptor( filterUPP );
			
			if( sfInfo.sfReply.sfGood )
			{
				*fileSpec = sfInfo.sfReply.sfFile;
			}
			else
			{
				err = kPGPError_UserAbort;
			}
		}
	}
	
	ExitPGPclientLib( &state );
	
	return( err );
}

	PGPError
PGPInitClientDecodeEventHandlerData(
	PGPContextRef 					context,
	PGPtlsContextRef				tlsContext,
	PGPKeySetRef 					allKeys,
	PGPClientDecodeEventHandlerData *data)
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidateParam( PGPContextRefIsValid( context ) );
	PGPValidatePtr( data );
	
	pgpClearMemory( data, sizeof( *data ) );
	
	data->context 		= context;
	data->tlsContext	= tlsContext;
	data->allKeys		= allKeys;
	
	err = PGPNewPassBuffer( PGPGetContextMemoryMgr( context ),
								&data->passBuffer);
	return( err );
}

	void
PGPCleanupClientDecodeEventHandlerData(
	PGPClientDecodeEventHandlerData *data)
{
	if( IsntNull( data ) )
	{
		if( PGPKeySetRefIsValid( data->recipientKeySet ) )
			(void) PGPFreeKeySet( data->recipientKeySet );

		if( PGPKeySetRefIsValid( data->newKeySet ) )
			(void) PGPFreeKeySet( data->newKeySet );

		if( IsntNull( data->recipientKeyIDList ) )
			PGPFreeData( data->recipientKeyIDList );

		if( PGPPassBufferRefIsValid( data->passBuffer ) )
			PGPFreePassBuffer( data->passBuffer );

		pgpClearMemory( data, sizeof( *data ) );
	}
}

	PGPError
PGPClientDecodeEventHandler(
	PGPContextRef 					context,
	struct PGPEvent 				*event,
	PGPClientDecodeEventHandlerData *data )
{
	PGPError	err = kPGPError_NoErr;
	
	PGPValidatePtr( data );
	PGPValidateParam( data->context == context );
	
	switch( event->type )
	{
		case kPGPEvent_BeginLexEvent:
		{
			pgpAssert( IsNull( data->outputBuffer ) );
			pgpAssert( data->outputBufferSize == 0 );
			
			pgpClearMemory( &data->signatureData, 
						sizeof( data->signatureData ) );
						
			data->signatureDataValid = FALSE;
			
			break;
		}
		
		case kPGPEvent_EndLexEvent:
		{
			if( IsntNull( data->outputBuffer ) )
				PGPFreeData( data->outputBuffer );
		
			data->outputBuffer 		= NULL;
			data->outputBufferSize	= 0;
			
			break;
		}
		
		case kPGPEvent_AnalyzeEvent:
		{
			data->sectionType = event->data.analyzeData.sectionType;
			break;
		}
			
		case kPGPEvent_OutputEvent:
		{
			err = PGPAddJobOptions( event->job,
						PGPOAllocatedOutputBuffer( context, &data->outputBuffer,
								MAX_PGPSize, &data->outputBufferSize ),
						PGPOLastOption( context ) );
			break;
		}
		
		case kPGPEvent_PassphraseEvent:
		{
			char	*passphrase = NULL;
			
			data->conventional 	= FALSE;
			data->split 		= FALSE;
			data->decryptionKey	= kInvalidPGPKeyRef;
			
			/* Clear previous passphrase attempt(s) */
			(void) PGPClearPassBuffer( data->passBuffer );

			if( event->data.passphraseData.fConventional )
			{
				err = PGPConventionalDecryptionPassphraseDialog( context,
							data->prompt[0] != 0 ?
								PGPOUIDialogPrompt( context, data->prompt ) :
								PGPONullOption( context ),
							PGPOUIOutputPassphrase( context, &passphrase ),
							PGPOLastOption( context ) );
			}
			else
			{
				PGPKeySetRef	newKeySet = kInvalidPGPKeySetRef;
				
				err = PGPClientDecryptionPassphraseDialog( context,
							data->tlsContext,
							data->prompt[0] != 0 ? data->prompt : NULL,
							data->recipientKeySet,
							data->recipientKeyIDList, data->recipientKeyIDCount,
							kInvalidPGPOptionListRef, &passphrase,
							&data->decryptionKey, &newKeySet );
				if( IsntPGPError( err ) && PGPKeySetRefIsValid( newKeySet ) )
				{
					if( PGPKeySetRefIsValid( data->newKeySet ) )
					{
						err = PGPAddKeys( data->newKeySet, newKeySet );
						if( IsntPGPError( err ) )
						{
							err = PGPCommitKeyRingChanges( newKeySet );
							
							PGPFreeKeySet( data->newKeySet );
						}
					}
				
					data->newKeySet = newKeySet;
				}
			}
			
			if( IsntPGPError( err ) )
			{
				data->conventional = event->data.passphraseData.fConventional;
				
				err = PGPPassBufferSetPassphrase( data->passBuffer, passphrase );
				if( IsntPGPError( err ) )
				{
					err = PGPAddJobOptions( event->job,
								PGPOPassphrase( context, passphrase ),
								PGPOLastOption( context ) );
				}
				
				PGPFreeData( passphrase );
			}
			else if( err == kPGPError_KeyUnusableForDecryption )
			{
				PGPByte		*passKey;
				PGPSize		passKeySize;
				
				data->split	= TRUE;
				
				err = PGPReconstitutionDialog( data->decryptionKey,
							data->allKeys, kInvalidPGPtlsContextRef,
							&passKey, &passKeySize);
				
				if( IsntPGPError( err ) )
				{
					pgpAssert( passKey != NULL );
					pgpAssert( passKeySize != 0 );
					
					err = PGPPassBufferSetPassKey( data->passBuffer, passKey,
										passKeySize );
					if( IsntPGPError( err ) )
					{
						err = PGPAddJobOptions( event->job,
									PGPOPasskeyBuffer( context, passKey,
										passKeySize ),
									PGPOLastOption( context ) );
					}
								
					PGPFreeData( passKey );
				}
			}
			
			break;
		}
	
		case kPGPEvent_RecipientsEvent:
		{
			pgpAssert( ! PGPKeySetRefIsValid( data->recipientKeySet ) );
			pgpAssert( IsNull( data->recipientKeyIDList ) );
			pgpAssert( data->recipientKeyIDCount == 0 );

			data->recipientKeySet = event->data.recipientsData.recipientSet;
			if( PGPKeySetRefIsValid( data->recipientKeySet ) )
			{
				err = PGPIncKeySetRefCount( data->recipientKeySet );
			}
			
			if( IsntPGPError( err ) &&
				IsntNull( event->data.recipientsData.keyIDArray ) )
			{
				PGPSize	dataSize;
				
				data->recipientKeyIDCount =
									event->data.recipientsData.keyCount;
				dataSize = data->recipientKeyIDCount * sizeof( PGPKeyID );
				
				data->recipientKeyIDList = (PGPKeyID *) PGPNewData(
									PGPGetContextMemoryMgr( data->context ),
									dataSize, 0 );
				if( IsntNull( data->recipientKeyIDList ) )
				{
					pgpCopyMemory( event->data.recipientsData.keyIDArray,
							data->recipientKeyIDList, dataSize );
				}
				else
				{
					err = kPGPError_OutOfMemory;
				}
			}
			
			break;
		}
		
		case kPGPEvent_SignatureEvent:
		{
			PGPBoolean		addedKeySetToJob;
			PGPKeySetRef	newKeySet;

			err = PGPSignerKeyLookup( context, event, kInvalidPGPKeySetRef,
								&addedKeySetToJob, &newKeySet);
			if( IsntPGPError( err ) )
			{
				if( addedKeySetToJob )
				{
					if( PGPKeySetRefIsValid( data->newKeySet ) )
					{
						err = PGPAddKeys( data->newKeySet, newKeySet );
						if( IsntPGPError( err ) )
						{
							err = PGPCommitKeyRingChanges( newKeySet );

							PGPFreeKeySet( data->newKeySet );
						}
					}

					data->newKeySet = newKeySet;
				}
				else
				{
					data->signatureData 		= event->data.signatureData;
					data->signatureDataValid 	= TRUE;
				}
			}

			break;
		}
		
		default:
			break;
	}
	
	return( err );
}



	PGPError
PGPCheckAutoUpdateKeysFromServer(
	PGPMemoryMgrRef	memoryMgr,
	PGPBoolean		launchKeys,
	PGPBoolean *	updateAllKeys,
	PGPBoolean *	updateTrustedIntroducers)
{
#if PGP_BUSINESS_SECURITY
	static const OSType	kPGPkeysType = 'APPL';
	static const OSType	kPGPkeysCreator = 'pgpK';
	PGPError			result;
	PGPUInt32			lastUpdate;
	PGPUInt32			updateInterval;
	UInt32				timeNow;
	PGPPrefRef			adminPrefs;
	PGPBoolean			shouldUpdateAllKeys = false;
	PGPBoolean			shouldUpdateTrustedIntroducers = false;
	FSSpec				libraryFileSpec;
	
	try {
		PGPValidateParam(PGPMemoryMgrRefIsValid(memoryMgr));
		
		result = PGPGetClientLibFSSpec(&libraryFileSpec);
		if (IsntPGPError(result)) {
			StSaveCurResFile	savedFile(libraryFileSpec);
			
			::GetDateTime(&timeNow);
			result = PGPOpenAdminPrefs(	memoryMgr,
										&adminPrefs);
			if (IsntPGPError(result)) {
				// Check trusted introducers
				result = PGPGetPrefNumber(
							adminPrefs,
							kPGPPrefDaysUpdateTrustedIntroducers,
							&updateInterval);
				if (IsntPGPError(result) && (updateInterval > 0)) {
					result = PGPGetPrefNumber(
								adminPrefs,
								kPGPPrefLastTrustedIntroducersUpdate,
								&lastUpdate);
					if (IsntPGPError(result) &&
					(((timeNow - lastUpdate) / 86400) >= updateInterval)) {
						shouldUpdateTrustedIntroducers = true;
					}
				}
				
				// Check all keys
				if (IsntPGPError(result)) {
					result = PGPGetPrefNumber(	adminPrefs,
												kPGPPrefDaysUpdateAllKeys,
												&updateInterval);
					if (IsntPGPError(result) && (updateInterval > 0)) {
						result = PGPGetPrefNumber(	adminPrefs,
													kPGPPrefLastAllKeysUpdate,
													&lastUpdate);
						if (IsntPGPError(result) &&
						(((timeNow - lastUpdate) / 86400) >= updateInterval)) {
							shouldUpdateAllKeys = true;
						}
					}
				}
				PGPClosePrefFile(adminPrefs);
				
				// Set booleans
				if (IsntPGPError(result)) {
					if (updateAllKeys != NULL) {
						*updateAllKeys = shouldUpdateAllKeys;
					}
					if (updateTrustedIntroducers != NULL) {
						*updateTrustedIntroducers = shouldUpdateTrustedIntroducers;
					}
				}
				
				// Launch keys if necessary
				if (IsntPGPError(result) && launchKeys &&
				(shouldUpdateAllKeys || shouldUpdateTrustedIntroducers)) {
					result = BringAppToFront(	kPGPkeysType,
												kPGPkeysCreator,
												kFindAppOnAllVolumes,
												true,
												nil,
												nil);
					
					// BringAppToFront returns an OSStatus,
					// so convert to PGPError
					if (result != noErr) {
						result = MacErrorToPGPError(result);
					}
				}
			}
		}
	}
	
	catch (...) {
		result = kPGPError_UnknownError;
	}
	
	return result;
#else
	(void) memoryMgr;
	(void) launchKeys;
	(void) updateAllKeys;
	(void) updateTrustedIntroducers;

	pgpAssert("PGP_BUSINESS_SECURITY not defined");
	return kPGPError_FeatureNotAvailable;
#endif
}



	PGPError
PGPSignerKeyLookup(
	PGPContextRef	context,
	PGPEvent *		event,
	PGPKeySetRef	defaultKeySet,
	PGPBoolean *	addedKeySetToJob,
	PGPKeySetRef *	newKeySet)
{
	PGPError		result = kPGPError_NoErr;
	PGPPrefRef		prefRef = kInvalidPGPPrefRef;
	PGPKeySetRef	signerKeySet = kInvalidPGPKeySetRef;
	PGPUInt32		count;
	PGPBoolean		sync;

	// Validation
	PGPValidatePtr(event);
	PGPValidateParam(event->type == kPGPEvent_SignatureEvent);
	PGPValidatePtr(addedKeySetToJob);
	*addedKeySetToJob = false;
	if (newKeySet != NULL) {
		PGPValidatePtr(newKeySet);
		*newKeySet = kInvalidPGPKeySetRef;
	}
	
	// Lookup key if necessary
	if (! PGPKeyRefIsValid(event->data.signatureData.signingKey)) {
		result = PGPOpenClientPrefs(	PGPGetContextMemoryMgr(context),
										&prefRef);
		if (IsntPGPError(result)) {
			result = PGPGetPrefBoolean(	prefRef,
										kPGPPrefKeyServerSyncOnVerify,
										&sync);
			PGPClosePrefFile(prefRef);
		}
		
		if (IsntPGPError(result) && sync) {
			result = PGPGetKeyIDFromServer(
						context,
						kInvalidPGPtlsContextRef,
						&event->data.signatureData.signingKeyID,
						&signerKeySet);
			if (IsntPGPError(result) && PGPKeySetRefIsValid(signerKeySet)) {
				result = PGPCountKeys(signerKeySet, &count);
			}
			if (IsntPGPError(result) && (count > 0)) {
				result = PGPAddJobOptions(	event->job,
											PGPOKeySetRef(	context,
															signerKeySet),
											PGPOLastOption(context));
				if (IsntPGPError(result)) {
					*addedKeySetToJob = true;
					
					// Do a selective import
					if (PGPKeySetRefIsValid(defaultKeySet)) {
						try {
							FSSpec	libraryFileSpec;
							
							result = PGPGetClientLibFSSpec(&libraryFileSpec);
							if (IsntPGPError(result)) {
								StSaveCurResFile	savedResFile(
														libraryFileSpec);
								PGPKeySetRef		selectedKeySet;
								
								result = PGPSelectKeysDialog(
											context,
											kPGPSelectKeysImportVariation,							
											CString(
												STRx_UtilStrings,
												kSignerKeyImportStringID),							
											signerKeySet,							
											defaultKeySet,
											&selectedKeySet);
								if (IsntPGPError(result)) {
									result = PGPSharedAddKeysToDefaultKeyring(
												selectedKeySet);
									PGPFreeKeySet(selectedKeySet);
								}
							}
						}
						
						catch (...) {
							result = kPGPError_UnknownError;
						}
					}
					
					// Return the keyset
					if (IsntNull(newKeySet)) {
						*newKeySet = signerKeySet;
						signerKeySet = kInvalidPGPKeySetRef;
					}
				}
			}
			if (PGPKeySetRefIsValid(signerKeySet)) {
				PGPFreeKeySet(signerKeySet);
			}
		}
	}
	
	return result;
}
	


	PGPError
PGPGuaranteeMinimumEntropy(
	PGPContextRef		context)
{
	PGPError	err = kPGPError_NoErr;
	
	if( ! PGPGlobalRandomPoolHasMinimumEntropy() )
	{
		PGPUInt32	haveBits;
		PGPUInt32	neededBits;
		
		haveBits 	= PGPGlobalRandomPoolGetEntropy();
		neededBits 	= PGPGlobalRandomPoolGetMinimumEntropy();
		
		pgpAssert( haveBits < neededBits );
		
		if( haveBits < neededBits )
		{
			neededBits = neededBits - haveBits + 100;
		}
		else
		{
			neededBits = 300;	/* Token amount */
		}
		
		err = PGPCollectRandomDataDialog( context, neededBits,
						PGPOLastOption( context ) );
	}
	
	return( err );
}



	PGPError
PGPRSASupported(
	PGPBoolean *	hasRSA)
{
	PGPError					result = kPGPError_NoErr;
	PGPUInt32					numAlgs;
	PGPUInt32					index = 0;
	PGPPublicKeyAlgorithmInfo	info;
	
	PGPValidatePtr(hasRSA);
	*hasRSA = false;
	
	result = PGPCountPublicKeyAlgorithms(&numAlgs);
	while (IsntPGPError(result) && (index < numAlgs)) {
		result = PGPGetIndexedPublicKeyAlgorithmInfo(	index,
														&info);
		if (IsntPGPError(result)) {
			if ((info.algID == kPGPPublicKeyAlgorithm_RSA)
			|| (info.algID == kPGPPublicKeyAlgorithm_RSAEncryptOnly)
			|| (info.algID == kPGPPublicKeyAlgorithm_RSASignOnly)) {
				*hasRSA = true;
				break;
			}
		}
		index++;
	}
	
	return result;
}



	PGPError
PGPIsAdminConfigured(
	PGPBoolean *	isConfigured)
{
	PGPError	result = kPGPError_NoErr;
	FSSpec		libraryFileSpec;
	
	PGPValidatePtr(isConfigured);
	*isConfigured = false;
	
	try {
		result = PGPGetClientLibFSSpec(&libraryFileSpec);
		if (IsntPGPError(result)) {
			StSaveCurResFile	savedFile(libraryFileSpec);
	
			if (ResourceExists(kAdminConfiguredType, 
			PGPa_Configured, true)) {
				*isConfigured = true;
			}
		}
	}

	catch (...) {
		result = kPGPError_UnknownError;
	}
	
	return result;
}
