/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	

	$Id: pgpMacCustomContextAlloc.c,v 1.8.8.1 1998/11/12 03:09:56 heller Exp $
____________________________________________________________________________*/

#include "pgpMacCustomContextAlloc.h"
#include "pgpMem.h"
#include "pgpMemoryMgr.h"
#include "MacEnvirons.h"

typedef struct BlockHeader
{
	/* we should keep this struct a multiple of 16 bytes.  For now,
	we use the extra space for extra magic fields */
	#define kMagic		'MACM'
	PGPUInt32		magic;
	// if temp-mem, then this is non-null
	Handle			tempMemHandle;
	PGPSize			userSize;
	PGPUInt32		magic2;
} BlockHeader;
#define BlockHeaderToData( hdr )	((void *)( ((BlockHeader *)hdr) + 1 ))
#define DataToBlockHeader( data )	\
	((BlockHeader *)(((char *)data) - sizeof( BlockHeader )))

#define IsTempMemBlock( hdr )		( IsntNull( (hdr)->tempMemHandle ) )

	static PGPBoolean
sHeaderIsValid( BlockHeader const *hdr )
{
	if ( IsNull( hdr ) )
		return( FALSE );
	return( hdr->magic == kMagic && hdr->magic2 == kMagic );
}

#define BlockIsValid( hdr )		( IsntNull( hdr ) && sHeaderIsValid( hdr ) )

/* anything larger than this will be allocated in temp mem */
#define kTempMemCutoff		( 4UL * 1024UL )


	static BlockHeader *
sAllocateBlock(
	PGPSize		requestSize,
	PGPBoolean	useTempMem )
{
	PGPSize			totalSize	= requestSize + sizeof( BlockHeader );
	BlockHeader *	header	= NULL;
	
	/* try getting a plain Ptr first, if that's what's desired */
	if ( requestSize < kTempMemCutoff )
	{
		header	= (BlockHeader *)NewPtr( totalSize );
		if ( IsntNull( header ) )
			header->tempMemHandle	= NULL;
	}
	
	if ( IsNull( header )  && useTempMem)
	{
		Handle	tempHandle;
		OSErr	err;

		tempHandle	= TempNewHandle( totalSize, &err );
		if ( IsntNull( tempHandle ) && IsntErr( err ) )
		{
			HLock( tempHandle );
			header	= (BlockHeader *)*tempHandle;
			header->tempMemHandle	= tempHandle;
		}
		
	}
	
	if ( IsntNull( header ) )
	{
		header->magic		= kMagic;
		header->magic2		= kMagic;
		header->userSize	= requestSize;
	}
	
	return( header );
}

	static void *
sMacCustomAllocator(
	PGPMemoryMgrRef 	mgr,
	PGPUserValue		userValue,
	PGPSize 			requestSize,
	PGPMemoryMgrFlags 	flags )
{
	BlockHeader *	header	= NULL;
	void *			mem	= NULL;
	
	(void)mgr;
	(void)flags;
	(void)userValue;
	
	header	= sAllocateBlock( requestSize, TRUE );
	
	if ( IsntNull( header ) )
		mem	= BlockHeaderToData( header );
	else
		mem	= NULL;
	return( mem );
}



	static PGPError
sMacCustomDealloc(
	PGPMemoryMgrRef mgr,
	PGPUserValue	userValue,
	void *			allocation,
	PGPSize 		allocationSize )
{
	PGPError		err	= kPGPError_NoErr;
	BlockHeader *	header	= NULL;
	
	(void) mgr;
	(void) userValue;
	(void) allocationSize;
	
	if ( IsNull( allocation ) )
	{
		pgpDebugMsg( "pgpMacCustomDealloc: NULL block" );
		return( kPGPError_BadParams );
	}
	
	header	= DataToBlockHeader( allocation );
	if ( ! BlockIsValid( header ) )
	{
		pgpDebugMsg( "pgpMacCustomDealloc: bad block" );
		return( kPGPError_BadParams );
	}
	
	if ( IsTempMemBlock( header ) )
	{
		Handle	theHandle;
		
		theHandle	= header->tempMemHandle;
		pgpAssert( (Ptr)header == (Ptr)*theHandle );
		
		MacDebug_FillWithGarbage( *theHandle ,
			GetHandleSize( theHandle ) );
			
		DisposeHandle( theHandle );
	}
	else
	{
		MacDebug_FillWithGarbage( header,
			GetPtrSize( (Ptr)header ) );
		
		DisposePtr( (Ptr)header );
	}

	
	return( err );
}



	static PGPError
sMacCustomRealloc( 
	PGPMemoryMgrRef 	mgr,
	PGPUserValue		userValue,
	void **				allocationPtr,
	PGPSize				newAllocationSize,
	PGPMemoryMgrFlags	flags,
	PGPSize 			existingSize)
{
	PGPError		err	= kPGPError_OutOfMemory;
	BlockHeader *	header	= NULL;
	PGPSize			totalSize;
	
	PGPValidatePtr( allocationPtr );
	header		= DataToBlockHeader( *allocationPtr );
	PGPValidateParam( BlockIsValid( header ) );
	if ( newAllocationSize == header->userSize )
		return( kPGPError_NoErr );
	
	totalSize	= newAllocationSize + sizeof(BlockHeader);
	
	/* CAUTION: if we fail, original must *not* move original */
	
	/* first, try resizing existing block in place */
	/* for temp mem, don't attempt to grow a locked block;
	  it may or may not be legal */
	err	= kPGPError_OutOfMemory;
	if ( IsTempMemBlock( header ))
	{
		if ( newAllocationSize <= header->userSize )
		{
			/* it's locked, but resizing may succeed */
			SetHandleSize( header->tempMemHandle, totalSize );
			if ( GetHandleSize( header->tempMemHandle ) == totalSize )
			{
				err	= kPGPError_NoErr;
			}
		}
	}
	else
	{
		SetPtrSize( (Ptr)header, totalSize );
		if ( GetPtrSize( (Ptr)header ) == totalSize )
		{
			err	= kPGPError_NoErr;
		}
	}
	
	if ( err == kPGPError_OutOfMemory )
	{
		/* resizing existing block in place failed */
		/* try allocating an all-new block */
		PGPSize		oldSize		= header->userSize;
		void *		newMem;
		
		newMem	= sMacCustomAllocator( mgr, userValue,
					newAllocationSize, flags  );
		if ( IsntNull( newMem ) )
		{
			err	= kPGPError_NoErr;
			
			pgpCopyMemory( *allocationPtr, newMem,
				pgpMin( newAllocationSize, oldSize ) );
				
			sMacCustomDealloc( mgr, userValue,
					*allocationPtr, existingSize );
			*allocationPtr	= newMem;
		}
	}
	
	if ( IsntPGPError( err ) )
	{
		header	= DataToBlockHeader( *allocationPtr );
		header->userSize	= newAllocationSize;
	}
	
	return( err );
}



	static void
sLockMemory(
	void *			mem,
	PGPSize			numBytes )
{
	if ( VirtualMemoryIsOn()  )
	{
		HoldMemory( mem, numBytes );
	}
}

	static void
sUnlockMemory(
	void *	mem,
	PGPSize	numBytes )
{
	if ( VirtualMemoryIsOn()  )
	{
		UnholdMemory( mem, numBytes );
	}
}


	static void *
sMacSecureMemoryAllocationProc(
	PGPMemoryMgrRef		mgr,
	PGPUserValue		userValue,
	PGPSize 			allocationSize,
	PGPMemoryMgrFlags	flags,
	PGPBoolean *		isNonPageable )
{
	void *			ptr	= NULL;
	BlockHeader *	header	= NULL;
	THz				saveZone	= GetZone();
	
	pgpAssert( IsntNull( isNonPageable ) );
	(void) mgr;
	(void) flags;
	(void) userValue;

	*isNonPageable	= TRUE;
	
	/* try system zone first */
	SetZone( SystemZone() );
	header	= (BlockHeader *)sAllocateBlock( allocationSize, FALSE );
	SetZone( saveZone );
	
	if ( IsNull( header ) )
	{
		/* system zone failed; try current zone and temp mem */
		header	= (BlockHeader *)sAllocateBlock( allocationSize, TRUE );
	}
		
	if ( IsntNull( header ) )
	{
		ptr	= BlockHeaderToData( header );
		/* lock user data; no need to lock header portion */
		sLockMemory( ptr, allocationSize );
	}
	
	return( ptr );
}




/*____________________________________________________________________________
	The default secure memory deallocator.
____________________________________________________________________________*/
	static PGPError
sMacSecureMemoryDeallocationProc(
	PGPMemoryMgrRef	mgr,
	PGPUserValue	userValue,
	void *			allocation,
	PGPSize			allocationSize,
	PGPBoolean		nonPageable )
{
	PGPError	err	= kPGPError_NoErr;
	
	(void) mgr;
	(void) allocationSize;
	(void) userValue;

	pgpAssert( nonPageable );
	if ( nonPageable )
	{
		/* unlock user data; no need to unlock header portion */
		sUnlockMemory( allocation, allocationSize );
	}
	
	err	= sMacCustomDealloc( mgr, userValue, (Ptr)allocation, allocationSize );
	
	return( kPGPError_NoErr );
}


	PGPError
pgpNewContextCustomMacAllocators(
	PGPContextRef *	context )
{
	PGPError				err	= kPGPError_NoErr;
	PGPMemoryMgrRef			memMgrRef;
	PGPNewMemoryMgrStruct	customMemMgr;
	
	pgpClearMemory( &customMemMgr, sizeof( customMemMgr ) );
	
	customMemMgr.sizeofStruct	= sizeof( customMemMgr );
	customMemMgr.allocProc		= sMacCustomAllocator;
	customMemMgr.reallocProc	= sMacCustomRealloc;
	customMemMgr.deallocProc	= sMacCustomDealloc;
	
	customMemMgr.secureAllocProc	= sMacSecureMemoryAllocationProc;
	customMemMgr.secureDeallocProc	= sMacSecureMemoryDeallocationProc;
	
	err = PGPNewMemoryMgrCustom( &customMemMgr, &memMgrRef );
	if( IsntPGPError( err ) )
	{
		PGPNewContextStruct		custom;
		
		pgpClearMemory( &custom, sizeof( custom ) );
		
		custom.sizeofStruct	= sizeof( custom );
		custom.memoryMgr	= memMgrRef;
	
		err	= PGPNewContextCustom( kPGPsdkAPIVersion, &custom,
					context );
	}			
	
	return( err );
}

				




























/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
