/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: CPrivateKeysPopup.cp,v 1.16.12.1 1998/11/12 03:20:53 heller Exp $
____________________________________________________________________________*/

#include <LArrayIterator.h>
#include <LComparator.h>

#include "MacStrings.h"
#include "pgpErrors.h"
#include "pgpKeys.h"
#include "pgpMem.h"
#include "pgpUserInterface.h"

#include "CPrivateKeysPopup.h"
#include "PGPsdkUILibDialogs.h"

const ResIDT	kSigningPopupMenuResID	= 4747;

typedef struct PopupKeyItem
{
	PGPKeyRef	keyRef;
	Str255		userID;		// Pascal string
	UInt16		algorithm;
	UInt16		keyBits;
	Boolean		isDefaultPrivateKey;

} PopupKeyItem; 

class CComparator : public LComparator
{
	virtual			~CComparator(void);
	virtual Int32	Compare(const void *inItemOne, const void *inItemTwo,
								Uint32 inSizeOne, Uint32 inSizeTwo) const;
};

CComparator::~CComparator(void)
{
}

	Int32
CComparator::Compare(
	const void*		inItemOne,
	const void*		inItemTwo,
	Uint32			inSizeOne,
	Uint32			inSizeTwo) const
{
	Int32				result;
	const PopupKeyItem	*keyItemOne = (PopupKeyItem *) inItemOne;
	const PopupKeyItem	*keyItemTwo = (PopupKeyItem *) inItemTwo;
	
	pgpAssertAddrValid( keyItemOne, PopupKeyItem );
	pgpAssertAddrValid( keyItemTwo, PopupKeyItem );
	
#if PGP_DEBUG
	pgpAssert( inSizeOne == sizeof( PopupKeyItem ) );
	pgpAssert( inSizeTwo == sizeof( PopupKeyItem ) );
#else
	#pragma unused( inSizeOne, inSizeTwo )
#endif
	
	result = CompareText( &keyItemOne->userID[1], &keyItemTwo->userID[1],
					keyItemOne->userID[0], keyItemTwo->userID[0], nil );

	return( result );
}

CPrivateKeysPopup::CPrivateKeysPopup(LStream *inStream)
	: LPopupButton( inStream )
{
	mKeyItemList = new( LArray )( sizeof(PopupKeyItem),
						new( CComparator ), TRUE );
}

CPrivateKeysPopup::~CPrivateKeysPopup()
{
	delete( mKeyItemList );
}

	Boolean
CPrivateKeysPopup::BuildKeyList(
	PGPKeySetRef 	allKeys,
	PGPKeyRef		defaultKey)
{
	PGPError		err;
	MenuHandle		menuH;
	PGPKeyListRef	keyList;
	Boolean			haveSecretKeys;
	
	pgpAssert( PGPKeyRefIsValid( defaultKey ) );
	
	haveSecretKeys = FALSE;
	
	// Find and add the private keys.
	
	err = PGPOrderKeySet( allKeys, kPGPAnyOrdering, &keyList );
	if( IsntPGPError( err ) )
	{
		PGPKeyIterRef	keyIterator;
		
		err = PGPNewKeyIter( keyList, &keyIterator );
		if( IsntPGPError( err ) )
		{
			PGPKeyRef	theKey;
			PGPKeyRef	defaultPrivateKey;
			
			if( IsPGPError( PGPGetDefaultPrivateKey( allKeys,
						&defaultPrivateKey ) ) )
			{
				defaultPrivateKey = (PGPKeyRef) kInvalidPGPKeyRef;
			}
			
			err = PGPKeyIterNext( keyIterator, &theKey );
			while( IsntPGPError( err ) )
			{
				PGPBoolean	canSign;
				
				err = PGPGetKeyBoolean( theKey, kPGPKeyPropCanSign, &canSign );
				if( IsntPGPError( err ) )
				{
					if( canSign )
					{
						PopupKeyItem	keyItem;
						PGPInt32		tempInt32;
						
						pgpClearMemory( &keyItem, sizeof( keyItem ) );
						
						keyItem.keyRef = theKey;
						
						if( theKey == defaultPrivateKey )
							keyItem.isDefaultPrivateKey = TRUE;
						
						err = PGPGetKeyNumber( theKey,
									kPGPKeyPropAlgID, &tempInt32 );
						if( IsntPGPError( err ) )
						{
							keyItem.algorithm = tempInt32;

							err = PGPGetKeyNumber( theKey,
										kPGPKeyPropBits, &tempInt32 );
							if( IsntPGPError( err ) )
							{
								PGPSize	nameSize;
								
								keyItem.keyBits = tempInt32;
								
								err = PGPGetPrimaryUserIDNameBuffer(
											theKey,
											sizeof( keyItem.userID ) - 1,
											(char *)&keyItem.userID[1],
											&nameSize );
								if( IsntPGPError( err ) )
								{
									/* nameSize includes the 0 terminator */
									keyItem.userID[0] = nameSize - 1;
								}
							}
						}
						
						// Add the key info to our array
						if( IsntPGPError( err ) )
						{
							mKeyItemList->InsertItemsAt( 1, 10000, &keyItem );
						}
					}
					
					err = PGPKeyIterNext( keyIterator, &theKey );
				}
			}
			
			if( err == kPGPError_EndOfIteration )
				err = kPGPError_NoErr;
				
			PGPFreeKeyIter( keyIterator );
		}
		
		PGPFreeKeyList( keyList );
	}

	// Walk our sorted array and setup the popup menu handle

	menuH = GetMenu( kSigningPopupMenuResID );
	AssertHandleIsValid( menuH, "" );
	if( IsntNull( menuH ) )
	{
		Str255	menuItemStr;
		UInt32	value;
		UInt32	maxValue;
		
		DetachResource( (Handle) menuH );
		
		value 		= 1;
		maxValue	= 0;
		
		haveSecretKeys = ( mKeyItemList->GetCount() > 0 );
		if( haveSecretKeys )
		{
			LArrayIterator	iterator( *mKeyItemList );
			PopupKeyItem	keyItem;
			
			FocusDraw();
			
			while( iterator.Next( &keyItem ) )
			{
				Str255	tempStr;
				short	strIndex;
				
				GetIndString( menuItemStr, kPGPLibDialogsStringListResID,
							kPrivateKeyPopupFormatStrIndex );
				CopyPString( keyItem.userID, tempStr );
				
				/* Truncate the string so we can add our key bits info */
				if( tempStr[0] > 225 )
					tempStr[0] = 225;
					
				PrintPString( menuItemStr, menuItemStr, tempStr );
				
				switch( keyItem.algorithm )
				{
					case kPGPPublicKeyAlgorithm_RSA:
						strIndex = kRSAAlgorithmStringIndex;
						break;
						
					case kPGPPublicKeyAlgorithm_ElGamal:
					case kPGPPublicKeyAlgorithm_DSA:
						strIndex = kDSAAlgorithmStringIndex;
						break;
						
					default:
						strIndex = kUnknownAlgorithmStringIndex;
						break;
				}
				
				GetIndString( tempStr, kPGPLibDialogsStringListResID,
							strIndex );
				PrintPString( menuItemStr, menuItemStr, tempStr );
				
				NumToString( keyItem.keyBits, tempStr );
				PrintPString( menuItemStr, menuItemStr, tempStr );
				
				#define kMaxPopupTextWidth	400
				
				if( StringWidth( menuItemStr ) > kMaxPopupTextWidth )
				{
					TruncString( kMaxPopupTextWidth, menuItemStr,
								smTruncMiddle );
				}
				
				++maxValue;
				
				::AppendMenu( menuH, "\pxx" );
				::SetMenuItemText( menuH, maxValue, menuItemStr );
				
				if( keyItem.isDefaultPrivateKey )
				{
					value = maxValue;
					SetItemStyle( menuH, maxValue, bold );
				}
			}
		}
		else
		{
			// Set the popup to the "no signing keys" found and disable
			// ourselves
			GetIndString( menuItemStr, kPGPLibDialogsStringListResID,
						kNoSigningKeysFoundStrIndex );
			::AppendMenu( menuH, menuItemStr );
			
			maxValue = 1;
		}
		
		// Recalculate the popup dimensions
		SetMacMenuH( menuH );

		SetMinValue( 1 );
		SetMaxValue( maxValue );
		SetValue( value );
		
		if( haveSecretKeys )
		{
			if( PGPKeyRefIsValid( defaultKey ) )
			{
				SetSigningKey( defaultKey );
			}
		}
		else
		{
			Disable();
		}
	}
	
	return( haveSecretKeys );
}

	PGPKeyRef
CPrivateKeysPopup::GetSigningKey(void)
{
	PGPKeyRef	signingKey;
	
	if( mKeyItemList->GetCount() > 0 )
	{
		PopupKeyItem	keyItem;
		
		mKeyItemList->FetchItemAt( GetValue(), &keyItem );

		signingKey = keyItem.keyRef;
	}
	else
	{
		pgpDebugMsg( "GetSigningKey(): No signing keys!" );
		
		signingKey = kInvalidPGPKeyRef;
	}
	
	return( signingKey );
}

	void
CPrivateKeysPopup::SetSigningKey(PGPKeyRef signingKey)
{
	if( mKeyItemList->GetCount() > 0 )
	{	
		LArrayIterator	iterator( *mKeyItemList );
		PopupKeyItem	keyItem;
		Int32			newValue;
		Int32			itemIndex;
		
		newValue 	= 0;
		itemIndex	= 1;

		// Walk the items and try to match the deault key spec. Note
		// that the user ID must be a fully expanded user ID.
		
		while( iterator.Next( &keyItem ) )
		{
			if( keyItem.keyRef == signingKey )
			{
				newValue = itemIndex;
				break;
			}
			
			++itemIndex;
		}

		if( newValue != GetValue() )
		{
			SetValue( newValue );
		}
	}
}



